/*! *********************************************************************************
 * \addtogroup Buzzer Control Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 * \file buzzer_control_interface.h
 * This file is the interface file for the Buzzer Control Service
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _BUZZER_CONTROL_INTERFACE_H_
#define _BUZZER_CONTROL_INTERFACE_H_

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
/*! Buzzer possible states */
typedef enum bcsBuzzerStatus{
  kBuzzerOff,                           /*!< Buzzer is off */
  kBuzzerOn                             /*!< Buzzer is on */
}bcs_buzzer_status_t;

/*! Input Report Service - Configuration */
typedef struct bcsConfig_tag
{
    uint16_t    serviceHandle;          /*!< Service handle for input report service */
    bool_t      buzzerStatus;           /*!< Current buzzer status */
} bcsConfig_t;

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/*!**********************************************************************************
* \brief        Starts Buzzer Control Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Bcs_Start(bcsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Stops Buzzer Control Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Bcs_Stop(bcsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Subscribes a GATT client to the Buzzer Control service
*
* \param[in]    pClient  Client Id in Device DB.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Bcs_Subscribe(deviceId_t clientDeviceId);

/*!**********************************************************************************
* \brief        Unsubscribes a GATT client from the Buzzer Control service
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Bcs_Unsubscribe();

/*!**********************************************************************************
* \brief        Records current Buzzer control on a specified service handle.
*
* \param[in]    serviceHandle          Service handle.
* \param[in]    newBuzzerControl       New buzzer state.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Bcs_RecordNewBuzzerControl (uint16_t serviceHandle, bool_t newBuzzerControl);

/*!**********************************************************************************
* \brief        Takes the new Buzzer value requested by the client and turns on/off
*               the buzzer accordingly.
*
* \param[in]    pBuzzerConfiguration   Buzzer configuration service handle
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Bcs_SetNewBuzzerValue (bcsConfig_t* pBuzzerConfiguration);

#ifdef __cplusplus
}
#endif

#endif /* _BUZZER_CONTROL_INTERFACE_H_ */

/*! **********************************************************************************
 * @}
 ************************************************************************************/
