/*! *********************************************************************************
 * \addtogroup Buzzer Control Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * \file buzzer_control_service.c
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
#include "FunctionLib.h"
#include "ble_general.h"
#include "gatt_db_app_interface.h"
#include "gatt_server_interface.h"
#include "gap_interface.h"
#include "buzzer_control_interface.h"
#include "buzzer_driver.h"
/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/
/*! Buzzer Control Service - Subscribed Client*/
static deviceId_t mBcs_SubscribedClientId;

//Characteristic declarations
const bleUuid_t buzzerCharacteristicUuid128 = {
  .uuid128 = {0xE0, 0x1C, 0x4B, 0x5E, 0x1E, 0xEB, 0xA1, 0x5C, 0xEE, 0xF4, 0x5E, 0xBA, 0x03, 0x57, 0xFF, 0x02}
};

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/
bleResult_t Bcs_Start (bcsConfig_t *pServiceConfig)
{    
    mBcs_SubscribedClientId = gInvalidDeviceId_c;
    
    return Bcs_SetNewBuzzerValue (pServiceConfig);
}

bleResult_t Bcs_Stop (bcsConfig_t *pServiceConfig)
{
    return Bcs_Unsubscribe();
}

bleResult_t Bcs_Subscribe(deviceId_t deviceId)
{
    mBcs_SubscribedClientId = deviceId;

    return gBleSuccess_c;
}

bleResult_t Bcs_Unsubscribe()
{
    mBcs_SubscribedClientId = gInvalidDeviceId_c;
    return gBleSuccess_c;
}

bleResult_t Bcs_RecordNewBuzzerControl (uint16_t serviceHandle, bool_t newBuzzerControl)
{
    uint16_t  handle;
    bleResult_t result;

    /* Get handle of buzzer characteristic */
    result = GattDb_FindCharValueHandleInService(serviceHandle,
        gBleUuidType128_c, (bleUuid_t*)&buzzerCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;

    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (bool_t*)&newBuzzerControl);

    if (result != gBleSuccess_c)
        return result;
    
    return gBleSuccess_c;
}

bleResult_t Bcs_SetNewBuzzerValue (bcsConfig_t* pBuzzerConfiguration){
    uint16_t  handle = pBuzzerConfiguration->serviceHandle;
    bleResult_t result;
    uint16_t byteCount;
    buzzer_driver_status_t buzzerStatus;
    
    //Update Buzzer
    buzzerStatus = buzzer_driver_change_buzzer_status (pBuzzerConfiguration->buzzerStatus);
    
    if(buzzerStatus != kBuzzerDriverOk)
      return gBleUnexpectedError_c;
    
    //Update Buzzer status in GATT database
    /* Get handle of Buzzer characteristic */
    result = GattDb_FindCharValueHandleInService(handle,
        gBleUuidType128_c, (bleUuid_t*)&buzzerCharacteristicUuid128, &handle);

    if (result != gBleSuccess_c)
        return result;
    
    /* Update characteristic value */
    result = GattDb_WriteAttribute(handle, sizeof(uint8_t), (uint8_t*)&pBuzzerConfiguration->buzzerStatus);

    if (result != gBleSuccess_c)
        return result;
    
    return gBleSuccess_c;
}


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/*! *********************************************************************************
 * @}
 ********************************************************************************** */
