/*! *********************************************************************************
 * \addtogroup LED Control Service
 * @{
 ********************************************************************************** */
/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 * \file led_control_interface.h
 * This file is the interface file for the LED Control Service
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _LED_CONTROL_INTERFACE_H_
#define _LED_CONTROL_INTERFACE_H_

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
/*! Available LED Commands */
typedef enum lcsLedCommand{
  kLedOff,                              /*!<Turn off the indicated LED */
  kLedOn,                               /*!<Turn on the indicated LED */
  kLedToggle,                           /*!<Toggle the indicated LED */
  kLedInvalid                           /*!<Invalid command. Calling this results in an error response from the server */
}lcs_led_command_t;     
  
/*! LED Control structure */    
typedef struct lcsControlStruct{        
  uint8_t ledNumber;                    /*!<LED number to change */
  uint8_t ledCommand;                   /*!<Command to execute as specified in @ref lcs_led_command_t */
}lcs_control_struct_t;

/*! LED Control Service - Configuration */
typedef struct lcsConfig_tag
{
    uint16_t    serviceHandle;          /*!<Service handle for LED control service */
    uint8_t     ledStatus;              /*!<LED status checkpoint */
    lcs_control_struct_t ledControl;    /*!<LED control command point */
} lcsConfig_t;

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/*!**********************************************************************************
* \brief        Starts LED Control Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lcs_Start(lcsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Stops LED Control Service functionality
*
* \param[in]    pServiceConfig  Pointer to structure that contains server 
*                               configuration information.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lcs_Stop(lcsConfig_t *pServiceConfig);

/*!**********************************************************************************
* \brief        Subscribes a GATT client to the LED Control service
*
* \param[in]    pClient  Client Id in Device DB.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lcs_Subscribe(deviceId_t clientDeviceId);

/*!**********************************************************************************
* \brief        Unsubscribes a GATT client from the LED Control service
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lcs_Unsubscribe();

/*!**********************************************************************************
* \brief        Records current LED Status on a specified service handle.
*
* \param[in]    serviceHandle   Service handle.
* \param[in]    ledStatus       Current LEDs status.
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lcs_RecordNewLedStatus (uint16_t serviceHandle, uint8_t ledStatus);

/*!**********************************************************************************
* \brief        Takes the new LED value requested by the client and sets the proper LEDs
* 
*               This function updates the current LED status in GATT database and
*               turns on/off the proper LEDs accordingly.
*
* \param[in]    pLedConfiguration   LED configuration service handle
*
* \return       gBleSuccess_c or error.
************************************************************************************/
bleResult_t Lcs_SetNewLedValue (lcsConfig_t* pLedConfiguration);

#ifdef __cplusplus
}
#endif

#endif /* _LED_CONTROL_INTERFACE_H_ */

/*! **********************************************************************************
 * @}
 ************************************************************************************/
