/*!
* @file FXOS8700CQ.c
*
* @author  
*
* @version 1.0
*
* @date Mar-14-2016
*
* @brief 
*
********************************************************************************
*
* Copyright (c) 2016, Freescale Semiconductor.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/************************************************************************************
* Headers
************************************************************************************/
#include "FXOS8700CQ.h"

/* KSDK */
#include "board.h"
#include "fsl_device_registers.h"
#include "fsl_i2c_custom_master_driver.h"

//Connectivity software includes (delete if connectivity software is not in use)
#include "MemManager.h"

/************************************************************************************
* Macros Resolution
************************************************************************************/
#if (FXOS8700CQ_TRANSPORT_I2C == 1)
  #define FXOS8700CQInitializeTransport                                                 FXOS8700CQInitializeTransportI2C
  #define FXOS8700CQReadRegisters(address,byteCount)                                    FXOS8700CQReadRegistersI2C(address,byteCount)
  #define FXOS8700CQWriteRegister(address,data)                                         FXOS8700CQWriteRegisterI2C(address,data)
  #define FXOS8700CQReadRegistersBlocking(startRegisterAddress, byteCount, pOutBuffer)  FXOS8700CQReadRegistersBlockingI2C(startRegisterAddress, byteCount, pOutBuffer)
  #define FXOS8700CQWriteRegisterBlocking(registerAddress, registerDataPtr)             FXOS8700CQWriteRegisterBlockingI2C(registerAddress, registerDataPtr)
#elif (FXOS8700CQ_TRANSPORT_SPI == 1)
  #define FXOS8700CQInitializeTransport                                                 FXOS8700CQInitializeTransportSPI
  #define FXOS8700CQReadRegisters(address,byteCount)                                    FXOS8700CQReadRegistersSPI(address,byteCount)
  #define FXOS8700CQWriteRegister(address,data)                                         FXOS8700CQWriteRegisterSPI(address,data)
#else
  #error "FXOS8700CQ.h no valid transport selected (I2C or SPI)"
#endif


/************************************************************************************
* Private Function Prototypes
************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQInitializeTransportI2C (void);
FXOS8700CQ_status_t FXOS8700CQReadRegistersI2C (uint8_t registerAddress, uint8_t byteCount);
FXOS8700CQ_status_t FXOS8700CQWriteRegisterI2C (uint8_t registerAddress, uint8_t* registerDataPtr);
FXOS8700CQ_status_t FXOS8700CQI2cErrorhandler (i2c_status_t i2cStatus);
FXOS8700CQ_status_t FXOS8700CQReadRegistersBlockingI2C(uint8_t startRegisterAddress, uint8_t byteCount, uint8_t* pOutBuffer);
FXOS8700CQ_status_t FXOS8700CQWriteRegisterBlockingI2C(uint8_t registerAddress, uint8_t* registerDataPtr);
extern void FXOS8700CQI2cCallbackHandler (void);        //Register this function in the proper IIC IRQ handler
void FXOS8700CQSoftReset(void);

/************************************************************************************
* Local Variables
************************************************************************************/
i2c_device_t FXOS8700CQMasterConfiguration = {
  .address = FXOS8700CQ_I2C_ADDRESS,
  .baudRate_kbps = FXOS8700CQ_I2C_BAUDRATE_KBPS
};

i2c_master_state_t FXOS8700CQMasterState;
uint8_t txBuffer[2];

FXOS8700CQ_callback_function_t userDefinedCallbackFunc;
uint8_t* memoryAllocationPointer;
uint8_t expectedDataSize;

/************************************************************************************
* Public Functions
************************************************************************************/
FXOS8700CQ_status_t FXOS8700CQ_init (FXOS8700CQ_config_t* pConfigStruct){
  FXOS8700CQ_status_t initializationError = kStatusSuccess;
  FXOS8700CQ_CTRL_REG1_map_t ctrlReg1 = {.ctrl_reg1 = 0x00u};
  FXOS8700CQ_M_CTRL_REG1_map_t mCtrlReg1 = {.m_ctrl_reg1 = 0x00u};
  FXOS8700CQ_M_CTRL_REG2_map_t mCtrlReg2 = {.m_ctrl_reg2 = 0x00u};
  
  //Initialize FXOS8700CQ transport
  initializationError = FXOS8700CQInitializeTransport();
  
  if(initializationError != kStatusSuccess)
    return initializationError;
  
  //Execute communications test
  initializationError = FXOS8700CQ_communication_test();
  
  if(initializationError != kStatusSuccess)
    return initializationError;
  
  //Reset FXOS8700CQ
  FXOS8700CQSoftReset();

  /* Initialize FXOS8700CQ */
  
  //Enable Sensors */
  mCtrlReg1.m_ctrl_reg1_map.m_hms = pConfigStruct->enabledSensors;
  
  if(pConfigStruct->enabledSensors == kHybridMode)
    mCtrlReg2.m_ctrl_reg2_map.hyb_autoinc_mode = 1;
  
  initializationError = FXOS8700CQ_set_register_blocking(FXOS8700CQ_M_CTRL_REG1, &mCtrlReg1.m_ctrl_reg1);
  if(initializationError != kStatusSuccess)
    return initializationError;
  
  initializationError = FXOS8700CQ_set_register_blocking(FXOS8700CQ_M_CTRL_REG2, &mCtrlReg2.m_ctrl_reg2);
  if(initializationError != kStatusSuccess)
    return initializationError;
  
  ctrlReg1.ctrl_reg1_map.dr = pConfigStruct->outputDataRate;
  ctrlReg1.ctrl_reg1_map.active = TRUE;
  
  initializationError = FXOS8700CQ_set_register_blocking(FXOS8700CQ_CTRL_REG1, &ctrlReg1.ctrl_reg1);
  
  return initializationError;
}

FXOS8700CQ_status_t FXOS8700CQ_start (void){
  FXOS8700CQ_CTRL_REG1_map_t ctrlReg1;
  FXOS8700CQ_status_t status;
  
  //Get CTRL_REG1 and store
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_CTRL_REG1, 0x01u, (uint8_t*)&ctrlReg1);
  if(status != kStatusSuccess)
    return status;
  
  //Start FXOS8700CQ (if not started)
  if(ctrlReg1.ctrl_reg1_map.active == 0){
    ctrlReg1.ctrl_reg1_map.active = 1;
    status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_CTRL_REG1, (uint8_t*)&ctrlReg1);
    if(status != kStatusSuccess)
      return status;  
  }
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQ_stop (void){
  FXOS8700CQ_CTRL_REG1_map_t ctrlReg1;
  FXOS8700CQ_status_t status;
  
  //Get CTRL_REG1 and store
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_CTRL_REG1, 0x01u, (uint8_t*)&ctrlReg1);
  if(status != kStatusSuccess)
    return status;
  
  //Stop FXOS8700CQ (if not stopped)
  if(ctrlReg1.ctrl_reg1_map.active){
    ctrlReg1.ctrl_reg1_map.active = 0;
    status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_CTRL_REG1, (uint8_t*)&ctrlReg1);
    if(status != kStatusSuccess)
      return status;  
  }
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQ_communication_test (void){
  uint8_t sensorTestRegisterResponse;
  FXOS8700CQ_status_t sensorStatus;
  
  sensorStatus = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_WHO_AM_I, 0x01u, &sensorTestRegisterResponse);
  if(sensorStatus != kStatusSuccess)
    return sensorStatus;
  
  if(sensorTestRegisterResponse != 0xC7)
    return kStatusCommunicationsError;
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQ_interrupt_configuration (FXOS8700CQ_interrupt_config_t* pConfigurationParameters){
  FXOS8700CQ_status_t status;
 
  //Stop FXOS8700CQ for configuration
  status = FXOS8700CQ_stop();
  if(status != kStatusSuccess)
    return status;
  
  //Configure CTRL_REG5 (Pin mapping)
  status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_CTRL_REG5, (uint8_t*)&pConfigurationParameters->interruptPinMap);
  if(status != kStatusSuccess)
    return status;
  
  //Configure CTRL_REG4 (Interrupt sources)
  status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_CTRL_REG4, (uint8_t*)&pConfigurationParameters->interruptSources);
  if(status != kStatusSuccess)
    return status;
  
  //Enable FXOS8700CQ again
  status = FXOS8700CQ_start();
  
  return status;
}

FXOS8700CQ_status_t FXOS8700CQ_get_interrupt_status (uint8_t* interruptStatus){
  FXOS8700CQ_status_t status;
  
  //Get INT_SOURCE register and store
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_INT_SOURCE, 0x01u, interruptStatus);
  
  return status;
}

FXOS8700CQ_status_t FXOS8700CQ_accelerometer_configuration (FXOS8700CQ_accelerometer_config_t* pConfigurationParameters){
  FXOS8700CQ_CTRL_REG1_map_t ctrlReg1;
  FXOS8700CQ_status_t status;
  
  //Stop FXOS8700CQ for configuration
  status = FXOS8700CQ_stop();
  if(status != kStatusSuccess)
    return status;
  
  /* Set sensitivity */
  //Get previous XYZ_DATA_CFG register configuration
  FXOS8700CQ_XYZ_DATA_CFG_map_t xyzDataCfg;
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_XYZ_DATA_CFG, 0x01u, (uint8_t*)&xyzDataCfg);
  if(status != kStatusSuccess)
    return status;
  //Set new sensitivity configuration
  xyzDataCfg.xyz_data_cfg_map.fs = pConfigurationParameters->sensitivity;
  //Write new sensitivity configuration 
  status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_XYZ_DATA_CFG, (uint8_t*)&xyzDataCfg);
  if(status != kStatusSuccess)
    return status;
  
  /* Set oversampling ratio */
  //Get previous CTRL_REG2 register configuration
  FXOS8700CQ_CTRL_REG2_map_t ctrlReg2;
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_CTRL_REG2, 0x01u, (uint8_t*)&ctrlReg2);
  if(status != kStatusSuccess)
    return status;
  ctrlReg2.ctrl_reg2_map.mods = pConfigurationParameters->oversamplingMod;
  //Write new oversampling configuration k
  status = FXOS8700CQ_set_register_blocking(FXOS8700CQ_CTRL_REG2, (uint8_t*)&ctrlReg2);
  if(status != kStatusSuccess)
    return status;
  
  //Get CTRL_REG1 and store
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_CTRL_REG1, 0x01u, (uint8_t*)&ctrlReg1);
  if(status != kStatusSuccess)
    return status;
  
  /* Update Fast mode and Low Noise configurations and restart the FXOS8700CQ */
  ctrlReg1.ctrl_reg1_map.f_read = pConfigurationParameters->fastModeEnabled;
  ctrlReg1.ctrl_reg1_map.lnoise = pConfigurationParameters->lowNoiseEnabled;
  ctrlReg1.ctrl_reg1_map.active = 0x01u;
  status = FXOS8700CQ_set_register_blocking(FXOS8700CQ_CTRL_REG1, (uint8_t*)&ctrlReg1);
  
  return status;
}

FXOS8700CQ_status_t FXOS8700CQ_get_accelerometer_readings (FXOS8700CQ_output_data_t* pAccelerometerData){
  FXOS8700CQ_status_t status;
  FXOS8700CQ_CTRL_REG1_map_t ctrlReg1;
  uint8_t accelerometerData[7];
  
  //Read CTRL_REG1
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_CTRL_REG1, 0x01u, &ctrlReg1.ctrl_reg1);
  if(status != kStatusSuccess)
    return status;
  
  //Read and store accelerometer data
  if(ctrlReg1.ctrl_reg1_map.f_read){
    //Fast read mode enabled (8-bits resolution)
    status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_STATUS, 0x04u, &accelerometerData[0]);
    if(status != kStatusSuccess)
      return status;
    
    pAccelerometerData->status = accelerometerData[0];
    pAccelerometerData->xAxisData = (int16_t)(accelerometerData[1]);
    pAccelerometerData->yAxisData = (int16_t)(accelerometerData[2]);
    pAccelerometerData->zAxisData = (int16_t)(accelerometerData[3]);
  }
  else{
    //Fast read mode disabled (14-bits resolution)
    status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_STATUS, 0x07u, &accelerometerData[0]);
    if(status != kStatusSuccess)
      return status;
    
    pAccelerometerData->status = accelerometerData[0];
    pAccelerometerData->xAxisData = (int16_t)((accelerometerData[1]<<8)|(accelerometerData[2]))/4;
    pAccelerometerData->yAxisData = (int16_t)((accelerometerData[3]<<8)|(accelerometerData[4]))/4;
    pAccelerometerData->zAxisData = (int16_t)((accelerometerData[5]<<8)|(accelerometerData[6]))/4;
  }
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQ_magnetometer_configuration (FXOS8700CQ_magnetometer_config_t* pConfigurationParameters){
  FXOS8700CQ_status_t status;
  uint8_t magnetometerControlRegisters[3];
  
  //Stop FXOS8700CQ
  status = FXOS8700CQ_stop();
  
  if(status != kStatusSuccess)
    return status;
  
  //Read current magnetometer control registers
  status = FXOS8700CQ_get_registers_blocking (FXOS8700CQ_M_CTRL_REG1, 0x03u, magnetometerControlRegisters);
  
  if(status != kStatusSuccess)
    return status;
  
  //Configure Oversampling Ratio and autocalibration
  FXOS8700CQ_M_CTRL_REG1_map_t* mCtrlReg1 = (FXOS8700CQ_M_CTRL_REG1_map_t*)&magnetometerControlRegisters[0];
  
  mCtrlReg1->m_ctrl_reg1_map.m_os = pConfigurationParameters->oversamplingRatio;
  mCtrlReg1->m_ctrl_reg1_map.m_acal = pConfigurationParameters->autoCalibrationEnabled;
  
  //Configure Magnetic Sensor Reset frequency
  FXOS8700CQ_M_CTRL_REG2_map_t* mCtrlReg2 = (FXOS8700CQ_M_CTRL_REG2_map_t*)&magnetometerControlRegisters[1];
  
  mCtrlReg2->m_ctrl_reg2_map.m_rst_cnt = pConfigurationParameters->autoSensorResetFreq;
  
  //Configure M_RAW bit (for autocalibration)
  FXOS8700CQ_M_CTRL_REG3_map_t* mCtrlReg3 = (FXOS8700CQ_M_CTRL_REG3_map_t*)&magnetometerControlRegisters[2];
  
  if(pConfigurationParameters->autoCalibrationEnabled)
    mCtrlReg3->m_ctrl_reg3_map.m_raw = 0;
  
  //Write registers to accelerometer
  status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_M_CTRL_REG1, (uint8_t*)mCtrlReg1);
  
  if(status != kStatusSuccess)
    return status;
  
  status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_M_CTRL_REG2, (uint8_t*)mCtrlReg2);
  
  if(status != kStatusSuccess)
    return status;
  
  status = FXOS8700CQ_set_register_blocking (FXOS8700CQ_M_CTRL_REG3, (uint8_t*)mCtrlReg3);
  
  if(status != kStatusSuccess)
    return status;

  //Restart FXOS8700CQ
  return FXOS8700CQ_start();
}

FXOS8700CQ_status_t FXOS8700CQ_get_magnetometer_readings (FXOS8700CQ_output_data_t* pMagnetometerData){
  FXOS8700CQ_status_t status;
  uint8_t magnetometerData[7];
  
  //Read magnetometer data registers
  status = FXOS8700CQ_get_registers_blocking (FXOS8700CQ_M_DR_STATUS, 0x07u, magnetometerData);
  
  if(status != kStatusSuccess)
    return status;
  
  //Store magnetometer data
  pMagnetometerData->status = magnetometerData[0];
  pMagnetometerData->xAxisData = (int16_t)((magnetometerData[1]<<8) | magnetometerData[2]);
  pMagnetometerData->yAxisData = (int16_t)((magnetometerData[3]<<8) | magnetometerData[4]);
  pMagnetometerData->zAxisData = (int16_t)((magnetometerData[5]<<8) | magnetometerData[6]);
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQ_get_hybrid_sensor_readings (FXOS8700CQ_output_data_t* pAccelerometerData, FXOS8700CQ_output_data_t* pMagnetometerData){
  FXOS8700CQ_status_t status;
  FXOS8700CQ_CTRL_REG1_map_t ctrlReg1;
  uint8_t sensorData[13];
  
  //Read CTRL_REG1
  status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_CTRL_REG1, 0x01u, &ctrlReg1.ctrl_reg1);
  if(status != kStatusSuccess)
    return status;
  
  //Read and store data
  if(ctrlReg1.ctrl_reg1_map.f_read){
    //Fast read mode enabled (8-bits resolution for accelerometer)
    status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_STATUS, 10u, sensorData);
    if(status != kStatusSuccess)
      return status;
    
    pAccelerometerData->status = sensorData[0];
    pAccelerometerData->xAxisData = (int16_t)(sensorData[1]);
    pAccelerometerData->yAxisData = (int16_t)(sensorData[2]);
    pAccelerometerData->zAxisData = (int16_t)(sensorData[3]);
    pMagnetometerData->xAxisData = (int16_t)((sensorData[4]<<8) | sensorData[5]);
    pMagnetometerData->yAxisData = (int16_t)((sensorData[6]<<8) | sensorData[7]);
    pMagnetometerData->zAxisData = (int16_t)((sensorData[8]<<8) | sensorData[9]);
  }
  else{
    //Fast read mode disabled (14-bits resolution for accelerometer)
    status = FXOS8700CQ_get_registers_blocking(FXOS8700CQ_STATUS, 13u, sensorData);
    if(status != kStatusSuccess)
      return status;
    
    pAccelerometerData->status = sensorData[0];
    pAccelerometerData->xAxisData = (int16_t)((sensorData[1]<<8)|(sensorData[2]))/4;
    pAccelerometerData->yAxisData = (int16_t)((sensorData[3]<<8)|(sensorData[4]))/4;
    pAccelerometerData->zAxisData = (int16_t)((sensorData[5]<<8)|(sensorData[6]))/4;
    pMagnetometerData->xAxisData = (int16_t)((sensorData[7]<<8) | sensorData[8]);
    pMagnetometerData->yAxisData = (int16_t)((sensorData[9]<<8) | sensorData[10]);
    pMagnetometerData->zAxisData = (int16_t)((sensorData[11]<<8) | sensorData[12]);
  }
  
  return kStatusSuccess;
}


FXOS8700CQ_status_t FXOS8700CQ_get_registers (uint8_t startRegisterAddress, uint8_t byteCount, FXOS8700CQ_callback_function_t onCompletionCallback){
  FXOS8700CQ_status_t sensorStatus;
  
  //Initiate read request
  sensorStatus = FXOS8700CQReadRegisters(startRegisterAddress, byteCount);
  if(sensorStatus != kStatusSuccess)
    return sensorStatus;
  
  //On success store user data
  userDefinedCallbackFunc = onCompletionCallback;
  expectedDataSize = byteCount;
  
  //Get data value
  return kStatusSuccess;  
}

FXOS8700CQ_status_t FXOS8700CQ_get_registers_blocking (uint8_t startRegisterAddress, uint8_t byteCount, uint8_t* pOutBuffer){
  FXOS8700CQ_status_t sensorStatus;
  
  //Initiate read request
  sensorStatus = FXOS8700CQReadRegistersBlocking(startRegisterAddress, byteCount, pOutBuffer);
  
  //Return status
  return sensorStatus;  
}

FXOS8700CQ_status_t FXOS8700CQ_set_register (uint8_t registerAddress, uint8_t* registerDataPtr){
  FXOS8700CQ_status_t sensorStatus;
  
  //Initiate write request
  sensorStatus = FXOS8700CQWriteRegister(registerAddress, registerDataPtr);
  if(sensorStatus != kStatusSuccess)
    return sensorStatus;
  
  //Get data value
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQ_set_register_blocking (uint8_t registerAddress, uint8_t* registerDataPtr){
  FXOS8700CQ_status_t sensorStatus;
  
  //Initiate write request
  sensorStatus = FXOS8700CQWriteRegisterBlocking(registerAddress, registerDataPtr);
  if(sensorStatus != kStatusSuccess)
    return sensorStatus;
  
  //Get data value
  return kStatusSuccess;  
}

/************************************************************************************
* Private Functions
************************************************************************************/
void FXOS8700CQSoftReset(void){
  FXOS8700CQ_CTRL_REG2_map_t resetByte = {
    .ctrl_reg2_map.rst = 1,
  };
  
  //Send reset signaling
  FXOS8700CQ_set_register_blocking (FXOS8700CQ_CTRL_REG2, (uint8_t*)&resetByte);
  
  //Wait 1ms aproximately for the system to reset
  for(int i=5000; i>0; i--){}
}

/*************************** I2C Functions ************************************/
FXOS8700CQ_status_t FXOS8700CQInitializeTransportI2C (void){
  
  //Initialize I2C Master
  if(kStatus_I2C_Success != I2C_DRV_MasterInit(FXOS8700CQ_I2C_INSTANCE, &FXOS8700CQMasterState, true))
    return kStatusInitializationError;
  
  //Set baud rate
  I2C_DRV_MasterSetBaudRate(FXOS8700CQ_I2C_INSTANCE, &FXOS8700CQMasterConfiguration);
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQReadRegistersI2C (uint8_t registerAddress, uint8_t byteCount){
  i2c_status_t i2cStatus;
  uint32_t remainingBytes;
  
  //Check that the receiver is not bussy 
  i2cStatus = I2C_DRV_MasterGetReceiveStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);
  
  //I2C Error handler
  if(i2cStatus != kStatus_I2C_Success)
    return FXOS8700CQI2cErrorhandler(i2cStatus);
  
  //Free previous allocated block (if any)
  if(memoryAllocationPointer != NULL){
#if defined(_MEM_MANAGER_H_)    //Use memory manager with the Connectivity stack
    MEM_BufferFree(memoryAllocationPointer);
#else
    free(memoryAllocationPointer);
#endif
    memoryAllocationPointer = NULL;
  }
  
  //Allocate memory for this reading
#if defined(_MEM_MANAGER_H_)    //Use memory manager with the Connectivity stack
  memoryAllocationPointer = MEM_BufferAlloc(byteCount);
#else
  memoryAllocationPointer = malloc(byteCount);
#endif
  
  //Check for a correct memory allocation
  if(memoryAllocationPointer == NULL)
    return kStatusMemoryAllocationError;
  
  //Read data
  i2cStatus = I2C_DRV_MasterReceiveData(FXOS8700CQ_I2C_INSTANCE, &FXOS8700CQMasterConfiguration, &registerAddress, 1u, memoryAllocationPointer, byteCount);
  
  //I2C Error handler
  if(i2cStatus != kStatus_I2C_Success)
    return FXOS8700CQI2cErrorhandler(i2cStatus);
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQReadRegistersBlockingI2C(uint8_t startRegisterAddress, uint8_t byteCount, uint8_t* pOutBuffer){
  i2c_status_t i2cStatus;
  uint32_t remainingBytes;
  
  /* Wait for I2C module to be free */
  //Wait for all receptions to complete
  do{
    i2cStatus = I2C_DRV_MasterGetReceiveStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);
  }
  while(i2cStatus == kStatus_I2C_Busy);
  //Wait for all sents to complete
  do{
    i2cStatus = I2C_DRV_MasterGetSendStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);
  }
  while(i2cStatus == kStatus_I2C_Busy);
  
  //Read address
  i2cStatus = I2C_DRV_MasterReceiveDataBlocking(FXOS8700CQ_I2C_INSTANCE, &FXOS8700CQMasterConfiguration, &startRegisterAddress, 1u, pOutBuffer, byteCount, FXOS8700CQ_TIMEOUT_MS);
  
  //I2C Error handler
  if(i2cStatus != kStatus_I2C_Success)
    return FXOS8700CQI2cErrorhandler(i2cStatus);
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQWriteRegisterI2C (uint8_t registerAddress, uint8_t* registerDataPtr){
  i2c_status_t i2cStatus;
  uint32_t remainingBytes;

  //Check if previous data is not being sent
  i2cStatus = I2C_DRV_MasterGetSendStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);

  //I2C Error handler
  if(i2cStatus != kStatus_I2C_Success)
    return FXOS8700CQI2cErrorhandler(i2cStatus);
  
  /* Fill data buffer to transmit */
   txBuffer[0] = registerAddress;
   txBuffer[1] = *registerDataPtr;
  
  //Transmit address and data
  i2cStatus = I2C_DRV_MasterSendData(FXOS8700CQ_I2C_INSTANCE, &FXOS8700CQMasterConfiguration, NULL, 0, txBuffer, 2u);
  
  //I2C Error handler
  if(i2cStatus != kStatus_I2C_Success)
    return FXOS8700CQI2cErrorhandler(i2cStatus);
  
  return kStatusSuccess;
}

FXOS8700CQ_status_t FXOS8700CQWriteRegisterBlockingI2C(uint8_t registerAddress, uint8_t* registerDataPtr){
  i2c_status_t i2cStatus;
  uint32_t remainingBytes;
  
   /* Wait for I2C module to be free */
  //Wait for all receptions to complete
  do{
    i2cStatus = I2C_DRV_MasterGetReceiveStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);
  }
  while(i2cStatus == kStatus_I2C_Busy);
  //Wait for all sents to complete
  do{
    i2cStatus = I2C_DRV_MasterGetSendStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);
  }
  while(i2cStatus == kStatus_I2C_Busy);
  
  /* Fill data buffer to transmit */
   txBuffer[0] = registerAddress;
   txBuffer[1] = *registerDataPtr;

  //Transmit address and data
  i2cStatus = I2C_DRV_MasterSendData(FXOS8700CQ_I2C_INSTANCE, &FXOS8700CQMasterConfiguration, NULL, 0, txBuffer, 2u);
  
  //I2C Error handler
  if(i2cStatus != kStatus_I2C_Success)
    return FXOS8700CQI2cErrorhandler(i2cStatus);

  return kStatusSuccess;  
}

FXOS8700CQ_status_t FXOS8700CQI2cErrorhandler (i2c_status_t i2cStatus){
  switch(i2cStatus){
  case kStatus_I2C_Timeout:
    return kStatusTimeOutError;
    break;
    
  case kStatus_I2C_Busy:
    return kStatusTransportBusyError;
    break;
    
  default:
    return kStatusCommunicationsError;
    break;
  }
}

void FXOS8700CQI2cCallbackHandler (void){
  i2c_status_t i2cStatus;
  uint32_t remainingBytes;
  
  //Check if last reception was completed 
  i2cStatus = I2C_DRV_MasterGetReceiveStatus(FXOS8700CQ_I2C_INSTANCE, &remainingBytes);
  
  //Exit if transaction has not been completed.
  if(i2cStatus != kStatus_I2C_Success)
    return;
  
  //Check that a user callback has been already registered
  if(userDefinedCallbackFunc == NULL)
    return;
  
  //On transaction completed call the user callback
  userDefinedCallbackFunc(memoryAllocationPointer, expectedDataSize);
  
  //Free allocated memory
#if defined(_MEM_MANAGER_H_)    //Use memory manager with the Connectivity stack
    MEM_BufferFree(memoryAllocationPointer);
#else
    free(memoryAllocationPointer);
#endif
  
  //Clear expected data size and user callback
  userDefinedCallbackFunc = NULL;
  expectedDataSize = 0;
}
          
/**********************************************************************************/