/*
 * TransceiverDrv.c
 *
 *  Created on: Nov 25, 2016
 *      Author: B50961
 */

#include "device.h"
#include "util.h"
#include "TransceiverDrv.h"
#include "TransceiverReg.h"
#include "RF_Rx_Appli.h"
#include "Interrupt.h"
#include "TPM.h" // for delays in reset function



SPI_MemMapPtr  SpiRegs;


/*****************************************************************************
******************************************************************************
* Public memory declarations
******************************************************************************
*****************************************************************************/


uint8_t const MKW01DefaultRegisterValues[] =
{
  /* Radio operation mode initialization @0x01*/
  MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy ) ,

  /* Radio Data mode and modulation initialization @0x02*/
  MKW01_Reg_DataModul, (uint8_t) ( DataModul_DataMode_Packet | DataModul_Modulation_Fsk | DataModul_ModulationShaping_NoShaping) , // Use NoShaping ?

  /* Radio bit rate initialization @0x03-0x04*/
  /* Default Bit Rate is 19200
   * In order to select another Bit Rate, you can call the function MKW01Drv_SetBaudRate in the RF init
   */
  MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_19200 ) ,
  MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_19200 ) ,

  /* Radio frequency deviation initialization @0x05-0x06*/
  /* Default Frequency Deviation is 50kHz
   * In order to select another Frequency Deviation, you can call the function MKW01Drv_SetFreqDev in the RF init
   */
  MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_50000 ) ,
  MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_50000 ) ,

  /* Radio RF frequency initialization @0x07-0x09*/
  /* Default Frequency is 434
   * In order to select another frequency, you can call the function MKW01Drv_SetFrequency in the RF init
   */
  MKW01_Reg_FrfMsb, (uint8_t) ( FrfMsb_434 ) ,
  MKW01_Reg_FrfMid, (uint8_t) ( FrfMid_434 ) ,
  MKW01_Reg_FrfLsb, (uint8_t) ( FrfLsb_434 ) ,

  /* Radio RegAfcCtrl initialization @0x0B*/
  MKW01_Reg_AfcCtrl, (uint8_t) AfcCtrl_AfcLowBeta_Off , // with modulation index < 2 we can use AfcCtrl_AfcLowBeta_On

  /* Radio output power initialization @0x11*/
  MKW01_Reg_PaLevel, (uint8_t) ( PaLevel_Pa0_On | PaLevel_Pa1_Off | PaLevel_Pa2_Off | 0x1F ) ,

  /* Radio Rise/Fall time of ramp up/down in FSK initialization @0x12*/
  MKW01_Reg_PaRamp, (uint8_t) (PaRamp_40) ,

  /* Radio overload current protection for PA initialization 0x13*/
  MKW01_Reg_Ocp, (uint8_t) ( Ocp_Ocp_On | Ocp_Ocp_Trim_110) ,

  /* Radio LNA gain and input impedance initialization @0x18*/
  MKW01_Reg_Lna, (uint8_t) ( Lna_LnaZin_200 | Lna_LnaGain_Agc ) ,

  /* Radio channel filter bandwidth initialization @0x19*/
  MKW01_Reg_RxBw, ( DccFreq_7 | RxBw_83300 ) ,

  /* Radio channel filter bandwidth for AFC operation initialization @0x1A*/
  MKW01_Reg_AfcBw, ( DccFreq_5 | RxBw_100000 ) ,

  /* Radio automatic frequency control initialization @0x1E*/
  MKW01_Reg_AfcFei, (uint8_t) ( AfcFei_AfcAuto_Off | AfcFei_AfcAutoClear_On ) ,

  /* Radio Rssi threshold initialization @0x29*/
  // RSSIthres = [-174 + NF +10*log(2*RxBw) + DemodSNR] dBm
  // NF = 7dB
  // DemodSnr = 8dB
  // RxBw depends on frequency bands and profiles
  MKW01_Reg_RssiThresh, (uint8_t) (0xE4) , // -101 dBm for 333.3 Khz singleside channel filter bandwith

  /* Interrupt pins configuration */
  MKW01_Reg_DioMapping1, (uint8_t) (DIO0_RxPayloadReady | DIO1_RxFifoNotEmpty | DIO2_RxFifoNotEmpty | DIO3_RxSyncAddres),
  MKW01_Reg_DioMapping2, (uint8_t) (DIO4_RxRxReady | DIO5_RxClkOut | ClkOutFxOsc_Div1),


  /* Radio RegTimeoutRxStart initialization @0x2A*/
  /* Radio RegTimeoutRssiThresh initialization @0x2B*/
  MKW01_Reg_RxTimeout1, (uint8_t) (RxTimeout_0) , //disable timeout rx start
  MKW01_Reg_RxTimeout2, (uint8_t) (RxTimeout_0) , //disable timeout rssi

  /* MKW01 preamble size initialization @0x2C-0x2D*/
  MKW01_Reg_PreambleMsb, (uint8_t) (PrmMSB_0),
  MKW01_Reg_PreambleLsb, (uint8_t) (PrmLSB_3),

  /* Radio sync word control and value initialization @0x2E-0x30*/
  MKW01_Reg_SyncConfig, (uint8_t) (SyncConfig_Sync_On | SyncConfig_FifioFill_ifSyncAddres | SyncConfig_SyncSize_4) ,
  MKW01_Reg_SyncValue1, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue2, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue3, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue4, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue5, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue6, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue7, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0
  MKW01_Reg_SyncValue8, (uint8_t) (SyncValue_1) , //SFD value for uncoded with phySUNMRFSKSFD = 0

  /* Radio packet mode config */
#if (RF_DATA_CORRUPTED == DISCARD_DATA)
  MKW01_Reg_PacketConfig1, (uint8_t) (PacketConfig1_PacketFormat_Variable|PacketConfig1_Crc_On|PacketConfig1_CrcAutoClear_On|PacketConfig1_AddresFiltering_Off),
#else
  MKW01_Reg_PacketConfig1, (uint8_t) (PacketConfig1_PacketFormat_Variable|PacketConfig1_Crc_On|PacketConfig1_CrcAutoClear_Off|PacketConfig1_AddresFiltering_Off),
#endif
  MKW01_Reg_PacketConfig2, (uint8_t) (PacketConfig2_AutoRxRestart_On),

  /* Radio payload length initialization */
  MKW01_Reg_PayloadLength, (uint8_t) (FIFO_SIZE), // Variable mode => max length in Rx

  MKW01_Reg_AutoModes, (uint8_t) (AutoModes_Disabled),

  /* FIFO threshold */
 MKW01_Reg_FifoThresh, (uint8_t) (FifoThresh_0),


 /* Below not used as we disabled address filtering */
 //MKW01_Reg_NodeAdrs,	     (uint8_t) (NodeAdrs_F0),
 //MKW01_Reg_BroadcastAdrs, (uint8_t) (BCastAdrs_FF),

};

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/


/******************************************************************************
* Description :SPI initialization for Radio driver interface
*
* Assumptions :
*
* Inputs      :none
* Output      :none
* Errors      :none
******************************************************************************/
void MKW01Drv_SpiInit(void)
{
   volatile uint8_t dummy;

        /* Enable Reset PORT clock */
  MKW01x_RESET_PORT_SIM_SCG |= cRESET_PORT_SIM_SCG_Config_c;

     /* Enable Reset */
  gMKW01RadioResetPort  &= ~PORT_PCR_MUX_MASK;
  gMKW01RadioResetPort  |= PORT_PCR_MUX(1);
  gRadioReset_DataAvailablePortDDirReg_c|=gMKW01ResetMask_c;

      /* Enable SPI PORT clock */
  MKW01x_SPI_PORT_SIM_SCG |= cSPI_PORT_SIM_SCG_Config_c;

    /* Enable SPI clock */

  MKW01x_SPI_SIM_SCG|= cSPI_SIM_SCG_Config_c;

    /* Initialize SPI Pin Control registers */

  /*Initialize PTD0 used as PCS line_Start*/
  PORTD_PCR0&= ~PORT_PCR_MUX_MASK;                                              //PTD0 used as PCS (Slave Select) line
  PORTD_PCR0|= PORT_PCR_MUX(1);                                                 //PTD0 used as PCS (Slave Select) line
  GPIOD_PDDR|=(1<<0);                                                           //PTD0 pin as output
  /*Initialize PTD0 used as PCS line_End*/

  SPI_SCK_PCR  &= ~PORT_PCR_MUX_MASK;
  SPI_SCK_PCR  |= PORT_PCR_MUX(SPI_SCK_ALT_FUNC);
  SPI_MOSI_PCR &= ~PORT_PCR_MUX_MASK;
  SPI_MOSI_PCR |= PORT_PCR_MUX(SPI_MOSI_ALT_FUNC);
  SPI_MISO_PCR &= ~PORT_PCR_MUX_MASK;
  SPI_MISO_PCR |= PORT_PCR_MUX(SPI_MOSI_ALT_FUNC);

  MKW01Drv_AssertReset(); //Keep Radio in reset

  /* Initialize the SPI peripheral base address */
  SpiRegs = SPI0_BASE_PTR;
  SpiRegs->C1 = cSPIC1_Init;
  SpiRegs->C2 = cSPIC2_Init;
  SPI0_BR = cSPIBR_Init;
  SpiRegs->C1 |= (uint8_t)(SPI_C1_SPE_MASK | SPI_C1_MSTR_MASK) ; //enable SPI system
  SPIS_DummyRead;
  DeAssertSS

}

/******************************************************************************
* Description :Make a pulse on Radio hardware reset PIN
*
* Assumptions :
*
* Inputs      :none
* Output      :none
* Errors      :none
******************************************************************************/
/*
 * Function below has been modified for KDS project
 */
void MKW01Drv_Reset(void)
{
  //@AC RESET sequence from SX1233 datasheet:
  // RESET high Z
  // RESET = 1, then wait 100us
  // RESET = 0 (high Z), then wait 5ms before using the Radio.

  MKW01Drv_AssertReset();
  Delay_TPM0_us(100);
  MKW01Drv_DeassertReset();
  Delay_TPM0_ms(5);
}


/*************************************************************************
*                       SPI Read function                                *
*************************************************************************/


/******************************************************************************
* Description :Read one 8bit data-location from Radio register "addr"
*              May be called from any context.
*              Protected from mcu/app irq. Includes irq status backup
* Assumptions :
*
* Inputs      :register address
* Output      :register value
* Errors      :none
******************************************************************************/
uint8_t MKW01Drv_ReadRegister(uint8_t addr) {
  uint8_t val;
  uint32_t CCR;

#if defined(IAR)
  CCR = IntDisableAll();
#elif defined (CW)
  IntDisableAll(CCR);
#endif

  //Added in KDS project
  DISABLE_NVIC_IT;

  SPIS_DummyRead;
  AssertSS;
  SpiRegs->DL = (unsigned char)(addr);                                          // Write Addr byte
  WaitSPITxBufferEmpty;
  SPID_DummyRead;
  SpiRegs->DL = 0x00;          /* Write data byte */
  WaitSPITxBufferEmpty;                                                         // Wait for room in holding register
  SPID_DummyRead;
  WaitSPIRxBufferFull;                                                          //Wait until the shift buffer has received the data from the Slave, this ensures that the Tx Data has been sent and the Slave has received it
  val = SpiRegs->DL;          /* Write data byte */
  DeAssertSS;
  //IntRestoreAll(CCR); // For IAR
  ENABLE_NVIC_IT;
  return val;
}


/*************************************************************************
*                      SPI Write functions                               *
*************************************************************************/

/******************************************************************************
* Description :Write one 8bit data into Radio at address "addr"
*              May be called from any context.
*              Protected from mcu/app irq. Includes irq status backup
* Assumptions :
*
* Inputs      :register address, register value
* Output      :none
* Errors      :none
******************************************************************************/
void MKW01Drv_WriteRegister(uint8_t addr, uint8_t val) {

  uint32_t CCR;

#if defined(IAR)
  CCR = IntDisableAll();
#elif defined (CW)
  IntDisableAll(CCR);
#endif

  //Added in KDS project
  DISABLE_NVIC_IT;

  SPIS_DummyRead;
  AssertSS;
  SpiRegs->DL = (unsigned char)(addr | 0x80);  /* Write Addr byte */
  WaitSPITxBufferEmpty;                        /* Wait for room in holding register */
  SPID_DummyRead;
  SpiRegs->DL = (unsigned char)(val);          /* Write data byte */
  WaitSPITxBufferEmpty;                                                         // Wait for room in holding register
  SPID_DummyRead;
  WaitSPIRxBufferFull;                                                          //Wait until the shift buffer has received the data from the Slave, this ensures that the Tx Data has been sent and the Slave has received it
  DeAssertSS;
  //IntRestoreAll(CCR); // For IAR
  ENABLE_NVIC_IT;
}

/******************************************************************************
* Description :Initialize Radio transceiver with default registers value
*
* Assumptions :
*
* Inputs      :none
* Output      :none
* Errors      :none
******************************************************************************/
void MKW01Drv_RFdefaultInit(void)
{
  uint8_t idx;
  for(idx = 0; idx < (sizeof(MKW01DefaultRegisterValues)/sizeof(MKW01DefaultRegisterValues[0])); idx += 2)
  {
    MKW01Drv_WriteRegister(MKW01DefaultRegisterValues[idx], MKW01DefaultRegisterValues[idx+1]);
  }

  return;
}

/*
 * Configure the RF frequency
 * Input parameter is one of the defined frequencies
 */
void MKW01Drv_SetFrequency(uint8_t freq)
{

	switch (freq)
	{
	case (FREQ_3135):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_3135 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_3135 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_3135 ));
			break;

	case (FREQ_314):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_314 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_314 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_314 ));
			break;

	case (FREQ_3145):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_3145 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_3145 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_3145 ));
			break;

	case (FREQ_315):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_315 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_315 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_315 ));
			break;

	case (FREQ_3155):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_3155 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_3155 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_3155 ));
			break;

	case (FREQ_316):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_316 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_316 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_316 ));
			break;

	case (FREQ_3165):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_3165 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_3165 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_3165 ));
			break;

	case (FREQ_4325):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_4325 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_4325 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_4325 ));
			break;

	case (FREQ_433):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_433 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_433 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_433 ));
			break;

	case (FREQ_4335):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_4335 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_4335 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_4335 ));
			break;

	case (FREQ_434):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_434 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_434 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_434 ));
			break;

	case (FREQ_4345):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_4345 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_4345 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_4345 ));
			break;

	case (FREQ_435):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_435 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_435 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_435 ));
			break;

	case (FREQ_4355):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_4355 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_4355 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_4355 ));
			break;

	case (FREQ_470):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_470 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_470 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_470 ));
			break;

	case (FREQ_865):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_865 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_865 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_865 ));
			break;

	case (FREQ_868):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_868 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_868 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_868 ));
			break;

	case (FREQ_915):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_915 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_915 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_915 ));
			break;

	case (FREQ_920):
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_920 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_920 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_920 ));
			break;

	default:
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMsb, (uint8_t)( FrfMsb_434 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfMid, (uint8_t)( FrfMid_434 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FrfLsb, (uint8_t)( FrfLsb_434 ));
			break;
	}

	return;
}

/*
 * Configures the RF baud rate
 * Input parameter is one of the defined baud rates
 */
void MKW01Drv_SetBaudRate(uint8_t baudRate)
{

	switch (baudRate)
	{
	case (BR_4800):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_4800 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_4800 )) ;
			break;

	case (BR_9600):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_9600 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_9600 )) ;
			break;

	case (BR_10000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_10000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_10000 )) ;
			break;

	case (BR_19200):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_19200 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_19200 )) ;
			break;

	case (BR_20000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_20000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_20000 )) ;
			break;

	case (BR_40000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_40000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_40000 )) ;
			break;

	case (BR_50000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_50000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_50000 )) ;
			break;

	case (BR_100000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_100000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_100000 )) ;
			break;

	case (BR_150000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_150000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_150000 )) ;
			break;

	case (BR_200000):
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_200000 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_200000 )) ;
			break;

	default:
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateMsb, (uint8_t) ( BitrateMsb_19200 )) ;
			MKW01Drv_WriteRegister(MKW01_Reg_BitrateLsb, (uint8_t) ( BitrateLsb_19200 )) ;
			break;

	}
	return;
}

/*
 * Configures the RF frequency deviation
 * Input parameter is one of the defined frequency deviations
 */
void MKW01Drv_SetFreqDev (uint8_t freqDev)
{

	switch (freqDev)
	{
	case (FREQ_DEV_5000):
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_5000 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_5000 ));
			break;

	case (FREQ_DEV_19000):
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_19000 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_19000 ));
			break;

	case (FREQ_DEV_25000):
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_25000 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_25000 ));
			break;

	case (FREQ_DEV_50000):
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_50000 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_50000 ));
			break;

	case (FREQ_DEV_50049):
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_50049 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_50049 ));
			break;

	case (FREQ_DEV_100000):
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_100000 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_100000 ));
			break;

	default:
			MKW01Drv_WriteRegister(MKW01_Reg_FdevMsb, (uint8_t) ( FdevMsb_50000 ));
			MKW01Drv_WriteRegister(MKW01_Reg_FdevLsb, (uint8_t) ( FdevLsb_50000 ));
			break;

	}
	return;
}


/************************************************************************
 * mkw01_drv_set_op_mode() configures operation mode of the receiver
 * Parameters - operation mode code
 * returns - nothing
 * ************************************************************************/
void mkw01_drv_set_op_mode(uint8_t opMode)
{
	if (opMode== RECEIVE)
	{
		MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_Receiver )); //RX
	}
	else
	{
		MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) (OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy));
	}

	return;
}



