/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "device.h"
#include "Interrupt.h"
#include "GPIO.h"
#include "TPM.h"
#include "RF_Rx_Appli.h"
#include "UART.h"
#include "TransceiverDrv.h"
#include "TransceiverReg.h"
#include "LF.h"
#include "Display.h"
#include "ExecuteAction.h"

/* Declaration of local functions */
void Init_General(void);
void Init_PORTS_General (void);
// End of declaration of local functions


/***************************************************************************************************************************
 * Demo configuration:
 * - Choose the main RF settings (frequency, baud rate, frequency deviation) in RF_InitRx (Rf_Rx_Appli.c)
 * 	 The remaining RF settings are in TransceiverDrv.c
 * - Choose the UART baud rate with UART0_SetBaudRate in Init_General (main.c)
 * - Choose the number of bytes to receive via UART (length of a command) with #define UART_NB_BYTES_TO_RECEIVE in UART.h
 * - In main you can choose between display on LabVIEW GUI or display on hyperterminal for debug (comment the appropriate line)
 *
 * Demo flow:
 * - When an RF frame is received it is displayed on hyperterminal or GUI.
 * - When 4bytes are received via UART, the command is executed. The list of commands/actions is in ExecuteAction.c
 * - When a RF frame is received, LED16 (PTC2) remains ON while it is processed and displayed.
 * - When a LF frame is sent, LED10 (PTE1) is ON while the frame is being sent.
 * - When an unknown command is received via UART, LED15 (PTE16) is ON during 200ms.
 ***************************************************************************************************************************/


int main(void)
{

	Init_General();

	(void)RF_EnableRx();

	UART0_ENABLE_RECEPTION;



    for (;;)
    {

       // Delay_TPM0_ms(200); // Remove all delays otherwise we will miss RF frames (we do polling and do not use interrupts)

		if (RF_PAYLOAD_READY)
		{
			LED16_ON;

			RF_StoreDataReceived(); // Store RF frame in RF_Rx_Buffer

			Display_SendDataToLabview(); // Display LabVIEW Sensor GUI
			//Display_SendDataHypertem(); // Display RF frame only on hyperterminal - for debug

			LED16_OFF;
		}

		if (UART0_RxBufferFull)
		{
			ExecuteAction_TPMS();
		}

    }
    /* Never leave main */
    return 0;
}



void Init_General (void)
{

    /* Enable interrupts with configurable priority (NVIC) */
	ENABLE_NVIC_IT;

	/* Enable clock gates + general port init */
	Init_PORTS_General();

	/* Init TPM0 */
	/* Need to init before RF as RF init uses delays based on TPM0 */
	Init_TPM0_General();

	/* Init UART (currently UART0) */
	Init_UART0_General();
	/* Available baud rates: UART_BAUD_RATE_9600, UART_BAUD_RATE_19200, UART_BAUD_RATE_115200 */
	UART0_SetBaudRate(UART_BAUD_RATE_115200);

	/* Init TPM1 for LF */
	/* If appli uses TPM1, move this init into Init_Tx_LF_PREMO
	 * otherwise let it here in order to save time in Init_Tx_LF_PREMO
	 * Uses PTB0 and PTB1
	 */
	LF_Init_General();

	/* Init RF */
	/* Uses PTE2 and PTE3 */
	RF_InitRx();

	/* Init LEDs used in ExecuteAction */
	/* LED4 is PTA4, LED5 is PTE0, LED10 is PTE1, LED15 is PTE16, LED16 is PTC2 */
	Init_LEDs();

	return;
}


void Init_PORTS_General (void)
{
	// Enable clock gate to PORTA,B,C,D,E
	SIM_SCGC5 |= ((uint32_t)SIM_SCGC5_PORTA_MASK) | ((uint32_t)SIM_SCGC5_PORTB_MASK) | ((uint32_t)SIM_SCGC5_PORTC_MASK) | ((uint32_t)SIM_SCGC5_PORTD_MASK) | ((uint32_t)SIM_SCGC5_PORTE_MASK);

	return;
}

/*
 	 PINS USED BY APPLICATION

	 PTA4  : Execute Action (LED4)

 	 PTB0  : LF
 	 PTB1  : LF

 	 PTC2  : Execute Action (LED16 - ON while an RF frame is received and displayed)
 	 PTC3  : physically connected to DIO0 on MRB board
 	 PTC4  : physically connected to DIO1 on MRB board

	 PTE0  : Execute Action (LED5)
	 PTE1  : Execute Action (LED10 - ON while LF frame is sent)
 	 PTE2  : RF (DIO0)
 	 PTE3  : RF (DIO1)
 	 PTE16 : Execute Action (LED15 - ON during 200ms when an unknown UART cmd is received)

 */

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
