/*********************************************************************************************************
**	LPC1100 Self Wake-up Sample Application
**
**	This project configures the watchdog oscillator at lowest frequency to clock the 16-bit Timer 0
**	in run and deep sleep modes. The device generates a repetitive self wake-up from deep-sleep mode and
**	toggles a port pin when in run mode.
**
**	Watchdog Osc configured at slowest freq (FREQSEL = 0x0; DIVSEL = 0x1F):	9600  Hz
**
**  The wake-up time can be measured by using the following pins:
**	Trigger pin (PIO0_8/CT16B0_MAT0)
**	Timer Output to get the device out of the deep-sleep mode.
**
**	Strobe pin (PIO0_7)
**	After wake-up, the device returns into run mode, and this pin is toggles 4 times
**	within the WAKEUP_IRQHANDLER() subroutine.
**
**  The wakeup time is the difference between the rising edge of the trigger pin and
**  the first falling edge of the Strobe pin.
**
**  Please note that once this code is downloaded into flash, the device enters and exists
**  deep-sleep mode on a repetitive basis.
**  User must put the device in ISP mode first if user wants to erase or reprogram the flash.
**  See LPC1100 UM for details on how to enter ISP mode.
**********************************************************************************************************/


#ifdef __USE_CMSIS
#include "LPC11xx.h"
#endif

#include "timer16_tk.h"

/*	ENABLE_CLKOUT = 0			PIO0_1 is configured as a GPIO
	ENABLE_CLKOUT = 1			PIO0_1 outputs CLKOUT */
#define ENABLE_CLKOUT		0

/*	DEEP_SLEEP_MODE = 0:		Device will go into Sleep Mode
	DEEP_SLEEP_MODE = 1:		Device will go into Deep Sleep Mode */
#define DEEP_SLEEP_MODE		1

/*	ENABLE TIMER specifies the specifies the self wake up feature */
#define ENABLE_TIMER		1

void wait(uint32_t time);
void config_ios(void);

void WAKEUP_IRQHandler(void){
	int i;

	for(i = 0; i < 4; i++){
		LPC_GPIO0->DATA ^= (1<<7);
		wait(1000);
	}

	Clear_MAT0(0);

	/* Clear the pending interrupt of the start logic */
	LPC_SYSCON->STARTRSRP0CLR	|=	(1<<(8));
	return;
}

int main(void){

	LPC_SYSCON->SYSAHBCLKCTRL	= /* Enable SYSCLK Peripherals */
									 (1<<0) 	//SYS Clock
									|(1<<1)		//ROM
									|(1<<2) 	//RAM
									|(1<<3) 	//FLASHREG
									|(1<<4) 	//FLASHARRAY
									|(1<<6) 	//GPIO
									|(1<<16)	//IOCON
								;

	LPC_SYSCON->SYSAHBCLKDIV = 1;

	/* Configure Wakeup I/O */
	/* Specify the start logic to allow the chip to be waken up */
	LPC_SYSCON->STARTAPRP0 		|=  (1<<(8));		// Rising edge
	LPC_SYSCON->STARTRSRP0CLR	|=	(1<<(8));		// Clear pending bit
	LPC_SYSCON->STARTERP0		|=	(1<<(8));		// Enable Start Logic

	/* Specify output pin to toggle an LED */
	LPC_GPIO0->DIR	|=	(1<<7);
	LPC_GPIO0->DATA &= ~(1<<7);

	#if ENABLE_TIMER
		/* Initialize 16-bit Timer 0 */
	Init_Timer(0, 1000, 100000);
	#endif /* ENABLE_TIMER */

	#if ENABLE_CLKOUT
		/* Output the Clk onto the CLKOUT Pin PIO0_1 to monitor the freq on a scope */
		LPC_IOCON->PIO0_1	= (1<<0);
		/* Select the MAIN clock as the clock out selection since it's driving the core */
		LPC_SYSCON->CLKOUTCLKSEL = 3;
		/* Set CLKOUTDIV to 6 */
		LPC_SYSCON->CLKOUTDIV = 10;		//	CLKOUT Divider = 10
		/* Enable CLKOUT */
		LPC_SYSCON->CLKOUTUEN = 0;
		LPC_SYSCON->CLKOUTUEN = 1;
		while (!(LPC_SYSCON->CLKOUTUEN & 0x01));
	#endif /* ENABLE_CLKOUT */

	/* Turn off all other peripheral dividers */
	LPC_SYSCON->SSP0CLKDIV = 0;
	LPC_SYSCON->SSP1CLKDIV = 0;
	LPC_SYSCON->WDTCLKDIV = 0;
	LPC_SYSCON->SYSTICKCLKDIV = 0;

	/* Turn on the watchdog oscillator */
	LPC_SYSCON->PDRUNCFG &= ~(1<<6);

	/* Run watchdog oscillator */
	LPC_SYSCON->WDTOSCCTRL = (0x1<<5) | 0x1F;	//SLOWEST WDT FREQ

	/* Switch MAINCLKSEL to Watchdog Oscillator */
	LPC_SYSCON->MAINCLKSEL	= 2;
	LPC_SYSCON->MAINCLKUEN  = 0;
	LPC_SYSCON->MAINCLKUEN  = 1;
	while (!(LPC_SYSCON->MAINCLKUEN & 0x01));

	/* Make sure that only the watchdog oscillator & Flash are running now */
	LPC_SYSCON->PDRUNCFG = ~((1<<6) | (1<<2) | (1<<9) | (1<<12));

	#if ENABLE_TIMER
		/* Initialize 16-bit Timer 0 */
		Enable_Timer(0);
	#endif /* ENABLE_TIMER */
	


	#if DEEP_SLEEP_MODE

		/* Clear the Deep Sleep Flag */
		LPC_PMU->PCON |= (1<<8);

		/* Specify peripherals to be powered up again when returning from deep sleep mode */
		LPC_SYSCON->PDAWAKECFG = LPC_SYSCON->PDRUNCFG;

		/* Configure PDSLEEPCFG */
		LPC_SYSCON->PDSLEEPCFG	= 0x000018BF;

		/* Specify Deep Sleep mode before entering mode */
		SCB->SCR	|=	(1<<2);		//Set SLEEPDEEP bit
	#endif /* DEEP_SLEEP_MODE */

	/* Reconfigure the IOs */
	config_ios();

	while(1){

		LPC_SYSCON->STARTRSRP0CLR |= (1<<(8));
		NVIC_ClearPendingIRQ(WAKEUP8_IRQn);
		NVIC_EnableIRQ(WAKEUP8_IRQn);

		/* Go to Sleep */
		__WFI();

	}
}

void wait(uint32_t time){
	while(time--);
}


void config_ios(void){

	LPC_IOCON->PIO2_6			= 0xC0;
	LPC_IOCON->PIO2_0			= 0xC0;
	LPC_IOCON->RESET_PIO0_0 	= 0xC1;      // RST_BOOT/P0_0
#if !ENABLE_CLKOUT
	LPC_IOCON->PIO0_1			= 0xC0;      // CLKOUT/CT32B0_MAT2/USB_SOF
#endif /* ENABLE_CLKOUT */
	LPC_IOCON->PIO1_8			= 0xC0;
	LPC_IOCON->PIO0_2			= 0xC0;      // SSEL/CT16B0_CAP0

	LPC_IOCON->PIO2_7			= 0xC0;
	LPC_IOCON->PIO2_8			= 0xC0;
	LPC_IOCON->PIO2_1			= 0xC0;
	LPC_IOCON->PIO0_3			= 0xC0;      // USB VBUS
	LPC_IOCON->PIO0_4			= 0xC0;      // I2C_SCL, no pull-up, inactive
	LPC_IOCON->PIO0_5			= 0xC0;      // I2C_SDA, no pull-up, inactive
	LPC_IOCON->PIO1_9			= 0xC0;      // CT16B1_MAT0
	LPC_IOCON->PIO3_4			= 0xC0;

	LPC_IOCON->PIO2_4			= 0xC0;
	LPC_IOCON->PIO2_5			= 0xC0;
	LPC_IOCON->PIO3_5			= 0xC0;
	LPC_IOCON->PIO0_6			= 0xC0;      // USB SoftConnect
	LPC_IOCON->PIO0_7			= 0xC0;
	LPC_IOCON->PIO2_9			= 0xC0;
	LPC_IOCON->PIO2_10			= 0xC0;
	LPC_IOCON->PIO2_2			= 0xC0;

#if !ENABLE_TIMER
	/* PIO0_8 needs to be configured to generate a Match output */
	LPC_IOCON->PIO0_8			= 0xC0;		// SSP_MISO/CT16B0_MAT0/TRACE_CLK
#endif /* ENABLE_TIMER */
	LPC_IOCON->PIO0_9			= 0xC0;		// SSP_MOSI/CT16B0_MAT1/TRACE_SWV
	LPC_IOCON->JTAG_TCK_PIO0_10	= 0xC1;		// JTAG_CLK/P0_10/SSP_CLK/CT16B0_MAT2
	LPC_IOCON->PIO1_10			= 0xC0;		// ADCIN6/CT16B1_MAT1
	LPC_IOCON->PIO2_11			= 0xC0;
	LPC_IOCON->JTAG_TDI_PIO0_11	= 0xC1;		// JTAG_TDI/P0_11/ADCIN0/CT32B0_MAT3
	LPC_IOCON->JTAG_TMS_PIO1_0  = 0xC1;		// JTAG_TMS/P1_0/ADCIN1/CT32B1_CAP0
	LPC_IOCON->JTAG_TDO_PIO1_1  = 0xC1;		// JTAG_TDO/P1_1/ADCIN2/CT32B1_MAT0

	LPC_IOCON->JTAG_nTRST_PIO1_2 = 0xC1;	// JTAG_TRST/P1_2/ADCIN3/CT32B1_MAT1
	LPC_IOCON->PIO3_0			= 0xC0;
	LPC_IOCON->PIO3_1			= 0xC0;
	LPC_IOCON->PIO2_3			= 0xC0;
	LPC_IOCON->ARM_SWDIO_PIO1_3	= 0xC1;		// ARM_SWD/P1_3/ADCIN4/CT32B1_MAT2
	LPC_IOCON->PIO1_4			= 0xC0;		// ADCIN5/CT32B1_MAT3/WAKEUP
	LPC_IOCON->PIO1_11			= 0xC0;		// ADCIN7
	LPC_IOCON->PIO3_2			= 0xC0;

	LPC_IOCON->PIO1_5			= 0xC0;      // UART_DIR/CT32B0_CAP0
	LPC_IOCON->PIO1_6			= 0xC0;      // UART_RXD/CT32B0_MAT0
	LPC_IOCON->PIO1_7			= 0xC0;      // UART_TXD/CT32B0_MAT1
	LPC_IOCON->PIO3_3			= 0xC0;

	LPC_GPIO0->DIR =\
	LPC_GPIO1->DIR =\
	LPC_GPIO2->DIR =\
	LPC_GPIO3->DIR = 0xFFFFFFFF;

	LPC_GPIO0->DATA  =\
	LPC_GPIO1->DATA  =\
	LPC_GPIO2->DATA  =\
	LPC_GPIO3->DATA  = 0;

	return;
}
