#include "LPC43xx.h"
#include "breakpoint.h"
#include "bbI2C.h"

int printf(const char *format, ...);

#define USE_SPIFI_LIB          // may need to be in C command line
//#define DEBUG_I2C		

#include "spifi_rom_api.h"
#include "system_LPC43xx.h"

#define  GPIO_PIN_SET			(1<<6)		// scu setting for slew rate, glitch..., AND ENABLE INPUT

/**********************************************************************
 ** SPIFI Flash comes here, copies itself to RAM and jumps there
 **   someday it will copy all RAM so multiple CPUs supported
 **
 ** the idea is to be able to take a snapshot of RAM -> SPIFI 
 **********************************************************************/

__asm int gotoR0(int x) {	//;		// sGPIO does LSB first, need to flip it -- this is in startup.S
 	bx	r0
} 	

__asm int whereAMi() {
	mov		r0,pc
	bx 		lr
}
 
#if 1  // LPC4357
#define RAM1start	0x10000000
#define RAM1size	0x8000
#define RAM2start	0x10080000
#define RAM2size	0x9000
#define RAM3start	0x20000000
#define RAM3size	0x8000
#define RAM4start	0
#define RAM4size	0



#else
#define RAM1start	0x10000000
#define RAM1size	0x20000
#define RAM2start	0x10080000
#define RAM2size	0x12000
#define RAM3start	0x20000000
#define RAM3size	0x10000
#define RAM4start	0x18000000
#define RAM4size	0x4800
#endif

// use 384K for snapshots of RAM, eventually allow lowspeed shared code to reside in the rest of the space

#define SPIFIsaveCODE	(0x80000000)			// 32K space
#define SPIFIsaveRAM1	(0x80100000 - 0x58000)	// 128K space
#define SPIFIsaveRAM2	(0x80100000 - 0x38000)	// 128K space
#define SPIFIsaveRAM3	(0x80100000 - 0x18000)	// 64K space
#define SPIFIsaveRAM4	(0x80100000 - 0x8000)	// 32K space

#define codeSTART	RAM2start		// sutomate this someday	

#define SPIFIstart	0x80000000	   		// grab this from SPIFI table obj someday


/////////////////////////////////////////////////////////////////////////////////
//
//	checkSPIFIstart called at startup -- if its called from other than SPIFI memory it just returns
//
//	if it is SPIFI, then copy the contents of SPIFI back into RAM and branch to it
//
//	DOH -- as this routine overwrites the stack it has to be inline code
//
/////////////////////////////////////////////////////////////////////////////////
 
void checkSPIFIstart(void) {
	int *ram, *spifi;

#ifdef RUN_IN_SPIFI
	return;
#else

	if ((whereAMi() & 0xff000000) != 0x80000000) return;

// from here on inline code only and only register access

	ram = (int *)RAM1start;
	spifi = (int *)SPIFIsaveRAM1;

	do {
		*ram++ = *spifi++;
	} while (ram < (int *)(RAM1start + RAM1size));
		
	ram = (int *)RAM2start;
	spifi = (int *)SPIFIsaveRAM2;

	do {
		*ram++ = *spifi++;
	} while (ram < (int *)(RAM2start + RAM2size));
	
	ram = (int *)RAM3start;
	spifi = (int *)SPIFIsaveRAM3;

	do {
		*ram++ = *spifi++;
	} while (ram < (int *)(RAM3start + RAM3size));

	// yes the next routine writes to the stack, but it never returns

	gotoR0 (((*(int *)0x80000004) & 0x0000FFFF) | codeSTART );	// this only works if the spifi image and ram image are the same for low memory which should be true.
#endif	
}
/**********************************************************************
 ** take a snapshot of RAM and move to Flash
 **
 ** use Keil to set PC to takeSnapshot routine and run
 **********************************************************************/

SPIFIobj obj;
SPIFI_RTNS * pSpifi;
SPIFIopers opers; 

#ifdef USE_SPIFI_LIB
extern SPIFI_RTNS spifi_table;
#endif	/* USE_SPIFI_LIB */

void copyRAMblock(int start, int SPIFIaddr, int size) {
	int i;
	
	opers.length = PROG_SIZE;
	opers.scratch = NULL;
	opers.protect = 0;
	opers.options = S_CALLER_ERASE;
	for ( i = 0 ; i < size / PROG_SIZE; i++ )
	{
        /* Write */
	    opers.dest = (char *)( SPIFIaddr + (i*PROG_SIZE) );
	    if (pSpifi->spifi_program (&obj, (char *)(start + (i*PROG_SIZE)), &opers)) 
			while (1);	 	// crash and burn on fail
	    /* Verify */
//	    if (memcmp((void *)SPIFIaddr,(void *)(start + (i*PROG_SIZE)),PROG_SIZE)) printf("a few expected miscompare %x\n",start + (i*PROG_SIZE));
	}	
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/**********************************************************************
 ** take a snapshot of RAM and move to Flash
 **
 ** use Keil to set PC to takeSnapshot routine and run from there, which copies the RAM contents to SPIFI
 **********************************************************************/
//  currently Keil shows a couple illegal accesses
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

void takeSnapshot(void) {
	int i;
	
	SystemInit(); 	// to take a SPIFI snapshot use Keil to set the PC here -- eventually I'll add a call to the breakpoint()-- allow boot RAM selection

	ClockInit();

	UART_init();  	// call to brucee code -- code in breakpoint
	
#ifdef USE_SPIFI_LIB
    pSpifi = &spifi_table;
#else
    pSpifi = (SPIFI_RTNS *)(SPIFI_ROM_TABLE);
#endif

	printf("Initializing SPIFI driver...");
	/* Initialize SPIFI driver */
	if (pSpifi->spifi_init(&obj, 3, S_RCVCLK | S_FULLCLK, 12)) while (1);
		
	/* set SPIFI clock */
	LPC_CGU->BASE_SPIFI0_CLK = 1<<24 | 1<<11; /* IRC 12 MHz is good enough for us */

	/* set up SPIFI I/O (undocumented bit 7 set as 1, Aug 2 2011) */
	LPC_SCU->SFSP3_3 = 0xF3; /* high drive for SCLK */
	/* IO pins */
	LPC_SCU->SFSP3_4=LPC_SCU->SFSP3_5=LPC_SCU->SFSP3_6=LPC_SCU->SFSP3_7 = 0xD3;
	LPC_SCU->SFSP3_8 = 0x13; /* CS doesn't need feedback */

	printf("OK\r\nErasing QSPI device...");
		/* Erase Entire SPIFI Device if required */
	for ( i = 0 ; i < obj.memSize / 4; i+=4 )
	{
	    if ( *((uint32_t *)(obj.base+i)) != 0xFFFFFFFF )
		{
		    opers.dest = (char *)(obj.base);
		    opers.length = obj.memSize;
      	    opers.scratch = NULL;
			opers.options = S_VERIFY_ERASE;
		    /* Erase Device */
			if (pSpifi->spifi_erase(&obj, &opers)) while (1);
			break;
		}
	}

	printf("OK\r\nProgramming + verifying QSPI device...");

	copyRAMblock(RAM1start, SPIFIsaveRAM1, RAM1size);
	copyRAMblock(RAM2start, SPIFIsaveRAM2, RAM2size);
	copyRAMblock(RAM3start, SPIFIsaveRAM3, RAM3size);
#if RAM4start != 0	
	copyRAMblock(RAM4start, SPIFIsaveRAM4, RAM4size);		// M0 memory
#endif	


	// copy code block vectors but change start address to SPIFI
	i = *(int *) (codeSTART + 4);
	*(int *) (codeSTART + 4) = (i & 0x0000ffff) | 0x80000000;	// reset vector in SPIFI

#if (codeSTART & 0xFFFF0000) == RAM1start
	copyRAMblock(RAM1start, SPIFIsaveCODE, RAM1size);			
#elif (codeSTART & 0xFFFF0000) == RAM2start
	copyRAMblock(RAM2start, SPIFIsaveCODE, RAM2size);		
#elif (codeSTART & 0xFFFF0000) == RAM3start
	copyRAMblock(RAM3start, SPIFIsaveCODE, RAM3size);	
#else
  #error  --  boot code not supported in this space		
#endif
	printf("SPIFI programmed OK!\r\n");

	
	while(1) {
		
		breakpoint();	
	}		
	
}


#ifndef __UART_H		// if not included then I need to do this myself
int getchar(void)
{
	if (LPC_USART0->LSR & 1) return LPC_USART0->RBR;
	else return -1;
}
int putchar(int ch) {
	while ((LPC_USART0->LSR & UART_LS_TX_EMPTY_ALL) == 0);
  	LPC_USART0->THR = (char)ch;

  	return ch;	
}
#endif

int outbyte(int);

int puts(char *s) {
	char lastc = 0;
	
	while (*s) {
		lastc = outbyte(*s);
		s++;
	}
	return lastc;
}

static void printchar(char **str, int c)
{
//	extern int outbyte(int c);
	if (str) {
		**str = c;
		++(*str);
	}
	else (void)outbyte(c);
}

#define PAD_RIGHT 1
#define PAD_ZERO 2

static int prints(char **out, const char *string, int width, int pad)
{
	register int pc = 0, padchar = ' ';

	if (width > 0) {
		register int len = 0;
		register const char *ptr;
		for (ptr = string; *ptr; ++ptr) ++len;
		if (len >= width) width = 0;
		else width -= len;
		if (pad & PAD_ZERO) padchar = '0';
	}
	if (!(pad & PAD_RIGHT)) {
		for ( ; width > 0; --width) {
			printchar (out, padchar);
			++pc;
		}
	}
	for ( ; *string ; ++string) {
		printchar (out, *string);
		++pc;
	}
	for ( ; width > 0; --width) {
		printchar (out, padchar);
		++pc;
	}

	return pc;
}

/* the following should be enough for 32 bit int */
#define PRINT_BUF_LEN 12

static int printi(char **out, int i, int b, int sg, int width, int pad, int letbase)
{
	char print_buf[PRINT_BUF_LEN];
	register char *s;
	register int t, neg = 0, pc = 0;
#ifdef FOR_BASIC	
	register int u = i;
#else	
	register unsigned int u = i;
#endif

	if (i == 0) {
		print_buf[0] = '0';
		print_buf[1] = '\0';
		return prints (out, print_buf, width, pad);
	}

	if (sg && b == 10 && i < 0) {
		neg = 1;
		u = -i;
	}

	s = print_buf + PRINT_BUF_LEN-1;
	*s = '\0';

	if (b == 16) {
		t = u & 0xF;
		if( t >= 10 )
			t += letbase - '0' - 10;
		*--s = t + '0';
		u = (u >> 4) & 0x0FFFFFFF;
	}
	while (u) {
		t = u % b;
		if( t >= 10 )
			t += letbase - '0' - 10;
		*--s = t + '0';
		u /= b;
	}

	if (neg) {
		if( width && (pad & PAD_ZERO) ) {
			printchar (out, '-');
			++pc;
			--width;
		}
		else {
			*--s = '-';
		}
	}

	return pc + prints (out, s, width, pad);
}

//static int print(char **out, int *varg)
int f_sprintf(char **out, int *varg)		// need to use it in BASIC
{
	register int width, pad;
	register int pc = 0;
	register char *format = (char *)(*varg++);
	char scr[2];

	for (; *format != 0; ++format) {
		if (*format == '%') {
			++format;
			width = pad = 0;
			if (*format == '\0') break;
			if (*format == '%') goto outc;
			if (*format == '-') {
				++format;
				pad = PAD_RIGHT;
			}
			while (*format == '0') {
				++format;
				pad |= PAD_ZERO;
			}
			for ( ; *format >= '0' && *format <= '9'; ++format) {
				width *= 10;
				width += *format - '0';
			}
			if( *format == 's' ) {
				register char *s = *((char **)varg++);
				pc += prints (out, s?s:"(null)", width, pad);
				continue;
			}
			if( *format == 'd' ) {
				pc += printi (out, *varg++, 10, 1, width, pad, 'a');
				continue;
			}
			if( *format == 'x' ) {
				pc += printi (out, *varg++, 16, 0, width, pad, 'a');
				continue;
			}
			if( *format == 'X' ) {
				pc += printi (out, *varg++, 16, 0, width, pad, 'A');
				continue;
			}
#ifndef FOR_BASIC			
			if( *format == 'u' ) {
				pc += printi (out, *varg++, 10, 0, width, pad, 'a');
				continue;
			}
#endif			
			if( *format == 'c' ) {
				/* char are converted to int then pushed on the stack */
				scr[0] = *varg++;
				scr[1] = '\0';
				pc += prints (out, scr, width, pad);
				continue;
			}
		}
		else {
		outc:
			printchar (out, *format);
			++pc;
		}
	}
	if (out) **out = '\0';
	return pc;
}

/* assuming sizeof(void *) == sizeof(int) */

int printf(const char *format, ...)
{
	register int *varg = (int *)(&format);
	return f_sprintf(0, varg);
}

int sprintf(char *out, const char *format, ...)
{
	register int *varg = (int *)(&format);
	return f_sprintf(&out, varg);
}

char line[80];

int hexdigit(char c) {
	if (c <= '9')	return (c - '0');
	c = c & 0x4F;			// crude upshift
	return (c - 'A' + 10);
}

void helpscreen(void) {
	printf("d D or @xxxx [yy] - dump memory at hex xxxx for yy words\n");
	printf("w W or !xxxx yyy  - write yyy to memory at hex xxxx\n");
	printf("i or I xx         - read I2C reg -- fixed address\n");
	printf("J or J xx yy      - write yy to I2C reg xx\n");
//	printf("g G or ^          - return to the program\n");
}

/////////////////////////////////////////////////////////////////////////////////
//
//	read a line -- handle backspace, tab
//
int getline(char *line, int max_len) {
	char c;
	int i, linelen; // , lineTruncated = 0;
		
	c = '\0';
	linelen = 0;

	while( 1 ) {
		//hang here waiting for a character
		i = -1;
		while(i == -1){
			i = getchar();
		}
		c = i & 0xFF;//safety
		if ((c == LF)||(c == CR)) {				// end on either CR or LF
			if (linelen) break;  // something to return
			continue;			 // must be a LF following a CR
		}
		
		if (c == BACKSPACE){
			linelen--;
			continue;
		} else if (c == TAB) {
			c = ' ';
		} 
		
		line[linelen++] = c;
		if(linelen >= max_len) {
			linelen--;
//			lineTruncated = 1;
		}
	}
	line[linelen] = '\0';

	return linelen;
}	

int debug_get_a, debug_get_b;		// these values set by scanning the input line

/////////////////////////////////////////////////////////////////////////////////
//
//	parse the line for upto 2 input values
//

int  get2values(char *line) {
	int i;
	
	if (line[1] == 0) return -1;	// no values
	debug_get_a = debug_get_b = 0;
	i = 1;
	while (line[i] == ' ') i++;		// skip spaces
	if (line[i] == 0) return -1;	// no values
	
	while (line[i]>' ') debug_get_a = (debug_get_a<<4) + hexdigit(line[i++]);
		
	if (line[i] == 0) return 0;		// one values

	while (line[i] == ' ') i++;		// skip spaces
	while (line[i]>' ') debug_get_b = (debug_get_b<<4) + hexdigit(line[i++]);
		
	return 1;
}

/////////////////////////////////////////////////////////////////////////////////
//
// dump a block of memory
//
void dumpmem(char *line) {
	int i;

	get2values(line);
	debug_get_a &= 0xfffffffc;		// word allign it

	for (i=0; i<(debug_get_b==0? 32: (debug_get_b<<2)-1); i += 4) {
		if ((i==0) || ((((debug_get_a + i)) & 0x1f) == 0)) printf("\n%08x:  ",debug_get_a+i);
		printf("%08X ", *(int *)(debug_get_a+i));		// no unsigned printf in BASIC
	}
	putchar('\n');

	if (debug_get_b) debug_get_a += (debug_get_b<<2);
}
  
/////////////////////////////////////////////////////////////////////////////////
//
//	change 1 word location
//
void changemem(char *line) {

	if (get2values(line) > 0) {
	
		// OK now write it out
	
		*(int *)debug_get_a = debug_get_b;
		putchar('\n');
		debug_get_b = 0;			// leave the start pointer alone, but clear this so dump works where it left off
	}
}


int hexdig2num(char c) {
	if (c <= '9')	return (c - '0');
	c = c & 0x4F;			// crude upshift
	return (c - 'A' + 10);
}

  

/////////////////////////////////////////////////////////////////////////////////
//
// a simple monitor, not much but pretty useful
//


void breakpoint(void) {
	int showhelp, i, cnt, start, block=0, stay_here=0;
	
	if ((LPC_USART0->LSR & 1) == 0) return;		// wait for a character in the buffer
	
	showhelp=1;
	
	do {		  		// fall through here so it looks "live"
		getline(line, 80);
		if( line[0] == '@' || line[0] == 'd' || line[0] == 'D')	     dumpmem(line);
		else if (line[0] == '!' || line[0] == 'w' || line[0] == 'W') changemem(line);
		else if (line[0] == '?') helpscreen();
		else if (line[0] == 0) 	 printf("\n");
		else if (line[0] == '^' || line[0] == 'g' || line[0] == 'G') stay_here = 0; //break;	// restart

// manipulate Flash buffer and read/write NAND flash
		
		else if( line[0] == '#')	{
			stay_here = 1;	// get ready for intel hex
			printf("ready to receive hex file\n");
		}

		else if( line[0] == ':')	{ 				// accept intel hex to load into 0x10000000
			stay_here = 1;
		
			cnt = 	(hexdig2num (line[1]) << 4)  + hexdig2num (line[2]);
			start = (hexdig2num (line[3]) << 12) +(hexdig2num (line[4]) << 8) + (hexdig2num (line[5]) << 4) +	hexdig2num (line[6]);

			if (cnt == 0) {
				block = start << 4;
				if (start == 0)	{
					printf("\ndone!\ntype g to display\n");
				}
			} else {
				for (i=0; i <= cnt; i++) {
					*(char *) (0x10000000 + block + start + i) = (hexdig2num (line[(i<<1) + 9]) <<  4) +	hexdig2num (line[(i<<1)+10]);
				}
			}
			printf(".");
		}



		
#ifdef DEBUG_I2C		
		else if (line[0] == 'i') {
			get2values(line);
				
			printf("\n\ni2c read %x := %x\n", debug_get_a, I2C_Read(I2C_SDA_PIN, I2C_SCL_PIN , I2C_SLAVE_ADDR, debug_get_a));
		}
		else if (line[0] == 'j') {
			get2values(line);
			I2C_Write(I2C_SDA_PIN, I2C_SCL_PIN, I2C_SLAVE_ADDR, debug_get_a, debug_get_b);
			
			printf("\n\ni2c write %x <= %x\n",debug_get_a,debug_get_b);
		}
		else if (line[0] == 'k') {

			printf("\n\nlooping on i2c...\n");
			while ((LPC_USART0->LSR & 1) == 0) {		// wait for a character in the buffer
				I2C_Read(I2C_SDA_PIN, I2C_SCL_PIN, I2C_SLAVE_ADDR, 0x0);
				WAITmicro(20000);
			}
			getchar();		// bit bucket the character
			printf("\n...done\n");
		}
#endif
		else {
			if (showhelp) {helpscreen();  showhelp = 0;}
			else showhelp = 1;
		}
	} while (stay_here);
}

void TIMER0_init(void) {

	LPC_TIMER0->PR = M4Frequency / 1000000;	// 12 MHz into TIMER0
	LPC_TIMER0->TCR = 1;					// turn it on
}

#define BAUD_RATE	125000

void UART_init(void) {
	int bauddiv;

	LPC_USART0->FDR = 0x10;	 	// boot ROM can change this  autobaud and fract divider were on
	LPC_USART0->ACR = 0x00;	 	// boot ROM can change this 

	SetClock(BASE_UART0_CLK, SRC_XTAL, DIV1);	// 

//	scu_pinmux(0xc,13,MD_EZI | MD_PUP, FUNC2);	  	//UART1
//	scu_pinmux(0xc,14,MD_EZI | MD_PUP, FUNC2);
//	scu_pinmux(0x2,0,MD_EZI | MD_PUP, FUNC1);	  	//UART0
//	scu_pinmux(0x2,1,MD_EZI | MD_PUP, FUNC1);
	LPC_SCU->SFSP2_0 = GPIO_PIN_SET | 1;				// function 1 -- UART0
	LPC_SCU->SFSP2_1 = GPIO_PIN_SET | 1;				// function 1 -- UART0
	
	LPC_USART0->IER = 0x00;                       // disable all interrupts

	bauddiv = ((XtalFrequency  / BAUD_RATE) + 7) >> 4;		// add 7 to round the result/16	  -- when running off 14.576 clock after ISP
	
	// set the baudrate
	LPC_USART0->LCR |= UART_LCR_DLAB_EN;         // select divisor latches 
	LPC_USART0->DLL = bauddiv;        				// set for baud low byte
	LPC_USART0->DLM = bauddiv >> 8;    				// set for baud high byte
	
	// set the number of characters and other
	// user specified operating parameters
	
	LPC_USART0->LCR = ((UART_CFG_DATABIT_8 | UART_CFG_STOPBIT_1 | UART_CFG_PARITY_NONE) & ~UART_LCR_DLAB_EN);
	LPC_USART0->FCR = UART_FCR_FIFO_EN | UART_CFG_FIFOTRG_8;		

	printf("hit a key to enter debugger\n");
	
}			
