/****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/

#include "LPC8xx.h"
#include "sct.h"

static volatile uint32_t IRQCount = 0;
#define IR_ERR 0
void SCT_IRQHandler(void)
{
	IRQCount--;
	if(IRQCount == 0) 
	{
		LPC_SCT->CTRL_L    |= (1 << 2);
		//NVIC_DisableIRQ(SCT_IRQn);
	}
	LPC_SCT->EVFLAG = (1 << 0);
}

void SCT_Init(void)
{
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 8);                 // enable the SCT clock

    LPC_SCT->CONFIG |= (1 << 17);                          // two 16-bit timers, auto limit
    LPC_SCT->CTRL_L |= (SystemCoreClock/12000000-1) << 5;   // set prescaler, SCT clock = 12 MHz
                                              
    LPC_SCT->MATCH[0].L    = 332;                         // match 0 @ 20/1MHz = 20 usec (50 KHz PWM freq)
    LPC_SCT->MATCHREL[0].L = 332;
    LPC_SCT->MATCH[1].L    = 220;                            // match 1 used for duty cycle (in 20 steps)
    LPC_SCT->MATCHREL[1].L = 220;

    LPC_SCT->EVENT[0].STATE = 0x00000001;                  // event 0 only happens in state 0
    LPC_SCT->EVENT[0].CTRL  = (0 << 0)  |                  // MATCHSEL[3:0] = related to match 0
                              (1 << 12) |                  // COMBMODE[13:12] = match condition only
                              (1 << 14) |                  // STATELD[14] = STATEV is loaded into state
                              (0 << 15);                   // STATEV[15] = 0

    LPC_SCT->EVENT[1].STATE = 0x00000001;                  // event 1 only happens in state 0
    LPC_SCT->EVENT[1].CTRL  = (1 << 0)  |                  // MATCHSEL[3:0] = related to match 1
                              (1 << 12) |                  // COMBMODE[13:12] = match condition only
                              (1 << 14) |                  // STATELD[14] = STATEV is loaded into state
                              (0 << 15);                   // STATEV[15] = 0

    //LPC_SCT->OUT[0].CLR = (1 << 0);                        // event 0 will set SCT_OUT_0
    //LPC_SCT->OUT[0].SET = (1 << 1);                        // event 1 will clear SCT_OUT_0
		
		LPC_SCT->EVEN = (1 << 0);
		NVIC_EnableIRQ(SCT_IRQn);                              // enable SCT interrupt
		NVIC_SetPriority(SysTick_IRQn, 1);

    //LPC_SCT->CTRL_L    &= ~(1 << 2);                       // unhalt it by clearing bit 2 of CTRL reg
}

void Start_SCT(uint32_t irqcount)
{
	IRQCount = irqcount;
	//NVIC_EnableIRQ(SCT_IRQn);
	
	LPC_SCT->CTRL_L    &= ~(1 << 2);                       // unhalt it by clearing bit 2 of CTRL reg
}

void Send_Bit_1_Data(void)
{
	Start_SCT(32); // 2t *16/t  (t: RC-6 444.44us)
	while(IRQCount)
	{
		// 1t: '1'
		if(IRQCount > 16)
		{
			LPC_SCT->OUT[0].CLR = (1 << 0);                        // event 0 will set SCT_OUT_0
			LPC_SCT->OUT[0].SET = (1 << 1);                        // event 1 will clear SCT_OUT_0
		}
		// 1t: '0'
		else// if(IRQCount == 16)
		{
			LPC_SCT->OUT[0].SET = (0 << 0);
		}
	}
}

void Send_Bit_0_Data(void)
{
	Start_SCT(32); // 2t *16/t  (t: RC-6 444.44us)
	while(IRQCount)
	{
		// 1t: '1'
		if(IRQCount < 17)
		{
			LPC_SCT->OUT[0].CLR = (1 << 0);                        // event 0 will set SCT_OUT_0
			LPC_SCT->OUT[0].SET = (1 << 1);                        // event 1 will clear SCT_OUT_0
		}
		// 1t: '0'
		else
		{
			LPC_SCT->OUT[0].SET = (0 << 0);
		}
	}
}

void Send_2Bytes_Data(uint16_t data)
{
	uint32_t i;
	for(i = 16; i > 0; i--)
	if(data & (0x01 << (i - 1)))
	{
		Send_Bit_1_Data();
	}
	else
	{
		Send_Bit_0_Data();
	}
}

void Send_Leader(void)
{
	Start_SCT(128); // 8t *16/t  (t: RC-6 444.44us)
	while(IRQCount)
	{
		// 1t: '1'
		if(IRQCount > 32)
		{
			LPC_SCT->OUT[0].CLR = (1 << 0);                        // event 0 will set SCT_OUT_0
			LPC_SCT->OUT[0].SET = (1 << 1);                        // event 1 will clear SCT_OUT_0
		}
		// 1t: '0'
		else// if(IRQCount == 16)
		{
			LPC_SCT->OUT[0].SET = (0 << 0);
		}
	}
}

void Send_SB(void)
{
	Send_Bit_1_Data();
}

void Send_ModeBit(void)
{
	Send_Bit_1_Data();
	Send_Bit_1_Data();
	Send_Bit_0_Data();
}

void Send_TR(void)
{
	Start_SCT(64); // 4t *16/t  (t: RC-6 444.44us)
	while(IRQCount)
	{
		// 1t: '1'
		if(IRQCount < 33)
		{
			LPC_SCT->OUT[0].CLR = (1 << 0);                        // event 0 will set SCT_OUT_0
			LPC_SCT->OUT[0].SET = (1 << 1);                        // event 1 will clear SCT_OUT_0
		}
		// 1t: '0'
		else
		{
			LPC_SCT->OUT[0].SET = (0 << 0);
		}
	}
}

// mode: 6A
void Send_Header(void)
{
	Send_Leader();
	Send_SB();
	Send_ModeBit();
	Send_TR();
}

// LCC: 0x000F
void Send_LCC(void)
{
	Send_2Bytes_Data(0x800F);
}

void Send_SignalFree(void)
{
	Start_SCT(100); // 6t *16/t  (t: RC-6 444.44us)
	LPC_SCT->OUT[0].SET = (0 << 0);
	while(IRQCount);
}

void Send_Data(uint16_t data)
{
	Send_Header();
	Send_LCC();
	Send_2Bytes_Data(data);
	Send_SignalFree();
}

