/***************************************************************************
*  Driver for the I2C hardware interface on the LPC800
*  A transfer consists of one or more linked messages separated by
*  repeated STARTs. To start a transfer call function "I2C_Transfer"
***************************************************************************/
#include "LPC82x.h"                                // LPC8xx definitions
#include "i2c.h"

static   I2C_MSG *msg;                             // ptr to active message block
static   uint8_t  cnt;                             // bytes send/received counter
volatile uint8_t  state;                           // driver state

void I2C_IRQHandler(void)
/***********************/
{
  uint32_t status = LPC_I2C->INTSTAT;

    if (status & (1 << 25))                        // SCL time-out?
//  if (status & (1 << 24))                        // event time-out?
    {
        LPC_I2C->STAT   = (1 << 25);               // clear flag
//        LPC_I2C->STAT   = (1 << 24);             // clear flag
//        LPC_I2C->STAT   = 0x800;                 // clear flag
        LPC_I2C->MSTCTL = 0x00000005;              // generate STOP and continue
        state = I2C_TIME_OUT;
    }
    else if (status & (1 << 6))                    // master start/stop error?
    {
        LPC_I2C->STAT = (1 << 6);
        state = I2C_ERR;
    }
    else if (status & (1 << 0))                    // master pending?
    {
        switch (LPC_I2C->STAT & 0xE)               // only check master states
        {
          case (0 << 1):                           // Idle
            LPC_I2C->INTENCLR = (1 << 0);          // disable master pending interrupt
            break;

/* RX */  case (1 << 1):                           // SLA+R transmitted ACK received or DATA received ACK returned
            msg->buf[cnt++] = LPC_I2C->MSTDAT;     // read data
            if (cnt != msg->nrBytes)               // last data?
            {
                LPC_I2C->MSTCTL = 0x00000001;      // no, master continue
            }
            else if (msg->next != 0)               // yes, any more messages?
            {
                cnt = 0;
                msg = (I2C_MSG *) msg->next;       // yes, point to next message
                LPC_I2C->MSTDAT = msg->address;
                LPC_I2C->MSTCTL = 0x00000003;      // generate (rep)START and continue
            }
            else                                   // ready
            {
                LPC_I2C->MSTCTL = 0x00000005;      // generate STOP and continue
                state = I2C_OK;
            }
            break;
	  
/* TX */  case (2 << 1):                           // SLA+W or DATA transmitted, ACK received
            if (cnt < msg->nrBytes)                // DATA, REP_START or STOP will be transmitted
            {
                LPC_I2C->MSTDAT = msg->buf[cnt++]; // sent byte
                LPC_I2C->MSTCTL = 0x00000001;      // master continue
            }
            else if (msg->next != 0)               // any more messages to send?
            {
                cnt = 0;
                msg = (I2C_MSG *) msg->next;       // next message
                LPC_I2C->MSTDAT = msg->address;
                LPC_I2C->MSTCTL = 0x00000003;      // generate (rep)START and continue
            }
            else                                   // ready
            {
                LPC_I2C->MSTCTL = 0x00000005;      // generate STOP and continue
                state = I2C_OK;
            }
            break;
	  
          case (3 << 1):                           // no ack on address
            LPC_I2C->MSTCTL = 0x00000005;          // generate STOP do not continue
            state = I2C_NACK_ON_ADDRESS;
            break;
	  
          case (4 << 1):                           // no ack on data
            LPC_I2C->MSTCTL = 0x00000005;          // generate STOP and continue
            state = I2C_NACK_ON_DATA;
            break;

          default:
            break;
        }	  
    }
}
	
void I2CMST_Init(void)
/********************/
{
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 5)         // I2C0 clock enable
                               | (1 << 7);        // enable clock to the Switch Matrix
    LPC_SYSCON->PRESETCTRL    |= (1 << 6);         // de-assert I2C0 reset

    LPC_SWM->PINENABLE0       &= ~(1 << 12);       // P0_10 is SCL
    LPC_SWM->PINENABLE0       &= ~(1 << 11);       // P0_11 is SDA

    LPC_I2C->TIMEOUT = 0x0000008F;                 // time-out of 8 x 16 I2C clock counts
  
    LPC_I2C->DIV = (SystemCoreClock / 200000)-1;   // I2C clock = 200000 Hz
    LPC_I2C->MSTTIME = 0;                          // SCL_low = SCL_high = clocks = 10usec (100kb/s)

    NVIC_EnableIRQ(I2C_IRQn);                      // enable I2C interrupt
    LPC_I2C->CFG |= (1<<0) | (1<<3);               // enable master and time-out function
}

uint8_t I2C_Transfer(I2C_MSG *p)
/*******************************
 * Start an I2C transfer, containing 1 or more LINKED messages.
 * The application must leave the message parameter block untouched.
 * The first I2C message is started with sending a START condition.
 ***************************************************************************/
{
//    if (state == I2C_BUSY || !(LPC_I2C->STAT & 1))
//        return I2C_BUSY;

    while (!(LPC_I2C->STAT & 1)) ;                         // wait if (previous) still pending

    msg = p;
    cnt = 0;
    state = I2C_BUSY;                                      // state of the I2C driver: BUSY

    LPC_I2C->MSTDAT   = msg->address;                      // Slave address + R/W
    LPC_I2C->MSTCTL   = 0x00000003;                        // generate START
    LPC_I2C->INTENSET = (1 << 0)                           // enable master pending interrupt
                      | (1 << 6)                           // enable master start/stop error interrupt
//                       (1 << 24) |               // enable event time-out interrupt
                      | (1 << 25);                 // enable SCL time-out interrupt

    while (state == I2C_BUSY) ;                            // wait till ready

    return state;
}
