/****************************************************************************
 *
 * MODULE:             Exposes module for Button driver on DK3 platform
 *
 * COMPONENT:          MatrixButton.h
 *
 * VERSION:            R and D Release 6 JAN 2011
 *
 * REVISION:           1.0
 *
 * DATED:              2014/03/28
 *
 * AUTHOR:             frobbe
 *
 *
 *
 ****************************************************************************
 *
 * This software is owned by Jennic and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on Jennic products. You, and any third parties must reproduce
 * the copyright and warranty notice and any other legend of ownership on each
 * copy or partial copy of the software.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS". JENNIC MAKES NO WARRANTIES, WHETHER
 * EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
 * ACCURACY OR LACK OF NEGLIGENCE. JENNIC SHALL NOT, IN ANY CIRCUMSTANCES,
 * BE LIABLE FOR ANY DAMAGES, INCLUDING, BUT NOT LIMITED TO, SPECIAL,
 * INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR ANY REASON WHATSOEVER.
 *
 * Copyright Jennic Ltd 2010. All rights reserved
 *
 ****************************************************************************/

#ifndef BUTTON_H_
#define BUTTON_H_

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

//#define MULTITOUCH		/* uncomment this line to manage multiple OneShot touches combination */

typedef enum
{
	eDisabled,				/* Button disabled, do not take it into account */
	eOneShot,				/* Button with short or long press function, but no repeat action when kept pressed */
	eRepeat,				/* Button with short function only, repeated if kept pressed */
	eModifier				/* Modifier button only (e.g. "shift"). Not taken into account when pressed alone. MULTITOUCH must be defined */
} teButtonType;



typedef struct
{
	 uint8 code;			/* code associated to the button by the application */
	 uint8 lp_code;			/* code associated to the long pressed button by the application */
	 teButtonType type;		/* type of button */
	 bool_t pressed;		/* set to TRUE by vButton_Update when button is enabled and pressed. Reset to FALSE otherwise */
	 uint16 time_pressed;	/* Reset by vButton_Update when the button is pressed and incremented by 1 each time vButton_Update detects the button is kept pressed */
	 uint16 time_released;	/* Reset by vButton_Update when the button is released and incremented by 1 each time vButton_Update detects the button is kept released */
	 uint8 bank;			/* bank number relative to the lsb of the mask of DIOs used for banks (e.g. if DIO banks mask is 0x0F00 and button is on DIO 8, bank is 0 */
	 uint16 line;			/* line mask of the button in the mask of DIO used for lines. (e.g. if DIO lines mask is 0x00F0 and button is on DIO 4, line is 0x01 */
} tsButtonDesc;



/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

 /****************************************************************************
  *
  * NAME: vMatrixButton_Configure
  *
  * DESCRIPTION:
  * Configures the Button matrix.
  * The matrix is defined as n banks of m lines.
  * Both n and m must be less or equal than 8.
  * Inputs are:
  * NumberOfButtons: number of buttons actually present on the matrix
  * MatrixBanksMask: mask for DIO used to select a bank. DIO must be consecutive.
  *                  The mask will be truncated to 8 bits starting from the first lsb set.
  * MatrixLinesMask: mask for DIO used to select a line. DIO must be consecutive.
  *                  The mask will be truncated to 8 bits starting from the first lsb set.
  *
  * RETURNS:
  * void
  *
  ****************************************************************************/

PUBLIC void vMatrixButton_Configure(uint8 NumberOfButtons, uint32 MatrixBanksMask, uint32 MatrixLinesMask);

/****************************************************************************
 *
 * NAME: vMatrixButton_UnConfigure
 *
 * DESCRIPTION:
 * Disable access to matrix update.
 * Release DIO used by the Button matrix.
 * RAZ pressed, time_pressed and time_released of the Button structure.
  * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vMatrixButton_UnConfigure(volatile tsButtonDesc* Button);

/****************************************************************************
 *
 * NAME: tsMatrixButton_Update
 *
 * DESCRIPTION:
 * Reads the matrix and update Button structure array in accordance
 * Input:
 * Button: pointer to an array of tsButtonDesc type
 * Output:
 * Button array updated.
 *
 * RETURNS:
 * Input pointer
 *
 ****************************************************************************/

PUBLIC volatile tsButtonDesc* tsMatrixButton_Update(volatile tsButtonDesc* Button);

/****************************************************************************/
/***        External Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Inlined Functions                                            ***/
/****************************************************************************/


#endif /*BUTTON_H_*/
