/*
 * @brief UART0 ROM API interrupt example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2013
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "board.h"
#include "string.h"

#ifdef __ICCARM__
const   
#include "user_app_example1_IAR.c"
const	
#include "user_app_example2_IAR.c"
#else
const
#include "user_app_example1_KEIL.c"
const
#include "user_app_example2_KEIL.c"
#endif

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
#define CODE_ADDR1					(0x00002000)
#define CODE_ADDR2					(0x00005000)	
	
#define		DATA_SIZE					(64) 
#define		LAST_DATA_PACK		(0xff) 
	
#define		MAX_RETRY_NUM			(3)

#define	MAX_TIMEOUT_NUM			(200)
#define	BASE_TIME_1MS				(24000)

typedef struct {
	uint8_t  	type;							/*!< message type: UPDATE_REQ: request to update; DATA_PACK: FW data package*/
	uint8_t	 	ctrl_flag;				/*!< control flag: LAST_DATA_PACK: the last data package; others reserved*/
	uint32_t	fragment_num;			/*!< number of fragment data package  */
	uint8_t		retry_num;				/*!< retry number  */
	uint8_t		data[DATA_SIZE];		/*!< data to be transmitted */
}FW_MESSAGE_T;

/* message transfer state */
typedef enum {
	MS_TX_START,
	MS_TX_WAIT,
	MS_FW_REQ_TX_OK,
	MS_FW_DATA_TX_OK,
	MS_TX_ERR,
	MS_TX_ACK,
	MS_TX_ACK_OK,
	MS_TX_FAILED,
}MS_STATUS_T;

FW_MESSAGE_T	message;

/* message type definitions */
#define 	FW_UPDATE_REQ 		(0x10)				/*!< firmware update request*/
#define		FW_DATA_PACK 			(0x11)				/*!< firmware data package*/
#define		FW_RSP_ACK 				(0x12)				/*!< firmware ack*/

/* UART handle and memory for ROM API */
static UART_HANDLE_T *uartHandle;

/* Use a buffer size larger than the expected return value of
   uart_get_mem_size() for the static UART handle type */
static uint32_t uartHandleMEM[0x10];

/* ASCII code for escapre key */
#define ESCKEY 27

/* Flag used to indicate callback fired */
static volatile bool gotCallback;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* UART Pin mux function - note that SystemInit() may already setup your
   pin muxing at system startup */
static void Init_UART0_PinMux(void)
{
#if defined(BOARD_MANLEY_11U68)
	/* UART signals on pins PIO0_18 (FUNC1, U0_TXD) and PIO0_19 (FUNC1, U0_RXD) */
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 18, (IOCON_FUNC1 | IOCON_MODE_INACT | IOCON_DIGMODE_EN));
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 19, (IOCON_FUNC1 | IOCON_MODE_INACT | IOCON_DIGMODE_EN));

#elif defined(BOARD_NXP_LPCXPRESSO_11U68)
	/* UART signals on pins PIO0_18 (FUNC1, U0_TXD) and PIO0_19 (FUNC1, U0_RXD) */
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 18, (IOCON_FUNC1 | IOCON_MODE_INACT | IOCON_DIGMODE_EN));
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 19, (IOCON_FUNC1 | IOCON_MODE_INACT | IOCON_DIGMODE_EN));

#else
	/* Configure your own UART pin muxing here if needed */
#warning "No UART pin muxing defined"
#endif
}

/* Turn on LED to indicate an error */
static void errorUART(void)
{
	Board_LED_Set(0, true);
	while (1) {}
}

/* Setup UART handle and parameters */
static void setupUART()
{
	uint32_t errCode;

	/* 115.2KBPS, 8N1, ASYNC mode, no errors, clock filled in later */
	UART_CONFIG_T cfg = {
		0,				/* U_PCLK frequency in Hz */
		115200,		/* Baud Rate in Hz */
		1,				/* 8N1 */
		0,				/* Asynchronous Mode */
		NO_ERR_EN	/* Enable No Errors */
	};

	/* Initialize UART0 */
	Chip_UART0_Init(LPC_USART0);

	/* Perform a sanity check on the storage allocation */
	if (LPC_UART0D_API->uart_get_mem_size() > sizeof(uartHandleMEM)) {
		/* Example only: this should never happen and probably isn't needed for
		   most UART code. */
		errorUART();
	}

	/* Setup the UART handle */
	uartHandle = LPC_UART0D_API->uart_setup((uint32_t) LPC_USART0, (uint8_t *) &uartHandleMEM);
	if (uartHandle == NULL) {
		errorUART();
	}

	/* Need to tell UART ROM API function the current UART peripheral clock
	     speed */
	cfg.sys_clk_in_hz = Chip_Clock_GetSystemClockRate();

	/* Initialize the UART with the configuration parameters */
	errCode = LPC_UART0D_API->uart_init(uartHandle, &cfg);
	if (errCode != LPC_OK) {
		/* Some type of error handling here */
		errorUART();
	}
}

/* UART callback */
static void waitCallback(uint32_t err_code, uint32_t n)
{
	gotCallback = true;
}
bool time_out;
/* Sleep until callback is called */
void sleepUntilCB(void)
{
//	uint32_t time_cnt, i;
//	/* Wait until the transmit callback occurs. When it hits, the
//	     transfer is complete. */	
//	time_cnt=0;
//	while(time_cnt < MAX_TIMEOUT_NUM) {
//		if(gotCallback == true) {
//			break;
//		}
//		for(i=0;i<BASE_TIME_1MS;i++);
//		time_cnt++;
//	}
//	if(time_cnt == MAX_TIMEOUT_NUM) {
//		time_out = true;
//	}
	while (gotCallback == false) {
		/* Sleep until the callback signals transmit completion */
		__WFI();
		
	}
}

/* Send a string on the UART terminated by a NULL character using
   interrupt mode. */
static void putLineUART(const char *send_data, uint32_t size)
{
	UART_PARAM_T param;
	
	param.buffer = (uint8_t *) send_data;
	param.size = size;

	/* Interrupt mode, do not append CR/LF to sent data */
	param.transfer_mode = TX_MODE_BUF_EMPTY;
	param.driver_mode = DRIVER_MODE_INTERRUPT;

	/* Setup the transmit callback, this will get called when the
	   transfer is complete */
	param.callback_func_pt = (UART_CALLBK_T) waitCallback;

	/* Transmit the data using interrupt mode, the function will
	   return */
	time_out = false;
	gotCallback = false;
	if (LPC_UART0D_API->uart_put_line(uartHandle, &param)) {
		errorUART();
	}

	/* Wait until the transmit callback occurs. When it hits, the
	   transfer is complete. */
	sleepUntilCB();
}

/* Receive a string on the UART terminated by a LF character using
   polling mode. */
static void getLineUART(char *receive_buffer, uint32_t length)
{
	UART_PARAM_T param;

	param.buffer = (uint8_t *) receive_buffer;
	param.size = length;

	/* Receive data up to the CR/LF character in polling mode. Will
	   truncate at length if too long.	*/
	param.transfer_mode = RX_MODE_BUF_FULL;
	param.driver_mode = DRIVER_MODE_INTERRUPT;

	/* Setup the receive callback, this will get called when the
	   transfer is complete */
	param.callback_func_pt = (UART_CALLBK_T) waitCallback;

	/* Receive the data */
	time_out = false;
	gotCallback = false;
	if (LPC_UART0D_API->uart_get_line(uartHandle, &param)) {
		errorUART();
	}

	/* Wait until the transmit callback occurs. When it hits, the
	   transfer is complete. */
	sleepUntilCB();
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/**
 * @brief	Handle UART interrupt
 * @return	Nothing
 */
// void USART0_IRQHandler(void)
void USART0_IRQHandler(void)
{
	LPC_UART0D_API->uart_isr(uartHandle);
}

void delay(uint32_t n)
{
	uint32_t i;
	
	while(n) {
		for(i=0;i<BASE_TIME_1MS;i++);
		n--;
	}
}
/**
 * @brief	Main UART program body
 * @return	Always returns 1
 */
void UART_put_message(uint8_t *ms_status)
{
	uint8_t status;

	status = *ms_status;
	switch (status) {
		case MS_TX_START:
			putLineUART((char *)&message, sizeof(message));	
			*ms_status = MS_TX_WAIT;				
		break;
		
		case MS_TX_WAIT:
			if(gotCallback == true) {
				switch (message.type) {
				case FW_UPDATE_REQ:
					*ms_status = MS_FW_REQ_TX_OK;
				break;
				case FW_DATA_PACK:
					*ms_status = MS_FW_DATA_TX_OK;
				break;
				default: 
					*ms_status = MS_TX_START;
				break;
				}
			} else {
				if(time_out == true) {
					*ms_status = MS_TX_ERR;
				}
			}
		break;
		
		case MS_FW_REQ_TX_OK:
		case MS_FW_DATA_TX_OK:
			getLineUART((char *)&message, sizeof(message));//get ack
			*ms_status = MS_TX_ACK;
			
		break;
		
		case MS_TX_ACK:
			if(gotCallback == true) {
				if(message.type!=FW_RSP_ACK) {
					message.retry_num++;
					if(message.retry_num == MAX_RETRY_NUM) {
						*ms_status = MS_TX_ERR;
					}
				} else {
					*ms_status = MS_TX_ACK_OK;
				}
			}
		break;
			
		case MS_TX_ACK_OK:
			*ms_status = MS_TX_START;
		break;
		
		case MS_TX_ERR:
			*ms_status = MS_TX_FAILED;	
		break;
			
		default:
			
		break;
	}
}

int main(void)
{
	static uint8_t ms_status;
	static uint32_t size, i;
	uint8_t *fw_p;
	uint32_t fw_s;
	uint32_t code_addr;
	
	SystemCoreClockUpdate();
	Board_Init();
	Init_UART0_PinMux();
	Board_LED_Set(0, true);

	/* Allocate UART handle, setup UART parameters, and initialize UART
	   clocking */
	setupUART();

	/* Enable the IRQ for the UART */
	NVIC_EnableIRQ(USART0_IRQn);
	
	ms_status = MS_TX_START;
	memset(&message, 0xff, sizeof(message));
	message.ctrl_flag = 0;		
	message.type = FW_UPDATE_REQ;
	while(1) {
		//send update request
		UART_put_message(&ms_status);
		
		//send data package
		if(ms_status == MS_TX_ACK_OK) 
		{
			memcpy((uint8_t *)&code_addr, message.data,sizeof(uint32_t));
			if(code_addr == CODE_ADDR1) { //if address is CODE_ADDR1
				fw_p = (uint8_t *)LR1;
				fw_s = sizeof(LR1);
			} else {
				fw_p = (uint8_t *)LR0;
				fw_s = sizeof(LR0);
			}
			memset(&message, 0xff, sizeof(message));
			message.type = FW_DATA_PACK;
			message.ctrl_flag = 0;	
			i = 0;
			size = DATA_SIZE;
			while(i<fw_s) {
				if((i+DATA_SIZE)>fw_s) {
					memset(&message.data, 0xff, DATA_SIZE);
					size = fw_s - i;
					message.ctrl_flag = LAST_DATA_PACK;
				}
				memcpy(&message.data, fw_p+i, size); 
				UART_put_message(&ms_status);
				if(ms_status == MS_TX_ACK_OK) {
					message.type = FW_DATA_PACK;
					i += DATA_SIZE;
					delay(10);
				} else if (ms_status == MS_TX_FAILED) {
					break; //exit data package TX
				}
			}
		}
	}
//	return 1;
}
