/*
 * @brief FLASH IAP programming & FLASH signature example using IAP commands
 * to write to FLASH memory and a FLASH signature generator
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2013
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"
#include <string.h>
#include <stdio.h>
#include <cr_section_macros.h>

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
/* IAP infomation */
//#define CODE_ADDR1					(0x00001000)
//#define CODE_ADDR2					(0x00004800)
//#define FIRMWARE_START			(0x10000000) 
//#define	VECT_SIZE						(0x000000C0)

//#define FW_BOOT_INFO_ADDR		(0x0F40)
//#define FW_CHECK_INFO_ADDR	(0x0F80)
#define CODE_ADDR1					(0x00002000)
#define CODE_ADDR2					(0x00005000)
#define FIRMWARE_START			(0x10000000) 
#define	VECT_SIZE						(0x000000C0)

#define FW_BOOT_INFO_ADDR		(0x1F40)
#define FW_CHECK_INFO_ADDR	(0x1F80)
#define	PAGE_BYTE						(64)
#define SECT_BYTE						(1024)
#define CHECK_INFO_PAGE			(FW_CHECK_INFO_ADDR/PAGE_BYTE)
#define CHECK_INFO_SECT			(FW_CHECK_INFO_ADDR/SECT_BYTE)
#define BOOT_INFO_PAGE			(FW_BOOT_INFO_ADDR/PAGE_BYTE)
#define BOOT_INFO_SECT			(FW_BOOT_INFO_ADDR/SECT_BYTE)


#define BLANK_VALUE_32B			(0xFFFFFFFF)
#define BLANK_VALUE_16B			(0xFFFF)

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
typedef struct {
	uint32_t  cur_addr;
	uint32_t  new_addr;
} FW_BOOT_INFO_T;

typedef struct {
	uint32_t  new_addr;
	uint16_t	start_sect;				/*!< */
	uint16_t  end_sect;				/*!<  */
	uint32_t  signatrue_val;				/*!<  */
} FW_CHECK_INFO_T;
__RODATA(FLASH_INFO1) FW_BOOT_INFO_T  RO_boot_addr = {CODE_ADDR1, CODE_ADDR2};
__RODATA(FLASH_INFO2) FW_CHECK_INFO_T boot_chk_info = {BLANK_VALUE_32B, BLANK_VALUE_16B, BLANK_VALUE_16B, BLANK_VALUE_32B};

FW_BOOT_INFO_T RW_boot_addr;
	
/*****************************************************************************
 * Private functions
 ****************************************************************************/
void remap_vectors(uint32_t addr)
{
    uint32_t* src,*dst;
    int32_t size;

    src = (uint32_t*)addr;
    dst = (uint32_t*)FIRMWARE_START;
    size = VECT_SIZE;

    while(size > 0)
    {
        *dst++ = *src++;
        size -= 4;
    }
		LPC_SYSCON->SYSMEMREMAP = 0x1;    /* remap to internal RAM */
}
typedef  void (*pFunction)(void);
pFunction Jump_To_Application;
static uint32_t JumpAddress;
static uint32_t *spp,spaddress;

void execute_new_firmware(uint32_t addr)
{
	remap_vectors(addr);
	JumpAddress = *(uint32_t*) (FIRMWARE_START + 4);
	/* Jump to user application */
	Jump_To_Application = (pFunction) JumpAddress;
	__set_MSP(0x10002000);
	Jump_To_Application();
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/
/**
 * @brief	Main program body
 * @return	Always returns 0
 */
uint8_t verify_fw(void)
{
	uint32_t block_num;
	
	block_num = boot_chk_info.end_sect - boot_chk_info.start_sect + 1;
	/* Start the signature generator for the f/w */
	Chip_FMC_ComputeSignatureBlocks(boot_chk_info.new_addr, (block_num / 16));

	/* Check for signature geenration completion */
	while (Chip_FMC_IsSignatureBusy()) {}

	if(boot_chk_info.signatrue_val == Chip_FMC_GetSignature(0)){//check the signature
		return 0;
	} else {
		return 1;
	}
}
uint8_t update_boot_info(void)
{
	uint8_t ret_code;
	uint32_t buf[PAGE_BYTE/4];
		
	memset(buf, 0xff, PAGE_BYTE);
	memcpy(buf, (uint8_t *)&RW_boot_addr, sizeof(RW_boot_addr));
	
	/* Disable interrupt mode so it doesn't fire during FLASH updates */
	__disable_irq();
	
	/* Note: to be reliable, must firstly update current f/w start address and then erase the new f/w boot infomation*/
	/* update current f/w start address */
	ret_code = Chip_IAP_PreSectorForReadWrite(BOOT_INFO_SECT, BOOT_INFO_SECT);
	if (ret_code != IAP_CMD_SUCCESS) {
		return ret_code;
	}
	/* Erase page */
  ret_code = Chip_IAP_ErasePage(BOOT_INFO_PAGE, BOOT_INFO_PAGE);
	if (ret_code != IAP_CMD_SUCCESS) {
		return ret_code;
	}
	
	/* Prepare to write/erase the sector */
	ret_code = Chip_IAP_PreSectorForReadWrite(BOOT_INFO_SECT, BOOT_INFO_SECT);
	if (ret_code != IAP_CMD_SUCCESS) {
		return ret_code;
	}

	/* Write to the page */
	ret_code = Chip_IAP_CopyRamToFlash(FW_BOOT_INFO_ADDR, buf, PAGE_BYTE);
	if (ret_code != IAP_CMD_SUCCESS) {
		return ret_code;
	}
	
	__enable_irq();
	
	return 0;
}

uint8_t update_chk_info(void)
{
	uint8_t ret_code;
	
	__disable_irq();
	/* Prepare to write/erase new f/w boot infomation */
	ret_code = Chip_IAP_PreSectorForReadWrite(CHECK_INFO_SECT, CHECK_INFO_SECT);
	if (ret_code != IAP_CMD_SUCCESS) {
		return ret_code;
	}
	/* Erase the new f/w boot infomation page */
  ret_code = Chip_IAP_ErasePage(CHECK_INFO_PAGE, CHECK_INFO_PAGE);
	if (ret_code != IAP_CMD_SUCCESS) {
		return ret_code;
	}	
	__enable_irq();
	
	return 0;
}
int main(void)
{
	uint8_t err;
	uint32_t fw_addr;
	
	/* Generic Initialization */
	SystemCoreClockUpdate();
	Board_Init();
	
	Board_LED_Set(0, true);
	
	while(1) {
		if(boot_chk_info.new_addr!=BLANK_VALUE_32B) {	//new firmware updated
			/* verify the new firmware */
			err = verify_fw();
			if(!err) {	//ready for new f/w execution
				if(RO_boot_addr.cur_addr != boot_chk_info.new_addr) {
//					fw_addr = boot_chk_info.new_addr;
					RW_boot_addr.new_addr = RO_boot_addr.cur_addr;
					RW_boot_addr.cur_addr = boot_chk_info.new_addr;
					/* NOTE: must firstly update f/w boot info then check info!!!*/
					update_boot_info();
				} 
				update_chk_info();
			}
		}			
		fw_addr = RO_boot_addr.cur_addr;
		/* execute firmware*/
		execute_new_firmware(fw_addr);
	}
}
