/*
 * @brief Wakeup example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2015
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"

static int curr_wakeup_method;
/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/* GPIO pin for PININT input */
#if defined(BOARD_NXP_LPCXPRESSO_54114)
#define GPIO_PININT_PIN     24						/* GPIO pin number mapped to PININT */
#define GPIO_PININT_PORT    0						/* GPIO port number mapped to PININT */
#define GPIO_PININT_INDEX   PININTSELECT0			/* PININT index used for GPIO mapping */
#define PININT_IRQ_HANDLER  PIN_INT0_IRQHandler		/* GPIO interrupt IRQ function name */
#define PININT_NVIC_NAME    PIN_INT0_IRQn			/* GPIO interrupt NVIC interrupt name */

#else
#error "Board not supported - setup GPIO for pin interrupt"
#endif


static const PINMUX_GRP_T pinmuxing[] = {

	{0, 0,  (IOCON_MODE_INACT | IOCON_FUNC1 | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF)},
	{0, 1,  (IOCON_MODE_INACT | IOCON_FUNC1 | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF)},
	{1, 6,  (IOCON_FUNC2 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 23, (IOCON_FUNC0 | IOCON_GPIO_MODE | IOCON_DIGITAL_EN)},
	{0, 24, (IOCON_FUNC0 | IOCON_GPIO_MODE | IOCON_DIGITAL_EN)},
	{0, 25, (IOCON_FUNC0 | IOCON_GPIO_MODE | IOCON_DIGITAL_EN)},
	{0, 26, (IOCON_FUNC0 | IOCON_GPIO_MODE | IOCON_DIGITAL_EN)},
	{0, 11, (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 12, (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 13, (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 14, (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 4,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 2,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 3,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 5,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 6,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 7,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 8,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},
	{0, 9,  (IOCON_FUNC0 | IOCON_MODE_PULLDOWN | IOCON_DIGITAL_EN)},

};

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
void periph_disable(){

#define PDRUNCFG1_PD_ALT_FLASH_IBG  (1UL << 28)
#define PDRUNCFG1_SEL_ALT_FLASH_IBG (1UL << 29)
#define PDRUNCFG_PD_FLASH_BG      (1 << 25)
#define PDRUNCFG_PD_VDDHV_ENA     (1 << 18)
  /* Use the alternative flash bandgap and turn off the original one*/
	LPC_SYSCON->PDRUNCFGCLR[1] = PDRUNCFG1_PD_ALT_FLASH_IBG;
  LPC_SYSCON->PDRUNCFGSET[1] = PDRUNCFG1_SEL_ALT_FLASH_IBG;
  Chip_SYSCON_PowerDown(PDRUNCFG_PD_FLASH_BG);

	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLASH);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SPIFI);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_INPUTMUX);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_IOCON);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_GPIO0);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_GPIO1);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_PINT);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_DMA);					
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_CRC);							
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_WWDT);								
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_RTC);								
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_MAILBOX);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_ADC0);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_MRT);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SCT0);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_UTICK);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM1);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM2);
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM3);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM4);	
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM5);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM6);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM7);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_DMIC);					
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER2);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_USB);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER0);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER1);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER2);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER3);			
	Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SYS_PLL); 
	Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_WDT_OSC);



Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SRAM0);
Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SRAM1);			
Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SRAM2);			
Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SRAMX);
Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SRAM1);
Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SRAM2);
Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SRAMX);

#ifdef RAM_SRAM0_SRAM0 
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FMC);							
	Chip_SYSCON_PowerDown(PDRUNCFG_PD_VDDHV_ENA);
#endif

	Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SRAM0);   
}

static POWER_MODE_T curr_pwr;
/* Get user option from UART */
static void get_option(void) {
	int ch = EOF;
	DEBUGSTR(
		"Select an option\r\n"
		"\t1. Sleep mode\r\n"
		"\t2. Deep Sleep mode\r\n"
		"\t3. Deep power down mode\r\n");
	while (1) {
		while ((ch = Board_UARTGetChar()) == EOF) {}
		if ((ch < '1') || (ch > '3')) {
			continue;
		}
		curr_pwr = (POWER_MODE_T) (ch - '1');
		break;
	}
}
static void get_wakeup_method(void){
	if((int)curr_pwr != 2){
		int ch = EOF;
		DEBUGSTR(
			"\r\nSelect wake-up method\r\n"
			"\t1. RTC wake-up\r\n"
			"\t2. Utick wake-up\r\n"
			"\t3. Pin interrupt wake-up\r\n"
			);
		
		while(1){
			while((ch = Board_UARTGetChar()) == EOF){}
				if((ch != '1') && (ch != '2') && (ch != '3')){
					continue;
				}
				curr_wakeup_method = (int)(ch - '0');
				break;
		}
	}
	else{
		int ch = EOF;
		DEBUGSTR(
			"\r\nSelect wake-up method\r\n"
			"\t1. RTC wake-up\r\n"
			"\t2. Restart\r\n");
		
		while(1){
			while((ch = Board_UARTGetChar()) == EOF){}
				if((ch != '1') && (ch != '2'))
				{
					continue;
				}
				if((int) ch - '0' == 1)
					curr_wakeup_method = (int)(ch - '0');
				else if((int) ch - '0' == 2)
					curr_wakeup_method = 0;
				break;
		}
	}
}
/*****************************************************************************
 * Public functions
 ****************************************************************************/

/* Pin interrupt IRQ handler */
void PININT_IRQ_HANDLER(void)
{
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 2, 1);
	Chip_PININT_ClearIntStatus(LPC_PININT, PININTCH(GPIO_PININT_INDEX));
}

/* RTC interrupt flags */
static volatile bool rtcWake, rtcAlarm;
void RTC_IRQHandler(void)
{
	uint32_t rtcStatus;

	/* Get RTC status register */
	rtcStatus = Chip_RTC_GetStatus(LPC_RTC);

	/* Check RTC 1KHz match interrupt */
	if (rtcStatus & RTC_CTRL_WAKE1KHZ) {
		/* RTC high resultiuon wakeup interrupt */
		rtcWake = true;
	}

	/* Check RTC 1Khz match interrupt */
	if (rtcStatus & RTC_CTRL_ALARM1HZ) {
		/* Alarm */
		rtcAlarm = true;
	}

	/* Clear only latched RTC status */
	Chip_RTC_ClearStatus(LPC_RTC, (rtcStatus & (RTC_CTRL_WAKE1KHZ | RTC_CTRL_ALARM1HZ)));
}
void RTCinit(){
	
	/* Enable the RTC oscillator, oscillator rate can be determined by
	   calling Chip_Clock_GetRTCOscRate()	*/
	Chip_Clock_EnableRTCOsc();
	
	/* Initialize RTC driver (enables RTC clocking) */
	Chip_RTC_Init(LPC_RTC);

	/* Enable RTC as a peripheral wakeup event */
	Chip_SYSCON_EnableWakeup(SYSCON_STARTER_RTC);

	/* RTC reset */
	Chip_RTC_Reset(LPC_RTC);

	/* Start RTC at a count of 0 when RTC is disabled. If the RTC is enabled, you
	   need to disable it before setting the initial RTC count. */
	Chip_RTC_Disable(LPC_RTC);
	Chip_RTC_SetCount(LPC_RTC, 0);

	/* Enable RTC and high resolution timer - this can be done in a single
	   call with Chip_RTC_EnableOptions(LPC_RTC, (RTC_CTRL_RTC1KHZ_EN | RTC_CTRL_RTC_EN)); */
	Chip_RTC_Enable1KHZ(LPC_RTC);
	Chip_RTC_Enable(LPC_RTC);

	/* Clear latched RTC interrupt statuses */
	Chip_RTC_ClearStatus(LPC_RTC, ( RTC_CTRL_ALARM1HZ | RTC_CTRL_WAKE1KHZ));

	/* Enable RTC interrupt */
	NVIC_EnableIRQ(RTC_IRQn);

	/* Enable RTC alarm interrupt */
	Chip_RTC_EnableWakeup(LPC_RTC, (RTC_CTRL_ALARMDPD_EN | RTC_CTRL_WAKEDPD_EN));

}
void UTICK_IRQHandler(void)
{
	Chip_UTICK_ClearInterrupt(LPC_UTICK);
	Board_LED_Set(1, true);
}
/* main function (C entry point) */
int main(void)
{
	const char *mname3[] =
	{"Sleep [Wake button SW1 to wakeup]", "Deep Sleep [Wake button SW1 to wakeup]", "Deep Powerdown [Reset to wakeup]"};
	const char *mname2[] =
	{"Sleep [Wait for Utick timer]", "Deep Sleep [Wait for Utick timer]", "Deep Powerdown [Wait for Utick timer]"};
	const char *mname1[] = 
	{"Sleep [Wait for RTC timer]", "Deep Sleep [Wait for RTC timer]", "Deep Powerdown [Wait for RTC timer]"};
	
	int loop = 1;	/* prevents unreachable statement warning */
	CHIP_SYSCON_MAINCLKSRC_T saved_clksrc;
	uint32_t saved_clkRate = 0;
	volatile uint32_t i;

	/* Generic Initialization */
	SystemCoreClockUpdate();

	Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_GPIO0);
	/* Sets up DEBUG UART */
	DEBUGINIT();

	/* Special pin muxing to lower power numbers */
	Chip_IOCON_SetPinMuxing(LPC_IOCON, pinmuxing, sizeof(pinmuxing) / sizeof(pinmuxing[0]));
	periph_disable();
	
	/* Show ROM version */
	DEBUGOUT("Power library ROM version = %04x\r\n", Chip_POWER_GetROMVersion());
	
	Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_GPIO0);
	
	Chip_GPIO_SetPinDIROutput(LPC_GPIO, 0, 2);
	
	Chip_GPIO_SetPinState(LPC_GPIO, 0, 2, 0);

	/* Wait for wakeup event */
	while (loop) {
		get_option();
		get_wakeup_method();
		
		Chip_GPIO_SetPinState(LPC_GPIO, 0, 2, 0);
		
		switch(curr_wakeup_method){
			case 0:
				DEBUGOUT("\r\nReset to get out of the Deep Power Down Mode\r\n");
				break;
			case 1:
				RTCinit();
			  Chip_RTC_SetWake(LPC_RTC, 2000);
				DEBUGOUT("Entering %s ...\r\n", mname1[curr_pwr]);
				break;
			case 2:
				/* Enable the power to the Watchdog Oscillator,
					 UTick timer ticks are driven by watchdog oscillator */
				Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_WDT_OSC);

				/* Initialize UTICK driver */
				Chip_UTICK_Init(LPC_UTICK);

				/* Clear UTICK interrupt status */
				Chip_UTICK_ClearInterrupt(LPC_UTICK);

				/* Set the UTick for a delay of 1000mS and in repeat mode */
				Chip_UTICK_SetDelayMs(LPC_UTICK, 5000, true);

				/* Enable Wake up from deep sleep mode due to UTick */
				Chip_SYSCON_EnableWakeup(SYSCON_STARTER_UTICK);

				/* Enable UTICK interrupt */
				NVIC_EnableIRQ(UTICK_IRQn);
				DEBUGOUT("Entering %s ...\r\n", mname2[curr_pwr]);
				break;
			case 3:
				Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_PINT);			// PINT	
				Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_INPUTMUX);	
				Chip_PININT_Init(LPC_PININT);

				/* Configure GPIO pin as input */
				Chip_GPIO_SetPinDIRInput(LPC_GPIO, GPIO_PININT_PORT, GPIO_PININT_PIN);

				/* Configure pin as GPIO */
				Chip_IOCON_PinMuxSet(LPC_IOCON, GPIO_PININT_PORT, GPIO_PININT_PIN,
						(IOCON_FUNC0 | IOCON_DIGITAL_EN  | IOCON_GPIO_MODE));

				/* Configure pin interrupt selection for the GPIO pin in Input Mux Block */
				Chip_INMUX_PinIntSel(GPIO_PININT_INDEX, GPIO_PININT_PORT, GPIO_PININT_PIN);

				/* Configure channel interrupt as edge sensitive and falling edge interrupt */
				Chip_PININT_ClearIntStatus(LPC_PININT, PININTCH(GPIO_PININT_INDEX));
				Chip_PININT_SetPinModeEdge(LPC_PININT, PININTCH(GPIO_PININT_INDEX));
				Chip_PININT_EnableIntLow(LPC_PININT, PININTCH(GPIO_PININT_INDEX));

				NVIC_DisableIRQ(PININT_NVIC_NAME);
				NVIC_ClearPendingIRQ(PIN_INT0_IRQn);
				/* Enable interrupt in the NVIC */
				NVIC_EnableIRQ(PININT_NVIC_NAME);
				//NVIC_DisableIRQ(PININT_NVIC_NAME);
				
				/* Enable wakeup for PININT0 */
				Chip_SYSCON_EnableWakeup(SYSCON_STARTER_PINT0);
				DEBUGOUT("Entering %s ...\r\n", mname3[curr_pwr]);
				break;
		}
		/* save the clock source, power down the PLL */		
	  saved_clksrc = Chip_Clock_GetMainClockSource();	
		
		/* Disable PLL, if previously enabled, prior to sleep */
		if (saved_clksrc == SYSCON_MAINCLKSRC_PLLOUT) {
			Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_FRO12MHZ);
			Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SYS_PLL);
		}
		else if (saved_clksrc == SYSCON_MAINCLKSRC_FROHF) {
			saved_clkRate = Chip_Clock_GetFROHFRate();
			Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_FRO12MHZ);
			LPC_SYSCON->FROCTRL &= ~(SYSCON_FROCTRL_MASK | SYSCON_FROCTRL_HSPDCLK);
			Chip_SYSCON_SetFLASHAccess(SYSCON_FLASH_1CYCLE);
		}

		/* Lower system voltages to current lock (likely IRC) */
		Chip_POWER_SetVoltage(Chip_Clock_GetMainClockRate());

		/* Wait for DEBUG Outputs to complete before entering reduced power mode */
		while (!(Chip_UART_GetStatus(DEBUG_UART) & UART_STAT_TXIDLE)) {}

		/* Go to sleep leaving SRAM powered during sleep. Use lower
		     voltage during sleep. */
		if(curr_wakeup_method == 2)
			Chip_POWER_EnterPowerMode(curr_pwr,
															(SYSCON_PDRUNCFG_PD_SRAM0 | SYSCON_PDRUNCFG_PD_WDT_OSC));								//OSC required to wakeup from uTICK
		else
			Chip_POWER_EnterPowerMode(curr_pwr, SYSCON_PDRUNCFG_PD_SRAM0);																	//Enter Low power mode with SRAM0 retention

		DEBUGSTR("Woken up!\r\n");

		/* On wakeup, restore PLL power if needed */
		if (saved_clksrc == SYSCON_MAINCLKSRC_PLLOUT) {
			Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SYS_PLL);

			/* Wait for PLL lock */
			while (!Chip_Clock_IsSystemPLLLocked()) {}

			Chip_POWER_SetVoltage(Chip_Clock_GetSystemPLLOutClockRate(false));

			/* Use PLL for system clock */
			Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_PLLOUT);
		}
		else if (saved_clksrc == SYSCON_MAINCLKSRC_FROHF) {
			Chip_POWER_SetVoltage(saved_clkRate);
			Chip_SetupFROClocking(saved_clkRate);
		}
	if(curr_wakeup_method == 1){
			Chip_RTC_DisableWakeup(LPC_RTC, (RTC_CTRL_ALARMDPD_EN | RTC_CTRL_WAKEDPD_EN));
			NVIC_DisableIRQ(RTC_IRQn);
			Chip_RTC_Disable(LPC_RTC);
			Chip_RTC_Disable1KHZ(LPC_RTC);
			Chip_SYSCON_DisableWakeup(SYSCON_STARTER_RTC);
			Chip_RTC_DeInit(LPC_RTC);
			Chip_Clock_DisableRTCOsc();
		}
		else if(curr_wakeup_method == 2){
			NVIC_DisableIRQ(UTICK_IRQn);
			Chip_UTICK_DeInit(LPC_UTICK);
			Chip_SYSCON_DisableWakeup(SYSCON_STARTER_UTICK);
			Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_WDT_OSC);
		}
		else if(curr_wakeup_method == 3){
			NVIC_DisableIRQ(PININT_NVIC_NAME);
			Chip_PININT_DeInit(LPC_PININT);
			Chip_SYSCON_DisableWakeup(SYSCON_STARTER_PINT0);
			Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_INPUTMUX);			
		}
	}
	
	return 0;
}
