#ifndef __PLATFORM_PERIPHERALS_H
#define __PLATFORM_PERIPHERALS_H

#pragma diag_suppress 368  // Suppresses warning from FreeRTOS code task.h file

#include <stdint.h>
#include "board.h"


/**
 * Initialize UART used for the terminal
 *
 * PORTME:  This function provides target specific initialization for the UART
 * that is used for test output.  It should setup the UART as follows:
 * Baud:9600, 8 data bits, 1 stop bit.
 * Typically this function would either directly reference CMSIS style registers 
 * or call library functions, e.g., LPCOpen functions, ROM APIs, xSW, etc.

 * If dual-core (MULTITHREAD == 2) the MASTER will setup the UART for both itself 
 * and the SLAVE
 */
void platform_uart_init(void);


/**
 * Initialize timer used to return timer ticks used by time functions
 *
 * PORTME:  This function provides target specific initialization for a system
 * timer.  The timer should be setup so that it provides a continuous
 * incrementing tick count that can be read by software as necessary.
 * Typically this function would either directly reference CMSIS style registers 
 * or call library functions, e.g., LPCOpen functions, ROM APIs, xSW, etc.
 *
 * If dual-core (MULTHREAD == 1) the MASTER will setup the timer for both itself 
 * and the SLAVE
 */
void platform_timer_init(void);


/**
 * Initialize system clocks
 *
 * PORTME:  This function provides target specific system clocks initialization
 * needed by the application.  Since different build environments, e.g.,
 * Keil, IAR, LPCXpresso may perform different system startup clock
 * initializations we assume nothing but instead initialization everything to a
 * known state in this function.
 * 
 * For the system clock this function must:
 *  Initialize PLL and set it as the system clock with desired frequency
 * 
 * Typically this function would either directly reference CMSIS style registers 
 * or call library functions, e.g., LPCOpen functions, ROM APIs, xSW, etc.
 * 
 * If dual-core (MULTITHREAD == 2) the MASTER will setup the clock for both 
 * itself and the slave
 *
 */
void platform_clocks_init(void);



/**
 * Send a character to the system UART
 * 
 * PORTME:  This function provides a target specific UART send routine.  
 * Also see init_UART()
 * 
 * @param c  Character to send
 */
void platform_uart_send_byte(uint8_t c);



void platform_power_init(void);

secs_ret platform_timer_get_time_in_secs(CORE_TICKS ticks);


/**
 * Return current system timer value
 *
 * PORTME:  This function returns a target specific count of the number of
 * timer ticks since the timer was initialized using the platform_timer_init() 
 * function.  Typically this function would either directly reference CMSIS style 
 * registers or call library functions, e.g., LPCOpen functions, ROM APIs, xSW etc.
 * 
 * @return  Returns current system timer value, i.e., system ticks
 */
uint32_t platform_timer_get_timer_count(void);


void platform_boot_cm0(uint32_t *coentry, uint32_t *costackptr);

#endif  // __PLATFORM_PERIPHERALS_H

