
#include "coremark.h"
#include "platform.h"

#include "power_lib_5411x.h"

#define BUFSIZE 							256


/* PORTME:  Define various clock parameters. Note that for CoreMark testing the
 * system clock will always be provided by the FRO.  Note that TIMER_RES_DIVIDER
 * is a prescaler used as input to the timer used for benchmark timing.  Set it
 * to a value to ensure that the timer doesn't wrap before the benchmark is complete 
 */
#define TIMER_RES_DIVIDER 8 // PCLK divisor for input to system timer (make it a power of 2)

//#define SYS_CLK_RATE_HZ 12000000
//#define SYS_CLK_RATE_HZ 48000000
#define SYS_CLK_RATE_HZ 96000000


#define BAUD_RATE  9600		// Used for user terminal

#define EE_TICKS_PER_SEC (SYS_CLK_RATE_HZ / TIMER_RES_DIVIDER)


/* Define multiprocessor bit positions */
#define MC_M4_BOOT            (1<<0)
#define MC_M4_SWD             (1<<1)
#define MC_CM4_CLK_ENABLE     (1<<2)
#define MC_CM0_CLK_ENABLE     (1<<3)
#define MC_CM4_RESET_ENABLE   (1<<4)
#define MC_CM0_RESET_ENABLE   (1<<5)
#define MC_CM4_SLEEPCON_OWNER (1<<6)

// CPU STAT register
#define MC_M4_SLEEPING        (1<<0)
#define MC_M0_SLEEPING        (1<<1)
#define MC_M4_LOCKUP          (1<<2)
#define MC_M0_LOCKUP          (1<<3)


/* Ring buffer size */
#define UART_RB_SIZE 64

/* Set the default UART, IRQ number, and IRQ handler name */
#define LPC_USART       LPC_USART0
#define LPC_IRQNUM      USART0_IRQn
#define LPC_UARTHNDLR   USART0_IRQHandler

volatile uint32_t delay;

void platform_timer_init(void) {

		/* Initialize Timer0 */
		Chip_TIMER_Init(LPC_TIMER0);
    /* Reset timer counter  */
		Chip_TIMER_Reset(LPC_TIMER0);
    /* Set prescaler */
		Chip_TIMER_PrescaleSet(LPC_TIMER0,TIMER_RES_DIVIDER - 1);
    /* Start counting */
    Chip_TIMER_Enable(LPC_TIMER0);

}


uint32_t platform_timer_get_timer_count() {

    return LPC_TIMER0->TC;

}

secs_ret platform_timer_get_time_in_secs(CORE_TICKS ticks) {
    secs_ret retval = ((secs_ret) ticks) / (secs_ret) EE_TICKS_PER_SEC;
    return retval;

}

void set_baud_rate(LPC_USART_T *pUART, uint32_t pclk, uint32_t baud_rate) {
	
	pUART->OSR = 0xf;
	pUART->BRG = (pclk / 16 / baud_rate) - 1;;
		
}


void platform_uart_init() {
  
		Chip_Clock_EnablePeriphClock(SYSCON_CLOCK_IOCON);
    Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 0, IOCON_MODE_INACT | IOCON_FUNC1 | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);
    Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 1, IOCON_MODE_INACT | IOCON_FUNC1 | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);

    /* Enable 12 MHz FRO  as clock source to Flexcomm 0 */
		Chip_Clock_SetFLEXCOMMClockSource(0,SYSCON_FLEXCOMMCLKSELSRC_FRO12MHZ);
		Chip_UART_Init(DEBUG_UART);

		/* Setup UART */
    set_baud_rate(LPC_USART, 12000000, BAUD_RATE);
    Chip_UART_ConfigData(LPC_USART, UART_CFG_DATALEN_8 | UART_CFG_PARITY_NONE | UART_CFG_STOPLEN_1);
    Chip_UART_Enable(LPC_USART);
	
}

void platform_uart_send_byte(uint8_t c) {
	
	Chip_UART_SendBlocking(LPC_USART, &c, 1);

}

void platform_clocks_init(void) {
	
#ifdef	COREMARK_SCORE_TEST 
	LPC_SYSCON->AUTOCGOR |= (SYSCON_AUTOCGOR_RAM0X | SYSCON_AUTOCGOR_RAM1 | SYSCON_AUTOCGOR_RAM2);
#endif
	LPC_SYSCON->MAINCLKSELA = 0x0;  	// Clock source is FRO 12 MHz;
	LPC_SYSCON->MAINCLKSELB = 0x0;  	// Clock source for main clock is MAINCLKSELA
	LPC_SYSCON->CLKOUTSELA = 0x0;     // Clock out source is main clock
	LPC_SYSCON->CLKOUTDIV = 3;				// Divide clock out by 4
	
#if SYS_CLK_RATE_HZ == 12000000

	  Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_FRO12MHZ);
    Chip_SYSCON_SetFLASHAccess(SYSCON_FLASH_1CYCLE);
	
#elif SYS_CLK_RATE_HZ == 48000000
		Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_FROHF);
    Chip_POWER_SetFROHFRate(SYSCON_FRO48MHZ_FREQ);
		Chip_SYSCON_SetFLASHAccess(SYSCON_FLASH_3CYCLE); 

#elif SYS_CLK_RATE_HZ == 96000000
		Chip_Clock_SetMainClockSource(SYSCON_MAINCLKSRC_FROHF);
    Chip_POWER_SetFROHFRate(SYSCON_FRO96MHZ_FREQ);
		Chip_SYSCON_SetFLASHAccess(SYSCON_FLASH_5CYCLE);
#else

#error Invalid SYS_RATE_HZ	
	
#endif

}

void platform_power_init(void) {

#if SYS_CLK_RATE_HZ == 48000000
	Chip_POWER_SetVoltage(SYS_CLK_RATE_HZ);
#elif SYS_CLK_RATE_HZ == 96000000
	Chip_POWER_SetVoltage(SYS_CLK_RATE_HZ);
#endif	

#define PDRUNCFG1_PD_ALT_FLASH_IBG  (1UL << 28)
#define PDRUNCFG1_SEL_ALT_FLASH_IBG (1UL << 29)

   // use the alternative flash bandgap and turn off the original one          
  LPC_SYSCON->PDRUNCFGCLR[1] = PDRUNCFG1_PD_ALT_FLASH_IBG;
  LPC_SYSCON->PDRUNCFGSET[1] = PDRUNCFG1_SEL_ALT_FLASH_IBG;
  Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_FLASH_BG);

	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SRAM2);    
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLASH);	  
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SPIFI);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_INPUTMUX);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_IOCON);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_GPIO0);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_GPIO1);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_PINT);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_GINT);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_DMA);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_CRC);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_WWDT);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_RTC);					
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_MAILBOX);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_ADC0);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_MRT);				
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_SCT0);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_UTICK);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM1);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM2);	
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM3);			
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM4);	
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM5);	
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM6);	
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FLEXCOMM7);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_DMIC);	
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER2);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_USB);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER1);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER3);		
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_TIMER4);		
	
	Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SYS_PLL);		
	Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SRAM1); 		
	Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_SRAM2);	
	Chip_SYSCON_PowerUp(SYSCON_PDRUNCFG_PD_SRAMX);		

#ifdef RAM_RAMX_SRAM0	 
#ifndef BOOT_FROM_FLASH
	Chip_Clock_DisablePeriphClock(SYSCON_CLOCK_FMC);			
	Chip_SYSCON_PowerDown(SYSCON_PDRUNCFG_PD_VDDA_ENA);
#endif
#endif

}
