/*
 * @brief Power Management Controller example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2013
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include <stdlib.h>
#include <string.h>
#include "board.h"
#include "chip.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

#define RTC_ALARM_TIME 2									/* RTC Alarm in seconds */
#define SRAMVECTORTABLE 0x10088000				/* Memory address for vector table copy when waking up to SRAM */
#define WakeUpToSRAM 0									/* Enables waking up to SRAM from Power-down mode */

void Set_Base_Clocks(void);
void Save_Base_Clocks(void);
void Optimize_Power(void);
void CopyInterruptToSRAM(void);

/* Structure for initial base clock states */
struct CLK_BASE_STATES {
	CHIP_CGU_BASE_CLK_T clk;	/* Base clock */
	CHIP_CGU_CLKIN_T clkin;	/* Base clock source, see UM for allowable souorces per base clock */
	bool autoblock_enab;/* Set to true to enable autoblocking on frequency change */
	bool powerdn;		/* Set to true if the base clock is initially powered down */
};


/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
static void PMC_Get_Wakeup_option(uint8_t *Wakeup_rtc)
{
	FlagStatus exitflag;
	uint8_t buffer = 0xFF;

	DEBUGSTR("\r\nSelect the Wake up signal\r\n- WAKEUP0 pin : Press 'W' or 'w' to select\r\n- RTC Alarm   : Press 'R' or 'r' to select    \r\n");

	exitflag = RESET;
	while (exitflag == RESET) {

		/* Get user input */
		buffer = DEBUGIN();
		if ((buffer == 'W') || (buffer == 'w')) {
			DEBUGSTR("WAKEUP0 pin selected \r\n");
			*Wakeup_rtc = 0;
			exitflag = SET;
		}

		if ((buffer == 'R') || (buffer == 'r')) {
			DEBUGSTR("RTC Alarm selected. Ready... \r\n");
			*Wakeup_rtc = 1;
			exitflag = SET;
		}
	}
}

/**
 * Set IRC as source clock fo all the output clocks & power down
 * before going to 'Deep Sleep'/'Power Down'/'Deep Power Down' modes
 */
static void PMC_Pre_SleepPowerDown(void)
{
	//int i;

#if (!defined(BOARD_NGX_XPLORER_1830) && !defined(BOARD_NGX_XPLORER_4330))
	/* SDRAM in self refresh mode only for Keil & Hitex boards */
	LPC_EMC->DYNAMICCONTROL |= (1 << 2);
	while (!(LPC_EMC->STATUS & (1 << 2))) {}
#endif

	/* Shutdown perpheral clocks with wake up enabled */
	Chip_Clock_StartPowerDown();

	/* Get state of individual base clocks & store them for restoring.
	 * Sets up the IRC as base clock source
	 */
	Save_Base_Clocks();

	/* Set IRC as clock source for SPIFI */
	Chip_Clock_SetBaseClock(CLK_BASE_SPIFI, CLKIN_IRC, true, false);

	/* Set IRC as source clock for Core */
	Chip_Clock_SetBaseClock(CLK_BASE_MX, CLKIN_IRC, true, false);

#if (defined(BOARD_KEIL_MCB_4357) || defined(BOARD_KEIL_MCB_1857))
	/* Set Flash acceleration */
	Chip_CREG_SetFlashAcceleration(OscRateIn);
#endif

	/* Power down the main PLL */
	Chip_Clock_DisableMainPLL();
	Chip_Clock_DisablePLL(CGU_USB_PLL);
	Chip_Clock_DisableCrystal();
}

/**
 * Post wake up initialisation function
 */
static void PMC_Post_Wakeup(uint8_t buffer)
{
#if (defined(BOARD_KEIL_MCB_4357) || defined(BOARD_KEIL_MCB_1857))
	/* Setup FLASH acceleration to target clock rate prior to clock switch */
	Chip_CREG_SetFlashAcceleration(MAX_CLOCK_FREQ);
#endif

	/* Reinitialise Clocks */
	Chip_Clock_SetBaseClock(CLK_BASE_MX, CLKIN_CRYSTAL, true, false);

	/* Setup PLL for 100MHz and switch main system clocking */
	Chip_Clock_SetupMainPLLHz(CLKIN_CRYSTAL, OscRateIn, 100 * 1000000, 100 * 1000000);
	Chip_Clock_SetBaseClock(CLK_BASE_MX, CLKIN_MAINPLL, true, false);

	/* Setup PLL for maximum clock */
	Chip_Clock_SetupMainPLLHz(CLKIN_CRYSTAL, OscRateIn, MAX_CLOCK_FREQ, MAX_CLOCK_FREQ);

	/* Restore the base clock states */
	Set_Base_Clocks();

	/* Reset and enable 32Khz oscillator */
	LPC_CREG->CREG0 &= ~((1 << 3) | (1 << 2));
	LPC_CREG->CREG0 |= (1 << 1) | (1 << 0);

	/* Setup a divider E for main PLL clock switch SPIFI clock to that divider.
	   Divide rate is based on CPU speed and speed of SPI FLASH part. */
#if (MAX_CLOCK_FREQ > 180000000)
	Chip_Clock_SetDivider(CLK_IDIV_E, CLKIN_MAINPLL, 5);
#else
	Chip_Clock_SetDivider(CLK_IDIV_E, CLKIN_MAINPLL, 4);
#endif
	Chip_Clock_SetBaseClock(CLK_BASE_SPIFI, CLKIN_IDIVE, true, false);

	/* Attach main PLL clock to divider C with a divider of 2 */
	Chip_Clock_SetDivider(CLK_IDIV_C, CLKIN_MAINPLL, 2);

	/* Re-Initialize debug UART
	 * ? 115200bps
	 * ? 8 data bit
	 * ? No parity
	 * ? 1 stop bit
	 * ? No flow control
	 */
	DEBUGINIT();

#if (!defined(BOARD_NGX_XPLORER_1830) && !defined(BOARD_NGX_XPLORER_4330))
	/* SDRAM in NORMAL mode, only for Keil & Hitex boards */
	LPC_EMC->DYNAMICCONTROL &= ~(1 << 2);
	while (LPC_EMC->STATUS & (1 << 2)) {}
#endif
		LPC_GPIO_PORT->CLR[4] = (1UL << 13);
}

/**
 * Event Router configure function
 */
static void PMC_Evrt_Configure(CHIP_EVRT_SRC_T Evrt_Src)
{
	/* Configure Interrupt signal from Evrt_Src pin to EVRT */
	if(Evrt_Src == EVRT_SRC_WAKEUP0)
		Chip_EVRT_ConfigIntSrcActiveType(Evrt_Src, EVRT_SRC_ACTIVE_FALLING_EDGE);
	else
		Chip_EVRT_ConfigIntSrcActiveType(Evrt_Src, EVRT_SRC_ACTIVE_RISING_EDGE);

	/* Clear any pending interrupt */
	Chip_EVRT_ClrPendIntSrc(Evrt_Src);

	/* Enable interrupt signal from Evrt_Src pin to EVRT */
	Chip_EVRT_SetUpIntSrc(Evrt_Src, ENABLE);
}

/** * Power State handler function
 */
static void PMC_PwrState_Handler(uint8_t buffer, uint8_t Wake_RTC)
{
	CHIP_EVRT_SRC_T Evrt_Src;
	CHIP_PMC_PWR_STATE_T Pwr_state;
	volatile uint32_t delay = 4000;

	if (Wake_RTC) {
		/* Disable WAKEUP0 pin to avoid conflicting settings */
		Chip_EVRT_SetUpIntSrc(EVRT_SRC_WAKEUP0, DISABLE);
		Chip_EVRT_ClrPendIntSrc(EVRT_SRC_WAKEUP0);

		/* Configure EVRT_SRC_RTC as wake up signal */
		Evrt_Src = EVRT_SRC_RTC;

		/* Disable interrupt signal from Evrt_Src pin to EVRT */
		Chip_EVRT_SetUpIntSrc(Evrt_Src, DISABLE);

		/* Initialize and configure RTC */
		Chip_RTC_Init(LPC_RTC);

		Chip_RTC_ResetClockTickCounter(LPC_RTC);
		Chip_RTC_SetTime(LPC_RTC, RTC_TIMETYPE_SECOND, 0);

		/* Set alarm time = RTC_ALARM_TIME seconds.
		 * So after each RTC_ALARM_TIME seconds, RTC will generate and wake-up system
		 */
		Chip_RTC_SetAlarmTime(LPC_RTC, RTC_TIMETYPE_SECOND, RTC_ALARM_TIME);
		Chip_RTC_CntIncrIntConfig(LPC_RTC, RTC_AMR_CIIR_IMSEC, DISABLE);

		/* Set the AMR for RTC_ALARM_TIME match alarm interrupt */
		Chip_RTC_AlarmIntConfig(LPC_RTC, RTC_AMR_CIIR_IMSEC, ENABLE);
		Chip_RTC_ClearIntPending(LPC_RTC, RTC_INT_ALARM);

	}
	else {
		Evrt_Src = EVRT_SRC_WAKEUP0;
	}

	/* Configure wake up signal */
	PMC_Evrt_Configure(Evrt_Src);

	/* Turn off LED6 to see that it is about to enter a low power state */
	LPC_GPIO_PORT->CLR[4] = (1UL << 13);

	switch (buffer) {
	case '1':
		DEBUGSTR("Entering 'Sleep' state ...\r\n");
		if (Wake_RTC) {
			DEBUGOUT("RTC alarm will wake up from 'Sleep' mode in %d seconds \r\n", RTC_ALARM_TIME);
			Chip_RTC_Enable(LPC_RTC, ENABLE);
		}
		else {
			DEBUGSTR("Press WAKEUP0 button to exit 'Sleep' mode \r\n");
		}
		/* Wait a bit for print statements to fully transfer before going into low power state */
		while(delay--);
		/* Set IRC as clock source for SPIFI */
		Chip_Clock_SetBaseClock(CLK_BASE_SPIFI, CLKIN_IRC, true, false);
		
		/* Set clock to 12MHz IRC for low power sleep mode */
		Chip_Clock_SetBaseClock(CLK_BASE_MX, CLKIN_IRC, true, false);

//		/* Disable flash wait states as they are not needed at 12MHz */
		Chip_CREG_SetFlashAcceleration(OscRateIn);

//		/* Turn off crystal to save power */
		Chip_Clock_DisableCrystal();
		
		/* Goto Sleep mode */
		Chip_PMC_Sleep();

		/* Execute Post Wake Up Process */
		Chip_RTC_DeInit(LPC_RTC);
		Chip_Clock_EnableCrystal();
		
		/* Post sleep mode is too quick, allow system to reset up clocks for UART transmission */
		delay = 4000;
		while(delay--);
		
		break;

	case '2':
		DEBUGSTR("Entering 'Deep Sleep' state ...\r\n");
		if (Wake_RTC) {
			DEBUGOUT("RTC alarm will wake up from 'Deep Sleep' mode in %d seconds \r\n", RTC_ALARM_TIME);
			Chip_RTC_Enable(LPC_RTC, ENABLE);
		}
		else {
			DEBUGSTR("Press WAKEUP0 button to exit 'Deep Sleep' mode \r\n");
		}
		/* Wait a bit for print statements to fully transfer before going into low power state */
		while(delay--);

		Pwr_state = PMC_DeepSleep;

		/* Call Pre SleepPowerDown function */
		PMC_Pre_SleepPowerDown();

		/* Goto Deep Sleep mode */
		Chip_PMC_Set_PwrState(Pwr_state);

		/* Execute Post Wake Up Process */
		Chip_Clock_EnableCrystal();
		PMC_Post_Wakeup(buffer);
		break;

	case '3':
		DEBUGSTR("Entering 'Power Down' state ...\r\n");
		if (Wake_RTC) {
			DEBUGOUT("RTC alarm will wake up from 'Power Down' mode in %d seconds \r\n", RTC_ALARM_TIME);
			Chip_RTC_Enable(LPC_RTC, ENABLE);
		}
		else {
			DEBUGSTR("Press WAKEUP0 button to exit 'Power Down' mode \r\n");
		}
		/* Wait a bit for print statements to fully transfer before going into low power state */
		while(delay--);

		Pwr_state = PMC_PowerDown;

		/* Call Pre SleepPowerDown function */
		PMC_Pre_SleepPowerDown();
		break;

	case '4':
		DEBUGSTR("Entering 'Deep Power Down' state ...\r\n");
		if (Wake_RTC) {
			DEBUGOUT("RTC alarm will wake up from 'Deep Power Down' mode in %d seconds \r\n", RTC_ALARM_TIME);
			Chip_RTC_Enable(LPC_RTC, ENABLE);
		}
		else {
			DEBUGSTR("Press WAKEUP0 button to exit 'Deep Power Down' mode \r\n");
		}
		/* Wait a bit for print statements to fully transfer before going into low power state */
		while(delay--);
		/* Set IRC as clock source for SPIFI */
		Chip_Clock_SetBaseClock(CLK_BASE_SPIFI, CLKIN_IRC, true, false);

		/* Set IRC as source clock for Core */
		Chip_Clock_SetBaseClock(CLK_BASE_MX, CLKIN_IRC, true, false);
		Chip_Clock_DisableCrystal();
		Pwr_state = PMC_DeepPowerDown;

		/* Goto Deep Power-down mode */
		Chip_PMC_Set_PwrState(Pwr_state);

		/* Wake up from Deep power down state is as good as RESET */
		while (1) {}

	default:
		break;
	}
}

/**
 * @brief	Main entry point
 * @return	Nothing
 */
int main(void)
{
	FlagStatus exitflag;
	uint8_t buffer = 0xFF;
	uint8_t Wake_RTC = 0;

	/* Initialize P9_1 to GPIO and set it to low for deep power-down wake-up measurements */
	Chip_GPIO_SetPinDIROutput(LPC_GPIO_PORT, 4, 13);
	LPC_GPIO_PORT->CLR[4] = (1UL << 13);

	/* Initialize MCU to necessary settings for low power example */
	SystemInit();
	Optimize_Power();
	SystemCoreClockUpdate();
	DEBUGINIT();

	/* Set ENA_EVENT0 bit in PD0_SLEEP0_HW_ENA regsiter to enter low power modes other than Sleep mode */
	LPC_PMC->PD0_SLEEP0_HW_ENA |= (1 << 0);
	
	/* Disable EVRT interrupt in NVIC */
	NVIC_DisableIRQ(EVENTROUTER_IRQn);

	/* preemption = 1, sub-priority = 1 */
	NVIC_SetPriority(EVENTROUTER_IRQn, ((0x01 << 3) | 0x01));

	/* Enable Event Router interrupt in NVIC */
	NVIC_EnableIRQ(EVENTROUTER_IRQn);

	DEBUGSTR("\r\n==========================================================================\r\n"
	"NXP Semiconductors LPC18xx Low Power Application Note Software Example      \r\n"
	"=============================================================================\r\n"
	"Select the power state test to be executed                           \r\n"
	" - Sleep state & Wake up test           : Press '1' to select        \r\n"
	" - Deep Sleep state & Wake up test      : Press '2' to select        \r\n"
	" - Power down state & Wake up test      : Press '3' to select        \r\n"
	" - Deep power down state & Wake up test : Press '4' to select        \r\n"
	"=============================================================================\r\n");

	exitflag = RESET;

	while (exitflag == RESET){
		/* Read user option from UART prompt */
		buffer = DEBUGIN();
		switch (buffer) {
		case '1':		/* Sleep/Wake up Mode */
			DEBUGSTR("'Sleep' state test selected \r\n");

			#if WakeUpToSRAM
			CopyInterruptToSRAM();
			SCB->VTOR = SRAMVECTORTABLE;
			#endif

			PMC_Get_Wakeup_option(&Wake_RTC);
			PMC_PwrState_Handler(buffer, Wake_RTC);
			DEBUGSTR("\r\nWoken up \r\n"
			"==========================================================================\r\n"
			"NXP Semiconductors LPC18xx Low Power Application Note Software Example      \r\n"
			"=============================================================================\r\n"
			"Select the power state test to be executed                           \r\n"
			" - Sleep state & Wake up test           : Press '1' to select        \r\n"
			" - Deep Sleep state & Wake up test      : Press '2' to select        \r\n"
			" - Power down state & Wake up test      : Press '3' to select        \r\n"
			" - Deep power down state & Wake up test : Press '4' to select        \r\n"
			"=============================================================================\r\n");
			break;

		case '2':		/* Deep sleep/Wakeup Mode */
			DEBUGSTR("'Deep Sleep' state test selected \r\n");

			#if WakeUpToSRAM
			CopyInterruptToSRAM();
			SCB->VTOR = SRAMVECTORTABLE;
			#endif

			PMC_Get_Wakeup_option(&Wake_RTC);
			PMC_PwrState_Handler(buffer, Wake_RTC);
			DEBUGSTR("\r\nWoken up \r\n"
			"==========================================================================\r\n"
			"NXP Semiconductors LPC18xx Low Power Application Note Software Example      \r\n"
			"=============================================================================\r\n"
			"Select the power state test to be executed                           \r\n"
			" - Sleep state & Wake up test           : Press '1' to select        \r\n"
			" - Deep Sleep state & Wake up test      : Press '2' to select        \r\n"
			" - Power down state & Wake up test      : Press '3' to select        \r\n"
			" - Deep power down state & Wake up test : Press '4' to select        \r\n"
			"=============================================================================\r\n");
			break;

		case '3':		/* Power Down/Wakeup Mode */
			DEBUGSTR("'Power Down' state test selected \r\n");

			#if WakeUpToSRAM
			CopyInterruptToSRAM();
			SCB->VTOR = SRAMVECTORTABLE;
			#endif

			PMC_Get_Wakeup_option(&Wake_RTC);
			PMC_PwrState_Handler(buffer, Wake_RTC);

			/* Execute Power-down code in main() as Power down shuts down SRAM and function calls might not return*/
			/* Goto Power-down mode */

			/* Set Deep sleep mode bit in System Control register of M4 core */
			SCB->SCR |= 0x4;

			/* Set power state in PMC */
			LPC_PMC->PD0_SLEEP0_MODE = (uint32_t) PMC_PowerDown;

			__WFI();

			/* Execute Post Wake Up Process */
			Chip_Clock_EnableCrystal();
			PMC_Post_Wakeup(buffer);

			DEBUGSTR("\r\nWoken up \r\n"
			"==========================================================================\r\n"
			"NXP Semiconductors LPC18xx Low Power Application Note Software Example      \r\n"
			"=============================================================================\r\n"
			"Select the power state test to be executed                           \r\n"
			" - Sleep state & Wake up test           : Press '1' to select        \r\n"
			" - Deep Sleep state & Wake up test      : Press '2' to select        \r\n"
			" - Power down state & Wake up test      : Press '3' to select        \r\n"
			" - Deep power down state & Wake up test : Press '4' to select        \r\n"
			"=============================================================================\r\n");
			break;

		case '4':		/* Deep Power Down/Wakeup Mode */
			DEBUGSTR("'Deep Power Down' state test selected \r\n");
			PMC_Get_Wakeup_option(&Wake_RTC);
			PMC_PwrState_Handler(buffer, Wake_RTC);
			/* Code will never reach here; MCU resets upon wake-up from Deep Power-down mode */
			DEBUGSTR("\r\nWoken up \r\n");
			break;
		}
	}
	while (1) {}
}

void Optimize_Power(void){
	Board_SetupClocking();	
			
	// Disable all peripheral clocks
	LPC_CGU->BASE_CLK[CLK_BASE_SAFE]				|=(1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_USB0]        |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_USB1]        |= (1<<0);
//	LPC_CGU->BASE_CLK[CLK_BASE_SPIFI]       |= (1<<0);// Keep spifi flash on for code space!
	LPC_CGU->BASE_CLK[CLK_BASE_PHY_RX]      |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_PHY_TX]      |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_APB1]        |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_APB3]        |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_LCD]         |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_SDIO]        |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_SSP0]        |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_SSP1]        |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_UART0]       |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_UART1]       |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_UART2]       |= (1<<0);
	//LPC_CGU->BASE_CLK[CLK_BASE_UART3]       |= (1<<0);	// UART3 stays on for GUI
	LPC_CGU->BASE_CLK[CLK_BASE_CGU_OUT0]    |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_CGU_OUT1]    |= (1<<0);
	LPC_CGU->BASE_CLK[CLK_BASE_APLL]				|= (1<<0);

	// Power down M3 branches
	Chip_Clock_Disable(CLK_MX_FLASHA);		// Keep flash on!
	Chip_Clock_Disable(CLK_MX_FLASHB);		// Keep flash on!
//	Chip_Clock_Disable(CLK_MX_SPIFI);			// Keep spifi flash on for code space!
	//Chip_Clock_Disable(CLK_MX_GPIO);			// Need GPIOs to measure wake up times
	Chip_Clock_Disable(CLK_MX_LCD);
	Chip_Clock_Disable(CLK_MX_ETHERNET);
	Chip_Clock_Disable(CLK_MX_USB0);
	//Chip_Clock_Disable(CLK_MX_EMC);				// Keep EMC on for pre-sleep/power down functions
	Chip_Clock_Disable(CLK_MX_DMA);
	Chip_Clock_Disable(CLK_MX_SCT);
	Chip_Clock_Disable(CLK_MX_USB1);
	//Chip_Clock_Disable(CLK_MX_EMC_DIV);		// Keep EMC on for pre-sleep/power down functions
	Chip_Clock_Disable(CLK_MX_EEPROM);
	Chip_Clock_Disable(CLK_MX_WWDT);
	Chip_Clock_Disable(CLK_MX_UART0);
	Chip_Clock_Disable(CLK_MX_UART1);
	Chip_Clock_Disable(CLK_MX_SSP0);
	Chip_Clock_Disable(CLK_MX_TIMER0);
	Chip_Clock_Disable(CLK_MX_TIMER1);
	//Chip_Clock_Disable(CLK_MX_SCU);				// Can't turn off SCU!
	Chip_Clock_Disable(CLK_MX_RITIMER);
	Chip_Clock_Disable(CLK_MX_UART2);
	//Chip_Clock_Disable(CLK_MX_UART3);			// UART3 stays on for GUI
	Chip_Clock_Disable(CLK_MX_TIMER2);
	Chip_Clock_Disable(CLK_MX_TIMER3);
	Chip_Clock_Disable(CLK_MX_SSP1);
	Chip_Clock_Disable(CLK_MX_QEI);

	LPC_CREG->CREG0 |= ((3<<12)|(3<<6));
}


void Set_Base_Clocks(void){
	struct CLK_BASE_STATES InitClkState[] = {
	{CLK_BASE_SAFE, CLKIN_IRC, true, false},
	{CLK_BASE_APB1, CLKIN_MAINPLL, true, false},
	{CLK_BASE_APB3, CLKIN_MAINPLL, true, false},
	{CLK_BASE_USB0, CLKIN_USBPLL, true, true},
#if defined(CHIP_LPC43XX)
	{CLK_BASE_PERIPH, CLKIN_MAINPLL, true, false},
	{CLK_BASE_SPI, CLKIN_MAINPLL, true, false},
#endif
#if defined(BOARD_NXP_LPCLINK2_4370)
	{CLK_BASE_ADCHS, CLKIN_MAINPLL, true, true},
#endif
	{CLK_BASE_PHY_TX, CLKIN_ENET_TX, true, false},
#if defined(USE_RMII)
	{CLK_BASE_PHY_RX, CLKIN_ENET_TX, true, false},
#else
	{CLK_BASE_PHY_RX, CLKIN_ENET_RX, true, false},
#endif
	{CLK_BASE_SDIO, CLKIN_MAINPLL, true, false},
	{CLK_BASE_SSP0, CLKIN_MAINPLL, true, false},
	{CLK_BASE_SSP1, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART0, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART1, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART2, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART3, CLKIN_MAINPLL, true, false},
	{CLK_BASE_LCD, CLKIN_IDIVC, true, false},
	{CLK_BASE_USB1, CLKIN_IDIVD, true, true},
};
	int i;
	for (i = 0; i < (sizeof(InitClkState) / sizeof(InitClkState[0])); i++) {
		Chip_Clock_SetBaseClock(InitClkState[i].clk, InitClkState[i].clkin,
								InitClkState[i].autoblock_enab, InitClkState[i].powerdn);

	}
}

void Save_Base_Clocks(void){
	int i;
	struct CLK_BASE_STATES InitClkState[] = {
	{CLK_BASE_SAFE, CLKIN_IRC, true, false},
	{CLK_BASE_APB1, CLKIN_MAINPLL, true, false},
	{CLK_BASE_APB3, CLKIN_MAINPLL, true, false},
	{CLK_BASE_USB0, CLKIN_USBPLL, true, true},
#if defined(CHIP_LPC43XX)
	{CLK_BASE_PERIPH, CLKIN_MAINPLL, true, false},
	{CLK_BASE_SPI, CLKIN_MAINPLL, true, false},
#endif
#if defined(BOARD_NXP_LPCLINK2_4370)
	{CLK_BASE_ADCHS, CLKIN_MAINPLL, true, true},
#endif
	{CLK_BASE_PHY_TX, CLKIN_ENET_TX, true, false},
#if defined(USE_RMII)
	{CLK_BASE_PHY_RX, CLKIN_ENET_TX, true, false},
#else
	{CLK_BASE_PHY_RX, CLKIN_ENET_RX, true, false},
#endif
	{CLK_BASE_SDIO, CLKIN_MAINPLL, true, false},
	{CLK_BASE_SSP0, CLKIN_MAINPLL, true, false},
	{CLK_BASE_SSP1, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART0, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART1, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART2, CLKIN_MAINPLL, true, false},
	{CLK_BASE_UART3, CLKIN_MAINPLL, true, false},
	{CLK_BASE_LCD, CLKIN_IDIVC, true, false},
	{CLK_BASE_USB1, CLKIN_IDIVD, true, true},
};
	for (i = 0; i < (sizeof(InitClkState) / sizeof(InitClkState[0])); i++) {
		/* Get the Base clock settings */
		Chip_Clock_GetBaseClockOpts(InitClkState[i].clk, &InitClkState[i].clkin,
									&InitClkState[i].autoblock_enab, &InitClkState[i].powerdn);

		/* Set IRC as clock input for all the base clocks */
		Chip_Clock_SetBaseClock(InitClkState[i].clk, CLKIN_IRC, InitClkState[i].autoblock_enab,
								InitClkState[i].powerdn);
	}
}
void CopyInterruptToSRAM(void)
{
	unsigned int * flashPtr, * ramPtr;
  unsigned int * uLimit = (unsigned int *) 0x200;

		ramPtr = (unsigned int *)SRAMVECTORTABLE;	//load RAM starting at 0x10088000,
		flashPtr = (unsigned int *)0x00;			//start of interrupt vector table
	  while(flashPtr < uLimit)
			{
				*ramPtr = *flashPtr;
				ramPtr++;
				flashPtr++;
			}
}
