/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file:	ATO.c
*
* @author:	B34195
*
* @date: 	June, 2017
*
* @brief: 	Source file with implementation of the Angle Tracking Observer.
* 			The ATO can be used to evaluate the position and speed from
* 			resolver, encoder or even sensorless control.
*
***************************************************************************/

#include "ATO.h"

/***************************************************************************//*!
@brief          Function clears the state variables in the ATO module

@param[in,out]  ptr	Pointer to ATO observer data structure

@return         tBool

@details
******************************************************************************/
tBool ATO_clear_module(ATO_observer_t *ptr){
	ptr->fltThRotEl 		= 0.F;
	ptr->fltThRotMec 		= 0.F;
	ptr->fltThRotErr 		= 0.F;
	ptr->fltThRotElOffset	= 0.F;

	ptr->wRotMec.filt		= 0.F;
	ptr->wRotMec.raw		= 0.F;
	ptr->wRotEl.filt		= 0.F;
	ptr->wRotEl.raw			= 0.F;

	// clear state variables of ATO
	AMCLIB_TrackObsrvInit_FLT(&ptr->ATO);
	// clear state variables of MAF
	GDFLIB_FilterMAInit_FLT(&ptr->filterMA);

	return (TRUE);
}

/***************************************************************************//*!
@brief          Function calculates the mechanical speed and position

@param[in,out]  ptr	Pointer to ATO observer data structure

@return         tBool

@details		Raw value of mechanical speed - output of ATO function
 	 	 	 	Mechanical position - output of ATO function
 	 	 	 	Due to the float implementation, the position in manually
 	 	 	 	wrapped to be within a range -PI to +PI
******************************************************************************/
tBool ATO_calculation(ATO_observer_t *ptr){

	// call ATO function from amclib.h, returns Mechanical speed and position
	AMCLIB_TrackObsrv_FLT( ptr->fltThRotErr, & ptr->fltThRotMec, &ptr->wRotMec.raw, &ptr->ATO);

	// Mechanical rotor position - transformation in to the range <-pi,pi> - float
	if (ptr->fltThRotMec > FLOAT_PI){
			ptr->ATO.pParamInteg.fltState = MLIB_Sub(ptr->ATO.pParamInteg.fltState, FLOAT_2_PI);
			ptr->fltThRotMec	= ptr->ATO.pParamInteg.fltState;
		}

	if (ptr->fltThRotMec<-FLOAT_PI){
			ptr->ATO.pParamInteg.fltState = MLIB_Add(ptr->ATO.pParamInteg.fltState, FLOAT_2_PI);
			ptr->fltThRotMec	= ptr->ATO.pParamInteg.fltState;
		}

	return (TRUE);
}

/***************************************************************************//*!
@brief          Function calculates the electrical motor position

@param[in,out]  ptr	Pointer to ATO observer data structure

@return         tBool

@details		Recalculate the mechanical position to electrical one, using
				an overflow feature of FIX32 arithmetic to wrap the position.
				The angle offset correction is possible.

******************************************************************************/

tBool ATO_get_position_el(ATO_observer_t *ptr){
	 static tFrac32 f32ThRotMe,f32ThRotEl,f32ThRotElOffset;

  	// Mechanical rotor position - convert from FLOAT to FIX, <-1,1> range
	f32ThRotMe 			= MLIB_ConvertPU_F32FLT(MLIB_Div(ptr->fltThRotMec,FLOAT_PI));

  	// Electrical rotor offset position - convert from FLOAT to FIX, <-1,1> range
	f32ThRotElOffset	= MLIB_ConvertPU_F32FLT(MLIB_Div(ptr->fltThRotElOffset,FLOAT_PI));

    // Electrical rotor position calculation respecting the resolver position offset - fix point arithmetic
    f32ThRotEl			= MLIB_ShL_F32(MLIB_MulSat_F32(f32ThRotMe,ptr->s32motorPPscale),ptr->s16motorPPscaleShift);
    f32ThRotEl			= MLIB_Sub_F32(f32ThRotEl,f32ThRotElOffset);

    // Electrical rotor position -> convert from FIX to FlOAT, <-pi, pi> range
    ptr->fltThRotEl		= MLIB_Mul(MLIB_ConvertPU_FLTF32(f32ThRotEl), FLOAT_PI);

	return (TRUE);
}

/***************************************************************************//*!
@brief          Function calculates the electrical motor speed

@param[in,out]  ptr	Pointer to ATO observer data structure

@return         tBool

@details		Recalculate the filtered mechanical speed to electrical one.
				Electrical speed is a product of mechanical speed and motor pole-pairs.

******************************************************************************/
tBool ATO_get_w_speed_el(ATO_observer_t *ptr){

	// Filtering (Moving average filter) of the mechanical speed value
	ptr->wRotMec.filt 	= GDFLIB_FilterMA(ptr->wRotMec.raw,&(ptr->filterMA));

	// calculate motor electrical position - raw value
	ptr->wRotEl.raw 	= MLIB_Mul(ptr->wRotMec.raw, ptr->fltmotorPP);
	// calculate motor electrical position - filtered value
	ptr->wRotEl.filt 	= MLIB_Mul(ptr->wRotMec.filt, ptr->fltmotorPP);

	return (TRUE);
}


/***************************************************************************//*!
@brief          Function calculates the ATO angle error for resolver

@param[in,out]  ato_ptr	Pointer to ATO observer data structure
				ptr		Pointer to resolver data structure

@return         tBool

@details		Calculate the angle error using a dedicated formula. This approach
  	  	  	  	is used mainly for resolver, where resolver SIN and COS signals
  	  	  	  	are measured
******************************************************************************/
tBool ATO_theta_err_calc_resolver(ATO_observer_t *ato_ptr, resolver_data_t *ptr ){
	static tFloat  fltCosEst;
	static tFloat  fltSinEst;

    /* Calculate phase error of the position signals based on sine angle difference formula 		*/
    /* sin(ThMeas-ThEst) = sin(ThMeas)cos(ThEst) - cos(ThMeas)sin(ThEst) 	*/
    /* sin(ThMeas-ThEst) = ThMeas-ThEst for error smaller than 7deg			*/
    fltSinEst                    = GFLIB_Sin(ato_ptr->fltThRotMec);
    fltCosEst                    = GFLIB_Cos(ato_ptr->fltThRotMec);

    ato_ptr->fltThRotErr = MLIB_Sub(MLIB_Mul(ptr->ResSin.filt,fltCosEst), MLIB_Mul(ptr->ResCos.filt,fltSinEst));

	return (TRUE);
}

/***************************************************************************//*!
@brief          Function calculates the ATO angle error for encoder

@param[in,out]  ato_ptr	Pointer to ATO observer data structure
				ptr		Pointer to encoder data structure

@return         tBool

@details		Calculate the angle error. This approach is used mainly for
				encoders, where the information of the position is directly
				measured
******************************************************************************/
tBool ATO_theta_err_calc_encoder(ATO_observer_t *ato_ptr, encoder_data_t *ptr){

    /* Calculate phase error of the position signals */
    /* ThErr = ThMeas - ThEst 	*/
	ato_ptr->fltThRotErr = MLIB_Sub(ptr->encEtmrRotorPosition, ato_ptr->fltThRotMec);

	return (TRUE);
}
