/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file:	Current_meas.c
*
* @author:	B34195
*
* @date: 	Aug 15, 2016
*
* @brief: 	Source file with implementation of functions initializing, interfacing
* 			on-chip ADC data registers to read phase current and dc-bus current.
*
***************************************************************************/
#ifndef BSP_CURRENT_MEAS_C_
#define BSP_CURRENT_MEAS_C_

#include "Current_meas.h"

/***************************************************************************//*!
@brief          Function for configuration the HW for phase currents measurement

@param[in,out]  hw_ptr	Pointer to ph-current measurement HW configuration structure
				currentID - select one of two available HW configurations

@return         tBool

@details		Initialize the HW configuration according the selected PICe
 	 	 	 	motor connector. 3 phase currents measurement
******************************************************************************/
tBool PhCurrent_meas_hw_init(ph_current_meas_hw_cfg_t *hw_ptr, tU16 currentID){
    switch (currentID)
    {
        case CURR_HW_PCIe_J1:
        	hw_ptr->pAdcIphAdata = (volatile tU32 *)(CURR_ADC0_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH11));
        	hw_ptr->pAdcIphBdata = (volatile tU32 *)(CURR_ADC1_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH12));
        	hw_ptr->pAdcIphCdata = (volatile tU32 *)(CURR_ADC0_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH13));
            break;
        case CURR_HW_PCIe_J200:
        	hw_ptr->pAdcIphAdata = (volatile tU32 *)(CURR_ADC2_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH0));
        	hw_ptr->pAdcIphBdata = (volatile tU32 *)(CURR_ADC3_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH1));
        	hw_ptr->pAdcIphCdata = (volatile tU32 *)(CURR_ADC3_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH4));
            break;
        default:
        	hw_ptr->pAdcIphAdata = (volatile tU32 *)(CURR_ADC0_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH11));
        	hw_ptr->pAdcIphBdata = (volatile tU32 *)(CURR_ADC1_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH12));
        	hw_ptr->pAdcIphCdata = (volatile tU32 *)(CURR_ADC0_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH13));
            break;
    }
	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for configuration the HW for DC bus current measurement

@param[in,out]  hw_ptr	Pointer to DC bus current measurement HW configuration structure
				currentID - select one of two available HW configurations

@return         tBool

@details		Initialize the HW configuration according the selected PICe
 	 	 	 	motor connector. DC bus current measurement.
******************************************************************************/
tBool DcbCurrent_meas_hw_init(dcb_current_meas_hw_cfg_t *hw_ptr, tU16 currentID){
    switch (currentID)
    {
        case CURR_HW_PCIe_J1:
        	hw_ptr->pAdcIdcdata = (volatile tU32 *)(CURR_ADC0_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH11));
            break;
        case CURR_HW_PCIe_J200:
        	hw_ptr->pAdcIdcdata = (volatile tU32 *)(CURR_ADC2_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH0));
            break;
        default:
        	hw_ptr->pAdcIdcdata = (volatile tU32 *)(CURR_ADC0_BASE + ADC_CDRn_OFFSET + (4 * CURR_ADC_CH11));
            break;
    }
	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for reading ADC data registers containing converted
                values of ph-current signals.

@param[in,out]  hw_ptr	Pointer to ph-current measurement HW configuration structure
				d_ptr	Pointer to ph-current measurement data structure

@return         tBool

@details		This approach allows to measure all three currents at any time.
				However, this cannot be used to 100% duty-cycle. User must ensure
				that minimal PWM pulse width is wider than ADC sampling time.
******************************************************************************/
tBool PhCurrent3Ph_get_data(ph_current_meas_hw_cfg_t *hw_ptr ,ph_current_meas_data_t *d_ptr, tFloat Imax){

	d_ptr->raw.fltArg1 = MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphAdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);
	d_ptr->raw.fltArg2 = MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphBdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);
	d_ptr->raw.fltArg3 = MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphCdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);


	d_ptr->filt.fltArg1 = MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg1,d_ptr->raw.fltArg1),2);
	d_ptr->filt.fltArg2 = MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg2,d_ptr->raw.fltArg2),2);
	d_ptr->filt.fltArg3 = MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg3,d_ptr->raw.fltArg3),2);

	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for reading ADC data registers containing converted
                values of ph-current signals.

@param[in,out]  hw_ptr	Pointer to ph-current measurement HW configuration structure
				d_ptr	Pointer to ph-current measurement data structure

@return         tBool

@details		This approach allows to measure only two currents at any time.
				This approach is used measure current even up to 100% duty-cycle.
				User must ensure proper calibration of ADC channels, since it is
				dependent to the PWM sector.
******************************************************************************/
tBool PhCurrent2Ph_get_data(ph_current_meas_hw_cfg_t *hw_ptr ,ph_current_meas_data_t *d_ptr, tFloat Imax){

	switch(d_ptr->svmSector)
		{
			case 2:
			case 3:
					{
						d_ptr->raw.fltArg1	= MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphAdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);
						d_ptr->raw.fltArg3	= MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphCdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);

						d_ptr->filt.fltArg1	= MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg1,d_ptr->raw.fltArg1),2);
						d_ptr->filt.fltArg3	= MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg3,d_ptr->raw.fltArg3),2);

						d_ptr->filt.fltArg2	= MLIB_Neg(MLIB_Add(d_ptr->filt.fltArg1,d_ptr->filt.fltArg3));
	                    return(1);
					}
			case 4:
			case 5:
	                {
						d_ptr->raw.fltArg1	= MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphAdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);
						d_ptr->raw.fltArg2	= MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphBdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);

						d_ptr->filt.fltArg1	= MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg1,d_ptr->raw.fltArg1),2);
						d_ptr->filt.fltArg2	= MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg2,d_ptr->raw.fltArg2),2);

						d_ptr->filt.fltArg3	= MLIB_Neg(MLIB_Add(d_ptr->filt.fltArg1,d_ptr->filt.fltArg2));

	                    return(1);
	                }
			case 1:
			case 6:
			default:
	                {
						d_ptr->raw.fltArg2	= MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphBdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);
						d_ptr->raw.fltArg3	= MLIB_Mul(((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIphCdata)) & 0x00000FFF), (tFloat)0x00000FFF)),Imax);

						d_ptr->filt.fltArg3	= MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg3,d_ptr->raw.fltArg3),2);
						d_ptr->filt.fltArg2	= MLIB_Mul(MLIB_Sub(d_ptr->dcOffset.fltArg2,d_ptr->raw.fltArg2),2);

						d_ptr->filt.fltArg1	= MLIB_Neg(MLIB_Add(d_ptr->filt.fltArg2,d_ptr->filt.fltArg3));

	                    return(1);
	                }
	    }
}
/***************************************************************************//*!
@brief          Function for reading ADC data registers containing converted
                values of ph-current signals.

@param[in,out]  hw_ptr	Pointer to ph-current measurement HW configuration structure
				d_ptr	Pointer to ph-current measurement data structure

@return         tBool

@details
******************************************************************************/
tBool DcbCurrent_get_data(dcb_current_meas_hw_cfg_t *hw_ptr ,dcb_current_meas_data_t *d_ptr, tFloat Imax){

	d_ptr->I_dcb = ((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcIdcdata)) & 0x00000FFF), (tFloat)0x00000FFF))*Imax;

	return(TRUE);
}
/***************************************************************************//*!
@brief          Function performs the calibration of all phase currents ADC inputs


@param[in,out]  hw_ptr	Pointer to ph-current measurement HW configuration structure
				d_ptr	Pointer to ph-current measurement data structure

@return         tBool

@details		Calibration of ph-current measurement ADC inputs to calibrate the DC offset
				of all phases

@note
******************************************************************************/
tBool PhCurrent_calibration(ph_current_meas_hw_cfg_t *hw_ptr, ph_current_meas_data_t *d_ptr){

	tBool statusPass = FALSE;

	if(!d_ptr->calib.calibInitDone){

		// Clearing the DC offset values of 3-phase currents
		d_ptr->dcOffset.fltArg1 = 0.F;
		d_ptr->dcOffset.fltArg2 = 0.F;
		d_ptr->dcOffset.fltArg3 = 0.F;

		d_ptr->calib.calibCntr = CURR_CALIB_CNTR;

		d_ptr->calib.calibInitDone 	= 1;
		d_ptr->calib.calibDone 		= 0;
	}

	if(d_ptr->calib.calibCntr>0){
		// 3-phase currents - DC offset data filtering using MA recursive filter
		d_ptr->dcOffset.fltArg1 = GDFLIB_FilterMA(d_ptr->raw.fltArg1, &d_ptr->MAF_PhA);
		d_ptr->dcOffset.fltArg2 = GDFLIB_FilterMA(d_ptr->raw.fltArg2, &d_ptr->MAF_PhB);
		d_ptr->dcOffset.fltArg3 = GDFLIB_FilterMA(d_ptr->raw.fltArg3, &d_ptr->MAF_PhC);

		d_ptr->calib.calibCntr--;
	}
	else{
		// set calibration done flag
		d_ptr->calib.calibDone 			= 1;

		// clear calibration init flag to allow next calibration
		d_ptr->calib.calibInitDone 		= 0;

		statusPass = TRUE;
	}

	return(statusPass);
}

/***************************************************************************//*!
@brief          Function clears the state variables in the phase current meas. module

@param[in,out]  ptr	Pointer to ph-current data structure

@return         tBool

@details
******************************************************************************/
tBool PhCurrent_module_clear(ph_current_meas_data_t *ptr){
	ptr->MAF_PhA.fltAcc 		= 0.F;
	ptr->MAF_PhB.fltAcc 		= 0.F;
	ptr->MAF_PhC.fltAcc 		= 0.F;

	ptr->raw.fltArg1			= 0.F;
	ptr->raw.fltArg2			= 0.F;
	ptr->raw.fltArg3			= 0.F;

	ptr->filt.fltArg1			= 0.F;
	ptr->filt.fltArg2			= 0.F;
	ptr->filt.fltArg3			= 0.F;

	ptr->dcOffset.fltArg1		= 0.F;
	ptr->dcOffset.fltArg2		= 0.F;
	ptr->dcOffset.fltArg3		= 0.F;

	return (TRUE);
}

#endif /* BSP_CURRENT_MEAS_C_ */
