/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file     MC33907_routines.c
*
* @author   B15651
* 
* @version  1.0.1.0
* 
* @date     Mar-12-2013
* 
* @brief    MC33907 PowerSBC Init function file, with routines for
*           setup and diagnostic.
*
*******************************************************************************
*
* Detailed Description of the file.
*
******************************************************************************/

/*****************************************************************************
* External objects
******************************************************************************/
#include "MC33907_routines.h"

/****************************************************************************\
* MODE functions
\****************************************************************************/

/***************************************************************************//*!
*   @brief Switches actual mode of PwSBC to the normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit Normal in the MODE register to switch
*					PwSBC into the Normal mode and checks whether the mode
*					was entered or not.
* 	@return 	0 - Device entered the Normal mode successfully. <br>
*				1 - Operation failed.
*	@remarks 	Normal mode has higher priority than the low power modes
*				(LPON, LPOFF).
********************************************************************************/
tU16 pwSbc_SetNormalMode(MC33907_SET_T *ptr)
{

	/* Set normal mode bit */
	ptr->RegConfigData.mode.B.Normal	= 1;
	/* Define Register Address */
	ptr->RegConfigData.mode.B.ADR		= MODE_ADR;
	/* Set Write bit */
	ptr->RegConfigData.mode.B.RW		= 1;

	ptr->dspiSet.transmitData = ptr->RegConfigData.mode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if((ptr->dspiSet.receivedData & 0x0004))
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}

}

/***************************************************************************//*!
*   @brief Switches actual mode of PwSBC to the low power ON mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit LPON in the MODE register to switch PwSBC
*					into the Low power Vreg ON mode and checks whether the mode
*					was entered or not.
* 	@return 	0 - Device entered the LPON mode successfully. <br>
*				1 - Operation failed.
*	@remarks 	LPON mode has a higher priority than LPOFF, but lower than
*				the Normal mode.
********************************************************************************/
tU16 pwSbc_ActivateLpOnMode(MC33907_SET_T *ptr){

	/* Set Write bit */
	ptr->RegConfigData.mode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.mode.B.LPON		= 1;
	/* Define Register Address */
	ptr->RegConfigData.mode.B.ADR		= MODE_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.mode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Switches actual mode of PwSBC to the low power OFF mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit LPOFF in the MODE register to switch PwSBC
*					into the Low power Vreg OFF mode and checks whether the mode
*					was entered or not.
* 	@return 	0 - Device entered the LPOFF mode successfuly. <br>
*				1 - Operation failed.
*	@remarks 	Normal and LPON modes have higher priority than LPOFF.
********************************************************************************/
tU16 pwSbc_ActivateLpOffMode(MC33907_SET_T *ptr)
{

	/* Set Write bit */
	ptr->RegConfigData.mode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.mode.B.LPOFF		= 1;
	/* Define Register Address */
	ptr->RegConfigData.mode.B.ADR		= MODE_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.mode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 1)
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Requests an interrupt.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit INT_request in the MODE register, which
*					will generate an interrupt.
*	@remarks 	INT_request will be cleared automatically when the interrupt is done.
********************************************************************************/
void pwSbc_RequestIntPuls(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.mode.B.RW			= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.mode.B.INT_request	= 1;
	/* Define Register Address */
	ptr->RegConfigData.mode.B.ADR			= MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.mode.R;

	PwSBC_SendSecureCmdW(ptr);
}

/****************************************************************************\
* Vreg_mode functions
\****************************************************************************/

/***************************************************************************//*!
*   @brief Enables Vcore regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit Vcore_EN in the Vreg Mode register, which
*					enables the Vcore power supply.
* 	@return 	0 - Vcore was successfuly enabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
//tU32 PwSBC_EnableVcore(MC33907_SET_T *ptr){
//	Vreg_mode_Tx_32B_tag cmd;
//	cmd.R = 0;
//
////	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
//	cmd.B.RW = 1;									//write command
//	cmd.B.ADR = VREG_MODE_ADR;						//set address
//	cmd.B.Vcore_EN = 1;								//enable Vcore
//	ptr->	= 1;
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(ptr->dspiSet.receivedData == 1){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}

/***************************************************************************//*!
*   @brief Disables Vcore regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit Vcore_EN in the Vreg Mode register,
*					which disables the Vcore power supply.
* 	@return 	0 - Vcore was successfuly disabled. <br>
*				1 - Operation failed.
*	@remarks 	Do not disable Vcore if it is configured as a safety critical.
********************************************************************************/
//tU32 PwSBC_DisableVcore(){
//	Vreg_mode_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	if(INTstruct.INIT_Supervisor1.B.Vcore_FS != 0){
//		return 2;									//error -> Vcore is safety critical
//	}
//
//	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
//	cmd.B.RW = 1;									//write command
//	cmd.B.ADR = VREG_MODE_ADR;						//set address
//	cmd.B.Vcore_EN = 0;								//disable Vcore
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_mode.B.Vcore_EN == 0){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}

/***************************************************************************//*!
*   @brief Enables Vcca regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit Vcca_EN in the Vreg Mode register,
*					which enables the Vcca power supply.
* 	@return 	0 - Vcca was successfuly enabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_EnableVcca(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.vregMode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.vregMode.B.Vcca_EN	= 1;
	/* Define Register Address */
	ptr->RegConfigData.vregMode.B.ADR		= VREG_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.vregMode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData & 0x4){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Disables Vcca regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit Vcca_EN in the Vreg Mode register,
*					which disables the Vcca power supply.
* 	@return 	0 - Vcca was successfuly disabled. <br>
*				1 - Operation failed.
*	@remarks 	Do not disable Vcca if it is configured as a safety critical.
********************************************************************************/
tU16 pwSbc_DisableVcca(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.vregMode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.vregMode.B.Vcca_EN	= 0;
	/* Define Register Address */
	ptr->RegConfigData.vregMode.B.ADR		= VREG_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.vregMode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(!(ptr->dspiSet.receivedData & 0x4)){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Enables Vaux regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit Vaux_EN in the Vreg Mode register, which
*					enables the Vaux power supply.
* 	@return 	0 - Vaux was successfuly enabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_EnableVaux(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.vregMode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.vregMode.B.Vaux_EN	= 1;
	/* Define Register Address */
	ptr->RegConfigData.vregMode.B.ADR		= VREG_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.vregMode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData & 0x2){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Disables Vaux regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit Vaux_EN in the Vreg Mode register,
*					which disables the Vaux power supply.
* 	@return 	0 - Vaux was successfuly disabled. <br>
*				1 - Operation failed.
*	@remarks 	Do not disable Vaux if it is configured as a safety critical.
********************************************************************************/
tU16 pwSbc_DisableVaux(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.vregMode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.vregMode.B.Vaux_EN	= 0;
	/* Define Register Address */
	ptr->RegConfigData.vregMode.B.ADR		= VREG_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.vregMode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(!(ptr->dspiSet.receivedData & 0x2)){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}


/***************************************************************************//*!
*   @brief Enables Vcan regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit Vcan_EN in the Vreg Mode register, which
*					enables the Vcan power supply.
* 	@return 	0 - Vcan was successfuly enabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_EnableVcan(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.vregMode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.vregMode.B.Vcan_EN	= 1;
	/* Define Register Address */
	ptr->RegConfigData.vregMode.B.ADR		= VREG_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.vregMode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData & 0x1)
	{
		return 0;									//success
	}
	else
	{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Disables Vcan regulator during Normal mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit Vcan_EN in the Vreg Mode register,
*					which disables the Vcan power supply.
* 	@return 	0 - Vcan was successfuly disabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_DisableVcan(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.vregMode.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.vregMode.B.Vcan_EN	= 0;
	/* Define Register Address */
	ptr->RegConfigData.vregMode.B.ADR		= VREG_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.vregMode.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(!(ptr->dspiSet.receivedData & 0x1))
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/****************************************************************************\
* IO_out - AMUX functions
\****************************************************************************/

/***************************************************************************//*!
*   @brief Enables output gate driver for IO pins (IO4 and IO5).
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit IO_out_4_EN (register IO_out-AMUX),
*					which enables output gate driver for IO_4.
* 	@return 	0 - Output gate driver was successfuly enabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_EnableOut4(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.ioOutAmux.B.IO_out_4_EN	= 1;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Sets specified IO output (logic High).
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit IO_out_4 (register IO_out-AMUX), which
*					sets IO_4 to logical high state.
* 	@return 	0 - Output was successfuly set to high state. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_SetOut4(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.ioOutAmux.B.IO_out_4	= 1;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Clears specified IO output (logic Low).
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit IO_out_4 (register IO_out-AMUX), which
*					sets IO_4 to logical low state.
* 	@return 	0 - Output was successfuly set to low state. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_ClearOut4(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.ioOutAmux.B.IO_out_4	= 0;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Enables output gate driver for IO pins (IO4 and IO5).
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit IO_out_5_EN (register IO_out-AMUX), which
*					enables output gate driver for IO_5.
* 	@return 	0 - Output gate driver was successfuly enabled. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_EnableOut5(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set normal mode bit */
	ptr->RegConfigData.ioOutAmux.B.IO_out_5_EN	= 1;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Sets specified IO output (logic High).
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit IO_out_5 (register IO_out-AMUX), which
*					sets IO_5 to logical high state.
* 	@return 	0 - Output was successfuly set to high state. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_SetOut5(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.ioOutAmux.B.IO_out_5	= 1;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Clears specified IO output (logic Low).
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit IO_out_5 (register IO_out-AMUX), which
*					sets IO_5 to logical low state.
* 	@return 	0 - Output was successfuly set to low state. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_ClearOut5(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.ioOutAmux.B.IO_out_5	= 0;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 			= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief Switches desired channel to the output of AMUX.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function switches AMUX channel to the desired one
*					(bits Amux_2:0 in the IO_out-AMUX register.
* 	@param[in] nbChannel - Number of the desired AMUX channel (from 0 till 7).
* 	@return 	0 - AMUX channel successfuly switched. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_switchAmuxChannel(MC33907_SET_T *ptr, tU16 nbChannel)
{
	if(nbChannel > 7)
	{
		return 2;									//error -> bad number of AMUX channel
	}
	/* Set Write bit */
	ptr->RegConfigData.ioOutAmux.B.RW		= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.ioOutAmux.B.Amux		= nbChannel;
	/* Define Register Address */
	ptr->RegConfigData.ioOutAmux.B.ADR		= IO_OUT_AMUX_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.ioOutAmux.R;

	PwSBC_SendCmdRW(ptr);
	if((ptr->dspiSet.receivedData & 0x7) == nbChannel)
	{
		return 0;									//success
	}
	else
	{
		return 1;									//error
	}
}

/***************************************************************************//*
* CAN-LIN mode functions
******************************************************************************/

/***************************************************************************//*!
*   @brief 			Sets CAN mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function changes CAN mode in the CAN-LIN mode register.
* 	@param[in] mode - Desired CAN mode (from 0 till 3).
* 	@return 	0 - CAN mode was successfuly changed. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_SetCanMode(MC33907_SET_T *ptr, tU16 mode)
{
	if(mode > 3)
	{
		return 2;									//error -> bad number of mode
	}
	/* Set Write bit */
	ptr->RegConfigData.canLinMode.B.RW		= 1;
	/* configure CAN mode */
	ptr->RegConfigData.canLinMode.B.CAN_mode	= mode;
	/* Define Register Address */
	ptr->RegConfigData.canLinMode.B.ADR		= CAN_LIN_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.canLinMode.R;

	PwSBC_SendCmdRW(ptr);
	if(((ptr->dspiSet.receivedData >> 6) & 0x3) == mode)
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief			Sets Automatic CAN Tx disable.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sets bit CAN_auto_dis in the CAN-LIN mode register.
*					This will enable CAN transceiver to switch its mode from
*					"Normal" to "Listen only" automatically when an event (CAN
*					overtemperature/TXD dominant/RXD recessive) occurs.
* 	@return 	0 - CAN_auto_dis bit was successfuly set. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_SetAutoCanTxDisable(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.canLinMode.B.RW		= 1;
	/* configure CAN Automatic disable bit */
	ptr->RegConfigData.canLinMode.B.CAN_auto_dis	= 1;
	/* Define Register Address */
	ptr->RegConfigData.canLinMode.B.ADR		= CAN_LIN_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.canLinMode.R;

	PwSBC_SendCmdRW(ptr);
	if(ptr->dspiSet.receivedData & 0x20)
	{
			return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief 			No Automatic CAN Tx disable.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit CAN_auto_dis in the CAN-LIN mode register.
*					In this configuration CAN transceiver will not change its
*					mode automatically if any event occurs.
* 	@return 	0 - CAN_auto_dis bit was successfuly cleared. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_ClearAutoCanTxDisable(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.canLinMode.B.RW		= 1;
	/* configure CAN Automatic disable bit */
	ptr->RegConfigData.canLinMode.B.CAN_auto_dis	= 0;
	/* Define Register Address */
	ptr->RegConfigData.canLinMode.B.ADR		= CAN_LIN_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.canLinMode.R;

	PwSBC_SendCmdRW(ptr);
	if(!(ptr->dspiSet.receivedData & 0x20))
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief 			Configure LIN mode.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function changes LIN mode in the CAN-LIN mode register.
* 	@param[in] mode - Desired LIN mode (from 0 till 3).
* 	@return 	0 - LIN mode was successfully changed. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_SetLinMode(MC33907_SET_T *ptr, tU16 mode)
{
	if(mode > 3)
	{
		return 2;									//error -> bad number of mode
	}
	/* Set Write bit */
	ptr->RegConfigData.canLinMode.B.RW		= 1;
	/* configure CAN mode */
	ptr->RegConfigData.canLinMode.B.LIN_mode	= mode;
	/* Define Register Address */
	ptr->RegConfigData.canLinMode.B.ADR		= CAN_LIN_MODE_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.canLinMode.R;

	PwSBC_SendCmdRW(ptr);
	if(((ptr->dspiSet.receivedData >> 3) & 0x3) == mode)
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*
* Vreg_LPON functions
******************************************************************************/

/**************************************************************************//*!
*   @brief Enables low power mode for Vcore.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function clears bit Vcore_LP (Vreg_LP register). In this
*					configuration Vcore will be automatically switched OFF in
*					LPON mode.
* 	@return 	0 - Vcore_LP bit was successfuly cleared. <br>
*				1 - Operation failed.
*	@remarks
******************************************************************************/
//tU32 PwSBC_EnableVcoreLowPower(){
//	Vreg_LPON_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	if(INTstruct.INIT_Supervisor1.B.Vcore_FS != 0){
//		return 2;										//error -> Vcore is safety critical
//	}
//
//	cmd.R = INTstruct.Vreg_LPON.R << 5;					//restore old register setting
//	cmd.B.RW = 1;										//write command
//	cmd.B.ADR = VREG_LPON_ADR;							//set address
//	cmd.B.Vcore_LP = 0;									//enable Low Power mode for Vcore
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_LPON.B.Vcore_LP == 0){
//		return 0;										//success
//	}
//	else{
//		return 1;										//error
//	}
//}

///***************************************************************************//*!
//*   @brief Disables low power mode for Vcore.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function sets bit Vcore_LP (Vreg_LP register). In this
//*					configuration Vcore will be ON during the LPON mode.
//* 	@return 	0 - Vcore_LP bit was successfuly set. <br>
//*				1 - Operation failed.
//*	@remarks
//********************************************************************************/
//tU32 PwSBC_DisableVcoreLowPower(){
//	Vreg_LPON_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	cmd.R = INTstruct.Vreg_LPON.R << 5;					//restore old register setting
//	cmd.B.RW = 1;										//write command
//	cmd.B.ADR = VREG_LPON_ADR;							//set address
//	cmd.B.Vcore_LP = 1;									//disable Low Power mode for Vcore
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_LPON.B.Vcore_LP == 1){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}
//
///***************************************************************************//*!
//*   @brief Enables low power mode for Vcca regulator.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function clears bit Vcca_LP (Vreg_LP register). In this
//*					configuration Vcca will be automatically switched OFF in
//*					LPON mode.
//* 	@return 	0 - Vcca_LP bit was successfuly cleared. <br>
//*				1 - Operation failed.
//*	@remarks
//********************************************************************************/
//tU32 PwSBC_EnableVccaLowPower(){
//	Vreg_LPON_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	if(INTstruct.INIT_Supervisor1.B.Vcca_FS != 0){
//		return 2;										//error -> Vcca is safety critical
//	}
//
//	cmd.R = INTstruct.Vreg_LPON.R << 5;					//restore old register setting
//	cmd.B.RW = 1;										//write command
//	cmd.B.ADR = VREG_LPON_ADR;							//set address
//	cmd.B.Vcca_LP = 0;											//enable Low Power mode for Vcca
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_LPON.B.Vcca_LP == 0){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}
//
///***************************************************************************//*!
//*   @brief Disables low power mode for Vcca regulator.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function sets bit Vcca_LP (Vreg_LP register). In this
//*					configuration Vcca will be ON during the LPON mode.
//* 	@return 	0 - Vcca_LP bit was successfuly set. <br>
//*				1 - Operation failed.
//*	@remarks
//********************************************************************************/
//tU32 PwSBC_DisableVccaLowPower(){
//	Vreg_LPON_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	cmd.R = INTstruct.Vreg_LPON.R << 5;					//restore old register setting
//	cmd.B.RW = 1;										//write command
//	cmd.B.ADR = VREG_LPON_ADR;							//set address
//	cmd.B.Vcca_LP = 1;									//disable Low Power mode for Vcca
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_LPON.B.Vcca_LP == 1){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}
//
///***************************************************************************//*!
//*   @brief Enables low power mode for Vaux regulator.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function clears bit Vaux_LP (Vreg_LP register). In this
//*					configuration Vaux will be automatically switched OFF in
//*					LPON mode.
//* 	@return 	0 - Vaux_LP bit was successfuly cleared. <br>
//*				1 - Operation failed.
//*	@remarks
//********************************************************************************/
//tU32 PwSBC_EnableVauxLowPower(){
//	Vreg_LPON_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	if(INTstruct.INIT_Supervisor2.B.Vaux_FS != 0){
//		return 2;										//error -> Vaux is safety critical
//	}
//
//	cmd.R = INTstruct.Vreg_LPON.R << 5;					//restore old register setting
//	cmd.B.RW = 1;										//write command
//	cmd.B.ADR = VREG_LPON_ADR;							//set address
//	cmd.B.Vaux_LP = 0;									//enable Low Power mode for Vaux
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_LPON.B.Vaux_LP == 0){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}
//
///***************************************************************************//*!
//*   @brief Disables low power mode for Vaux regulator.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function sets bit Vaux_LP (Vaux_LP register). In this
//*					configuration Vaux will be ON during the LPON mode.
//* 	@return 	0 - Vaux_LP bit was successfuly set. <br>
//*				1 - Operation failed.
//*	@remarks
//********************************************************************************/
//tU32 PwSBC_DisableVauxLowPower(){
//	Vreg_LPON_Tx_32B_tag cmd;
//	cmd.R = 0;
//
//	cmd.R = INTstruct.Vreg_LPON.R << 5;					//restore old register setting
//	cmd.B.RW = 1;										//write command
//	cmd.B.ADR = VREG_LPON_ADR;							//set address
//	cmd.B.Vaux_LP = 1;									//disable Low Power mode for Vaux
//	PwSBC_SendSecureCmdRW(cmd.R);
//	if(INTstruct.Vreg_LPON.B.Vaux_LP == 1){
//		return 0;									//success
//	}
//	else{
//		return 1;									//error
//	}
//}

/****************************************************************************\
* WD_window functions
\****************************************************************************/

/***************************************************************************//*!
*   @brief Changes WD window duration.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function changes WD window duration.
* 	@param[in] duration - WD window duration coded in binary (from 0 till 15).
* 	@return 	0 - WD window duration was successfuly set. <br>
*				1 - Operation failed.
*	@remarks
********************************************************************************/
tU16 pwSbc_ChangeWdWindow(MC33907_SET_T *ptr, tU16 duration){

	if(duration > 15)
	{
		return 2;										//error -> given parameter is OUT of range
	}

	/* Set Write bit */
	ptr->RegConfigData.wdWindow.B.RW		= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.wdWindow.B.WD_window	= duration;
	/* Define Register Address */
	ptr->RegConfigData.wdWindow.B.ADR		= WD_WINDOW_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.wdWindow.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData == duration){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*
* WD_LFSR functions
******************************************************************************/

/**************************************************************************//*!
*   @brief 			Sends seed for LFSR from MCU to PwSBC.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sends new seed for the implemented LFSR.
* 	@param[in] seed - 	New seed for the LFSR. This can be any 8-bit value
*						excluding 0.
* 	@return 		0 - Seed was successfuly changed. <br>
*					1 - Operation failed. <br>
*					2 - Wrong seed parameter (cannot be 0).
*	@remarks 		Zero state is not allowed for this type of LFSR, because it
*					should never evolve from this state.
******************************************************************************/
tU16 pwSbc_SendSeed(MC33907_SET_T *ptr, tU16 seed)
{

	if(seed == 0)
	{
		return 2;				//error ->	seed CANNOT be zero
	}

	/* Set Write bit */
	ptr->RegConfigData.wdLfsr.B.RW			= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.wdLfsr.B.WD_LFSR		= seed;
	/* Define Register Address */
	ptr->RegConfigData.wdLfsr.B.ADR			= WD_LFSR_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.wdLfsr.R;

	PwSBC_SendSecureCmdRW(ptr);
	if(ptr->dspiSet.receivedData == seed)
	{
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*
* WD_answer functions
******************************************************************************/

/**************************************************************************//*!
*   @brief 			Refreshes WD.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function refreshes WD using the given WD answer. This is
*					done by writing into the WD answer register.
* 	@param[in] answer - 8-bit WD answer to be sent.
* 	@return 	0 - WD was successfuly refreshed. <br>
*				(10)D - SPI disconnected or no SPI answer. <br>
*				(11)D - SPI_G error detected.
*	@remarks 	8-bit answer has to be constisted of the control computations
*				that have been done on the actual LFSR content.
********************************************************************************/
tU16 pwSbc_RefreshWd(MC33907_SET_T *ptr)
{
	tU16 resp = 0;
	tU16 newLfsr = 0;

	/* Obtain actual WD LFSR register value */
	newLfsr = PwSBC_GetWDLFSR(ptr);

	ptr->RegConfigData.wdAnswer.B.RW		= 1;

	ptr->RegConfigData.wdAnswer.B.ADR		= WD_ANSWER_ADR;
	/* compute new LFSR value based on the actual register value */
	ptr->RegConfigData.wdAnswer.B.WD_answer	= pwSbc_ComputeLFSR(newLfsr);

	ptr->dspiSet.transmitData				= ptr->RegConfigData.wdAnswer.R;

	ptr->pwrSbcRegValues.wdAnswer.R 		=  PwSBC_SendCmdRW(ptr);

	ptr->pwrSbcRegValues.wdCounter.R		= PwSBC_GetWDcounter(ptr);
	ptr->pwrSbcRegValues.ioInputState.R		= PwSBC_GetIOinputState(ptr);

	return resp;										//returns error status from the previous function
}

/////***************************************************************************//*!
////*   @brief 	Generates, stores and returns a new state of LFSR from the previous
////*			one, that is stored in memory.
////*	@par Include:
////*					PwSBC.h
////* 	@par Description:
////*					Function evolves LFSR implemented in the MCU into the
////*					following state and stores the new value in a global
////*					structure.
////* 	@return 	New LFSR state.
////*	@remarks 	If this function is used, then the synchronization between LFSR
////*				implemented in the MCU and the one in PwSBC has to be guaranteed.
////********************************************************************************/
////tU32 PwSBC_GenerateLFSR(){
////	register32_struct gate;
////	gate.R = 0;
////
////	gate.B.bit0 = ~(PITstruct.currentLFSR.B.bit7 ^ PITstruct.currentLFSR.B.bit5);
////	gate.B.bit0 = ~(PITstruct.currentLFSR.B.bit4 ^ gate.B.bit0);
////	gate.B.bit0 = ~(PITstruct.currentLFSR.B.bit3 ^ gate.B.bit0);
////
////	PITstruct.currentLFSR.R <<= 1;
////	PITstruct.currentLFSR.B.bit0 = gate.B.bit0;
////
////	PITstruct.currentLFSR.R &= 0xFF;		//mask out only the lowest Byte
////
////	return PITstruct.currentLFSR.R;
////}
///********DEBUG***************************
//tU32 PwSBC_GenerateLFSR(){
//	register32_struct gate;
//	gate.R = 0;
//
//	gate.B.bit7 = ~(PITstruct.currentLFSR.B.bit0 ^ PITstruct.currentLFSR.B.bit2);
//	gate.B.bit7 = ~(PITstruct.currentLFSR.B.bit3 ^ gate.B.bit7);
//	gate.B.bit7 = ~(PITstruct.currentLFSR.B.bit4 ^ gate.B.bit7);
//
//	PITstruct.currentLFSR.R >>= 1;
//	PITstruct.currentLFSR.B.bit7 = gate.B.bit7;
//
//	return PITstruct.currentLFSR.R;
//}
//********************END DEBUG********************************/

/***************************************************************************//*!
*   @brief Computes, stores and returns test based from actual LFSR.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function makes control computations with the given LFSR and
*					returns result of the computation on the least 8 bits.
* 	@param[in] actualLFSR - 8-bit LFSR value on which will be applied control
*						computations.
* 	@return 	Result of the control computations.
*	@remarks 	Control computations are made in assembler using instructions
*				to prove ALU using basic operations (*, /, +, -).
********************************************************************************/
tU16 pwSbc_ComputeLFSR(tU16	actualLFSR)
{
	tU8	newLfsr;

	newLfsr		= (actualLFSR & 0xFF)<<2;
	newLfsr		+= 6;
	newLfsr		-= 4;
	newLfsr		= ((~newLfsr)>>2) & 0xFF;
	return(newLfsr);
}
//tU32 PwSBC_ComputeLFSR(tU32	actualLFSR){
//	/**
//	* ~(((lfsr*4 + 6 - 4)/4
//	*/
//	asm{
//		#if __option(vle)
//			se_li 		r25,0x04		//load nb.4 -> r25
//			mullw 		r3,r25,r3		//lfsr * 4 	-> r3
//			e_add16i	r3,r3,6			//r3 + 6 		-> r3
//			se_subi 	r3,4				//r3 - 4 		-> r3
//			se_not 	r3						//NOT r3 		-> r3
//			e_li 		r24,0xFFFF			//mask 			-> r24
//			se_and 	r3,r24				//r24 & r3  -> r3
//			divwu		r3,r3,r25			//r3 / 4		-> r3	-> as a return value
//			e_li 		r24,0xFF			//mask 			-> r24
//			se_and 	r3,r24				//store only lower 8 bits -> r3
//		#else
//			li r25,0x04						//load nb.4 -> r25
//			mullw r3,r25,r3				//lfsr * 4 	-> r3
//			addi r3,r3,6					//r3 + 6		-> r3
//			li r25,4							//load nb.4 -> r25
//			subf r3,r25,r3				//r3 - r25	-> r3
//			li r25,0xFFFF						//mask for negation	-> r25
//			nand r3,r3,r25				//r25 NAND r3 -> r3
//			li r25,4							//load nb.r3 -> r25
//			divwu r3,r3,r25				//r3 / r25	->	r3 -> as a return value
//			clrlwi r3,r3,24				//store only 8 lower bits
//
//		#endif
//	}

///*********************DEBUG********************
//	return (actualLFSR^0xFF);
//*********************END DEBUG***************/
//}

/***************************************************************************//*
* Fail_Safe_OUT functions
******************************************************************************/

/**************************************************************************//*!
*   @brief 			Releases Fail Safe outputs FS0b and FS1.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function swaps actual LFSR content (stored in the LFSR)
*					and sends command to release safe outputs (FS0b and FS1).
*	@remarks 	Command should be applied to release fail safe outputs when RST
*				error counter goes back to the state no. 0.
******************************************************************************/
void pwSbc_ReleaseFsOut(MC33907_SET_T *ptr)
{
	/* Get current LSFR register content */
	PwSBC_GetWDLFSR(ptr);

	/* Set Write bit */
	ptr->RegConfigData.fsOut.B.RW			= 1;
	/* Fill FS_OUT register with right values */
	ptr->RegConfigData.fsOut.B.FS_out_0		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_7;
	ptr->RegConfigData.fsOut.B.FS_out_1		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_6;
	ptr->RegConfigData.fsOut.B.FS_out_2		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_5;
	ptr->RegConfigData.fsOut.B.FS_out_3		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_4;
	ptr->RegConfigData.fsOut.B.FS_out_4		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_3;
	ptr->RegConfigData.fsOut.B.FS_out_5		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_2;
	ptr->RegConfigData.fsOut.B.FS_out_6		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_1;
	ptr->RegConfigData.fsOut.B.FS_out_7		= ~ptr->pwrSbcRegValues.wdLfsr.B.WD_LFSR_0;
	/* Define Register Address */
	ptr->RegConfigData.fsOut.B.ADR			= FS_OUT_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.fsOut.R;

	PwSBC_SendCmdW(ptr);
}

/****************************************************************************\
* RSTb_request functions
\****************************************************************************/

/***************************************************************************//*!
*   @brief Requests a low pulse on RSTb - MCU reset.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function requests one Reset pulse.
*	@remarks 	Reset can cause a data loss thanks to the restart of the MCU.
********************************************************************************/
void pwSbc_RequestReset(MC33907_SET_T *ptr)
{
	/* Set Write bit */
	ptr->RegConfigData.rstb.B.RW			= 1;
	/* Set IO4 to 1 */
	ptr->RegConfigData.rstb.B.RSTb_request	= 1;
	/* Define Register Address */
	ptr->RegConfigData.rstb.B.ADR			= RSTB_REQUEST_ADR;

	ptr->dspiSet.transmitData 				= ptr->RegConfigData.rstb.R;

	PwSBC_SendSecureCmdRW(ptr);

}

/****************************************************************************\
* Getters
\****************************************************************************\

\***************************************************************************//*!
*   @brief Changes actual mode to the desired one.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Functions return content of the XXX register.
* 	@return 	Content of the XXX register.
*	@remarks
********************************************************************************/

tU16 PwSBC_GetInitVreg1(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_VREG1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitVreg2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_VREG2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitCANLIN(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_CAN_LIN_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitIOWU1(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_IO_WU1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitIOWU2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_IO_WU2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitINT(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_INT_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetHWconfig(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (HW_CONFIG_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.hwConfig.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetWUsource(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (WU_SOURCE_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.wuSource.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetIOinputState(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (IO_INPUT_STATE_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.ioInputState.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetStatusVreg1(MC33907_SET_T *ptr)
{

	ptr->dspiSet.transmitData	= (STATUS_VREG1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.statusVreg1.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetStatusVreg2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (STATUS_VREG2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.initVreg2.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagVreg1(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (DIAG_VREG1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagVreg1.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagVreg2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (DIAG_VREG2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagVreg2.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagVreg3(MC33907_SET_T *ptr){
	ptr->dspiSet.transmitData	= (DIAG_VREG3_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagVreg3.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagCAN(MC33907_SET_T *ptr){
	ptr->dspiSet.transmitData	= (DIAG_CAN_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagCan.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagCANLIN(MC33907_SET_T *ptr){
	ptr->dspiSet.transmitData	= (DIAG_CAN_LIN_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagCanLin.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagSPI(MC33907_SET_T *ptr){

	ptr->dspiSet.transmitData	= (DIAG_SPI_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagSpi.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetMode(MC33907_SET_T *ptr){

	ptr->dspiSet.transmitData	= (MODE_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.mode.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetVregMode(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (VREG_MODE_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.vregMode.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetIOAMUX(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (IO_OUT_AMUX_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.ioOutAmux.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetCANLINmode(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (CAN_LIN_MODE_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.canLinMode.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetVregLP(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (VREG_LPON_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.vregLpOn.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitSupervisor1(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_SUPERVISOR1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.initSupervisor1.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitSupervisor2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_SUPERVISOR2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.initSupervisor2.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitSupervisor3(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_SUPERVISOR3_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.initSupervisor3.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitFSSM1(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_FSSM1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.initFssm1.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetInitFSSM2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (INIT_FSSM2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.initFssm2.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetWDwindow(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (WD_WINDOW_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.wdWindow.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetWDLFSR(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (WD_LFSR_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.wdLfsr.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetWDanswer(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (WD_ANSWER_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.wdAnswer.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagFS1(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (DIAG_FS1_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagFs1.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetWDcounter(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (WD_COUNTER_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.wdCounter.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}

tU16 PwSBC_GetDiagFS2(MC33907_SET_T *ptr)
{
	ptr->dspiSet.transmitData	= (DIAG_FS2_ADR << 9);
	PwSBC_SendCmdR(ptr);    //send a read command created from register address
	// update register value
	ptr->pwrSbcRegValues.diagFs2.R = ptr->dspiSet.receivedData;
	return (ptr->dspiSet.receivedData);
}


/****************************************************************************\
* Setters for INIT functions
\****************************************************************************\

\***************************************************************************//*!
*   @brief Changes actual mode to the desired one.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Functions set content of the XXX register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	0 - Register was successfuly set. <br>
*				1 - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity
*				setting and security bits in this command can be neglected,
*				because they will be recomputed by the function automatically.
********************************************************************************/

tU16 PwSBC_SetInitVreg1(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
	else
	{
		if((ptr->dspiSet.receivedData & 0x00FF) == (ptr->dspiSet.txCmd & 0x00FF)){		//compare lower Byte only
			return 0;							//success
		}
		else{
			return 1;							//error -> register content NOT verified
		}
	}
}

tU16 PwSBC_SetInitVreg2(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_Vreg2.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
//			return 0;							//success
//		}
//		else{
//			return 1;							//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitCANLIN(MC33907_SET_T *ptr){
	tU16 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_CAN_LIN.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
//			return 0;							//success
//		}
//		else{
//			return 1;							//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitIOWU1(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_IO_WU1.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
//			return 0;							//success
//		}
//		else{
//			return 1;							//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitIOWU2(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//
//		if((INTstruct.INIT_IO_WU2.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
//			return 0;							//success
//		}
//		else{
//			return 1;							//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitINT(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_INT.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitSupervisor1(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_Supervisor1.B.Vcca_FS == cmdTx.B.Vcca_FS) && (INTstruct.INIT_Supervisor1.B.Vcore_FS == cmdTx.B.Vcore_FS)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitSupervisor2(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_Supervisor2.B.Vaux_FS == cmdTx.B.Vaux_FS) && (INTstruct.INIT_Supervisor2.B.DIS_8s == cmdTx.B.DIS_8s)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitSupervisor3(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((INTstruct.INIT_Supervisor3.B.Vcore_5D == cmdTx.B.Vcore_5D) && (INTstruct.INIT_Supervisor3.B.Vcca_5D == cmdTx.B.Vcca_5D) && (INTstruct.INIT_Supervisor3.B.Vaux_5D == cmdTx.B.Vaux_5D)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitFSSM1(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if(((cmd & 0x000000FF) >> 4) == (INTstruct.INIT_FSSM1.R & 0x0000000F)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetInitFSSM2(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if(((cmd & 0x000000FF) >> 4) == (INTstruct.INIT_FSSM2.R & 0x0000000F)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

//tU32 PwSBC_SetMode(tU32 cmd){
//	tU32 errCode;
//
//	errCode = PwSBC_SendSecureCmdRW(cmd);
//
//	if(errCode != 0){
//		return errCode;							//error in the communication
//	}
//	else
//	{
//		if((((cmd & 0x000000FF) >> 5) == (INTstruct.MODE.R & 0x00000007)) || ((cmd & 0x0010) == 0x0010)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
//}

tU16 PwSBC_SetVregMode(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if(((cmd & 0x000000FF) >> 4) == (INTstruct.Vreg_mode.R & 0x0000000F)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetIOAMUX(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if(((cmd & 0x000000FF) & 0x000000F7) == (INTstruct.IO_OUT_AMUX.R & 0x000000F7)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

//tU32 PwSBC_SetCANLINmode(tU32 cmd){
//	tU32 errCode;
//
//	errCode =  PwSBC_SendCmdRW(cmd);
//
//	if(errCode != 0){
//		return errCode;							//error in the communication
//	}
//	else
//	{
//		if(((cmd & 0x000000FF) & 0x000000FC) == (INTstruct.CAN_LIN_mode.R & 0x000000FC)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
//}
//
//tU32 PwSBC_SetVregLP(tU32 cmd){
//	tU32 errCode;
//
//	errCode = PwSBC_SendSecureCmdRW(cmd);
//
//	if(errCode != 0){
//		return errCode;							//error in the communication
//	}
//	else
//	{
//		if(((cmd & 0x000000FF) >> 5) == (INTstruct.Vreg_LPON.R & 0x00000007)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
//}
//
tU16 PwSBC_SetWDwindow(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode = PwSBC_SendSecureCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if(((cmd & 0x000000FF) >> 4) == (INTstruct.WD_window.R & 0x0000000F)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

tU16 PwSBC_SetWDLFSR(MC33907_SET_T *ptr){
	tU32 errCode;

	errCode =  PwSBC_SendCmdRW(ptr);

	if(errCode != 0){
		return errCode;							//error in the communication
	}
//	else
//	{
//		if((cmd & 0x000000FF) == (INTstruct.WD_LFSR.R & 0x000000FF)){
//			return 0;									//success
//		}
//		else{
//			return 1;									//error -> register content NOT verified
//		}
//	}
	return errCode;							//error in the communication
}

/****************************************************************************\
* INIT PwSBC function
\****************************************************************************\

\* ************************************************************************ */
/*!
*   @brief 	Initializes PwSBC registers.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function initializes PwSBC registers that are marked as
*					INIT. This initialization is made using default setting
*					that can be changed in the global defines. Initialization
*					of the INIT registers is finished by writing to INIT_INT
*					register. Then the global structures are refreshed by reading
*					content of all registers and first WD answer is computed and
*					sent. This WD refresh closes the whole init phase and
*					function ends successfuly.
*	@remarks 	Function tries to write into each INIT register and verify this
*				operation. If the write operation does not succeed, then program
*				tries to repeat the same operation until it succeeds. If the
*				program is not able to write into any of INIT registers
*				successfuly, program can stay blocked in an infinite loop till
*				the first reset (RSTb activation) arrives. This case can occur
*				in case of any fatal error on the SPI bus is presented (then
*				reset activation is the only solution).
********************************************************************************/
tU16 MC33907_Init(MC33907_SET_T *ptr){
	tU8		firstAnswer = 0;
	tU16	status = 0;

	/* set transmitted FS command */
	ptr->dspiSet.transmitData	= INIT_SUPERVISOR1_DEF;
	status = PwSBC_SetInitSupervisor1(ptr);		//FS command
	ptr->dspiSet.transmitData	= INIT_VREG1_DEF;
	status = PwSBC_SetInitVreg1(ptr);
	ptr->dspiSet.transmitData	= INIT_SUPERVISOR2_DEF;
	status = PwSBC_SetInitSupervisor2(ptr);		//FS command
	ptr->dspiSet.transmitData	= INIT_VREG2_DEF;
	status = PwSBC_SetInitVreg2(ptr);
	ptr->dspiSet.transmitData	= INIT_SUPERVISOR3_DEF;
	status = PwSBC_SetInitSupervisor3(ptr);		//FS command
	ptr->dspiSet.transmitData	= INIT_CAN_LIN_DEF;
	status = PwSBC_SetInitCANLIN(ptr);
	ptr->dspiSet.transmitData	= INIT_FSSM1_DEF;
	status = PwSBC_SetInitFSSM1(ptr);					//FS command
	ptr->dspiSet.transmitData	= INIT_IO_WU1_DEF;
	status = PwSBC_SetInitIOWU1(ptr);
	ptr->dspiSet.transmitData	= INIT_FSSM2_DEF;
	status = PwSBC_SetInitFSSM2(ptr);					//FS command
	ptr->dspiSet.transmitData	= INIT_IO_WU2_DEF;
	status = PwSBC_SetInitIOWU2(ptr);
	ptr->dspiSet.transmitData	= WD_WINDOW_DEF;
	status = PwSBC_SetWDwindow(ptr);						//FS command 25Ons between commands
	ptr->dspiSet.transmitData	= WD_LFSR_DEF;
	status = PwSBC_SetWDLFSR(ptr);							//FS command
	ptr->dspiSet.transmitData	= INIT_INT_DEF;
	status = PwSBC_SetInitINT(ptr);

//	/* clear under-voltage flags that are 1 by default */
	ptr->pwrSbcRegValues.diagVreg1.R	= PwSBC_GetDiagVreg1(ptr);
	ptr->pwrSbcRegValues.diagVreg2.R	= PwSBC_GetDiagVreg2(ptr);
	ptr->pwrSbcRegValues.diagVreg3.R	= PwSBC_GetDiagVreg3(ptr);
//	/* clear over-current flags that are 1 on the beginning*/
	ptr->pwrSbcRegValues.statusVreg1.R	= PwSBC_GetStatusVreg1(ptr);
	ptr->pwrSbcRegValues.statusVreg2.R	= PwSBC_GetStatusVreg2(ptr);
//	/* initialize structure by reading all necessary registers, that have not been initialized yet */
	ptr->pwrSbcRegValues.hwConfig.R		= PwSBC_GetHWconfig(ptr);
	ptr->pwrSbcRegValues.wuSource.R		= PwSBC_GetWUsource(ptr);
	ptr->pwrSbcRegValues.diagFs1.R		= PwSBC_GetDiagFS1(ptr);				//FS command
	ptr->pwrSbcRegValues.ioInputState.R	= PwSBC_GetIOinputState(ptr);
	ptr->pwrSbcRegValues.statusVreg1.R	= PwSBC_GetStatusVreg1(ptr);
	ptr->pwrSbcRegValues.wdCounter.R	= PwSBC_GetWDcounter(ptr);				//FS command
	ptr->pwrSbcRegValues.statusVreg2.R	= PwSBC_GetStatusVreg2(ptr);
	ptr->pwrSbcRegValues.diagVreg1.R	= PwSBC_GetDiagVreg1(ptr);
	ptr->pwrSbcRegValues.diagFs2.R		= PwSBC_GetDiagFS2(ptr);				//FS command
	ptr->pwrSbcRegValues.diagVreg2.R	= PwSBC_GetDiagVreg2(ptr);
	ptr->pwrSbcRegValues.diagVreg3.R	= PwSBC_GetDiagVreg3(ptr);
	ptr->pwrSbcRegValues.diagCan.R		= PwSBC_GetDiagCAN(ptr);
	ptr->pwrSbcRegValues.canLinMode.R	= PwSBC_GetDiagCANLIN(ptr);
	ptr->pwrSbcRegValues.diagSpi.R		= PwSBC_GetDiagSPI(ptr);
	ptr->pwrSbcRegValues.mode.R			= PwSBC_GetMode(ptr);
	ptr->pwrSbcRegValues.diagSpi.R 		= PwSBC_GetDiagSPI(ptr);
	ptr->pwrSbcRegValues.mode.R			= PwSBC_GetMode(ptr);
	ptr->pwrSbcRegValues.vregMode.R		= PwSBC_GetVregMode(ptr);
	ptr->pwrSbcRegValues.ioOutAmux.R	= PwSBC_GetIOAMUX(ptr);
	ptr->pwrSbcRegValues.canLinMode.R	= PwSBC_GetCANLINmode(ptr);
	ptr->pwrSbcRegValues.vregLpOn.R		= PwSBC_GetVregLP(ptr);

	firstAnswer = pwSbc_ComputeLFSR(WD_LFSR_DEF);			//first WD refresh computations
	ptr->dspiSet.transmitData	= firstAnswer;
	ptr->pwrSbcRegValues.wdAnswer.R	= pwSbc_RefreshWd(ptr);

	ptr->dspiSet.transmitData	= VREG_MODE_DEF;
	PwSBC_SetVregMode(ptr);


	return(status);
}

/***************************************************************************//*!
*   @brief 	Computes parity for a specified command and returns the whole
*			command.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function computes odd parity for the given command and
*					returns modified command.
* 	@param[in] cmd - 16-bit command.
* 	@return 	Modified 16-bit command including odd parity.
*	@remarks 	If a secured command is being used, then the
*				PwSBC_ComputeSecurityBits function has to be applied before
*				computing parity.
********************************************************************************/
tU16 PwSBC_ComputeParity(tU16 cmd){
	tU8 sum = 0;
	tU8 i = 0;

	for(i = 0;i < 16;i++)
	{
		if(i != 8)
		{
			sum += (tU8)((cmd >> i) & 0x0001);
		}
	}
	if(sum%2 == 0)
	{			//even number -> P = 1
		cmd = cmd | 0x0100;
	}
	else
	{
		cmd = cmd & 0xFEFF;
	}
	return cmd;
}

/***************************************************************************//*!
*   @brief 	Computes security bits for a specified command and returns
*			the whole command.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function computes 4 security bits for the given command and
*					returns modified command.
* 	@param[in] cmd - 16-bit command.
* 	@return 	Modified 16-bit command including 4 security bits.
*	@remarks 	This function should be used only for secure commands and has
*				to be used before applying PwSBC_ComputeParity function.
********************************************************************************/
tU16 PwSBC_ComputeSecurityBits(tU16 cmd)
{
	REG16_T command;

	command.R = cmd;
	command.B.bit0 = command.B.bit6;
	command.B.bit1 = command.B.bit7;
	command.B.bit2 = ~command.B.bit4;
	command.B.bit3 = ~command.B.bit5;

	return command.R;
}

///***************************************************************************//*!
//*   @brief  Reads value from ADC and recomputes it into voltage considering
//*			a tight voltage range ratio.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function reads value from ADC and recomputes it into voltage
//*					considering tight voltage range ratio.
//* 	@return 	Voltage [V] that was measured on the active AMUX channel.
//*	@remarks 	This function works only when the tight range is being used.
//*				This range allows to measure voltages from 2 till 10 V. If
//*				a wider range is necessary, then PwSBC_GetVoltageWide should be
//*				used instead.
//********************************************************************************/
//float PwSBC_GetVoltageTight(){
//	double voltage = 0;
//	tU32 x;
//	float treshold = 4.15;
//
//	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
//	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
//		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
//		if(ADC_SOURCE_CALIB < treshold){													//choose between 3.3V and 5V ratio for computations
//			voltage *= VAMUX_TG_33;
//		}
//		else{
//			voltage *= VAMUX_TG_5;
//		}
//	#else																										//reference voltage is set to Vcca
//		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
//		 		voltage = (3.3 / ADC_RESOLUTION) * x;
//		 		voltage *= VAMUX_TG_33;
//		 }
//		 else{
//		 		voltage = (5.0 / ADC_RESOLUTION) * x;
//		 		voltage *= VAMUX_TG_5;
//		 }
//	#endif
//
//	return (float)voltage;
//}
//
///***************************************************************************//*!
//*   @brief 	Reads value from ADC and recomputes it into voltage considering
//*			a wide voltage range ratio.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function reads value from ADC and recomputes it into voltage
//*					considering wide voltage range ratio.
//* 	@return 	Voltage [V] that was measured on the active AMUX channel.
//*	@remarks 	This function works for wide range of voltages
//*				(from 2 till 24 V). If the voltage is less than 10 V than usage
//*				of tight range (function PwSBC_GetVoltageTight) will be more
//*				precise.
//********************************************************************************/
//float PwSBC_GetVoltageWide(){
//	double voltage = 0;
//	tU32 x;
//	float treshold = 4.15;
//
//	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
//	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
//		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
//		if(ADC_SOURCE_CALIB < treshold){													//choose between 3.3V and 5V ratio for computations
//			voltage *= VAMUX_WD_33;
//		}
//		else{
//			voltage *= VAMUX_WD_5;
//		}
//	#else																										//reference voltage is set to Vcca
//		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
//		 		voltage = (3.3 / ADC_RESOLUTION) * x;
//		 		voltage *= VAMUX_WD_33;
//		 }
//		 else{
//		 		voltage = (5.0 / ADC_RESOLUTION) * x;
//		 		voltage *= VAMUX_WD_5;
//		 }
//	#endif
//
//	return (float)voltage;
//}
//
///***************************************************************************//*!
//*   @brief  Reads value from ADC and recomputes it into voltage considering
//*			no voltage ratio.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function reads value from ADC and recomputes it into voltage
//*					considering no voltage ratio.
//* 	@return 	Voltage [V] that was measured on the active AMUX channel.
//*	@remarks 	This function should be used for voltages between 0 and
//*				Vcca - voltages that does not use any ratio (Vref, Die
//*				temperature sensor). For other volatges please use either
//*				PwSBC_GetVoltageTight or PwSBC_GetVoltageWide.
//********************************************************************************/
//float PwSBC_GetVoltage(){
//	double voltage = 0;
//	tU32 x;
//
//	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
//	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
//		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
//	#else																										//reference voltage is set to Vcca
//		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
//		 		voltage = (3.3 / ADC_RESOLUTION) * x;
//		 }
//		 else{
//		 		voltage = (5.0 / ADC_RESOLUTION) * x;
//		 }
//	#endif
//
//	return (float)voltage;
//}
//
///***************************************************************************//*!
//*   @brief  Reads value from ADC and recomputes it into temperature
//*			(no voltage ratio).
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function reads value from ADC and recomputes it into
//*					temperature.
//* 	@return 	Temperature in C.
//*	@remarks 	Temperature coefficients are set by default in the global
//*				defines.
//********************************************************************************/
//float PwSBC_GetTemperature(){
//	double voltage = 0;
//	double temp = 0;
//	tU32 x;
//
//	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
//	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
//		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
//	#else																										//reference voltage is set to Vcca
//		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
//		 		voltage = (3.3 / ADC_RESOLUTION) * x;
//		 }
//		 else{
//		 		voltage = (5.0 / ADC_RESOLUTION) * x;
//		 }
//	#endif
//
//	// Temperature computations from input voltage:
//	// temp				temperature in C
//	// VAMUX_TP_0		nominal voltage at 0 C
//	// VAMUX_TP_CO		coefficient in mV/C
//
//	temp = ((voltage * 1000) - VAMUX_TP_0)/VAMUX_TP_CO;
//
//	return (float)temp;
//}

///***************************************************************************//*!
//*   @brief Treats data received on the SPI MISO line.
//*	@par Include:
//*					PwSBC.h
//* 	@par Description:
//*					Function reads word received on the SPI MISO line,
//*					decodes the frame address and stores received word in
//*					the global structures.
//*	@remarks 	Function is called everytime by the PwSBC_SendCmdXX and
//*				PwSBC_SendSecureCmdXX. This calling causes that the global
//*				structures are updated with every command sent to the PwSBC.
//********************************************************************************/
//void PwSBC_ProcessSPI(){
//
//	tU32 address = 0;
//
//	SPIstruct.response = DSPI_Read(DSPI_NB);
//	SPIstruct.statusPwSBC.R = SPIstruct.response >> 8;
//
//	address = (SPIstruct.readCmd & 0x00007E00) >> 9;									//mask register address from the read command
//	switch(address){
//		case	INIT_VREG1_ADR				:	INTstruct.INIT_Vreg1.R = SPIstruct.response; break;
//		case	INIT_VREG2_ADR				:	INTstruct.INIT_Vreg2.R = SPIstruct.response; break;
//		case	INIT_CAN_LIN_ADR			:	INTstruct.INIT_CAN_LIN.R = SPIstruct.response; break;
//		case	INIT_IO_WU1_ADR				:	INTstruct.INIT_IO_WU1.R = SPIstruct.response; break;
//		case	INIT_IO_WU2_ADR				:	INTstruct.INIT_IO_WU2.R = SPIstruct.response; break;
//		case	INIT_INT_ADR				:	INTstruct.INIT_INT.R = SPIstruct.response; break;
//		case	HW_CONFIG_ADR				:	INTstruct.HW_Config.R = SPIstruct.response; break;
//		case	WU_SOURCE_ADR				:	INTstruct.WU_source.R = SPIstruct.response; break;
//		case	IO_INPUT_STATE_ADR			:	INTstruct.IO_input_state.R = SPIstruct.response; break;
//		case	STATUS_VREG1_ADR			:	INTstruct.Status_Vreg1.R = SPIstruct.response; break;
//		case	STATUS_VREG2_ADR			:	INTstruct.Status_Vreg2.R = SPIstruct.response; break;
//		case	DIAG_VREG1_ADR				:	INTstruct.Diag_Vreg1.R = SPIstruct.response; break;
//		case	DIAG_VREG2_ADR				:	INTstruct.Diag_Vreg2.R = SPIstruct.response; break;
//		case	DIAG_VREG3_ADR				:	INTstruct.Diag_Vreg3.R = SPIstruct.response; break;
//		case	DIAG_CAN_ADR				:	INTstruct.Diag_CAN.R = SPIstruct.response; break;
//		case	DIAG_CAN_LIN_ADR			:	INTstruct.Diag_CAN_LIN.R = SPIstruct.response; break;
//		case	DIAG_SPI_ADR				:	INTstruct.Diag_SPI.R = SPIstruct.response; break;
//		case	MODE_ADR					:	INTstruct.MODE.R = SPIstruct.response; break;
//		case	VREG_MODE_ADR				:	INTstruct.Vreg_mode.R = SPIstruct.response; break;
//		case	IO_OUT_AMUX_ADR				:	INTstruct.IO_OUT_AMUX.R = SPIstruct.response; break;
//		case	CAN_LIN_MODE_ADR			:	INTstruct.CAN_LIN_mode.R = SPIstruct.response; break;
//		case	VREG_LPON_ADR				:	INTstruct.Vreg_LPON.R = SPIstruct.response; break;
//		case	INIT_SUPERVISOR1_ADR		:	INTstruct.INIT_Supervisor1.R = SPIstruct.response; break;
//		case	INIT_SUPERVISOR2_ADR		:	INTstruct.INIT_Supervisor2.R = SPIstruct.response; break;
//		case	INIT_SUPERVISOR3_ADR		:	INTstruct.INIT_Supervisor3.R = SPIstruct.response; break;
//		case	INIT_FSSM1_ADR				:	INTstruct.INIT_FSSM1.R = SPIstruct.response; break;
//		case	INIT_FSSM2_ADR				:	INTstruct.INIT_FSSM2.R = SPIstruct.response; break;
//		case	WD_WINDOW_ADR				:	INTstruct.WD_window.R = SPIstruct.response; break;
//		case	WD_LFSR_ADR					:	INTstruct.WD_LFSR.R = SPIstruct.response; break;
//		case	WD_ANSWER_ADR				:	INTstruct.WD_answer.R = SPIstruct.response; break;
//		case	DIAG_FS1_ADR				:	INTstruct.Diag_FS1.R = SPIstruct.response; break;
//		case	WD_COUNTER_ADR				:	INTstruct.WD_counter.R = SPIstruct.response; break;
//		case	DIAG_FS2_ADR				:	INTstruct.Diag_FS2.R = SPIstruct.response; break;
//	}
//}

/***************************************************************************//*!
*   @brief Sends a read command and waits until the end of transmission.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sends a read command to the PwSBC and waits
*					the end of transmission. Data received on the MISO pin are
*					decoded and the global structure is refreshed using this
*					value.
* 	@param[in] cmd - 	16-bit read command (parity bit should be 0 for all read
*						commands).
* 	@return 	0 - Command was sent without any error. <br>
*				(10)D - SPI disconnected or no SPI answer. <br>
*				(11)D - SPI_G error detected.
*	@remarks 	This presents the lowest level of the PwSBC driver to send
*				commands to PwSBC and is being used by the other functions.
*				Function is designed just for read commands.
********************************************************************************/
tU16 PwSBC_SendCmdR(MC33907_SET_T *ptr){
//	tU32 stockPriority = 0;
	Status_16B_tag sbcStatus;
	tU16	cmd = 0;

//	stockPriority = INTC.CPR.B.PRI;			//save current priority
//	INTC.CPR.B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource

	/* create read command */
	cmd	= (ptr->dspiSet.transmitData & 0x7F00);
	// read fromSBC register, function waits till transmit is done
	mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);
	sbcStatus.R	= ptr->dspiSet.receivedData;
//	PwSBC_ProcessSPI();														//read received cmd and save it in the global structure

	if(sbcStatus.B.SPI_G == 1){
//		if(SPIstruct.response == 0xFFFF){
//			SPIstruct.writeCmd = 0;								//NO write cmd
//			SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
//			DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
//			PwSBC_ProcessSPI();
//			if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//				return 11;										//error -> SPI_G error
//			}
//			else{
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//				//PwSBC_SignalError(ERR_SPI_NO_COMMUN);
//				return 10;										//error -> no answer on SPI
//			}
//		}
//		else{
//			INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
			return 11;										//error -> SPI_G error
//		}
	}
	else{
//		INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
		return 0;
	}
}

/***************************************************************************//*!
*   @brief Sends a write command and waits until the end of transmission.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sends a write command to the PwSBC and waits
*					the end of transmission. Data received on the MISO pin are
*					decoded and the global structure is refreshed using this
*					value.
* 	@param[in] cmd - 	16-bit write command. Parity bit will be computed and
*						set automatically by this function.
* 	@return 	0 - Command was sent without any error. <br>
*				(10)D - SPI disconnected or no SPI answer. <br>
*				(11)D - SPI_G error detected.
*	@remarks 	This presents the lowest level of the PwSBC driver to send
*				commands to PwSBC and is being used by the other functions.
*				This function simply sends a write command, but does not apply
*				any verification after writing.
********************************************************************************/
tU16 PwSBC_SendCmdW(MC33907_SET_T *ptr){
//	tU32 stockPriority = 0;
//	Status_32B_tag sbcStatus;
	tU32	cmd = 0;

//	stockPriority = INTC.CPR.B.PRI;			//save current priority
//	INTC.CPR.B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource

	/* create write command */
	cmd	= ptr->dspiSet.txCmd | 0x8000;

	cmd = PwSBC_ComputeParity(cmd);
	// write to SBC register, function waits till transmit is done
	mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);
	/* read result */
//	sbcStatus = (Status_32B_tag)ptr->dspiSet.receivedData;

//	PwSBC_ProcessSPI();														//read received cmd and save it in the global structure

		if(ptr->dspiSet.receivedData == 1){
//			if(SPIstruct.response == 0xFFFF){
//				SPIstruct.writeCmd = 0;								//NO write cmd
//				SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
//				DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
//				PwSBC_ProcessSPI();
//				if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
//					INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//					return 11;										//error -> SPI_G error
//				}
//				else{
//					INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//					//PwSBC_SignalError(ERR_SPI_NO_COMMUN);
//					return 10;										//error -> no answer on SPI
//				}
//			}
//			else{
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
				return 11;										//error -> SPI_G error
//			}
		}
		else{
//			INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
			return 0;
		}
}


/***************************************************************************//*!
*   @brief 	Sends a write and then a read command and waits until the end
*			of transmission.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sends a write command followed by a read command
*					to PwSBC. Data sent on the MISO during the read command is
*					stocked in global structure, so it can be used by functions
*					on higher level to verify register content (if the write
*					command was applied correctly or not).
* 	@param[in] cmd - 	16-bit write or read command (function recomputes
*						the R/W bit automatically so either read or write
*						command can be given). Parity bit will be computed
*						and set automatically by this function.
* 	@return 	0 - Command was sent without any error. <br>
*				(10)D - SPI disconnected or no SPI answer. <br>
*				(11)D - SPI_G error detected.
*	@remarks 	This presents the lowest level of the PwSBC driver to send
*				commands to PwSBC and is being used by the other functions.
*				This function should be used for Read/Write commands.
********************************************************************************/
tU16  PwSBC_SendCmdRW(MC33907_SET_T *ptr){
//	tU32 stockPriority = 0;
//	Status_32B_tag sbcStatus;
	tU16	cmd = 0;

//	stockPriority = INTC.CPR.B.PRI;			//save current priority
//	INTC.CPR.B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource

	/* create write command */
	cmd	= ptr->dspiSet.transmitData | 0x8000;
	/* calculate parity bit of the command */
	cmd = PwSBC_ComputeParity(cmd);
	// write to SBC register, function waits till transmit is done
	mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);
	/* read result */
	ptr->dspiSet.devStatus	= ptr->dspiSet.receivedData;

//	sbcStatus = ptr->dspiSet.receivedData;
//		SPIstruct.response = DSPI_Read(DSPI_NB);							//read result and release inp. buffer
//		SPIstruct.statusPwSBC.R = SPIstruct.response >> 8;

	/* create read command */
	cmd	= ptr->dspiSet.transmitData & 0x7E00;
	// read fromSBC register, function waits till transmit is done
	mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);

//	PwSBC_ProcessSPI();														//read received cmd and save it in the global structure

	if(ptr->dspiSet.receivedData & 0x8000){
//		if(SPIstruct.response == 0xFFFF){
//			SPIstruct.writeCmd = 0;								//NO write cmd
//			SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
//			DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
//			PwSBC_ProcessSPI();
//			if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//				return 11;										//error -> SPI_G error
//			}
//			else{
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//				//PwSBC_SignalError(ERR_SPI_NO_COMMUN);
//				return 10;										//error -> no answer on SPI
//			}
//		}
//		else{
//			INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
			return 11;										//error -> SPI_G error
//		}
	}
	else{
//		INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
		return 0;
	}
}


/***************************************************************************//*!
*   @brief 	Sends a write command with secure bits and then a read command
			and waits until the end of transmission.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sends a secured write command (commands that use
*					four security bits) followed by a read command to PwSBC.
*					Data sent on the MISO during the read command is stocked in
*					global structure, so it can be used by functions on higher
*					level to verify register content (if the write command was
*					applied correctly or not).
* 	@param[in] cmd - 	16-bit write or read command (function recomputes
*						the R/W bit automatically so either read or write command
*						can be given). Parity bit and security bits will be
*						computed and set automatically by this function.
* 	@return 	0 - Command was sent without any error. <br>
*				(10)D - SPI disconnected or no SPI answer. <br>
*				(11)D - SPI_G error detected.
*	@remarks 	This presents the lowest level of the PwSBC driver to send
*				commands to PwSBC and is being used by the other functions.
*				This function should be used for secured Read/Write commands.
********************************************************************************/
tU16 PwSBC_SendSecureCmdRW(MC33907_SET_T *ptr)
{
//		tU32 stockPriority = 0;
//		Status_32B_tag sbcStatus;
		tU16	cmd = 0;

//		stockPriority = INTC.CPR.B.PRI;			//save current priority
//		INTC.CPR.B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource

		/* create write command */
		cmd	= (ptr->dspiSet.transmitData | 0x8000);

		cmd = PwSBC_ComputeSecurityBits(cmd);
		cmd = PwSBC_ComputeParity(cmd);
		/* assign SPI command bits to */
//		cmd |= ptr->dspiSet.txCmd;
		// write to SBC register, function waits till transmit is done
		mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);
		/* store device status answer */
		ptr->dspiSet.devStatus	= ptr->dspiSet.receivedData;
//		sbcStatus = ptr->dspiSet.receivedData;
//		SPIstruct.response = DSPI_Read(DSPI_NB);							//read result and release inp. buffer
//		SPIstruct.statusPwSBC.R = SPIstruct.response >> 8;

		/* assign 16-bit data to TX_FIFO, read from device command */
		cmd	= ptr->dspiSet.transmitData & 0x7E00;
		/* assign 16-bit command to TX_FIFO, for DSPI module */
//		cmd |= ptr->dspiSet.txCmd;
		// read fromSBC register, function waits till transmit is done
		mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);

//		PwSBC_ProcessSPI();														//read received cmd and save it in the global structure

		if(ptr->dspiSet.receivedData & 0x8000)
		{
//			if(SPIstruct.response == 0xFFFF){
//				SPIstruct.writeCmd = 0;								//NO write cmd
//				SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
//				DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
//				PwSBC_ProcessSPI();
//				if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
//					INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//					return 11;										//error -> SPI_G error
//				}
//				else{
//					INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//					//PwSBC_SignalError(ERR_SPI_NO_COMMUN);
//					return 10;										//error -> no answer on SPI
//				}
//			}
//			else{
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
				return 11;										//error -> SPI_G error
//			}
		}
		else{
//			INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
			return 0;
		}
}

/***************************************************************************//*!
*   @brief 	Sends a write command with secure bits and waits until the end
*			of transmission.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function sends a secured write command (commands that use
*					four security bits) to the PwSBC and waits the end of
*					transmission. Data received on the MISO pin are decoded
*					and the global structure is refreshed using this value.
* 	@param[in] cmd - 	16-bit write command. Parity bit and four security bits
*						will be computed and set automatically by this function.
* 	@return 	0 - Command was sent without any error. <br>
*				(10)D - SPI disconnected or no SPI answer. <br>
*				(11)D - SPI_G error detected.
*	@remarks 	This presents the lowest level of the PwSBC driver to send
*				commands to PwSBC and is being used by the other functions.
*				This function simply sends a write command, but does not apply
*				any verification after writing.
********************************************************************************/
tU16 PwSBC_SendSecureCmdW(MC33907_SET_T *ptr){
//	tU32 stockPriority = 0;
//	Status_32B_tag sbcStatus;
	tU16	cmd = 0;

//	stockPriority = INTC.CPR.B.PRI;			//save current priority
//	INTC.CPR.B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource

	/* create write command */
	cmd	= ptr->dspiSet.transmitData | 0x8000;

	cmd = PwSBC_ComputeSecurityBits(cmd);
	cmd = PwSBC_ComputeParity(cmd);
	// write to SBC register, function waits till transmit is done
	mc33907SpiSendWord(&ptr->dspiSet, cmd, &ptr->dspiSet.receivedData);
	/* read result */
//	sbcStatus = ptr->dspiSet.receivedData;

//	PwSBC_ProcessSPI();														//read received cmd and save it in the global structure

	if(ptr->dspiSet.receivedData == 1){
//		if(SPIstruct.response == 0xFFFF){
//			SPIstruct.writeCmd = 0;								//NO write cmd
//			SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
//			DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
//			PwSBC_ProcessSPI();
//			if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//				return 11;										//error -> SPI_G error
//			}
//			else{
//				INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
//				//PwSBC_SignalError(ERR_SPI_NO_COMMUN);
//				return 10;										//error -> no answer on SPI
//			}
//		}
//		else{
//			INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
			return 11;										//error -> SPI_G error
//		}
	}
	else{
//		INTC.CPR.B.PRI = stockPriority;					//release DSPI resource
		return 0;
	}
}



/******************************************************************************
* DSPI functions
*****************************************************************************/

///**************************************************************************//*!
//@brief          DSPI byte transmision
//
//@param[in,out]  void
//@param[in]      void
//
//@return         void
//
//@details        The function transmit 8bits from SPI module and get
//
//                \image latex foc.pdf "My application" width=10cm
//
//@note           This function transmit 8bits from DSPI1 unit
//
//@warning
//******************************************************************************/
//tBool dspiSendByte(DSPI_SET_T *ptr, tU8 u8DataOut, tU8 *ptDataIn)
//{
//    static  tU32 data_out;
//
//    // data preparation - concatenate transfer command and data into 32bit transfer register
//    data_out            = (tU32)u8DataOut;
//    data_out            |= ptr->txCmd;      //(ASSERT_CS0 | ASSERT_CTAR0);  ptr->DspiBaseAdr
//
//    /* clear transfer flag */
//    (*(vuint32_t *)(ptr->dspiBaseAdr + DSPI_SR_OFFSET)) = (DSPI_TCF);
//    (*(vuint32_t *)(ptr->dspiBaseAdr + DSPI_PUSHR_OFFSET)) = (data_out);
//    /* Wait end of transfer in MASTER */
//    while(((*(vuint32_t *)(ptr->dspiBaseAdr + DSPI_SR_OFFSET)) & (DSPI_TCF))==0);
//
//    /* read received data */
//    *ptDataIn           = (tU8)(((vuint32_t *)(ptr->dspiBaseAdr + DSPI_POPR_OFFSET)));
//
//
//    return (1);
//}


/**************************************************************************//*!
@brief          DSPI transmit word

@param[in,out]  u16DspiOut
@param[in]      void

@return         ptDspiIn

@details        The function transmit 16bits from SPI module and get 16 bit \
                from external device

                \image latex foc.pdf "My application" width=10cm

@note           This function transmit 16bits from DSPIx unit

@warning
******************************************************************************/
tU16 mc33907SpiSendWord(MC33907_DRV_SET_T *ptr, tU16 u16DspiOut, tU16 *ptDspiIn)
{
    static  tU32 data_out;

    // data preparation - concatenate transfer command and data into 32bit transfer register
    data_out            = (tU32)u16DspiOut;
    data_out            |= ptr->txCmd;

    /* clear transfer flag */
    (*(volatile tU32 *)(ptr->dspiBaseAdr + DSPI_SR_OFFSET)) = (DSPI_TCF);
    (*(volatile tU32 *)(ptr->dspiBaseAdr + DSPI_PUSHR_OFFSET)) = (data_out);
    /* Wait end of transfer in MASTER */
    while((*((vuint32_t *)(ptr->dspiBaseAdr + DSPI_SR_OFFSET)) & (DSPI_TCF)) == 0);

    /* read received data */
    data_out			= (*((volatile tU32 *)(ptr->dspiBaseAdr + DSPI_POPR_OFFSET)));
    *ptDspiIn			= (tU16)data_out;

    return (1);
}

/****************************************************************************\
* COMMUNICATION functions
\****************************************************************************/

/***************************************************************************//*!
*   @brief 	Gets actual content of all necessary registers and stores it into
*			structure.
*	@par Include:
*					PwSBC.h
* 	@par Description:
*					Function updates data structures that contain states
*					of PwSBC registers. This is done for all PwSBC registers.
*	@remarks
********************************************************************************/
void	PwSBC_UpdateRegisters(MC33907_SET_T *ptr)
{
	PwSBC_GetInitVreg1(ptr);
	PwSBC_GetInitVreg2(ptr);
	PwSBC_GetInitCANLIN(ptr);
	PwSBC_GetInitIOWU1(ptr);
	PwSBC_GetInitIOWU2(ptr);
	PwSBC_GetInitINT(ptr);
	PwSBC_GetHWconfig(ptr);
	PwSBC_GetWUsource(ptr);
	PwSBC_GetIOinputState(ptr);
	PwSBC_GetStatusVreg1(ptr);
	PwSBC_GetStatusVreg2(ptr);
	PwSBC_GetDiagVreg1(ptr);
	PwSBC_GetDiagVreg2(ptr);
	PwSBC_GetDiagVreg3(ptr);
	PwSBC_GetDiagCAN(ptr);
	PwSBC_GetDiagCANLIN(ptr);
	PwSBC_GetDiagSPI(ptr);
	PwSBC_GetMode(ptr);
	PwSBC_GetVregMode(ptr);
	PwSBC_GetIOAMUX(ptr);
	PwSBC_GetCANLINmode(ptr);
	PwSBC_GetVregLP(ptr);
	PwSBC_GetInitSupervisor1(ptr);
	PwSBC_GetInitSupervisor2(ptr);
	PwSBC_GetInitSupervisor3(ptr);
	PwSBC_GetDiagFS1(ptr);					//FS command
	PwSBC_GetDiagFS2(ptr);					//FS command
	PwSBC_GetWDwindow(ptr);
	PwSBC_GetWDLFSR(ptr);
	PwSBC_GetWDanswer(ptr);
	PwSBC_GetDiagFS1(ptr);
	PwSBC_GetWDcounter(ptr);				//FS command
	PwSBC_GetDiagFS2(ptr);
}



/* END OF FILE */
