/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file:	MC33937_routines.c
*
* @author:	B34195
*
* @date: 	Aug 17, 2016
*
* @brief: 	Source file to configure the MC33937 according to selected HW configuration
*
***************************************************************************/

#include "MC33937_routines.h"

static volatile tU8    empty;
static volatile tU8    command;

/***************************************************************************//*!
@brief          Function for HW initialization of the MC33937

@param[in,out]  hw_ptr	Pointer to MC33937 HW configuration structure
				mc33937ID - select one of two available HW configurations

@return         tBool

@details		Initialize the HW configuration according the selected PICe
 	 	 	 	motor connector.
******************************************************************************/
tBool MC33937_hw_init(MC33937_hw_cfg_t *hw_ptr, tU16 mc33937ID){
    switch (mc33937ID)
    {
        case MC33937_HW_PCIe_J1:
        	hw_ptr->pSiulGpdoEN 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_EN1);
        	hw_ptr->pSiulGpdoRST 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_RST1);
        	hw_ptr->pSiulGpdoCS 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_CS1);

        	hw_ptr->pSiulMscrCS 	= (volatile tU32 *)(SIUL_BASE + SIUL_MSCR_OFFSET + 4*MC33937_MSCR_CS1);

        	hw_ptr->pSiulGpdiINT 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDI_OFFSET + MC33937_MSCR_INT1);

            MC33937_dspi_hw_init(&hw_ptr->spi, MC33937_DSPI_HW_PCIe_J1);

        	break;
        case MC33937_HW_PCIe_J200:
        	hw_ptr->pSiulGpdoEN 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_EN2);
        	hw_ptr->pSiulGpdoRST 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_RST2);
        	hw_ptr->pSiulGpdoCS 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_CS2);

        	hw_ptr->pSiulMscrCS 	= (volatile tU32 *)(SIUL_BASE + SIUL_MSCR_OFFSET + 4*MC33937_MSCR_CS2);

        	hw_ptr->pSiulGpdiINT 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDI_OFFSET + MC33937_MSCR_INT2);

        	MC33937_dspi_hw_init(&hw_ptr->spi, MC33937_DSPI_HW_PCIe_J200);
            break;
        default:
        	hw_ptr->pSiulGpdoEN 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_EN1);
        	hw_ptr->pSiulGpdoRST 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_RST1);
        	hw_ptr->pSiulGpdoCS 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDO_OFFSET + MC33937_MSCR_CS1);

        	hw_ptr->pSiulMscrCS 	= (volatile tU32 *)(SIUL_BASE + SIUL_MSCR_OFFSET + 4*MC33937_MSCR_CS1);

        	hw_ptr->pSiulGpdiINT 	= (volatile tU8 *)(SIUL_BASE + SIUL_GPDI_OFFSET + MC33937_MSCR_INT1);

        	MC33937_dspi_hw_init(&hw_ptr->spi, MC33937_DSPI_HW_PCIe_J1);
            break;
    }

    MC33937_PIT_hw_init(&hw_ptr->pit, 50);

	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for SW initialization of the MC33937 functionality

@param[in,out]  hw_ptr	Pointer to MC33937 HW configuration structure
				sw_ptr  Pointer to MC33937 SW configuration structure

@return         tBool

@details		Initialize the SW configuration according the application
				requirements (dead-time, interrupts ... )
******************************************************************************/
tBool MC33937_sw_init(MC33937_hw_cfg_t *hw_ptr, MC33937_T *sw_ptr){
	MC33937_disable_output(hw_ptr, sw_ptr);

    sw_ptr->data.requiredDeadTimeNs            = sw_ptr->configData.deadtime;
    sw_ptr->data.mc33937mask0                  = 0x20 | (tU8)(sw_ptr->configData.interruptEnable.R);
    sw_ptr->data.mc33937mask1                  = 0x30 | (tU8)(sw_ptr->configData.interruptEnable.R >> 8);
    sw_ptr->data.mc33937mode                   = 0x40 | (tU8)sw_ptr->configData.mode.R;

    MC33937_configure(hw_ptr, sw_ptr);

    return (TRUE);

}
/***************************************************************************//*!
@brief			Set/Clear ENable pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool SetEN(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    *((volatile tU8 *)ptr->pSiulGpdoEN) = (tU8)0x1;

    return(statusPass);
}

tBool ClearEN(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    *((volatile tU8 *)ptr->pSiulGpdoEN) = (tU8)0x0;

    return(statusPass);
}

/***************************************************************************//*!
@brief			Set/Clear RESET pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool SetRESET(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    *((volatile tU8 *)ptr->pSiulGpdoRST) = (tU8)0x1;

    return(statusPass);
}

tBool ClearRESET(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    *((volatile tU8 *)ptr->pSiulGpdoRST) = (tU8)0x0;

    return(statusPass);
}

/***************************************************************************//*!
@brief			Set/Clear chip select pin if manual control mode enabled

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool SetCS(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    *((volatile tU8 *)ptr->pSiulGpdoCS) = (tU8)0x1;

    return(statusPass);
}

tBool ClearCS(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    *((volatile tU8 *)ptr->pSiulGpdoCS) = (tU8)0x0;

    return(statusPass);
}

/***************************************************************************//*!
@brief			Read INTerrupt pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool ReadINT(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    statusPass &= ~(*((volatile tU8 *)ptr->pSiulGpdiINT));

    return(statusPass);
}

/***************************************************************************//*!
@brief			Manual control of ChipSelect pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool EnableCSModeManual(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    // configure CS pad as SIUL GPIO signal => Output + SSS value = 0
   // *(volatile tU32 *)(ptr->pSiulGpdoCS) = 0x02840000UL;
    *(volatile tU32 *)(ptr->pSiulMscrCS) = 0x02840000UL;
    return(statusPass);
}

/***************************************************************************//*!
@brief			DSPI control of ChipSelect pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool EnableCSModeDSPI(MC33937_hw_cfg_t * ptr)
{
    tBool statusPass = TRUE;

    // configure CS pad as DSPI0 CSx => Output + SSS value = 1
    *(volatile tU32 *)(ptr->pSiulMscrCS) = 0x02840001UL;

    return(statusPass);
}

/***************************************************************************//*!
@brief			MC33937 fault clearing by setting CLINT0/1 specific commands

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_clear_faults(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr){
    tBool statusPass = TRUE;

    /* CLINT0_COMMAND = 0x6F */
    /* CLINT1_COMMAND = 0x7F */
    command     = 0x6F;
    statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);
    command     = 0x7F;
    statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

    ptr->status.sw.B.clearingError = ~statusPass;

    return(statusPass);
}

/***************************************************************************//*!
@brief			Configure MC33937 -> initialization, dead-time set, read status registers

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_configure(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr){
	volatile tU8     	cnt;
	volatile tBool 		statusPass;
	tU32            	cntrCompValue;

	statusPass = TRUE;

	 statusPass &= ClearRESET(hw_ptr);

    statusPass &= ClearEN(hw_ptr);
    statusPass &= SetRESET(hw_ptr);

    // Start system counter if not already running
    if(!GetStatus_PIT(&hw_ptr->pit))
    	Start_PIT(&hw_ptr->pit);

    // wait 2ms = 2 000 000 ns
    cntrCompValue = ReadValueNs_PIT(&hw_ptr->pit);
    while((ReadValueNs_PIT(&hw_ptr->pit)-cntrCompValue)<=2000000);

    // Clear all faults
    statusPass &= MC33937_clear_faults(hw_ptr, ptr);

    // Initialize MASK register 0
    command     = ptr->data.mc33937mask0;
    statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

    // Initialize MASK register 1
    command     = ptr->data.mc33937mask1;
    statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

    // Dead-time configuration
    cnt = 5;
	do
	{
		ptr->status.sw.B.deadtimeSetupError = ~MC33937_set_deadtime(hw_ptr, ptr,(tU16)(ptr->data.requiredDeadTimeNs));
		cnt--;
	} while (ptr->status.sw.B.deadtimeSetupError && cnt);

	statusPass &= ~(ptr->status.sw.B.deadtimeSetupError);

	// Clear all faults
	statusPass &= MC33937_clear_faults(hw_ptr, ptr);

	// Read status registers
	statusPass &= MC33937_get_status(hw_ptr, ptr);

	// Send MODE command
	command     = ptr->data.mc33937mode;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

	ptr->status.sw.B.configurationError = ~statusPass;

    return (statusPass);

}

/***************************************************************************//*!
@brief			Set dead-time following the procedure described in MC33937 datasheet

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_set_deadtime(MC33937_hw_cfg_t * hw_ptr, MC33937_T * ptr, tU16 deadtimeNs){
	volatile tU8 		actualDeadtime;
	volatile tFrac16 	delta;
	volatile tBool		statusPass;
	tU32            	cntrCompValue;
	tU32            	dt;

	statusPass = TRUE;

	/* Start system counter if not already running */
    if(!GetStatus_PIT(&hw_ptr->pit))
    	Start_PIT(&hw_ptr->pit);

	/* Send DEADTIME command for dead time configuration with zero calibration */
	command     = 0x81;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

	/* wait 20us = 20 000 ns*/
    cntrCompValue = ReadValueNs_PIT(&hw_ptr->pit);
    while((ReadValueNs_PIT(&hw_ptr->pit)-cntrCompValue)<=20000);

	/* configure CS pad as GPIO for deadtime configuration */
	statusPass &= EnableCSModeManual(hw_ptr);

	/* recalculate dead-time for CS pulse width as required DTx16		*/
	dt = ((tU32)deadtimeNs)<<4;

	/* Read PIT compare value before CS pulling-down */
	cntrCompValue = ReadValueNs_PIT(&hw_ptr->pit);

	/* pull-down CS to configure dead time */
	statusPass &= ClearCS(hw_ptr);

	/* Apply delay in order to keep CS down for duration of req. deadtime x 16 */
	while((ReadValueNs_PIT(&hw_ptr->pit)-cntrCompValue)<=dt);

	/* pull-up CS to configure dead time */
	statusPass &= SetCS(hw_ptr);

	/* configure CS pad back as DSPI chip select */
	statusPass &= EnableCSModeDSPI(hw_ptr);

	/*
	 * Check whether dead-time is already configured
	 * STATUS3_COMMAND  0x3
	 */

	/* read deadtime status register 3 from MC33937 */
	command     = 0x3;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

	command     = 0x0;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &actualDeadtime);

	ptr->data.actualDeadTimeNs = (tU16)((tU16)actualDeadtime * (tU16)50);

	delta       = MLIB_Abs_F16((tFrac16)(deadtimeNs) - (tFrac16)(ptr->data.actualDeadTimeNs));
	statusPass  &= (delta > 100)?FALSE:TRUE;

	return(statusPass);

}


/***************************************************************************//*!
@brief			Enable MC33937 output by setting Enable pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_enable_output(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr){
    tBool statusPass = TRUE;

    /* EN pins go HIGH, RST stays HIGH, device goes to ENABLE MODE */
    statusPass &= SetEN(hw_ptr);
    statusPass &= SetRESET(hw_ptr); // RST pin remains high, it goes only low in sleep mode

    ptr->status.sw.B.outputEnablingError = ~statusPass;

    return(statusPass);
}

/***************************************************************************//*!
@brief			Disable MC33937 output by clearing Enable pin

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_disable_output(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr){
    tBool statusPass = TRUE;

    /* EN pins go LOW, RST stays HIGH, device goes to STANDBY MODE */
    statusPass &= ClearEN(hw_ptr);
    statusPass &= SetRESET(hw_ptr); // RST pin remains high, it goes only low in sleep mode

    ptr->status.sw.B.outputDisablingError = ~statusPass;

    return(statusPass);
}

/***************************************************************************//*!
@brief			Get status of MC33937 and store it in status registers

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_get_status(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr){
    tBool statusPass = TRUE;
    /*
     * Check whether dead-time is already configured
     * STATUS3_COMMAND  0x3
     */
    /* read status registers from MC33937 */
    command     = 0x0;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

	command     = 0x1;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &ptr->status.sr0);

	command     = 0x2;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &ptr->status.sr1);

	command     = 0x3;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &ptr->status.sr2);

	command     = 0x0;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &ptr->status.sr3);

    return(statusPass);
}

/***************************************************************************//*!
@brief			Get general status of MC33937 by reading status registers 0

@param[in,out]  this    Pointer to the current object.

@return         tBool

@details
******************************************************************************/
tBool MC33937_get_status_SR0(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr){
    tBool statusPass = TRUE;

    /* read status register SR0 from MC33937 */
    command     = 0x0;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &empty);

	command     = 0x0;
	statusPass  &= SendCommand(&hw_ptr->spi, &command, &ptr->status.sr0);

    return(statusPass);
}
