/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file:	MC33937_routines.h
*
* @author:	B34195
*
* @date: 	Aug 17, 2016
*
* @brief: 	Header file to define the HW configuration for MC33937
*
***************************************************************************/
#ifndef BSP_MC33937_ROUTINES_H_
#define BSP_MC33937_ROUTINES_H_

#include "SWLIBS_Typedefs.h"
#include "mlib.h"
#include "MC33937_spi_routines.h"
#include "MC33937_pit_routines.h"

/*! Different HW configuration available */
#define MC33937_HW_PCIe_J1			1
#define MC33937_HW_PCIe_J200		2

/*! MSCR register numbers of MC33937 control signals */
#define MC33937_MSCR_EN1			11
#define MC33937_MSCR_EN2			132
#define MC33937_MSCR_RST1			10
#define MC33937_MSCR_RST2			49
#define MC33937_MSCR_INT1			13
#define MC33937_MSCR_INT2			14
#define MC33937_MSCR_CS1			79
#define MC33937_MSCR_CS2			54

/*! Macro containing base address of SIUL module */
#define SIUL_BASE					0xFFFC0000

/*! Macro containing offset of GPDI/O, MSCR SIUL registers */
#define SIUL_GPDO_OFFSET			0x1300
#define SIUL_GPDI_OFFSET			0x1500
#define SIUL_MSCR_OFFSET			0x240

/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef struct{
    volatile tU8       * pSiulGpdoEN;
    volatile tU8       * pSiulGpdoRST;
    volatile tU8       * pSiulGpdoCS;
    volatile tU32	   * pSiulMscrCS;

    volatile tU8       * pSiulGpdiINT;

    MC33937_SPI_hw_cfg_t	spi;
    MC33937_PIT_hw_cfg_t	pit;
}MC33937_hw_cfg_t;


/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef union
{
    tU16 R;
    struct {
        tU16                   :1;
        tU16                   :1;
        tU16                   :1;
        tU16                   :1;
        tU16    resetEvent     :1;
        tU16    writeErr       :1;
        tU16    framingErr     :1;
        tU16    phaseErr       :1;
        tU16                   :1;
        tU16                   :1;
        tU16                   :1;
        tU16                   :1;
        tU16    overCurrent    :1;
        tU16    lowVls         :1;
        tU16    desaturation   :1;
        tU16    overTemp       :1;
    }B;
}mc33937ConfigMask_t;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef union
{
    tU8 R;
    struct {
        tU8                 :1;
        tU8                 :1;
        tU8                 :1;
        tU8                 :1;
        tU8    disableDesat :1;
        tU8                 :1;
        tU8    enableFullOn :1;
        tU8    lock         :1;
    }B;
}mc33937Mode_t;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing status register 0 image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef union {
    tU8 R;
    struct {
        tU8     resetEvent   :1;    /*!< reset event flag, is set upon exiting /RST */
        tU8     writeErr     :1;    /*!< Write Error After the Lock flag */
        tU8     framingErr   :1;    /*!< Framing error flag */
        tU8     phaseErr     :1;    /*!< Phase error flag */
        tU8     overCurrent  :1;    /*!< Over-current event flag */
        tU8     lowVls       :1;    /*!< Low VLS voltage flag */
        tU8     desaturation :1;    /*!< DESAT flag detected on any channel */
        tU8     overTemp     :1;    /*!< TLIM flag detected on any channel */
    } B;
}mc33937SR0_t;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing status register 1 image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef union {
    tU8 R;
    struct {
        tU8                  :1;     /*!< reserved */
        tU8     desat_mode   :1;     /*!< current state of the Desaturation/Phase Error turn-off mode */
        tU8     zds          :1;     /*!< Zero deadtime is commanded */
        tU8     calib_overfl :1;     /*!< flag for a Deadtime Calibration Overflow */
        tU8     deadtime_cal :1;     /*!< Deadtime calibration occurred */
        tU8                  :1;     /*!< reserved */
        tU8     fullon       :1;     /*!< present status of FULLON MODE */
        tU8     lockbit      :1;     /*!< LockBit indicates the IC regs are locked */
    } B;
}mc33937SR1_t;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing status register 2 image of the MC33937 device
*//*-------------------------------------------------------------------------*/
typedef union {
    tU8 R;
    struct {
        tU8     mask1_3      :1;     /*!< status of the MASK1.3 bit */
        tU8     mask1_2      :1;     /*!< status of the MASK1.2 bit */
        tU8     mask1_1      :1;     /*!< status of the MASK1.1 bit */
        tU8     mask1_0      :1;     /*!< status of the MASK1.0 bit */
        tU8     mask0_3      :1;     /*!< status of the MASK0.3 bit */
        tU8     mask0_2      :1;     /*!< status of the MASK0.2 bit */
        tU8     mask0_1      :1;     /*!< status of the MASK0.1 bit */
        tU8     mask0_0      :1;     /*!< status of the MASK0.0 bit */
    } B;
}mc33937SR2_t;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing status of the software driver for mc33937
*//*-------------------------------------------------------------------------*/
typedef union {
    tU32 R;
    struct {
        tU32                     :26; /* RESERVED */
        tU32 outputDisablingError:1;
        tU32 outputEnablingError :1;
        tU32 configurationError  :1;
        tU32 clearingError       :1;
        tU32 initializationError :1;
        tU32 deadtimeSetupError  :1;
    } B;
}mc33937StatusSW_t;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef struct {
    mc33937SR0_t                sr0;    /*!< status register 0 */
    mc33937SR1_t                sr1;    /*!< status register 1 */
    mc33937SR2_t                sr2;    /*!< status register 2 */
    tU8                         sr3;    /*!< status register 3 */
    mc33937StatusSW_t           sw;
}MC33937_STATUS_T;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef struct{
    tU16                        deadtime;       /*!< define dead time of HS and LS transistors, value in [ns]*/
    mc33937ConfigMask_t         interruptEnable;
    mc33937Mode_t               mode;
}MC33937_CONFIG_DATA_T;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef struct{
    tU16            requiredDeadTimeNs;
    tU16            actualDeadTimeNs;
    tU8        		mc33937mask0;
    tU8         	mc33937mask1;
    tU8         	mc33937mode;
}MC33937_DATA_T;

/*------------------------------------------------------------------------*//*!
@brief  Structure containing
*//*-------------------------------------------------------------------------*/
typedef struct{
    MC33937_DATA_T              data;
    MC33937_STATUS_T            status;
    MC33937_CONFIG_DATA_T       configData;
}MC33937_T;

/******************************************************************************
| Exported function prototypes
-----------------------------------------------------------------------------*/
extern tBool MC33937_clear_faults(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr);  //
extern tBool MC33937_configure(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr);
extern tBool MC33937_get_status(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr);     // read status of internal IC settings
extern tBool MC33937_get_status_SR0(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr);     // read general status in SR0
extern tBool MC33937_enable_output(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr);  //
extern tBool MC33937_disable_output(MC33937_hw_cfg_t * hw_ptr, MC33937_T *ptr); //
extern tBool MC33937_set_deadtime(MC33937_hw_cfg_t * hw_ptr, MC33937_T * ptr, tU16 deadtimeNs);

extern tBool SetEN(MC33937_hw_cfg_t * ptr);
extern tBool ClearEN(MC33937_hw_cfg_t * ptr);
extern tBool SetRESET(MC33937_hw_cfg_t * ptr);
extern tBool ClearRESET(MC33937_hw_cfg_t * ptr);
extern tBool SetCS(MC33937_hw_cfg_t * ptr);
extern tBool ClearCS(MC33937_hw_cfg_t * ptr);
extern tBool ReadINT(MC33937_hw_cfg_t * ptr);
extern tBool ReadOC(MC33937_hw_cfg_t * ptr);
extern tBool EnableCSModeManual(MC33937_hw_cfg_t * ptr);
extern tBool EnableCSModeDSPI(MC33937_hw_cfg_t * ptr);

extern tBool MC33937_hw_init(MC33937_hw_cfg_t *hw_ptr, tU16 mc33937ID);
extern tBool MC33937_sw_init(MC33937_hw_cfg_t *hw_ptr, MC33937_T *sw_ptr);


#endif /* BSP_MC33937_ROUTINES_H_ */
