/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file:	Resolver.c
*
* @author:	B34195
*
* @date: 	Aug 9, 2016
*
* @brief: 	Source file with implementation of functions initializing, interfacing
* 			on-chip ADC data registers and eTimer register for controlling the resolver excitation.
*
***************************************************************************/
#ifndef BSP_RESOLVER_C_
#define BSP_RESOLVER_C_

#include "Resolver.h"

/***************************************************************************//*!
@brief          Function for configuration the HW for resolver Sin/Cos
				measurement and resolver excitation

@param[in,out]  hw_ptr	Pointer to resolver HW configuration structure
				resolverID - select one of two available HW configurations

@return         tBool

@details		Initialize the HW configuration according the selected PICe
 	 	 	 	motor connector.
******************************************************************************/
tBool Resolver_hw_init(resolver_hw_cfg_t *hw_ptr, tU16 resolverID){
    switch (resolverID)
    {
        case RES_HW_PCIe_J1:
        	// Resolver input signals measurement
        	hw_ptr->pAdcSinData = (volatile tU32 *)(RES_ADC0_BASE + ADC_CDRn_OFFSET + (4 * RES_ADC_CH0));
        	hw_ptr->pAdcCosData = (volatile tU32 *)(RES_ADC1_BASE + ADC_CDRn_OFFSET + (4 * RES_ADC_CH0));

        	// Resolver excitation configuration via eTimer
        	hw_ptr->pResExcitEtimer = (volatile tU16 *)(RES_ETIMER0_BASE + RES_ETIMER_ENBL);
        	hw_ptr->eTimerChannel = RES_ETIMER_CH5;
            break;
        case RES_HW_PCIe_J200:
        	hw_ptr->pAdcSinData = (volatile tU32 *)(RES_ADC2_BASE + ADC_CDRn_OFFSET + (4 * RES_ADC_CH2));
        	hw_ptr->pAdcCosData = (volatile tU32 *)(RES_ADC3_BASE + ADC_CDRn_OFFSET + (4 * RES_ADC_CH3));

        	hw_ptr->pResExcitEtimer = (volatile tU16 *)(RES_ETIMER1_BASE + RES_ETIMER_ENBL);
        	hw_ptr->eTimerChannel = RES_ETIMER_CH5;
            break;
        default:
        	hw_ptr->pAdcSinData = (volatile tU32 *)(RES_ADC0_BASE + ADC_CDRn_OFFSET + (4 * RES_ADC_CH0));
        	hw_ptr->pAdcCosData = (volatile tU32 *)(RES_ADC1_BASE + ADC_CDRn_OFFSET + (4 * RES_ADC_CH0));

        	hw_ptr->pResExcitEtimer = (volatile tU16 *)(RES_ETIMER0_BASE + RES_ETIMER_ENBL);
        	hw_ptr->eTimerChannel = RES_ETIMER_CH5;
            break;
    }
	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for reading ADC data registers containing converted
                values of resolver sin and cos signals.

@param[in,out]  hw_ptr	Pointer to resolver HW configuration structure
				d_ptr	Pointer to resolver data structure

@return         tBool

@details
******************************************************************************/
tBool Resolver_get_data(resolver_hw_cfg_t *hw_ptr ,resolver_data_t *d_ptr){

	d_ptr->ResSin.raw = ((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcSinData)) & 0x00000FFF), (tFloat)0x00000FFF))*2;
	d_ptr->ResCos.raw = ((tFloat)MLIB_Div((tFloat)((*(hw_ptr->pAdcCosData)) & 0x00000FFF), (tFloat)0x00000FFF))*2;

	d_ptr->ResSin.filt = MLIB_Sub(d_ptr->ResSin.raw,d_ptr->ResSinDcOffset);
	d_ptr->ResCos.filt = MLIB_Sub(d_ptr->ResCos.raw,d_ptr->ResCosDcOffset);

	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for enabling the resolver excitation


@param[in,out]  hw_ptr	Pointer to resolver HW configuration structure

@return         tBool

@details		Enabling resolver excitation by enabling the appropriate
				eTimer module and channel
******************************************************************************/
tBool Resolver_enable_excitation(resolver_hw_cfg_t *hw_ptr){

	// Set nth-bit in ENBL reg. according to the selected eTimer nth-chnl
	*hw_ptr->pResExcitEtimer |= (1<<hw_ptr->eTimerChannel);
	return(TRUE);
}

/***************************************************************************//*!
@brief          Function for disabling the resolver excitation


@param[in,out]  hw_ptr	Pointer to resolver HW configuration structure

@return         tBool

@details		Disabling resolver excitation by disabling the appropriate
				eTimer module and channel
******************************************************************************/
tBool Resolver_disable_excitation(resolver_hw_cfg_t *hw_ptr){

	// ETIMER_n_ENBL = 0x3F - all eTimer_n channels enabled
	// Clear nth-bit in ENBL reg. according to selected eTimer nth-chnl
	*hw_ptr->pResExcitEtimer &= (0x3F ^ (1<<hw_ptr->eTimerChannel));

	return(TRUE);
}

/***************************************************************************//*!
@brief          Function performs the calibration of both resolver ADC inputs


@param[in,out]  hw_ptr	Pointer to resolver HW configuration structure
				d_ptr	Pointer to resolver data structure

@return         tBool

@details		Calibration of resolver ADC inputs to calibrate the DC offset
				in sine and cosine signals

@note			Before calibration, the resolver excitation must be disabled
				first. Once the calibration is done, the excitation must be
				enabled again.
******************************************************************************/
tBool Resolver_inputs_calibration(resolver_hw_cfg_t *hw_ptr, resolver_data_t *d_ptr){

	tBool statusPass = FALSE;

	if(!d_ptr->calib.calibInitDone){

		// Clearing the DC offset values of both Sin/Cos signals
		d_ptr->ResSinDcOffset = 0.F;
		d_ptr->ResCosDcOffset = 0.F;

		d_ptr->calib.calibCntr = RES_CALIB_CNTR;

		d_ptr->calib.calibInitDone 	= 1;
		d_ptr->calib.calibDone 		= 0;
	}

	if(d_ptr->calib.calibCntr>0){
		// resolver Sin/Cos - DC offset data filtering using MA recursive filter
		d_ptr->ResSinDcOffset = GDFLIB_FilterMA(d_ptr->ResSin.raw, &d_ptr->MAF_Sin);
		d_ptr->ResCosDcOffset = GDFLIB_FilterMA(d_ptr->ResCos.raw, &d_ptr->MAF_Cos);

		d_ptr->calib.calibCntr--;
	}
	else{
		//Enable excitation after successful resolver ADC inputs calibration
		Resolver_enable_excitation(hw_ptr);

		// set calibration done flag
		d_ptr->calib.calibDone = 1;

		// clear calibration init flag to allow next calibration
		d_ptr->calib.calibInitDone 		= 0;

		statusPass = TRUE;
	}

	return(statusPass);
}

#endif /* BSP_RESOLVER_C_ */
