/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*
*
* @file:	Voltage_gener.c
*
* @author:	B34195
*
* @date: 	Aug 22, 2016
*
* @brief: 	Source file with implementation of functions initializing, interfacing
* 			on-chip FlexPWM update registers to apply calculated dutycycles (voltages)
**
***************************************************************************/

#include "Voltage_gener.h"


/***************************************************************************//*!
@brief          Function for configuration the HW for voltage actuator
				utilizing the flexPWM module

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure
				flexPwmID - select one of two available HW configurations

@return         tBool

@details
******************************************************************************/
tBool FLEXPWM_hw_init(flexPWM_hw_cfg_t * hw_ptr, tU16 flexPwmID){
	volatile tS32 FLEXPWM_X_BASE;

    switch (flexPwmID)
    {
        case FLEXPWM_HW_PCIe_J1:
        	FLEXPWM_X_BASE = FLEXPWM_0_BASE;
            break;

        case FLEXPWM_HW_PCIe_J200:
        	FLEXPWM_X_BASE = FLEXPWM_1_BASE;
            break;

        default:
        	FLEXPWM_X_BASE = FLEXPWM_0_BASE;
        	break;
    }

	// phase A
	hw_ptr->s16Sub0.pFlexPwmINIT	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_INIT_OFFSET + (0x50 * FLEXPWM_SUB_0));
	hw_ptr->s16Sub0.pFlexPwmVAL0	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL0_OFFSET + (0x50 * FLEXPWM_SUB_0));
	hw_ptr->s16Sub0.pFlexPwmVAL1	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL1_OFFSET + (0x50 * FLEXPWM_SUB_0));
	hw_ptr->s16Sub0.pFlexPwmVAL2	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL2_OFFSET + (0x50 * FLEXPWM_SUB_0));
	hw_ptr->s16Sub0.pFlexPwmVAL3	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL3_OFFSET + (0x50 * FLEXPWM_SUB_0));
	hw_ptr->s16Sub0.pFlexPwmVAL4	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL4_OFFSET + (0x50 * FLEXPWM_SUB_0));
	hw_ptr->s16Sub0.pFlexPwmVAL5	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL5_OFFSET + (0x50 * FLEXPWM_SUB_0));

	// phase B
	hw_ptr->s16Sub1.pFlexPwmINIT	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_INIT_OFFSET + (0x50 * FLEXPWM_SUB_1));
	hw_ptr->s16Sub1.pFlexPwmVAL0	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL0_OFFSET + (0x50 * FLEXPWM_SUB_1));
	hw_ptr->s16Sub1.pFlexPwmVAL1	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL1_OFFSET + (0x50 * FLEXPWM_SUB_1));
	hw_ptr->s16Sub1.pFlexPwmVAL2	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL2_OFFSET + (0x50 * FLEXPWM_SUB_1));
	hw_ptr->s16Sub1.pFlexPwmVAL3	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL3_OFFSET + (0x50 * FLEXPWM_SUB_1));
	hw_ptr->s16Sub1.pFlexPwmVAL4	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL4_OFFSET + (0x50 * FLEXPWM_SUB_1));
	hw_ptr->s16Sub1.pFlexPwmVAL5	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL5_OFFSET + (0x50 * FLEXPWM_SUB_1));

	// phase C
	hw_ptr->s16Sub2.pFlexPwmINIT	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_INIT_OFFSET + (0x50 * FLEXPWM_SUB_2));
	hw_ptr->s16Sub2.pFlexPwmVAL0	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL0_OFFSET + (0x50 * FLEXPWM_SUB_2));
	hw_ptr->s16Sub2.pFlexPwmVAL1	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL1_OFFSET + (0x50 * FLEXPWM_SUB_2));
	hw_ptr->s16Sub2.pFlexPwmVAL2	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL2_OFFSET + (0x50 * FLEXPWM_SUB_2));
	hw_ptr->s16Sub2.pFlexPwmVAL3	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL3_OFFSET + (0x50 * FLEXPWM_SUB_2));
	hw_ptr->s16Sub2.pFlexPwmVAL4	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL4_OFFSET + (0x50 * FLEXPWM_SUB_2));
	hw_ptr->s16Sub2.pFlexPwmVAL5	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_VAL5_OFFSET + (0x50 * FLEXPWM_SUB_2));

	hw_ptr->pFlexPwmOUTEN	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_OUTEN_OFFSET);
	hw_ptr->pFlexPwmMASK	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_MASK_OFFSET);
	hw_ptr->pFlexPwmMCTRL	= (volatile tS16 *)(FLEXPWM_X_BASE + FLEXPWM_MCTRL_OFFSET);

	return(TRUE);
}

/***************************************************************************//*!
@brief          Function to read PWM modulo from VAL1 register

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure
				sw_ptr  Pointer to flexPWM SW configuration structure

@return         tBool

@details
******************************************************************************/
tBool FLEXPWM_get_modulo(flexPWM_hw_cfg_t * hw_ptr,flexPWM_sw_t * sw_ptr){

	// phase A
	sw_ptr->moduloSub0 = (tS16)*hw_ptr->s16Sub0.pFlexPwmVAL1;
	// phase B
	sw_ptr->moduloSub1 = (tS16)*hw_ptr->s16Sub1.pFlexPwmVAL1;
	// phase C
	sw_ptr->moduloSub2 = (tS16)*hw_ptr->s16Sub2.pFlexPwmVAL1;

	return (TRUE);
}

/***************************************************************************//*!
@brief          Set Load Okay bit

@param[in,out]  hw_ptr	Pointer to FlexPWM HW configuration structure

@return         tBool

@details		LDOK bit in FlexPWM_MCTRL register
******************************************************************************/
tBool FLEXPWM_set_LDOK(flexPWM_hw_cfg_t * hw_ptr){

	*hw_ptr->pFlexPwmMCTRL = ((*hw_ptr->pFlexPwmMCTRL) & (~0x000F)) |(0x1 | 0x2 | 0x4);

	return (TRUE);
}

/***************************************************************************//*!
@brief          Recalculate the application (Float) DutyCycles to flexPWM (s16) DutyCycles

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure
				sw_ptr  Pointer to flexPWM SW configuration structures

@return         tBool

@details
******************************************************************************/
tBool FLEXPWM_calc_3ph_DutyCycles(flexPWM_hw_cfg_t * hw_ptr,flexPWM_sw_t * sw_ptr){

	FLEXPWM_get_modulo(hw_ptr, sw_ptr);

	sw_ptr->DutyCycle.u16Sub0 = (tU16)(MLIB_Mul((tFloat)sw_ptr->moduloSub0,sw_ptr->PhABC.fltArg1));
	sw_ptr->DutyCycle.u16Sub1 = (tU16)(MLIB_Mul((tFloat)sw_ptr->moduloSub1,sw_ptr->PhABC.fltArg2));
	sw_ptr->DutyCycle.u16Sub2 = (tU16)(MLIB_Mul((tFloat)sw_ptr->moduloSub2,sw_ptr->PhABC.fltArg3));

	return (TRUE);
}

/***************************************************************************//*!
@brief          Update related VALx registers - CENTER ALIGNED PWM

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure
				sw_ptr  Pointer to flexPWM SW configuration structures

@return         tBool

@details
******************************************************************************/
tBool FLEXPWM_set_3ph_pwmVal(flexPWM_hw_cfg_t * hw_ptr,flexPWM_sw_t * sw_ptr){

	// phase A
	*hw_ptr ->s16Sub0.pFlexPwmVAL2 = MLIB_NegSat_F16(sw_ptr->DutyCycle.u16Sub0);
	*hw_ptr ->s16Sub0.pFlexPwmVAL3 = sw_ptr->DutyCycle.u16Sub0;

	// phase B
	*hw_ptr ->s16Sub1.pFlexPwmVAL2 = MLIB_NegSat_F16(sw_ptr->DutyCycle.u16Sub1);
	*hw_ptr ->s16Sub1.pFlexPwmVAL3 = sw_ptr->DutyCycle.u16Sub1;

	// phase C
	*hw_ptr ->s16Sub2.pFlexPwmVAL2 = MLIB_NegSat_F16(sw_ptr->DutyCycle.u16Sub2);
	*hw_ptr ->s16Sub2.pFlexPwmVAL3 = sw_ptr->DutyCycle.u16Sub2;


	return (TRUE);
}

/***************************************************************************//*!
@brief          Enable the output of flexPWM

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure

@return         tBool

@details
******************************************************************************/
tBool FLEXPWM_enable_output(flexPWM_hw_cfg_t * hw_ptr){
	/* FlexPWM
	     * OUTEN register = mcPWM_BASE + $140
	     * MASK register = mcPWM_BASE + $142
	     *
	     * The OUTEN bits take effect immediately after writing.
	     * PWM_OUTEN = 0 - PWMA output disabled.
	     * PWM_OUTEN = 1 - PWMA output enabled.
	     * |--------------------------------|
	     * |RESERVED| PWM_A | PWM_B | PWM_X |
	     * |        |  sub  |  sub  |  sub  |
	     * |        |3|2|1|0|3|2|1|0|3|2|1|0|
	     * |--------------------------------|
	     * | 0|0|0|0|       |       |       |
	     * |--------------------------------|
	     *
	     * The MASKx bits are double buffered and do not take effect until a
	     * FORCE_OUT event occurs within the appropriate submodule.
	     * PWM_MASK = 0 - PWMA output normal.
	     * PWM_MASK = 1 - PWMA output masked.
	     * |--------------------------------|
	     * |RESERVED| PWM_A | PWM_B | PWM_X |
	     * |        |  sub  |  sub  |  sub  |
	     * |        |3|2|1|0|3|2|1|0|3|2|1|0|
	     * |--------------------------------|
	     * | 0|0|0|0|       |       |       |
	     * |--------------------------------|
	     */

	*hw_ptr->pFlexPwmOUTEN	= 0x0FF0; 	// PWM_A & B in all submodules are enabled
	*hw_ptr->pFlexPwmMASK	= 0x0;

	return (TRUE);
}

/***************************************************************************//*!
@brief          Disable the output of flexPWM

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure

@return         tBool

@details
******************************************************************************/
tBool FLEXPWM_disable_output(flexPWM_hw_cfg_t * hw_ptr){

	/* FlexPWM
	     * OUTEN register = mcPWM_BASE + $140
	     * MASK register = mcPWM_BASE + $142
	     *
	     * The OUTEN bits take effect immediately after writing.
	     * PWM_OUTEN = 0 - PWMA output disabled.
	     * PWM_OUTEN = 1 - PWMA output enabled.
	     * |--------------------------------|
	     * |RESERVED| PWM_A | PWM_B | PWM_X |
	     * |        |  sub  |  sub  |  sub  |
	     * |        |3|2|1|0|3|2|1|0|3|2|1|0|
	     * |--------------------------------|
	     * | 0|0|0|0|       |       |       |
	     * |--------------------------------|
	     *
	     * The MASKx bits are double buffered and do not take effect until a
	     * FORCE_OUT event occurs within the appropriate submodule.
	     * PWM_MASK = 0 - PWMA output normal.
	     * PWM_MASK = 1 - PWMA output masked.
	     * |--------------------------------|
	     * |RESERVED| PWM_A | PWM_B | PWM_X |
	     * |        |  sub  |  sub  |  sub  |
	     * |        |3|2|1|0|3|2|1|0|3|2|1|0|
	     * |--------------------------------|
	     * | 0|0|0|0|       |       |       |
	     * |--------------------------------|
	     */

	*hw_ptr->pFlexPwmMASK = 0x0777;		// PWM A/B/X output masked in sub0/1/2

	return (TRUE);
}

/***************************************************************************//*!
@brief          Function performs calculation of DutyCycles and updates flexPWM VALx
				register

@param[in,out]  hw_ptr	Pointer to flexPWM HW configuration structure
				sw_ptr  Pointer to flexPWM SW configuration structures

@return         tBool

@details
******************************************************************************/
tBool apply_voltages(flexPWM_hw_cfg_t * hw_ptr,flexPWM_sw_t * sw_ptr){
	FLEXPWM_calc_3ph_DutyCycles(hw_ptr, sw_ptr);
	FLEXPWM_set_3ph_pwmVal(hw_ptr, sw_ptr);

	return (TRUE);
}
