/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2013 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file     sys.c
*
* @author   b15651
* 
* @version  1.0.1.0
* 
* @date     Aug-28-2013
* 
* @brief    System initialization functions. Magic Carpet initialization.
*           Interrupt initialization.
*
*******************************************************************************
*
* Detailed Description of the file.
*
*******************************************************************************/

/*******************************************************************************
* External objects
*******************************************************************************/
#include "MPC5744P_qs.h"
#include "sys.h"

/***************************************************************************//*!
@brief          MPC5744P Reset generation module initialization

@param[in,out]  void
@param[in]      void

@return         void

@details        The function setup MPC5744P MC_RGM, which includes clock
                generation module (MC_CGM), mode entry (MC_ME), reset generation
                module (RGM), power control unit (PCU).

@warning        
******************************************************************************/
void mc_mode_init(void)
{   
    vuint32_t      temp,timer;

    /* Disable Software Watchdog (SWT) */
    SWT_0.SR.R						= 0xc520;
    SWT_0.SR.R                      = 0xd928;
    SWT_0.CR.R                      = 0x8000010A;

    /* Clear all "Functional Event Status reset sources" to avoid staying in SAFE mode */
    MC_RGM.DES.R = 0xFFFFFFFF;               //write 1 to clear dest reset events
    MC_RGM.FES.R = 0xFFFFFFFF;               //write 1 to clear func reset events
    MC_ME.ME.R = 0x000005FF;   				//enable all modes

    MC_ME.ME.R = 0x000005FF;

    // Enable the EXTAL clk
    MC_ME.DRUN_MC.B.XOSCON=1;
    /* RE enter the drun mode, to update the configuration */
    MC_ME.MCTL.R = 0x30005AF0;                       /* Mode & Key */
    MC_ME.MCTL.R = 0x3000A50F;                /* Mode & Key inverted */
    while(MC_ME.GS.B.S_MTRANS == 1);                /* Wait for mode entry to complete */
    while(MC_ME.GS.B.S_CURRENT_MODE != 0x3);       /* Check DRUN mode has been entered */
    /* Wait for clock to stabilize */
    while(!MC_ME.GS.B.S_XOSC);

    AIPS_0.MPRA.R = 0x77777777;
    AIPS_1.MPRA.R = 0x77777777;

    /* ====================================================
    * Auxiliary Clock 0
    * - motor control clock MC_PLL_CLK   divider 0
    * - SGEN clock                       divider 1
    * - ADC clock						divider 2
    * ====================================================*/
    MC_CGM.AC0_SC.B.SELCTL			= AC0SC_SELCTL_PLL0;		/* Select PLL0 PHI as AUX clock 0 source */

    MC_CGM.AC0_DC0.B.DE             = DIVIDER_ENABLE;     		/* MOTC_CLK divider enable */
    MC_CGM.AC0_DC0.B.DIV			= 0;                   		/* PLL0 PHI div. by 1 as MOTC_CLK, 160MHz */

    MC_CGM.AC0_DC1.B.DE				= DIVIDER_ENABLE;     		/* SWG_CLK clock divider enable */
    MC_CGM.AC0_DC1.B.DIV			= 7;                        /* PLL0 PHI div. by 8 as SWG_CLK, 20MHz */

    MC_CGM.AC0_DC2.B.DE				= DIVIDER_ENABLE;     		/* ADC_CLK clock divider enable */
    MC_CGM.AC0_DC2.B.DIV			= 1;                        /* PLL0 PHI div. by 2 as ADC_CLK, 80MHz */

    /*====================================================
     Auxiliary Clock 1 (FlexRay clock ) configuration
     ====================================================*/
//    CGM.AC1_SC.B.SELCTL              = AC1SC_CLKSRC_SECPLL;
//    CGM.AC1_DC0_3.B.DE0              = AC1DC_DIVIDER_ENABLE;
//    CGM.AC1_DC0_3.B.DIV0             = 0;                        // Aux1 clock divider   0 = 1:1;  1 = 1:2;  2 = 1:3 ... 15 = 1:16


    /*====================================================
     Auxiliary Clock 2 (Can clock) configuration
     ====================================================*/
    MC_CGM.AC2_DC0.B.DE				= DIVIDER_ENABLE;			/* CAN_CLK clock divider enable */
    MC_CGM.AC2_DC0.B.DIV			= 3;						/* PLL0 PHI div. by 4 as CAN_CLK, 40MHz */

    /*====================================================
     Auxiliary Clock 3 Clock source for FMPLL0
     ====================================================*/
    MC_CGM.AC3_SC.B.SELCTL			= 1;						/* Select XOSC as PLL0 clock source */

    /* XOSC settings */
    XOSC.CTL.R = 0x40800000UL;      /* Loop controlled pierce mode, end of count */

    /*====================================================
     Auxiliary Clock 4 Clock source for FMPLL1
     ====================================================*/
    MC_CGM.AC4_SC.B.SELCTL			= 1;						/* Select XOSC as PLL1 clock source */

    /* XOSC settings */
    XOSC.CTL.R = 0x40800000UL;      /* Loop controlled pierce mode, end of count */

    /*====================================================
     Auxiliary Clock 5 Clock source for
     - LFAST PLL   divider 0
     ====================================================*/
    /* CLKOUT clock source selection */
    MC_CGM.AC5_SC.B.SELCTL          = 1;						/* Select XOSC as AUX clock 5 source */
    /* Output clock enable */
    MC_CGM.AC5_DC0.B.DE				= DIVIDER_ENABLE;
    /* System clock division */
    MC_CGM.AC5_DC0.B.DIV			= 0;						/* XOSC div. by 1 as LFAST_REF_CLK, 40MHz */

    /*====================================================
     Auxiliary Clock 6 - External clock (Clockout) config.
     ====================================================*/
    /* CLKOUT clock source selection */
    MC_CGM.AC6_SC.B.SELCTL          = AC6_SC_CLKOUT_SRC_PLL1;	/* Select PLL1 PHI as AUX clock 6 source */
    /* Output clock enable */
    MC_CGM.AC6_DC0.B.DE				= CLKOUT_ENABLE;
    /* System clock division */
    MC_CGM.AC6_DC0.B.DIV			= 0x13;						/* PLL1 PHI div. by 20 as CLKOUT0, 10MHz */

    /*====================================================
     Auxiliary Clock 10 - ENET_CLK clock configuration
     ====================================================*/
    MC_CGM.AC11_SC.B.SELCTL			= 3;						/* PLL0 PHI1 */

    MC_CGM.AC11_DC0.B.DE			= DIVIDER_DISABLE;
    MC_CGM.AC11_DC0.B.DIV			= 3;						/* PLL0 PHI1 div. by 4 as ENET_CLK, 40MHz */

    /*====================================================
     Auxiliary Clock 11 - ENET_TIME_CLK clock configuration
     ====================================================*/
    MC_CGM.AC10_SC.B.SELCTL			= 3;						/* PLL0 PHI1 */

    MC_CGM.AC10_DC0.B.DE			= DIVIDER_DISABLE;
    MC_CGM.AC10_DC0.B.DIV			= 3;						/* PLL0 PHI1 div. by 4 as ENET_TIME_CLK, 40MHz */

    MC_CGM.PCS_SDUR.R = 0xFF;                               // Configure Switch Duration

    MC_ME.DRUN_MC.B.SYSCLK = 1;            /* Select system clock as PLL PHI */
    MC_ME.MCTL.R = 0x30005AF0;                /* Mode & Key */
    MC_ME.MCTL.R = 0x3000A50F;             /* Mode & Key inverted */

    while(MC_ME.GS.B.S_MTRANS==1);   /* Wait for mode entry to complete */
    while(MC_ME.GS.B.S_CURRENT_MODE!=0x3); /* Check DRUN mode has been entered */
    while(MC_ME.GS.B.S_SYSCLK != 1);       /* fail is stuck here         */

#ifdef XOSC40MHZ
    #ifdef PLL0_160MHZ
    /* Setup PLL0 - 160MHz @ 40MHz XOSC */
    PLLDIG.PLL0DV.R = 0x20041010UL; /* RFDPHI1 = 4, RFDPHI = 4, PREDIV = 1,
                                       MFD = 16, fVCO = 640MHz */
	#endif
#endif
#ifdef XOSC40MHZ
	#ifdef PLL1_200MHZ
    /* Setup PLL1 - 200MHz @ 40MHz XOSC */
    PLLDIG.PLL1DV.R = 0x00020014UL; /* RFDPHI = 2 (divider=4), MFD = 20,
                                       fVCO = 800MHz */
	#endif
#endif

    /* PLL PULSE to lock PLL device */
    MC_ME.DRUN_MC.B.PLL0ON = 0;           // Disable the PLL0
    MC_ME.DRUN_MC.B.PLL1ON = 0;           // Disable the PLL1
    /* RE  enter the drun mode, to update the configuration */
    MC_ME.MCTL.R = 0x30005AF0;                /* Mode & Key */
    MC_ME.MCTL.R = 0x3000A50F;                /* Mode & Key inverted */
    while(MC_ME.GS.B.S_MTRANS == 1);                          /* Wait for mode entry to complete */
    while(MC_ME.GS.B.S_CURRENT_MODE != 0x3);       /* Check DRUN mode has been entered */

    MC_ME.DRUN_MC.B.PLL0ON = 1;           // Enable the PLL0
    MC_ME.DRUN_MC.B.PLL1ON = 1;           // Enable the PLL1
    /* RE  enter the drun mode, to update the configuration */
    MC_ME.MCTL.R = 0x30005AF0;                        /* Mode & Key */
    MC_ME.MCTL.R = 0x3000A50F;                /* Mode & Key inverted */
    while(MC_ME.GS.B.S_MTRANS == 1);                          /* Wait for mode entry to complete */
    while(MC_ME.GS.B.S_CURRENT_MODE != 0x3);       /* Check DRUN mode has been entered */
    //waiting for PLL0 to lock
    while(PLLDIG.PLL0SR.B.LOCK ==0);
    //waiting for PLL1 to lock
    while(PLLDIG.PLL1SR.B.LOCK ==0);

    /* RUN0 & System Clock */
    MC_ME.RUN0_MC.R = 0x000300E4UL;   /* FLASH in normal mode, PLL0 ON, PLL1
                                           ON, XOSC ON, system clock is sourced
                                           from secondary PLL (PLL1) */
    MC_CGM.SC_DC0.R = 0x80030000UL; /* PLL1 PHI div. by 4 as PBRIDGEx_CLK,
                                       50MHz */
    
    MC_ME.RUN_PC[0].R = 0x000000FEUL;   /* Peripherals ON in every mode */
    
    /* Enter RUN0 mode */
    MC_ME.MCTL.R					= 0x40005AF0;   // RUN0 Mode & Key
    MC_ME.MCTL.R					= 0x4000A50F;   // RUN0 Mode & Key
    while(MC_ME.GS.B.S_MTRANS == 1);                /* Wait for mode entry to complete */
    while(MC_ME.GS.B.S_CURRENT_MODE != 0x4);		/* Check RUN0 mode has been entered */

}

/**************************************************************************//*!
@brief          This function intializes the INTC priority for software vector mode.

@param[in,out]  void
@param[in]      void

@return         void

@details        Set up INTC vector table base address to the  address defined
                in the linker. Satellite transmission interrupt has priority 2.
                Diagnostic transmission interrupt has priority 3.

@note           
                
@warning        
******************************************************************************/
void intc_init(void)
{    
	volatile uint32_t dummy = 0;
    /* Initialize INTC for software vector mode */
    INTC_0.BCR.B.HVEN0 = 0;

    // Set INTC ISR vector table base addr.
    INTC_0.IACKR0.R = (uint32_t) &IntcIsrVectorTable[0];

    #ifdef IRQ_0_PRIORITY        /* ISR 0 */
       INTC_0.PSR[0].B.PRIN=IRQ_0_PRIORITY;
    #endif
    #ifdef IRQ_1_PRIORITY        /* ISR 1 */
       INTC_0.PSR[1].B.PRIN=IRQ_1_PRIORITY;
    #endif
    #ifdef IRQ_2_PRIORITY        /* ISR 2 */
       INTC_0.PSR[2].B.PRIN=IRQ_2_PRIORITY;
    #endif
    #ifdef IRQ_3_PRIORITY        /* ISR 3 */
       INTC_0.PSR[3].B.PRIN=IRQ_3_PRIORITY;
    #endif
    #ifdef IRQ_4_PRIORITY        /* ISR 4 */
       INTC_0.PSR[4].B.PRIN=IRQ_4_PRIORITY;
    #endif
    #ifdef IRQ_5_PRIORITY        /* ISR 5 */
       INTC_0.PSR[5].B.PRIN=IRQ_5_PRIORITY;
    #endif
    #ifdef IRQ_6_PRIORITY        /* ISR 6 */
       INTC_0.PSR[6].B.PRIN=IRQ_6_PRIORITY;
    #endif
    #ifdef IRQ_7_PRIORITY        /* ISR 7 */
       INTC_0.PSR[7].B.PRIN=IRQ_7_PRIORITY;
    #endif
    #ifdef IRQ_8_PRIORITY        /* ISR 8 */
       INTC_0.PSR[8].B.PRIN=IRQ_8_PRIORITY;
    #endif
    #ifdef IRQ_9_PRIORITY        /* ISR 9 */
       INTC_0.PSR[9].B.PRIN=IRQ_9_PRIORITY;
    #endif
    #ifdef IRQ_10_PRIORITY       /* ISR 10 */
       INTC_0.PSR[10].B.PRIN=IRQ_10_PRIORITY;
    #endif
    #ifdef IRQ_11_PRIORITY       /* ISR 11 */
       INTC_0.PSR[11].B.PRIN=IRQ_11_PRIORITY;
    #endif
    #ifdef IRQ_12_PRIORITY       /* ISR 12 */
       INTC_0.PSR[12].B.PRIN=IRQ_12_PRIORITY;
    #endif
    #ifdef IRQ_13_PRIORITY       /* ISR 13 */
       INTC_0.PSR[13].B.PRIN=IRQ_13_PRIORITY;
    #endif
    #ifdef IRQ_14_PRIORITY       /* ISR 14 */
       INTC_0.PSR[14].B.PRIN=IRQ_14_PRIORITY;
    #endif
    #ifdef IRQ_15_PRIORITY       /* ISR 15 */
       INTC_0.PSR[15].B.PRIN=IRQ_15_PRIORITY;
    #endif
    #ifdef IRQ_16_PRIORITY       /* ISR 16 */
       INTC_0.PSR[16].B.PRIN=IRQ_16_PRIORITY;
    #endif
    #ifdef IRQ_17_PRIORITY       /* ISR 17 */
       INTC_0.PSR[17].B.PRIN=IRQ_17_PRIORITY;
    #endif
    #ifdef IRQ_18_PRIORITY       /* ISR 18 */
       INTC_0.PSR[18].B.PRIN=IRQ_18_PRIORITY;
    #endif
    #ifdef IRQ_19_PRIORITY       /* ISR 19 */
       INTC_0.PSR[19].B.PRIN=IRQ_19_PRIORITY;
    #endif
    #ifdef IRQ_20_PRIORITY       /* ISR 20 */
       INTC_0.PSR[20].B.PRIN=IRQ_20_PRIORITY;
    #endif
    #ifdef IRQ_21_PRIORITY       /* ISR 21 */
       INTC_0.PSR[21].B.PRIN=IRQ_21_PRIORITY;
    #endif
    #ifdef IRQ_22_PRIORITY       /* ISR 22 */
       INTC_0.PSR[22].B.PRIN=IRQ_22_PRIORITY;
    #endif
    #ifdef IRQ_23_PRIORITY       /* ISR 23 */
       INTC_0.PSR[23].B.PRIN=IRQ_23_PRIORITY;
    #endif
    #ifdef IRQ_24_PRIORITY       /* ISR 24 */
       INTC_0.PSR[24].B.PRIN=IRQ_24_PRIORITY;
    #endif
    #ifdef IRQ_25_PRIORITY       /* ISR 25 */
       INTC_0.PSR[25].B.PRIN=IRQ_25_PRIORITY;
    #endif
    #ifdef IRQ_26_PRIORITY       /* ISR 26 */
       INTC_0.PSR[26].B.PRIN=IRQ_26_PRIORITY;
    #endif
    #ifdef IRQ_27_PRIORITY       /* ISR 27 */
       INTC_0.PSR[27].B.PRIN=IRQ_27_PRIORITY;
    #endif
    #ifdef IRQ_28_PRIORITY       /* ISR 28 */
       INTC_0.PSR[28].B.PRIN=IRQ_28_PRIORITY;
    #endif
    #ifdef IRQ_29_PRIORITY       /* ISR 29 */
       INTC_0.PSR[29].B.PRIN=IRQ_29_PRIORITY;
    #endif
    #ifdef IRQ_30_PRIORITY       /* ISR 30 */
       INTC_0.PSR[30].B.PRIN=IRQ_30_PRIORITY;
    #endif
    #ifdef IRQ_31_PRIORITY       /* ISR 31 */
       INTC_0.PSR[31].B.PRIN=IRQ_31_PRIORITY;
    #endif
    #ifdef IRQ_32_PRIORITY       /* ISR 32 */
       INTC_0.PSR[32].B.PRIN=IRQ_32_PRIORITY;
    #endif
    #ifdef IRQ_33_PRIORITY       /* ISR 33 */
       INTC_0.PSR[33].B.PRIN=IRQ_33_PRIORITY;
    #endif
    #ifdef IRQ_34_PRIORITY       /* ISR 34 */
       INTC_0.PSR[34].B.PRIN=IRQ_34_PRIORITY;
    #endif
    #ifdef IRQ_35_PRIORITY       /* ISR 35 */
       INTC_0.PSR[35].B.PRIN=IRQ_35_PRIORITY;
    #endif
    #ifdef IRQ_36_PRIORITY       /* ISR 36 */
       INTC_0.PSR[36].B.PRIN=IRQ_36_PRIORITY;
    #endif
    #ifdef IRQ_37_PRIORITY       /* ISR 37 */
       INTC_0.PSR[37].B.PRIN=IRQ_37_PRIORITY;
    #endif
    #ifdef IRQ_38_PRIORITY       /* ISR 38 */
       INTC_0.PSR[38].B.PRIN=IRQ_38_PRIORITY;
    #endif
    #ifdef IRQ_39_PRIORITY       /* ISR 39 */
       INTC_0.PSR[39].B.PRIN=IRQ_39_PRIORITY;
    #endif
    #ifdef IRQ_40_PRIORITY       /* ISR 40 */
       INTC_0.PSR[40].B.PRIN=IRQ_40_PRIORITY;
    #endif
    #ifdef IRQ_41_PRIORITY       /* ISR 41 */
       INTC_0.PSR[41].B.PRIN=IRQ_41_PRIORITY;
    #endif
    #ifdef IRQ_42_PRIORITY       /* ISR 42 */
       INTC_0.PSR[42].B.PRIN=IRQ_42_PRIORITY;
    #endif
    #ifdef IRQ_43_PRIORITY       /* ISR 43 */
       INTC_0.PSR[43].B.PRIN=IRQ_43_PRIORITY;
    #endif
    #ifdef IRQ_44_PRIORITY       /* ISR 44 */
       INTC_0.PSR[44].B.PRIN=IRQ_44_PRIORITY;
    #endif
    #ifdef IRQ_45_PRIORITY       /* ISR 45 */
       INTC_0.PSR[45].B.PRIN=IRQ_45_PRIORITY;
    #endif
    #ifdef IRQ_46_PRIORITY       /* ISR 46 */
       INTC_0.PSR[46].B.PRIN=IRQ_46_PRIORITY;
    #endif
    #ifdef IRQ_47_PRIORITY       /* ISR 47 */
       INTC_0.PSR[47].B.PRIN=IRQ_47_PRIORITY;
    #endif
    #ifdef IRQ_48_PRIORITY       /* ISR 48 */
       INTC_0.PSR[48].B.PRIN=IRQ_48_PRIORITY;
    #endif
    #ifdef IRQ_49_PRIORITY       /* ISR 49 */
       INTC_0.PSR[49].B.PRIN=IRQ_49_PRIORITY;
    #endif
    #ifdef IRQ_50_PRIORITY       /* ISR 50 */
       INTC_0.PSR[50].B.PRIN=IRQ_50_PRIORITY;
    #endif
    #ifdef IRQ_51_PRIORITY       /* ISR 51 */
       INTC_0.PSR[51].B.PRIN=IRQ_51_PRIORITY;
    #endif
    #ifdef IRQ_52_PRIORITY       /* ISR 52 */
       INTC_0.PSR[52].B.PRIN=IRQ_52_PRIORITY;
    #endif
    #ifdef IRQ_53_PRIORITY       /* ISR 53 */
       INTC_0.PSR[53].B.PRIN=IRQ_53_PRIORITY;
    #endif
    #ifdef IRQ_54_PRIORITY       /* ISR 54 */
       INTC_0.PSR[54].B.PRIN=IRQ_54_PRIORITY;
    #endif
    #ifdef IRQ_55_PRIORITY       /* ISR 55 */
       INTC_0.PSR[55].B.PRIN=IRQ_55_PRIORITY;
    #endif
    #ifdef IRQ_56_PRIORITY       /* ISR 56 */
       INTC_0.PSR[56].B.PRIN=IRQ_56_PRIORITY;
    #endif
    #ifdef IRQ_57_PRIORITY       /* ISR 57 */
       INTC_0.PSR[57].B.PRIN=IRQ_57_PRIORITY;
    #endif
    #ifdef IRQ_58_PRIORITY       /* ISR 58 */
       INTC_0.PSR[58].B.PRIN=IRQ_58_PRIORITY;
    #endif
    #ifdef IRQ_59_PRIORITY       /* ISR 59 */
       INTC_0.PSR[59].B.PRIN=IRQ_59_PRIORITY;
    #endif
    #ifdef IRQ_60_PRIORITY       /* ISR 60 */
       INTC_0.PSR[60].B.PRIN=IRQ_60_PRIORITY;
    #endif
    #ifdef IRQ_61_PRIORITY       /* ISR 61 */
       INTC_0.PSR[61].B.PRIN=IRQ_61_PRIORITY;
    #endif
    #ifdef IRQ_62_PRIORITY       /* ISR 62 */
       INTC_0.PSR[62].B.PRIN=IRQ_62_PRIORITY;
    #endif
    #ifdef IRQ_63_PRIORITY       /* ISR 63 */
       INTC_0.PSR[63].B.PRIN=IRQ_63_PRIORITY;
    #endif
    #ifdef IRQ_64_PRIORITY       /* ISR 64 */
       INTC_0.PSR[64].B.PRIN=IRQ_64_PRIORITY;
    #endif
    #ifdef IRQ_65_PRIORITY       /* ISR 65 */
       INTC_0.PSR[65].B.PRIN=IRQ_65_PRIORITY;
    #endif
    #ifdef IRQ_66_PRIORITY       /* ISR 66 */
       INTC_0.PSR[66].B.PRIN=IRQ_66_PRIORITY;
    #endif
    #ifdef IRQ_67_PRIORITY       /* ISR 67 */
       INTC_0.PSR[67].B.PRIN=IRQ_67_PRIORITY;
    #endif
    #ifdef IRQ_68_PRIORITY       /* ISR 68 */
       INTC_0.PSR[68].B.PRIN=IRQ_68_PRIORITY;
    #endif
    #ifdef IRQ_69_PRIORITY       /* ISR 69 */
       INTC_0.PSR[69].B.PRIN=IRQ_69_PRIORITY;
    #endif
    #ifdef IRQ_70_PRIORITY       /* ISR 70 */
       INTC_0.PSR[70].B.PRIN=IRQ_70_PRIORITY;
    #endif
    #ifdef IRQ_71_PRIORITY       /* ISR 71 */
       INTC_0.PSR[71].B.PRIN=IRQ_71_PRIORITY;
    #endif
    #ifdef IRQ_72_PRIORITY       /* ISR 72 */
       INTC_0.PSR[72].B.PRIN=IRQ_72_PRIORITY;
    #endif
    #ifdef IRQ_73_PRIORITY       /* ISR 73 */
       INTC_0.PSR[73].B.PRIN=IRQ_73_PRIORITY;
    #endif
    #ifdef IRQ_74_PRIORITY       /* ISR 74 */
       INTC_0.PSR[74].B.PRIN=IRQ_74_PRIORITY;
    #endif
    #ifdef IRQ_75_PRIORITY       /* ISR 75 */
       INTC_0.PSR[75].B.PRIN=IRQ_75_PRIORITY;
    #endif
    #ifdef IRQ_76_PRIORITY       /* ISR 76 */
       INTC_0.PSR[76].B.PRIN=IRQ_76_PRIORITY;
    #endif
    #ifdef IRQ_77_PRIORITY       /* ISR 77 */
       INTC_0.PSR[77].B.PRIN=IRQ_77_PRIORITY;
    #endif
    #ifdef IRQ_78_PRIORITY       /* ISR 78 */
       INTC_0.PSR[78].B.PRIN=IRQ_78_PRIORITY;
    #endif
    #ifdef IRQ_79_PRIORITY       /* ISR 79 */
       INTC_0.PSR[79].B.PRIN=IRQ_79_PRIORITY;
    #endif
    #ifdef IRQ_80_PRIORITY       /* ISR 80 */
       INTC_0.PSR[80].B.PRIN=IRQ_80_PRIORITY;
    #endif
    #ifdef IRQ_81_PRIORITY       /* ISR 81 */
       INTC_0.PSR[81].B.PRIN=IRQ_81_PRIORITY;
    #endif
    #ifdef IRQ_82_PRIORITY       /* ISR 82 */
       INTC_0.PSR[82].B.PRIN=IRQ_82_PRIORITY;
    #endif
    #ifdef IRQ_83_PRIORITY       /* ISR 83 */
       INTC_0.PSR[83].B.PRIN=IRQ_83_PRIORITY;
    #endif
    #ifdef IRQ_84_PRIORITY       /* ISR 84 */
       INTC_0.PSR[84].B.PRIN=IRQ_84_PRIORITY;
    #endif
    #ifdef IRQ_85_PRIORITY       /* ISR 85 */
       INTC_0.PSR[85].B.PRIN=IRQ_85_PRIORITY;
    #endif
    #ifdef IRQ_86_PRIORITY       /* ISR 86 */
       INTC_0.PSR[86].B.PRIN=IRQ_86_PRIORITY;
    #endif
    #ifdef IRQ_87_PRIORITY       /* ISR 87 */
       INTC_0.PSR[87].B.PRIN=IRQ_87_PRIORITY;
    #endif
    #ifdef IRQ_88_PRIORITY       /* ISR 88 */
       INTC_0.PSR[88].B.PRIN=IRQ_88_PRIORITY;
    #endif
    #ifdef IRQ_89_PRIORITY       /* ISR 89 */
       INTC_0.PSR[89].B.PRIN=IRQ_89_PRIORITY;
    #endif
    #ifdef IRQ_90_PRIORITY       /* ISR 90 */
       INTC_0.PSR[90].B.PRIN=IRQ_90_PRIORITY;
    #endif
    #ifdef IRQ_91_PRIORITY       /* ISR 91 */
       INTC_0.PSR[91].B.PRIN=IRQ_91_PRIORITY;
    #endif
    #ifdef IRQ_92_PRIORITY       /* ISR 92 */
       INTC_0.PSR[92].B.PRIN=IRQ_92_PRIORITY;
    #endif
    #ifdef IRQ_93_PRIORITY       /* ISR 93 */
       INTC_0.PSR[93].B.PRIN=IRQ_93_PRIORITY;
    #endif
    #ifdef IRQ_94_PRIORITY       /* ISR 94 */
       INTC_0.PSR[94].B.PRIN=IRQ_94_PRIORITY;
    #endif
    #ifdef IRQ_95_PRIORITY       /* ISR 95 */
       INTC_0.PSR[95].B.PRIN=IRQ_95_PRIORITY;
    #endif
    #ifdef IRQ_96_PRIORITY       /* ISR 96 */
       INTC_0.PSR[96].B.PRIN=IRQ_96_PRIORITY;
    #endif
    #ifdef IRQ_97_PRIORITY       /* ISR 97 */
       INTC_0.PSR[97].B.PRIN=IRQ_97_PRIORITY;
    #endif
    #ifdef IRQ_98_PRIORITY       /* ISR 98 */
       INTC_0.PSR[98].B.PRIN=IRQ_98_PRIORITY;
    #endif
    #ifdef IRQ_99_PRIORITY       /* ISR 99 */
       INTC_0.PSR[99].B.PRIN=IRQ_99_PRIORITY;
    #endif
    #ifdef IRQ_100_PRIORITY      /* ISR 100 */
       INTC_0.PSR[100].B.PRIN=IRQ_100_PRIORITY;
    #endif
    #ifdef IRQ_101_PRIORITY      /* ISR 101 */
       INTC_0.PSR[101].B.PRIN=IRQ_101_PRIORITY;
    #endif
    #ifdef IRQ_102_PRIORITY      /* ISR 102 */
       INTC_0.PSR[102].B.PRIN=IRQ_102_PRIORITY;
    #endif
    #ifdef IRQ_103_PRIORITY      /* ISR 103 */
       INTC_0.PSR[103].B.PRIN=IRQ_103_PRIORITY;
    #endif
    #ifdef IRQ_104_PRIORITY      /* ISR 104 */
       INTC_0.PSR[104].B.PRIN=IRQ_104_PRIORITY;
    #endif
    #ifdef IRQ_105_PRIORITY      /* ISR 105 */
       INTC_0.PSR[105].B.PRIN=IRQ_105_PRIORITY;
    #endif
    #ifdef IRQ_106_PRIORITY      /* ISR 106 */
       INTC_0.PSR[106].B.PRIN=IRQ_106_PRIORITY;
    #endif
    #ifdef IRQ_107_PRIORITY      /* ISR 107 */
       INTC_0.PSR[107].B.PRIN=IRQ_107_PRIORITY;
    #endif
    #ifdef IRQ_108_PRIORITY      /* ISR 108 */
       INTC_0.PSR[108].B.PRIN=IRQ_108_PRIORITY;
    #endif
    #ifdef IRQ_109_PRIORITY      /* ISR 109 */
       INTC_0.PSR[109].B.PRIN=IRQ_109_PRIORITY;
    #endif
    #ifdef IRQ_110_PRIORITY      /* ISR 110 */
       INTC_0.PSR[110].B.PRIN=IRQ_110_PRIORITY;
    #endif
    #ifdef IRQ_111_PRIORITY      /* ISR 111 */
       INTC_0.PSR[111].B.PRIN=IRQ_111_PRIORITY;
    #endif
    #ifdef IRQ_112_PRIORITY      /* ISR 112 */
       INTC_0.PSR[112].B.PRIN=IRQ_112_PRIORITY;
    #endif
    #ifdef IRQ_113_PRIORITY      /* ISR 113 */
       INTC_0.PSR[113].B.PRIN=IRQ_113_PRIORITY;
    #endif
    #ifdef IRQ_114_PRIORITY      /* ISR 114 */
       INTC_0.PSR[114].B.PRIN=IRQ_114_PRIORITY;
    #endif
    #ifdef IRQ_115_PRIORITY      /* ISR 115 */
       INTC_0.PSR[115].B.PRIN=IRQ_115_PRIORITY;
    #endif
    #ifdef IRQ_116_PRIORITY      /* ISR 116 */
       INTC_0.PSR[116].B.PRIN=IRQ_116_PRIORITY;
    #endif
    #ifdef IRQ_117_PRIORITY      /* ISR 117 */
       INTC_0.PSR[117].B.PRIN=IRQ_117_PRIORITY;
    #endif
    #ifdef IRQ_118_PRIORITY      /* ISR 118 */
       INTC_0.PSR[118].B.PRIN=IRQ_118_PRIORITY;
    #endif
    #ifdef IRQ_119_PRIORITY      /* ISR 119 */
       INTC_0.PSR[119].B.PRIN=IRQ_119_PRIORITY;
    #endif
    #ifdef IRQ_120_PRIORITY      /* ISR 120 */
       INTC_0.PSR[120].B.PRIN=IRQ_120_PRIORITY;
    #endif
    #ifdef IRQ_121_PRIORITY      /* ISR 121 */
       INTC_0.PSR[121].B.PRIN=IRQ_121_PRIORITY;
    #endif
    #ifdef IRQ_122_PRIORITY      /* ISR 122 */
       INTC_0.PSR[122].B.PRIN=IRQ_122_PRIORITY;
    #endif
    #ifdef IRQ_123_PRIORITY      /* ISR 123 */
       INTC_0.PSR[123].B.PRIN=IRQ_123_PRIORITY;
    #endif
    #ifdef IRQ_124_PRIORITY      /* ISR 124 */
       INTC_0.PSR[124].B.PRIN=IRQ_124_PRIORITY;
    #endif
    #ifdef IRQ_125_PRIORITY      /* ISR 125 */
       INTC_0.PSR[125].B.PRIN=IRQ_125_PRIORITY;
    #endif
    #ifdef IRQ_126_PRIORITY      /* ISR 126 */
       INTC_0.PSR[126].B.PRIN=IRQ_126_PRIORITY;
    #endif
    #ifdef IRQ_127_PRIORITY      /* ISR 127 */
       INTC_0.PSR[127].B.PRIN=IRQ_127_PRIORITY;
    #endif
    #ifdef IRQ_128_PRIORITY      /* ISR 128 */
       INTC_0.PSR[128].B.PRIN=IRQ_128_PRIORITY;
    #endif
    #ifdef IRQ_129_PRIORITY      /* ISR 129 */
       INTC_0.PSR[129].B.PRIN=IRQ_129_PRIORITY;
    #endif
    #ifdef IRQ_130_PRIORITY      /* ISR 130 */
       INTC_0.PSR[130].B.PRIN=IRQ_130_PRIORITY;
    #endif
    #ifdef IRQ_131_PRIORITY      /* ISR 131 */
       INTC_0.PSR[131].B.PRIN=IRQ_131_PRIORITY;
    #endif
    #ifdef IRQ_132_PRIORITY      /* ISR 132 */
       INTC_0.PSR[132].B.PRIN=IRQ_132_PRIORITY;
    #endif
    #ifdef IRQ_133_PRIORITY      /* ISR 133 */
       INTC_0.PSR[133].B.PRIN=IRQ_133_PRIORITY;
    #endif
    #ifdef IRQ_134_PRIORITY      /* ISR 134 */
       INTC_0.PSR[134].B.PRIN=IRQ_134_PRIORITY;
    #endif
    #ifdef IRQ_135_PRIORITY      /* ISR 135 */
       INTC_0.PSR[135].B.PRIN=IRQ_135_PRIORITY;
    #endif
    #ifdef IRQ_136_PRIORITY      /* ISR 136 */
       INTC_0.PSR[136].B.PRIN=IRQ_136_PRIORITY;
    #endif
    #ifdef IRQ_137_PRIORITY      /* ISR 137 */
       INTC_0.PSR[137].B.PRIN=IRQ_137_PRIORITY;
    #endif
    #ifdef IRQ_138_PRIORITY      /* ISR 138 */
       INTC_0.PSR[138].B.PRIN=IRQ_138_PRIORITY;
    #endif
    #ifdef IRQ_139_PRIORITY      /* ISR 139 */
       INTC_0.PSR[139].B.PRIN=IRQ_139_PRIORITY;
    #endif
    #ifdef IRQ_140_PRIORITY      /* ISR 140 */
       INTC_0.PSR[140].B.PRIN=IRQ_140_PRIORITY;
    #endif
    #ifdef IRQ_141_PRIORITY      /* ISR 141 */
       INTC_0.PSR[141].B.PRIN=IRQ_141_PRIORITY;
    #endif
    #ifdef IRQ_142_PRIORITY      /* ISR 142 */
       INTC_0.PSR[142].B.PRIN=IRQ_142_PRIORITY;
    #endif
    #ifdef IRQ_143_PRIORITY      /* ISR 143 */
       INTC_0.PSR[143].B.PRIN=IRQ_143_PRIORITY;
    #endif
    #ifdef IRQ_144_PRIORITY      /* ISR 144 */
       INTC_0.PSR[144].B.PRIN=IRQ_144_PRIORITY;
    #endif
    #ifdef IRQ_145_PRIORITY      /* ISR 145 */
       INTC_0.PSR[145].B.PRIN=IRQ_145_PRIORITY;
    #endif
    #ifdef IRQ_146_PRIORITY      /* ISR 146 */
       INTC_0.PSR[146].B.PRIN=IRQ_146_PRIORITY;
    #endif
    #ifdef IRQ_147_PRIORITY      /* ISR 147 */
       INTC_0.PSR[147].B.PRIN=IRQ_147_PRIORITY;
    #endif
    #ifdef IRQ_148_PRIORITY      /* ISR 148 */
       INTC_0.PSR[148].B.PRIN=IRQ_148_PRIORITY;
    #endif
    #ifdef IRQ_149_PRIORITY      /* ISR 149 */
       INTC_0.PSR[149].B.PRIN=IRQ_149_PRIORITY;
    #endif
    #ifdef IRQ_150_PRIORITY      /* ISR 150 */
       INTC_0.PSR[150].B.PRIN=IRQ_150_PRIORITY;
    #endif
    #ifdef IRQ_151_PRIORITY      /* ISR 151 */
       INTC_0.PSR[151].B.PRIN=IRQ_151_PRIORITY;
    #endif
    #ifdef IRQ_152_PRIORITY      /* ISR 152 */
       INTC_0.PSR[152].B.PRIN=IRQ_152_PRIORITY;
    #endif
    #ifdef IRQ_153_PRIORITY      /* ISR 153 */
       INTC_0.PSR[153].B.PRIN=IRQ_153_PRIORITY;
    #endif
    #ifdef IRQ_154_PRIORITY      /* ISR 154 */
       INTC_0.PSR[154].B.PRIN=IRQ_154_PRIORITY;
    #endif
    #ifdef IRQ_155_PRIORITY      /* ISR 155 */
       INTC_0.PSR[155].B.PRIN=IRQ_155_PRIORITY;
    #endif
    #ifdef IRQ_156_PRIORITY      /* ISR 156 */
       INTC_0.PSR[156].B.PRIN=IRQ_156_PRIORITY;
    #endif
    #ifdef IRQ_157_PRIORITY      /* ISR 157 */
       INTC_0.PSR[157].B.PRIN=IRQ_157_PRIORITY;
    #endif
    #ifdef IRQ_158_PRIORITY      /* ISR 158 */
       INTC_0.PSR[158].B.PRIN=IRQ_158_PRIORITY;
    #endif
    #ifdef IRQ_159_PRIORITY      /* ISR 159 */
       INTC_0.PSR[159].B.PRIN=IRQ_159_PRIORITY;
    #endif
    #ifdef IRQ_160_PRIORITY      /* ISR 160 */
       INTC_0.PSR[160].B.PRIN=IRQ_160_PRIORITY;
    #endif
    #ifdef IRQ_161_PRIORITY      /* ISR 161 */
       INTC_0.PSR[161].B.PRIN=IRQ_161_PRIORITY;
    #endif
    #ifdef IRQ_162_PRIORITY      /* ISR 162 */
       INTC_0.PSR[162].B.PRIN=IRQ_162_PRIORITY;
    #endif
    #ifdef IRQ_163_PRIORITY      /* ISR 163 */
       INTC_0.PSR[163].B.PRIN=IRQ_163_PRIORITY;
    #endif
    #ifdef IRQ_164_PRIORITY      /* ISR 164 */
       INTC_0.PSR[164].B.PRIN=IRQ_164_PRIORITY;
    #endif
    #ifdef IRQ_165_PRIORITY      /* ISR 165 */
       INTC_0.PSR[165].B.PRIN=IRQ_165_PRIORITY;
    #endif
    #ifdef IRQ_166_PRIORITY      /* ISR 166 */
       INTC_0.PSR[166].B.PRIN=IRQ_166_PRIORITY;
    #endif
    #ifdef IRQ_167_PRIORITY      /* ISR 167 */
       INTC_0.PSR[167].B.PRIN=IRQ_167_PRIORITY;
    #endif
    #ifdef IRQ_168_PRIORITY      /* ISR 168 */
       INTC_0.PSR[168].B.PRIN=IRQ_168_PRIORITY;
    #endif
    #ifdef IRQ_169_PRIORITY      /* ISR 169 */
       INTC_0.PSR[169].B.PRIN=IRQ_169_PRIORITY;
    #endif
    #ifdef IRQ_170_PRIORITY      /* ISR 170 */
       INTC_0.PSR[170].B.PRIN=IRQ_170_PRIORITY;
    #endif
    #ifdef IRQ_171_PRIORITY      /* ISR 171 */
       INTC_0.PSR[171].B.PRIN=IRQ_171_PRIORITY;
    #endif
    #ifdef IRQ_172_PRIORITY      /* ISR 172 */
       INTC_0.PSR[172].B.PRIN=IRQ_172_PRIORITY;
    #endif
    #ifdef IRQ_173_PRIORITY      /* ISR 173 */
       INTC_0.PSR[173].B.PRIN=IRQ_173_PRIORITY;
    #endif
    #ifdef IRQ_174_PRIORITY      /* ISR 174 */
       INTC_0.PSR[174].B.PRIN=IRQ_174_PRIORITY;
    #endif
    #ifdef IRQ_175_PRIORITY      /* ISR 175 */
       INTC_0.PSR[175].B.PRIN=IRQ_175_PRIORITY;
    #endif
    #ifdef IRQ_176_PRIORITY      /* ISR 176 */
       INTC_0.PSR[176].B.PRIN=IRQ_176_PRIORITY;
    #endif
    #ifdef IRQ_177_PRIORITY      /* ISR 177 */
       INTC_0.PSR[177].B.PRIN=IRQ_177_PRIORITY;
    #endif
    #ifdef IRQ_178_PRIORITY      /* ISR 178 */
       INTC_0.PSR[178].B.PRIN=IRQ_178_PRIORITY;
    #endif
    #ifdef IRQ_179_PRIORITY      /* ISR 179 */
       INTC_0.PSR[179].B.PRIN=IRQ_179_PRIORITY;
    #endif
    #ifdef IRQ_180_PRIORITY      /* ISR 180 */
       INTC_0.PSR[180].B.PRIN=IRQ_180_PRIORITY;
    #endif
    #ifdef IRQ_181_PRIORITY      /* ISR 181 */
       INTC_0.PSR[181].B.PRIN=IRQ_181_PRIORITY;
    #endif
    #ifdef IRQ_182_PRIORITY      /* ISR 182 */
       INTC_0.PSR[182].B.PRIN=IRQ_182_PRIORITY;
    #endif
    #ifdef IRQ_183_PRIORITY      /* ISR 183 */
       INTC_0.PSR[183].B.PRIN=IRQ_183_PRIORITY;
    #endif
    #ifdef IRQ_184_PRIORITY      /* ISR 184 */
       INTC_0.PSR[184].B.PRIN=IRQ_184_PRIORITY;
    #endif
    #ifdef IRQ_185_PRIORITY      /* ISR 185 */
       INTC_0.PSR[185].B.PRIN=IRQ_185_PRIORITY;
    #endif
    #ifdef IRQ_186_PRIORITY      /* ISR 186 */
       INTC_0.PSR[186].B.PRIN=IRQ_186_PRIORITY;
    #endif
    #ifdef IRQ_187_PRIORITY      /* ISR 187 */
       INTC_0.PSR[187].B.PRIN=IRQ_187_PRIORITY;
    #endif
    #ifdef IRQ_188_PRIORITY      /* ISR 188 */
       INTC_0.PSR[188].B.PRIN=IRQ_188_PRIORITY;
    #endif
    #ifdef IRQ_189_PRIORITY      /* ISR 189 */
       INTC_0.PSR[189].B.PRIN=IRQ_189_PRIORITY;
    #endif
    #ifdef IRQ_190_PRIORITY      /* ISR 190 */
       INTC_0.PSR[190].B.PRIN=IRQ_190_PRIORITY;
    #endif
    #ifdef IRQ_191_PRIORITY      /* ISR 191 */
       INTC_0.PSR[191].B.PRIN=IRQ_191_PRIORITY;
    #endif
    #ifdef IRQ_192_PRIORITY      /* ISR 192 */
       INTC_0.PSR[192].B.PRIN=IRQ_192_PRIORITY;
    #endif
    #ifdef IRQ_193_PRIORITY      /* ISR 193 */
       INTC_0.PSR[193].B.PRIN=IRQ_193_PRIORITY;
    #endif
    #ifdef IRQ_194_PRIORITY      /* ISR 194 */
       INTC_0.PSR[194].B.PRIN=IRQ_194_PRIORITY;
    #endif
    #ifdef IRQ_195_PRIORITY      /* ISR 195 */
       INTC_0.PSR[195].B.PRIN=IRQ_195_PRIORITY;
    #endif
    #ifdef IRQ_196_PRIORITY      /* ISR 196 */
       INTC_0.PSR[196].B.PRIN=IRQ_196_PRIORITY;
    #endif
    #ifdef IRQ_197_PRIORITY      /* ISR 197 */
       INTC_0.PSR[197].B.PRIN=IRQ_197_PRIORITY;
    #endif
    #ifdef IRQ_198_PRIORITY      /* ISR 198 */
       INTC_0.PSR[198].B.PRIN=IRQ_198_PRIORITY;
    #endif
    #ifdef IRQ_199_PRIORITY      /* ISR 199 */
       INTC_0.PSR[199].B.PRIN=IRQ_199_PRIORITY;
    #endif
    #ifdef IRQ_200_PRIORITY      /* ISR 200 */
       INTC_0.PSR[200].B.PRIN=IRQ_200_PRIORITY;
    #endif
    #ifdef IRQ_201_PRIORITY      /* ISR 201 */
       INTC_0.PSR[201].B.PRIN=IRQ_201_PRIORITY;
    #endif
    #ifdef IRQ_202_PRIORITY      /* ISR 202 */
       INTC_0.PSR[202].B.PRIN=IRQ_202_PRIORITY;
    #endif
    #ifdef IRQ_203_PRIORITY      /* ISR 203 */
       INTC_0.PSR[203].B.PRIN=IRQ_203_PRIORITY;
    #endif
    #ifdef IRQ_204_PRIORITY      /* ISR 204 */
       INTC_0.PSR[204].B.PRIN=IRQ_204_PRIORITY;
    #endif
    #ifdef IRQ_205_PRIORITY      /* ISR 205 */
       INTC_0.PSR[205].B.PRIN=IRQ_205_PRIORITY;
    #endif
    #ifdef IRQ_206_PRIORITY      /* ISR 206 */
       INTC_0.PSR[206].B.PRIN=IRQ_206_PRIORITY;
    #endif
    #ifdef IRQ_207_PRIORITY      /* ISR 207 */
       INTC_0.PSR[207].B.PRIN=IRQ_207_PRIORITY;
    #endif
    #ifdef IRQ_208_PRIORITY      /* ISR 208 */
       INTC_0.PSR[208].B.PRIN=IRQ_208_PRIORITY;
    #endif
    #ifdef IRQ_209_PRIORITY      /* ISR 209 */
       INTC_0.PSR[209].B.PRIN=IRQ_209_PRIORITY;
    #endif
    #ifdef IRQ_210_PRIORITY      /* ISR 210 */
       INTC_0.PSR[210].B.PRIN=IRQ_210_PRIORITY;
    #endif
    #ifdef IRQ_211_PRIORITY      /* ISR 211 */
       INTC_0.PSR[211].B.PRIN=IRQ_211_PRIORITY;
    #endif
    #ifdef IRQ_212_PRIORITY      /* ISR 212 */
       INTC_0.PSR[212].B.PRIN=IRQ_212_PRIORITY;
    #endif
    #ifdef IRQ_213_PRIORITY      /* ISR 213 */
       INTC_0.PSR[213].B.PRIN=IRQ_213_PRIORITY;
    #endif
    #ifdef IRQ_214_PRIORITY      /* ISR 214 */
       INTC_0.PSR[214].B.PRIN=IRQ_214_PRIORITY;
    #endif
    #ifdef IRQ_215_PRIORITY      /* ISR 215 */
       INTC_0.PSR[215].B.PRIN=IRQ_215_PRIORITY;
    #endif
    #ifdef IRQ_216_PRIORITY      /* ISR 216 */
       INTC_0.PSR[216].B.PRIN=IRQ_216_PRIORITY;
    #endif
    #ifdef IRQ_217_PRIORITY      /* ISR 217 */
       INTC_0.PSR[217].B.PRIN=IRQ_217_PRIORITY;
    #endif
    #ifdef IRQ_218_PRIORITY      /* ISR 218 */
       INTC_0.PSR[218].B.PRIN=IRQ_218_PRIORITY;
    #endif
    #ifdef IRQ_219_PRIORITY      /* ISR 219 */
       INTC_0.PSR[219].B.PRIN=IRQ_219_PRIORITY;
    #endif
    #ifdef IRQ_220_PRIORITY      /* ISR 220 */
       INTC_0.PSR[220].B.PRIN=IRQ_220_PRIORITY;
    #endif
    #ifdef IRQ_221_PRIORITY      /* ISR 221 */
       INTC_0.PSR[221].B.PRIN=IRQ_221_PRIORITY;
    #endif
	#ifdef IRQ_223_PRIORITY      /* ISR 223 */
	   INTC_0.PSR[223].B.PRIN=IRQ_223_PRIORITY;
	#endif
	#ifdef IRQ_224_PRIORITY      /* ISR 224 */
	   INTC_0.PSR[224].B.PRIN=IRQ_224_PRIORITY;
	#endif
	#ifdef IRQ_225_PRIORITY      /* ISR 225 */
	   INTC_0.PSR[225].B.PRIN=IRQ_225_PRIORITY;
	#endif
	#ifdef IRQ_226_PRIORITY      /* ISR 226 */
	   INTC_0.PSR[226].B.PRIN=IRQ_226_PRIORITY;
	#endif
	#ifdef IRQ_227_PRIORITY      /* ISR 227 */
	   INTC_0.PSR[227].B.PRIN=IRQ_227_PRIORITY;
	#endif
	#ifdef IRQ_228_PRIORITY      /* ISR 228 */
	   INTC_0.PSR[228].B.PRIN=IRQ_228_PRIORITY;
	#endif
	#ifdef IRQ_229_PRIORITY      /* ISR 229 */
	   INTC_0.PSR[229].B.PRIN=IRQ_229_PRIORITY;
	#endif
	#ifdef IRQ_230_PRIORITY      /* ISR 230 */
	   INTC_0.PSR[230].B.PRIN=IRQ_230_PRIORITY;
	#endif
	#ifdef IRQ_231_PRIORITY      /* ISR 231 */
	   INTC_0.PSR[231].B.PRIN=IRQ_231_PRIORITY;
	#endif
	#ifdef IRQ_232_PRIORITY      /* ISR 232 */
	   INTC_0.PSR[232].B.PRIN=IRQ_232_PRIORITY;
	#endif
	#ifdef IRQ_233_PRIORITY      /* ISR 233 */
	   INTC_0.PSR[233].B.PRIN=IRQ_233_PRIORITY;
	#endif
	#ifdef IRQ_234_PRIORITY      /* ISR 234 */
	   INTC_0.PSR[234].B.PRIN=IRQ_234_PRIORITY;
	#endif
	#ifdef IRQ_235_PRIORITY      /* ISR 235 */
	   INTC_0.PSR[235].B.PRIN=IRQ_235_PRIORITY;
	#endif
	#ifdef IRQ_236_PRIORITY      /* ISR 236 */
	   INTC_0.PSR[236].B.PRIN=IRQ_236_PRIORITY;
	#endif
	#ifdef IRQ_237_PRIORITY      /* ISR 237 */
	   INTC_0.PSR[237].B.PRIN=IRQ_237_PRIORITY;
	#endif
	#ifdef IRQ_238_PRIORITY      /* ISR 238 */
	   INTC_0.PSR[238].B.PRIN=IRQ_238_PRIORITY;
	#endif
	#ifdef IRQ_239_PRIORITY      /* ISR 239 */
	   INTC_0.PSR[239].B.PRIN=IRQ_239_PRIORITY;
	#endif
	#ifdef IRQ_240_PRIORITY      /* ISR 240 */
	   INTC_0.PSR[240].B.PRIN=IRQ_240_PRIORITY;
	#endif
	#ifdef IRQ_241_PRIORITY      /* ISR 241 */
	   INTC_0.PSR[241].B.PRIN=IRQ_241_PRIORITY;
	#endif
	#ifdef IRQ_242_PRIORITY      /* ISR 242 */
	   INTC_0.PSR[242].B.PRIN=IRQ_242_PRIORITY;
	#endif
	#ifdef IRQ_243_PRIORITY      /* ISR 243 */
	   INTC_0.PSR[243].B.PRIN=IRQ_243_PRIORITY;
	#endif
	#ifdef IRQ_244_PRIORITY      /* ISR 244 */
	   INTC_0.PSR[244].B.PRIN=IRQ_244_PRIORITY;
	#endif
	#ifdef IRQ_245_PRIORITY      /* ISR 245 */
	   INTC_0.PSR[245].B.PRIN=IRQ_245_PRIORITY;
	#endif
	#ifdef IRQ_246_PRIORITY      /* ISR 246 */
	   INTC_0.PSR[246].B.PRIN=IRQ_246_PRIORITY;
	#endif
	#ifdef IRQ_247_PRIORITY      /* ISR 247 */
	   INTC_0.PSR[247].B.PRIN=IRQ_247_PRIORITY;
	#endif
	#ifdef IRQ_248_PRIORITY      /* ISR 248 */
	   INTC_0.PSR[248].B.PRIN=IRQ_248_PRIORITY;
	#endif
	#ifdef IRQ_249_PRIORITY      /* ISR 249 */
	   INTC_0.PSR[249].B.PRIN=IRQ_249_PRIORITY;
	#endif
	#ifdef IRQ_250_PRIORITY      /* ISR 250 */
	   INTC_0.PSR[250].B.PRIN=IRQ_250_PRIORITY;
	#endif
	#ifdef IRQ_251_PRIORITY      /* ISR 251 */
	   INTC_0.PSR[251].B.PRIN=IRQ_251_PRIORITY;
	#endif
	#ifdef IRQ_252_PRIORITY      /* ISR 252 */
	   INTC_0.PSR[252].B.PRIN=IRQ_252_PRIORITY;
	#endif
	#ifdef IRQ_253_PRIORITY      /* ISR 253 */
	   INTC_0.PSR[253].B.PRIN=IRQ_253_PRIORITY;
	#endif
	#ifdef IRQ_254_PRIORITY      /* ISR 254 */
	   INTC_0.PSR[254].B.PRIN=IRQ_254_PRIORITY;
	#endif
	#ifdef IRQ_255_PRIORITY      /* ISR 255 */
	   INTC_0.PSR[255].B.PRIN=IRQ_255_PRIORITY;
	#endif
	#ifdef IRQ_256_PRIORITY      /* ISR 256 */
	   INTC_0.PSR[256].B.PRIN=IRQ_256_PRIORITY;
	#endif
	#ifdef IRQ_257_PRIORITY      /* ISR 257 */
	   INTC_0.PSR[257].B.PRIN=IRQ_257_PRIORITY;
	#endif
	#ifdef IRQ_258_PRIORITY      /* ISR 258 */
	   INTC_0.PSR[258].B.PRIN=IRQ_258_PRIORITY;
	#endif
	#ifdef IRQ_259_PRIORITY      /* ISR 259 */
	   INTC_0.PSR[259].B.PRIN=IRQ_259_PRIORITY;
	#endif
	#ifdef IRQ_260_PRIORITY      /* ISR 260 */
	   INTC_0.PSR[260].B.PRIN=IRQ_260_PRIORITY;
	#endif
	#ifdef IRQ_261_PRIORITY      /* ISR 261 */
	   INTC_0.PSR[261].B.PRIN=IRQ_261_PRIORITY;
	#endif
	#ifdef IRQ_262_PRIORITY      /* ISR 262 */
	   INTC_0.PSR[262].B.PRIN=IRQ_262_PRIORITY;
	#endif
	#ifdef IRQ_263_PRIORITY      /* ISR 263 */
	   INTC_0.PSR[263].B.PRIN=IRQ_263_PRIORITY;
	#endif
	#ifdef IRQ_264_PRIORITY      /* ISR 264 */
	   INTC_0.PSR[264].B.PRIN=IRQ_264_PRIORITY;
	#endif
	#ifdef IRQ_265_PRIORITY      /* ISR 265 */
	   INTC_0.PSR[265].B.PRIN=IRQ_265_PRIORITY;
	#endif
	#ifdef IRQ_266_PRIORITY      /* ISR 266 */
	   INTC_0.PSR[266].B.PRIN=IRQ_266_PRIORITY;
	#endif
	#ifdef IRQ_267_PRIORITY      /* ISR 267 */
	   INTC_0.PSR[267].B.PRIN=IRQ_267_PRIORITY;
	#endif
	#ifdef IRQ_268_PRIORITY      /* ISR 268 */
	   INTC_0.PSR[268].B.PRIN=IRQ_268_PRIORITY;
	#endif
	#ifdef IRQ_269_PRIORITY      /* ISR 269 */
	   INTC_0.PSR[269].B.PRIN=IRQ_269_PRIORITY;
	#endif
	#ifdef IRQ_270_PRIORITY      /* ISR 270 */
	   INTC_0.PSR[270].B.PRIN=IRQ_270_PRIORITY;
	#endif
	#ifdef IRQ_271_PRIORITY      /* ISR 271 */
	   INTC_0.PSR[271].B.PRIN=IRQ_271_PRIORITY;
	#endif
	#ifdef IRQ_272_PRIORITY      /* ISR 272 */
	   INTC_0.PSR[272].B.PRIN=IRQ_272_PRIORITY;
	#endif
	#ifdef IRQ_273_PRIORITY      /* ISR 273 */
	   INTC_0.PSR[273].B.PRIN=IRQ_273_PRIORITY;
	#endif
	#ifdef IRQ_274_PRIORITY      /* ISR 274 */
	   INTC_0.PSR[274].B.PRIN=IRQ_274_PRIORITY;
	#endif
	#ifdef IRQ_275_PRIORITY      /* ISR 275 */
	   INTC_0.PSR[275].B.PRIN=IRQ_275_PRIORITY;
	#endif
	#ifdef IRQ_276_PRIORITY      /* ISR 276 */
	   INTC_0.PSR[276].B.PRIN=IRQ_276_PRIORITY;
	#endif
	#ifdef IRQ_277_PRIORITY      /* ISR 277 */
	   INTC_0.PSR[277].B.PRIN=IRQ_277_PRIORITY;
	#endif
	#ifdef IRQ_278_PRIORITY      /* ISR 278 */
	   INTC_0.PSR[278].B.PRIN=IRQ_278_PRIORITY;
	#endif
	#ifdef IRQ_279_PRIORITY      /* ISR 279 */
	   INTC_0.PSR[279].B.PRIN=IRQ_279_PRIORITY;
	#endif
	#ifdef IRQ_280_PRIORITY      /* ISR 280 */
	   INTC_0.PSR[280].B.PRIN=IRQ_280_PRIORITY;
	#endif
	#ifdef IRQ_281_PRIORITY      /* ISR 281 */
	   INTC_0.PSR[281].B.PRIN=IRQ_281_PRIORITY;
	#endif
	#ifdef IRQ_282_PRIORITY      /* ISR 282 */
	   INTC_0.PSR[282].B.PRIN=IRQ_282_PRIORITY;
	#endif
	#ifdef IRQ_283_PRIORITY      /* ISR 283 */
	   INTC_0.PSR[283].B.PRIN=IRQ_283_PRIORITY;
	#endif
	#ifdef IRQ_284_PRIORITY      /* ISR 284 */
	   INTC_0.PSR[284].B.PRIN=IRQ_284_PRIORITY;
	#endif
	#ifdef IRQ_285_PRIORITY      /* ISR 285 */
	   INTC_0.PSR[285].B.PRIN=IRQ_285_PRIORITY;
	#endif
	#ifdef IRQ_286_PRIORITY      /* ISR 286 */
	   INTC_0.PSR[286].B.PRIN=IRQ_286_PRIORITY;
	#endif
	#ifdef IRQ_287_PRIORITY      /* ISR 287 */
	   INTC_0.PSR[287].B.PRIN=IRQ_287_PRIORITY;
	#endif
	#ifdef IRQ_288_PRIORITY      /* ISR 288 */
	   INTC_0.PSR[288].B.PRIN=IRQ_288_PRIORITY;
	#endif
	#ifdef IRQ_289_PRIORITY      /* ISR 289 */
	   INTC_0.PSR[289].B.PRIN=IRQ_289_PRIORITY;
	#endif
	#ifdef IRQ_290_PRIORITY      /* ISR 290 */
	   INTC_0.PSR[290].B.PRIN=IRQ_290_PRIORITY;
	#endif
	#ifdef IRQ_291_PRIORITY      /* ISR 291 */
	   INTC_0.PSR[291].B.PRIN=IRQ_291_PRIORITY;
	#endif
	#ifdef IRQ_292_PRIORITY      /* ISR 292 */
	   INTC_0.PSR[292].B.PRIN=IRQ_292_PRIORITY;
	#endif
	#ifdef IRQ_293_PRIORITY      /* ISR 293 */
	   INTC_0.PSR[293].B.PRIN=IRQ_293_PRIORITY;
	#endif
	#ifdef IRQ_294_PRIORITY      /* ISR 294 */
	   INTC_0.PSR[294].B.PRIN=IRQ_294_PRIORITY;
	#endif
	#ifdef IRQ_295_PRIORITY      /* ISR 295 */
	   INTC_0.PSR[295].B.PRIN=IRQ_295_PRIORITY;
	#endif
	#ifdef IRQ_296_PRIORITY      /* ISR 296 */
	   INTC_0.PSR[296].B.PRIN=IRQ_296_PRIORITY;
	#endif
	#ifdef IRQ_297_PRIORITY      /* ISR 297 */
	   INTC_0.PSR[297].B.PRIN=IRQ_297_PRIORITY;
	#endif
	#ifdef IRQ_298_PRIORITY      /* ISR 298 */
	   INTC_0.PSR[298].B.PRIN=IRQ_298_PRIORITY;
	#endif
	#ifdef IRQ_299_PRIORITY      /* ISR 299 */
	   INTC_0.PSR[299].B.PRIN=IRQ_299_PRIORITY;
	#endif
	#ifdef IRQ_300_PRIORITY      /* ISR 300 */
	   INTC_0.PSR[300].B.PRIN=IRQ_300_PRIORITY;
	#endif
	#ifdef IRQ_301_PRIORITY      /* ISR 301 */
	   INTC_0.PSR[301].B.PRIN=IRQ_301_PRIORITY;
	#endif
	#ifdef IRQ_302_PRIORITY      /* ISR 302 */
	   INTC_0.PSR[302].B.PRIN=IRQ_302_PRIORITY;
	#endif
	#ifdef IRQ_303_PRIORITY      /* ISR 303 */
	   INTC_0.PSR[303].B.PRIN=IRQ_303_PRIORITY;
	#endif
	#ifdef IRQ_304_PRIORITY      /* ISR 304 */
	   INTC_0.PSR[304].B.PRIN=IRQ_304_PRIORITY;
	#endif
	#ifdef IRQ_305_PRIORITY      /* ISR 305 */
	   INTC_0.PSR[305].B.PRIN=IRQ_305_PRIORITY;
	#endif
	#ifdef IRQ_306_PRIORITY      /* ISR 306 */
	   INTC_0.PSR[306].B.PRIN=IRQ_306_PRIORITY;
	#endif
	#ifdef IRQ_307_PRIORITY      /* ISR 307 */
	   INTC_0.PSR[307].B.PRIN=IRQ_307_PRIORITY;
	#endif
	#ifdef IRQ_308_PRIORITY      /* ISR 308 */
	   INTC_0.PSR[308].B.PRIN=IRQ_308_PRIORITY;
	#endif
	#ifdef IRQ_309_PRIORITY      /* ISR 309 */
	   INTC_0.PSR[309].B.PRIN=IRQ_309_PRIORITY;
	#endif
	#ifdef IRQ_310_PRIORITY      /* ISR 310 */
	   INTC_0.PSR[310].B.PRIN=IRQ_310_PRIORITY;
	#endif
	#ifdef IRQ_311_PRIORITY      /* ISR 311 */
	   INTC_0.PSR[311].B.PRIN=IRQ_311_PRIORITY;
	#endif
	#ifdef IRQ_312_PRIORITY      /* ISR 312 */
	   INTC_0.PSR[312].B.PRIN=IRQ_312_PRIORITY;
	#endif
	#ifdef IRQ_313_PRIORITY      /* ISR 313 */
	   INTC_0.PSR[313].B.PRIN=IRQ_313_PRIORITY;
	#endif
	#ifdef IRQ_314_PRIORITY      /* ISR 314 */
	   INTC_0.PSR[314].B.PRIN=IRQ_314_PRIORITY;
	#endif
	#ifdef IRQ_315_PRIORITY      /* ISR 315 */
	   INTC_0.PSR[315].B.PRIN=IRQ_315_PRIORITY;
	#endif
	#ifdef IRQ_316_PRIORITY      /* ISR 316 */
	   INTC_0.PSR[316].B.PRIN=IRQ_316_PRIORITY;
	#endif
	#ifdef IRQ_317_PRIORITY      /* ISR 317 */
	   INTC_0.PSR[317].B.PRIN=IRQ_317_PRIORITY;
	#endif
	#ifdef IRQ_318_PRIORITY      /* ISR 318 */
	   INTC_0.PSR[318].B.PRIN=IRQ_318_PRIORITY;
	#endif
	#ifdef IRQ_319_PRIORITY      /* ISR 319 */
	   INTC_0.PSR[319].B.PRIN=IRQ_319_PRIORITY;
	#endif
	#ifdef IRQ_320_PRIORITY      /* ISR 320 */
	   INTC_0.PSR[320].B.PRIN=IRQ_320_PRIORITY;
	#endif
	#ifdef IRQ_321_PRIORITY      /* ISR 321 */
	   INTC_0.PSR[321].B.PRIN=IRQ_321_PRIORITY;
	#endif
	#ifdef IRQ_323_PRIORITY      /* ISR 323 */
	   INTC_0.PSR[323].B.PRIN=IRQ_323_PRIORITY;
	#endif
	#ifdef IRQ_324_PRIORITY      /* ISR 324 */
	   INTC_0.PSR[324].B.PRIN=IRQ_324_PRIORITY;
	#endif
	#ifdef IRQ_325_PRIORITY      /* ISR 325 */
	   INTC_0.PSR[325].B.PRIN=IRQ_325_PRIORITY;
	#endif
	#ifdef IRQ_326_PRIORITY      /* ISR 326 */
	   INTC_0.PSR[326].B.PRIN=IRQ_326_PRIORITY;
	#endif
	#ifdef IRQ_327_PRIORITY      /* ISR 327 */
	   INTC_0.PSR[327].B.PRIN=IRQ_327_PRIORITY;
	#endif
	#ifdef IRQ_328_PRIORITY      /* ISR 328 */
	   INTC_0.PSR[328].B.PRIN=IRQ_328_PRIORITY;
	#endif
	#ifdef IRQ_329_PRIORITY      /* ISR 329 */
	   INTC_0.PSR[329].B.PRIN=IRQ_329_PRIORITY;
	#endif
	#ifdef IRQ_330_PRIORITY      /* ISR 330 */
	   INTC_0.PSR[330].B.PRIN=IRQ_330_PRIORITY;
	#endif
	#ifdef IRQ_331_PRIORITY      /* ISR 331 */
	   INTC_0.PSR[331].B.PRIN=IRQ_331_PRIORITY;
	#endif
	#ifdef IRQ_332_PRIORITY      /* ISR 332 */
	   INTC_0.PSR[332].B.PRIN=IRQ_332_PRIORITY;
	#endif
	#ifdef IRQ_333_PRIORITY      /* ISR 333 */
	   INTC_0.PSR[333].B.PRIN=IRQ_333_PRIORITY;
	#endif
	#ifdef IRQ_334_PRIORITY      /* ISR 334 */
	   INTC_0.PSR[334].B.PRIN=IRQ_334_PRIORITY;
	#endif
	#ifdef IRQ_335_PRIORITY      /* ISR 335 */
	   INTC_0.PSR[335].B.PRIN=IRQ_335_PRIORITY;
	#endif
	#ifdef IRQ_336_PRIORITY      /* ISR 336 */
	   INTC_0.PSR[336].B.PRIN=IRQ_336_PRIORITY;
	#endif
	#ifdef IRQ_337_PRIORITY      /* ISR 337 */
	   INTC_0.PSR[337].B.PRIN=IRQ_337_PRIORITY;
	#endif
	#ifdef IRQ_338_PRIORITY      /* ISR 338 */
	   INTC_0.PSR[338].B.PRIN=IRQ_338_PRIORITY;
	#endif
	#ifdef IRQ_339_PRIORITY      /* ISR 339 */
	   INTC_0.PSR[339].B.PRIN=IRQ_339_PRIORITY;
	#endif
	#ifdef IRQ_340_PRIORITY      /* ISR 340 */
	   INTC_0.PSR[340].B.PRIN=IRQ_340_PRIORITY;
	#endif
	#ifdef IRQ_341_PRIORITY      /* ISR 341 */
	   INTC_0.PSR[341].B.PRIN=IRQ_341_PRIORITY;
	#endif
	#ifdef IRQ_342_PRIORITY      /* ISR 342 */
	   INTC_0.PSR[342].B.PRIN=IRQ_342_PRIORITY;
	#endif
	#ifdef IRQ_343_PRIORITY      /* ISR 343 */
	   INTC_0.PSR[343].B.PRIN=IRQ_343_PRIORITY;
	#endif
	#ifdef IRQ_344_PRIORITY      /* ISR 344 */
	   INTC_0.PSR[344].B.PRIN=IRQ_344_PRIORITY;
	#endif
	#ifdef IRQ_345_PRIORITY      /* ISR 345 */
	   INTC_0.PSR[345].B.PRIN=IRQ_345_PRIORITY;
	#endif
	#ifdef IRQ_346_PRIORITY      /* ISR 346 */
	   INTC_0.PSR[346].B.PRIN=IRQ_346_PRIORITY;
	#endif
	#ifdef IRQ_347_PRIORITY      /* ISR 347 */
	   INTC_0.PSR[347].B.PRIN=IRQ_347_PRIORITY;
	#endif
	#ifdef IRQ_348_PRIORITY      /* ISR 348 */
	   INTC_0.PSR[348].B.PRIN=IRQ_348_PRIORITY;
	#endif
	#ifdef IRQ_349_PRIORITY      /* ISR 349 */
	   INTC_0.PSR[349].B.PRIN=IRQ_349_PRIORITY;
	#endif
	#ifdef IRQ_350_PRIORITY      /* ISR 350 */
	   INTC_0.PSR[350].B.PRIN=IRQ_350_PRIORITY;
	#endif
	#ifdef IRQ_351_PRIORITY      /* ISR 351 */
	   INTC_0.PSR[351].B.PRIN=IRQ_351_PRIORITY;
	#endif
	#ifdef IRQ_352_PRIORITY      /* ISR 352 */
	   INTC_0.PSR[352].B.PRIN=IRQ_352_PRIORITY;
	#endif
	#ifdef IRQ_353_PRIORITY      /* ISR 353 */
	   INTC_0.PSR[353].B.PRIN=IRQ_353_PRIORITY;
	#endif
	#ifdef IRQ_354_PRIORITY      /* ISR 354 */
	   INTC_0.PSR[354].B.PRIN=IRQ_354_PRIORITY;
	#endif
	#ifdef IRQ_355_PRIORITY      /* ISR 355 */
	   INTC_0.PSR[355].B.PRIN=IRQ_355_PRIORITY;
	#endif
	#ifdef IRQ_356_PRIORITY      /* ISR 356 */
	   INTC_0.PSR[356].B.PRIN=IRQ_356_PRIORITY;
	#endif
	#ifdef IRQ_357_PRIORITY      /* ISR 357 */
	   INTC_0.PSR[357].B.PRIN=IRQ_357_PRIORITY;
	#endif
	#ifdef IRQ_358_PRIORITY      /* ISR 358 */
	   INTC_0.PSR[358].B.PRIN=IRQ_358_PRIORITY;
	#endif
	#ifdef IRQ_359_PRIORITY      /* ISR 359 */
	   INTC_0.PSR[359].B.PRIN=IRQ_359_PRIORITY;
	#endif
	#ifdef IRQ_360_PRIORITY      /* ISR 360 */
	   INTC_0.PSR[360].B.PRIN=IRQ_360_PRIORITY;
	#endif
	#ifdef IRQ_361_PRIORITY      /* ISR 361 */
	   INTC_0.PSR[361].B.PRIN=IRQ_361_PRIORITY;
	#endif
	#ifdef IRQ_362_PRIORITY      /* ISR 362 */
	   INTC_0.PSR[362].B.PRIN=IRQ_362_PRIORITY;
	#endif
	#ifdef IRQ_363_PRIORITY      /* ISR 363 */
	   INTC_0.PSR[363].B.PRIN=IRQ_363_PRIORITY;
	#endif
	#ifdef IRQ_364_PRIORITY      /* ISR 364 */
	   INTC_0.PSR[364].B.PRIN=IRQ_364_PRIORITY;
	#endif
	#ifdef IRQ_365_PRIORITY      /* ISR 365 */
	   INTC_0.PSR[365].B.PRIN=IRQ_365_PRIORITY;
	#endif
	#ifdef IRQ_366_PRIORITY      /* ISR 366 */
	   INTC_0.PSR[366].B.PRIN=IRQ_366_PRIORITY;
	#endif
	#ifdef IRQ_367_PRIORITY      /* ISR 367 */
	   INTC_0.PSR[367].B.PRIN=IRQ_367_PRIORITY;
	#endif
	#ifdef IRQ_368_PRIORITY      /* ISR 368 */
	   INTC_0.PSR[368].B.PRIN=IRQ_368_PRIORITY;
	#endif
	#ifdef IRQ_369_PRIORITY      /* ISR 369 */
	   INTC_0.PSR[369].B.PRIN=IRQ_369_PRIORITY;
	#endif
	#ifdef IRQ_370_PRIORITY      /* ISR 370 */
	   INTC_0.PSR[370].B.PRIN=IRQ_370_PRIORITY;
	#endif
	#ifdef IRQ_371_PRIORITY      /* ISR 371 */
	   INTC_0.PSR[371].B.PRIN=IRQ_371_PRIORITY;
	#endif
	#ifdef IRQ_372_PRIORITY      /* ISR 372 */
	   INTC_0.PSR[372].B.PRIN=IRQ_372_PRIORITY;
	#endif
	#ifdef IRQ_373_PRIORITY      /* ISR 373 */
	   INTC_0.PSR[373].B.PRIN=IRQ_373_PRIORITY;
	#endif
	#ifdef IRQ_374_PRIORITY      /* ISR 374 */
	   INTC_0.PSR[374].B.PRIN=IRQ_374_PRIORITY;
	#endif
	#ifdef IRQ_375_PRIORITY      /* ISR 375 */
	   INTC_0.PSR[375].B.PRIN=IRQ_375_PRIORITY;
	#endif
	#ifdef IRQ_376_PRIORITY      /* ISR 376 */
	   INTC_0.PSR[376].B.PRIN=IRQ_376_PRIORITY;
	#endif
	#ifdef IRQ_377_PRIORITY      /* ISR 377 */
	   INTC_0.PSR[377].B.PRIN=IRQ_377_PRIORITY;
	#endif
	#ifdef IRQ_378_PRIORITY      /* ISR 378 */
	   INTC_0.PSR[378].B.PRIN=IRQ_378_PRIORITY;
	#endif
	#ifdef IRQ_379_PRIORITY      /* ISR 379 */
	   INTC_0.PSR[379].B.PRIN=IRQ_379_PRIORITY;
	#endif
	#ifdef IRQ_380_PRIORITY      /* ISR 380 */
	   INTC_0.PSR[380].B.PRIN=IRQ_380_PRIORITY;
	#endif
	#ifdef IRQ_381_PRIORITY      /* ISR 381 */
	   INTC_0.PSR[381].B.PRIN=IRQ_381_PRIORITY;
	#endif
	#ifdef IRQ_382_PRIORITY      /* ISR 382 */
	   INTC_0.PSR[382].B.PRIN=IRQ_382_PRIORITY;
	#endif
	#ifdef IRQ_383_PRIORITY      /* ISR 383 */
	   INTC_0.PSR[383].B.PRIN=IRQ_383_PRIORITY;
	#endif
	#ifdef IRQ_384_PRIORITY      /* ISR 384 */
	   INTC_0.PSR[384].B.PRIN=IRQ_384_PRIORITY;
	#endif
	#ifdef IRQ_385_PRIORITY      /* ISR 385 */
	   INTC_0.PSR[385].B.PRIN=IRQ_385_PRIORITY;
	#endif
	#ifdef IRQ_386_PRIORITY      /* ISR 386 */
	   INTC_0.PSR[386].B.PRIN=IRQ_386_PRIORITY;
	#endif
	#ifdef IRQ_387_PRIORITY      /* ISR 387 */
	   INTC_0.PSR[387].B.PRIN=IRQ_387_PRIORITY;
	#endif
	#ifdef IRQ_388_PRIORITY      /* ISR 388 */
	   INTC_0.PSR[388].B.PRIN=IRQ_388_PRIORITY;
	#endif
	#ifdef IRQ_389_PRIORITY      /* ISR 389 */
	   INTC_0.PSR[389].B.PRIN=IRQ_389_PRIORITY;
	#endif
	#ifdef IRQ_390_PRIORITY      /* ISR 390 */
	   INTC_0.PSR[390].B.PRIN=IRQ_390_PRIORITY;
	#endif
	#ifdef IRQ_391_PRIORITY      /* ISR 391 */
	   INTC_0.PSR[391].B.PRIN=IRQ_391_PRIORITY;
	#endif
	#ifdef IRQ_392_PRIORITY      /* ISR 392 */
	   INTC_0.PSR[392].B.PRIN=IRQ_392_PRIORITY;
	#endif
	#ifdef IRQ_393_PRIORITY      /* ISR 393 */
	   INTC_0.PSR[393].B.PRIN=IRQ_393_PRIORITY;
	#endif
	#ifdef IRQ_394_PRIORITY      /* ISR 394 */
	   INTC_0.PSR[394].B.PRIN=IRQ_394_PRIORITY;
	#endif
	#ifdef IRQ_395_PRIORITY      /* ISR 395 */
	   INTC_0.PSR[395].B.PRIN=IRQ_395_PRIORITY;
	#endif
	#ifdef IRQ_396_PRIORITY      /* ISR 396 */
	   INTC_0.PSR[396].B.PRIN=IRQ_396_PRIORITY;
	#endif
	#ifdef IRQ_397_PRIORITY      /* ISR 397 */
	   INTC_0.PSR[397].B.PRIN=IRQ_397_PRIORITY;
	#endif
	#ifdef IRQ_398_PRIORITY      /* ISR 398 */
	   INTC_0.PSR[398].B.PRIN=IRQ_398_PRIORITY;
	#endif
	#ifdef IRQ_399_PRIORITY      /* ISR 399 */
	   INTC_0.PSR[399].B.PRIN=IRQ_399_PRIORITY;
	#endif
	#ifdef IRQ_400_PRIORITY      /* ISR 400 */
	   INTC_0.PSR[400].B.PRIN=IRQ_400_PRIORITY;
	#endif
	#ifdef IRQ_401_PRIORITY      /* ISR 401 */
	   INTC_0.PSR[401].B.PRIN=IRQ_401_PRIORITY;
	#endif
	#ifdef IRQ_402_PRIORITY      /* ISR 402 */
	   INTC_0.PSR[402].B.PRIN=IRQ_402_PRIORITY;
	#endif
	#ifdef IRQ_403_PRIORITY      /* ISR 403 */
	   INTC_0.PSR[403].B.PRIN=IRQ_403_PRIORITY;
	#endif
	#ifdef IRQ_404_PRIORITY      /* ISR 404 */
	   INTC_0.PSR[404].B.PRIN=IRQ_404_PRIORITY;
	#endif
	#ifdef IRQ_405_PRIORITY      /* ISR 405 */
	   INTC_0.PSR[405].B.PRIN=IRQ_405_PRIORITY;
	#endif
	#ifdef IRQ_406_PRIORITY      /* ISR 406 */
	   INTC_0.PSR[406].B.PRIN=IRQ_406_PRIORITY;
	#endif
	#ifdef IRQ_407_PRIORITY      /* ISR 407 */
	   INTC_0.PSR[407].B.PRIN=IRQ_407_PRIORITY;
	#endif
	#ifdef IRQ_408_PRIORITY      /* ISR 408 */
	   INTC_0.PSR[408].B.PRIN=IRQ_408_PRIORITY;
	#endif
	#ifdef IRQ_409_PRIORITY      /* ISR 409 */
	   INTC_0.PSR[409].B.PRIN=IRQ_409_PRIORITY;
	#endif
	#ifdef IRQ_410_PRIORITY      /* ISR 410 */
	   INTC_0.PSR[410].B.PRIN=IRQ_410_PRIORITY;
	#endif
	#ifdef IRQ_411_PRIORITY      /* ISR 411 */
	   INTC_0.PSR[411].B.PRIN=IRQ_411_PRIORITY;
	#endif
	#ifdef IRQ_412_PRIORITY      /* ISR 412 */
	   INTC_0.PSR[412].B.PRIN=IRQ_412_PRIORITY;
	#endif
	#ifdef IRQ_413_PRIORITY      /* ISR 413 */
	   INTC_0.PSR[413].B.PRIN=IRQ_413_PRIORITY;
	#endif
	#ifdef IRQ_414_PRIORITY      /* ISR 414 */
	   INTC_0.PSR[414].B.PRIN=IRQ_414_PRIORITY;
	#endif
	#ifdef IRQ_415_PRIORITY      /* ISR 415 */
	   INTC_0.PSR[415].B.PRIN=IRQ_415_PRIORITY;
	#endif
	#ifdef IRQ_416_PRIORITY      /* ISR 416 */
	   INTC_0.PSR[416].B.PRIN=IRQ_416_PRIORITY;
	#endif
	#ifdef IRQ_417_PRIORITY      /* ISR 417 */
	   INTC_0.PSR[417].B.PRIN=IRQ_417_PRIORITY;
	#endif
	#ifdef IRQ_418_PRIORITY      /* ISR 418 */
	   INTC_0.PSR[418].B.PRIN=IRQ_418_PRIORITY;
	#endif
	#ifdef IRQ_419_PRIORITY      /* ISR 419 */
	   INTC_0.PSR[419].B.PRIN=IRQ_419_PRIORITY;
	#endif
	#ifdef IRQ_420_PRIORITY      /* ISR 420 */
	   INTC_0.PSR[420].B.PRIN=IRQ_420_PRIORITY;
	#endif
	#ifdef IRQ_421_PRIORITY      /* ISR 421 */
	   INTC_0.PSR[421].B.PRIN=IRQ_421_PRIORITY;
	#endif
	#ifdef IRQ_423_PRIORITY      /* ISR 423 */
	   INTC_0.PSR[423].B.PRIN=IRQ_423_PRIORITY;
	#endif
	#ifdef IRQ_424_PRIORITY      /* ISR 424 */
	   INTC_0.PSR[424].B.PRIN=IRQ_424_PRIORITY;
	#endif
	#ifdef IRQ_425_PRIORITY      /* ISR 425 */
	   INTC_0.PSR[425].B.PRIN=IRQ_425_PRIORITY;
	#endif
	#ifdef IRQ_426_PRIORITY      /* ISR 426 */
	   INTC_0.PSR[426].B.PRIN=IRQ_426_PRIORITY;
	#endif
	#ifdef IRQ_427_PRIORITY      /* ISR 427 */
	   INTC_0.PSR[427].B.PRIN=IRQ_427_PRIORITY;
	#endif
	#ifdef IRQ_428_PRIORITY      /* ISR 428 */
	   INTC_0.PSR[428].B.PRIN=IRQ_428_PRIORITY;
	#endif
	#ifdef IRQ_429_PRIORITY      /* ISR 429 */
	   INTC_0.PSR[429].B.PRIN=IRQ_429_PRIORITY;
	#endif
	#ifdef IRQ_430_PRIORITY      /* ISR 430 */
	   INTC_0.PSR[430].B.PRIN=IRQ_430_PRIORITY;
	#endif
	#ifdef IRQ_431_PRIORITY      /* ISR 431 */
	   INTC_0.PSR[431].B.PRIN=IRQ_431_PRIORITY;
	#endif
	#ifdef IRQ_432_PRIORITY      /* ISR 432 */
	   INTC_0.PSR[432].B.PRIN=IRQ_432_PRIORITY;
	#endif
	#ifdef IRQ_433_PRIORITY      /* ISR 433 */
	   INTC_0.PSR[433].B.PRIN=IRQ_433_PRIORITY;
	#endif
	#ifdef IRQ_434_PRIORITY      /* ISR 434 */
	   INTC_0.PSR[434].B.PRIN=IRQ_434_PRIORITY;
	#endif
	#ifdef IRQ_435_PRIORITY      /* ISR 435 */
	   INTC_0.PSR[435].B.PRIN=IRQ_435_PRIORITY;
	#endif
	#ifdef IRQ_436_PRIORITY      /* ISR 436 */
	   INTC_0.PSR[436].B.PRIN=IRQ_436_PRIORITY;
	#endif
	#ifdef IRQ_437_PRIORITY      /* ISR 437 */
	   INTC_0.PSR[437].B.PRIN=IRQ_437_PRIORITY;
	#endif
	#ifdef IRQ_438_PRIORITY      /* ISR 438 */
	   INTC_0.PSR[438].B.PRIN=IRQ_438_PRIORITY;
	#endif
	#ifdef IRQ_439_PRIORITY      /* ISR 439 */
	   INTC_0.PSR[439].B.PRIN=IRQ_439_PRIORITY;
	#endif
	#ifdef IRQ_440_PRIORITY      /* ISR 440 */
	   INTC_0.PSR[440].B.PRIN=IRQ_440_PRIORITY;
	#endif
	#ifdef IRQ_441_PRIORITY      /* ISR 441 */
	   INTC_0.PSR[441].B.PRIN=IRQ_441_PRIORITY;
	#endif
	#ifdef IRQ_442_PRIORITY      /* ISR 442 */
	   INTC_0.PSR[442].B.PRIN=IRQ_442_PRIORITY;
	#endif
	#ifdef IRQ_443_PRIORITY      /* ISR 443 */
	   INTC_0.PSR[443].B.PRIN=IRQ_443_PRIORITY;
	#endif
	#ifdef IRQ_444_PRIORITY      /* ISR 444 */
	   INTC_0.PSR[444].B.PRIN=IRQ_444_PRIORITY;
	#endif
	#ifdef IRQ_445_PRIORITY      /* ISR 445 */
	   INTC_0.PSR[445].B.PRIN=IRQ_445_PRIORITY;
	#endif
	#ifdef IRQ_446_PRIORITY      /* ISR 446 */
	   INTC_0.PSR[446].B.PRIN=IRQ_446_PRIORITY;
	#endif
	#ifdef IRQ_447_PRIORITY      /* ISR 447 */
	   INTC_0.PSR[447].B.PRIN=IRQ_447_PRIORITY;
	#endif
	#ifdef IRQ_448_PRIORITY      /* ISR 448 */
	   INTC_0.PSR[448].B.PRIN=IRQ_448_PRIORITY;
	#endif
	#ifdef IRQ_449_PRIORITY      /* ISR 449 */
	   INTC_0.PSR[449].B.PRIN=IRQ_449_PRIORITY;
	#endif
	#ifdef IRQ_450_PRIORITY      /* ISR 450 */
	   INTC_0.PSR[450].B.PRIN=IRQ_450_PRIORITY;
	#endif
	#ifdef IRQ_451_PRIORITY      /* ISR 451 */
	   INTC_0.PSR[451].B.PRIN=IRQ_451_PRIORITY;
	#endif
	#ifdef IRQ_452_PRIORITY      /* ISR 452 */
	   INTC_0.PSR[452].B.PRIN=IRQ_452_PRIORITY;
	#endif
	#ifdef IRQ_453_PRIORITY      /* ISR 453 */
	   INTC_0.PSR[453].B.PRIN=IRQ_453_PRIORITY;
	#endif
	#ifdef IRQ_454_PRIORITY      /* ISR 454 */
	   INTC_0.PSR[454].B.PRIN=IRQ_454_PRIORITY;
	#endif
	#ifdef IRQ_455_PRIORITY      /* ISR 455 */
	   INTC_0.PSR[455].B.PRIN=IRQ_455_PRIORITY;
	#endif
	#ifdef IRQ_456_PRIORITY      /* ISR 456 */
	   INTC_0.PSR[456].B.PRIN=IRQ_456_PRIORITY;
	#endif
	#ifdef IRQ_457_PRIORITY      /* ISR 457 */
	   INTC_0.PSR[457].B.PRIN=IRQ_457_PRIORITY;
	#endif
	#ifdef IRQ_458_PRIORITY      /* ISR 458 */
	   INTC_0.PSR[458].B.PRIN=IRQ_458_PRIORITY;
	#endif
	#ifdef IRQ_459_PRIORITY      /* ISR 459 */
	   INTC_0.PSR[459].B.PRIN=IRQ_459_PRIORITY;
	#endif
	#ifdef IRQ_460_PRIORITY      /* ISR 460 */
	   INTC_0.PSR[460].B.PRIN=IRQ_460_PRIORITY;
	#endif
	#ifdef IRQ_461_PRIORITY      /* ISR 461 */
	   INTC_0.PSR[461].B.PRIN=IRQ_461_PRIORITY;
	#endif
	#ifdef IRQ_462_PRIORITY      /* ISR 462 */
	   INTC_0.PSR[462].B.PRIN=IRQ_462_PRIORITY;
	#endif
	#ifdef IRQ_463_PRIORITY      /* ISR 463 */
	   INTC_0.PSR[463].B.PRIN=IRQ_463_PRIORITY;
	#endif
	#ifdef IRQ_464_PRIORITY      /* ISR 464 */
	   INTC_0.PSR[464].B.PRIN=IRQ_464_PRIORITY;
	#endif
	#ifdef IRQ_465_PRIORITY      /* ISR 465 */
	   INTC_0.PSR[465].B.PRIN=IRQ_465_PRIORITY;
	#endif
	#ifdef IRQ_466_PRIORITY      /* ISR 466 */
	   INTC_0.PSR[466].B.PRIN=IRQ_466_PRIORITY;
	#endif
	#ifdef IRQ_467_PRIORITY      /* ISR 467 */
	   INTC_0.PSR[467].B.PRIN=IRQ_467_PRIORITY;
	#endif
	#ifdef IRQ_468_PRIORITY      /* ISR 468 */
	   INTC_0.PSR[468].B.PRIN=IRQ_468_PRIORITY;
	#endif
	#ifdef IRQ_469_PRIORITY      /* ISR 469 */
	   INTC_0.PSR[469].B.PRIN=IRQ_469_PRIORITY;
	#endif
	#ifdef IRQ_470_PRIORITY      /* ISR 470 */
	   INTC_0.PSR[470].B.PRIN=IRQ_470_PRIORITY;
	#endif
	#ifdef IRQ_471_PRIORITY      /* ISR 471 */
	   INTC_0.PSR[471].B.PRIN=IRQ_471_PRIORITY;
	#endif
	#ifdef IRQ_472_PRIORITY      /* ISR 472 */
	   INTC_0.PSR[472].B.PRIN=IRQ_472_PRIORITY;
	#endif
	#ifdef IRQ_473_PRIORITY      /* ISR 473 */
	   INTC_0.PSR[473].B.PRIN=IRQ_473_PRIORITY;
	#endif
	#ifdef IRQ_474_PRIORITY      /* ISR 474 */
	   INTC_0.PSR[474].B.PRIN=IRQ_474_PRIORITY;
	#endif
	#ifdef IRQ_475_PRIORITY      /* ISR 475 */
	   INTC_0.PSR[475].B.PRIN=IRQ_475_PRIORITY;
	#endif
	#ifdef IRQ_476_PRIORITY      /* ISR 476 */
	   INTC_0.PSR[476].B.PRIN=IRQ_476_PRIORITY;
	#endif
	#ifdef IRQ_477_PRIORITY      /* ISR 477 */
	   INTC_0.PSR[477].B.PRIN=IRQ_477_PRIORITY;
	#endif
	#ifdef IRQ_478_PRIORITY      /* ISR 478 */
	   INTC_0.PSR[478].B.PRIN=IRQ_478_PRIORITY;
	#endif
	#ifdef IRQ_479_PRIORITY      /* ISR 479 */
	   INTC_0.PSR[479].B.PRIN=IRQ_479_PRIORITY;
	#endif
	#ifdef IRQ_480_PRIORITY      /* ISR 480 */
	   INTC_0.PSR[480].B.PRIN=IRQ_480_PRIORITY;
	#endif
	#ifdef IRQ_481_PRIORITY      /* ISR 481 */
	   INTC_0.PSR[481].B.PRIN=IRQ_481_PRIORITY;
	#endif
	#ifdef IRQ_482_PRIORITY      /* ISR 482 */
	   INTC_0.PSR[482].B.PRIN=IRQ_482_PRIORITY;
	#endif
	#ifdef IRQ_483_PRIORITY      /* ISR 483 */
	   INTC_0.PSR[483].B.PRIN=IRQ_483_PRIORITY;
	#endif
	#ifdef IRQ_484_PRIORITY      /* ISR 484 */
	   INTC_0.PSR[484].B.PRIN=IRQ_484_PRIORITY;
	#endif
	#ifdef IRQ_485_PRIORITY      /* ISR 485 */
	   INTC_0.PSR[485].B.PRIN=IRQ_485_PRIORITY;
	#endif
	#ifdef IRQ_486_PRIORITY      /* ISR 486 */
	   INTC_0.PSR[486].B.PRIN=IRQ_486_PRIORITY;
	#endif
	#ifdef IRQ_487_PRIORITY      /* ISR 487 */
	   INTC_0.PSR[487].B.PRIN=IRQ_487_PRIORITY;
	#endif
	#ifdef IRQ_488_PRIORITY      /* ISR 488 */
	   INTC_0.PSR[488].B.PRIN=IRQ_488_PRIORITY;
	#endif
	#ifdef IRQ_489_PRIORITY      /* ISR 489 */
	   INTC_0.PSR[489].B.PRIN=IRQ_489_PRIORITY;
	#endif
	#ifdef IRQ_490_PRIORITY      /* ISR 490 */
	   INTC_0.PSR[490].B.PRIN=IRQ_490_PRIORITY;
	#endif
	#ifdef IRQ_491_PRIORITY      /* ISR 491 */
	   INTC_0.PSR[491].B.PRIN=IRQ_491_PRIORITY;
	#endif
	#ifdef IRQ_492_PRIORITY      /* ISR 492 */
	   INTC_0.PSR[492].B.PRIN=IRQ_492_PRIORITY;
	#endif
	#ifdef IRQ_493_PRIORITY      /* ISR 493 */
	   INTC_0.PSR[493].B.PRIN=IRQ_493_PRIORITY;
	#endif
	#ifdef IRQ_494_PRIORITY      /* ISR 494 */
	   INTC_0.PSR[494].B.PRIN=IRQ_494_PRIORITY;
	#endif
	#ifdef IRQ_495_PRIORITY      /* ISR 495 */
	   INTC_0.PSR[495].B.PRIN=IRQ_495_PRIORITY;
	#endif
	#ifdef IRQ_496_PRIORITY      /* ISR 496 */
	   INTC_0.PSR[496].B.PRIN=IRQ_496_PRIORITY;
	#endif
	#ifdef IRQ_497_PRIORITY      /* ISR 497 */
	   INTC_0.PSR[497].B.PRIN=IRQ_497_PRIORITY;
	#endif
	#ifdef IRQ_498_PRIORITY      /* ISR 498 */
	   INTC_0.PSR[498].B.PRIN=IRQ_498_PRIORITY;
	#endif
	#ifdef IRQ_499_PRIORITY      /* ISR 499 */
	   INTC_0.PSR[499].B.PRIN=IRQ_499_PRIORITY;
	#endif
	#ifdef IRQ_500_PRIORITY      /* ISR 500 */
	   INTC_0.PSR[500].B.PRIN=IRQ_500_PRIORITY;
	#endif
	#ifdef IRQ_501_PRIORITY      /* ISR 501 */
	   INTC_0.PSR[501].B.PRIN=IRQ_501_PRIORITY;
	#endif
	#ifdef IRQ_502_PRIORITY      /* ISR 502 */
	   INTC_0.PSR[502].B.PRIN=IRQ_502_PRIORITY;
	#endif
	#ifdef IRQ_503_PRIORITY      /* ISR 503 */
	   INTC_0.PSR[503].B.PRIN=IRQ_503_PRIORITY;
	#endif
	#ifdef IRQ_504_PRIORITY      /* ISR 504 */
	   INTC_0.PSR[504].B.PRIN=IRQ_504_PRIORITY;
	#endif
	#ifdef IRQ_505_PRIORITY      /* ISR 505 */
	   INTC_0.PSR[505].B.PRIN=IRQ_505_PRIORITY;
	#endif
	#ifdef IRQ_506_PRIORITY      /* ISR 506 */
	   INTC_0.PSR[506].B.PRIN=IRQ_506_PRIORITY;
	#endif
	#ifdef IRQ_507_PRIORITY      /* ISR 507 */
	   INTC_0.PSR[507].B.PRIN=IRQ_507_PRIORITY;
	#endif
	#ifdef IRQ_508_PRIORITY      /* ISR 508 */
	   INTC_0.PSR[508].B.PRIN=IRQ_508_PRIORITY;
	#endif
	#ifdef IRQ_509_PRIORITY      /* ISR 509 */
	   INTC_0.PSR[509].B.PRIN=IRQ_509_PRIORITY;
	#endif
	#ifdef IRQ_510_PRIORITY      /* ISR 510 */
	   INTC_0.PSR[510].B.PRIN=IRQ_510_PRIORITY;
	#endif
	#ifdef IRQ_511_PRIORITY      /* ISR 511 */
	   INTC_0.PSR[511].B.PRIN=IRQ_511_PRIORITY;
	#endif
	#ifdef IRQ_512_PRIORITY      /* ISR 512 */
	   INTC_0.PSR[512].B.PRIN=IRQ_512_PRIORITY;
	#endif
	#ifdef IRQ_513_PRIORITY      /* ISR 513 */
	   INTC_0.PSR[513].B.PRIN=IRQ_513_PRIORITY;
	#endif
	#ifdef IRQ_514_PRIORITY      /* ISR 514 */
	   INTC_0.PSR[514].B.PRIN=IRQ_514_PRIORITY;
	#endif
	#ifdef IRQ_515_PRIORITY      /* ISR 515 */
	   INTC_0.PSR[515].B.PRIN=IRQ_515_PRIORITY;
	#endif
	#ifdef IRQ_516_PRIORITY      /* ISR 516 */
	   INTC_0.PSR[516].B.PRIN=IRQ_516_PRIORITY;
	#endif
	#ifdef IRQ_517_PRIORITY      /* ISR 517 */
	   INTC_0.PSR[517].B.PRIN=IRQ_517_PRIORITY;
	#endif
	#ifdef IRQ_518_PRIORITY      /* ISR 518 */
	   INTC_0.PSR[518].B.PRIN=IRQ_518_PRIORITY;
	#endif
	#ifdef IRQ_519_PRIORITY      /* ISR 519 */
	   INTC_0.PSR[519].B.PRIN=IRQ_519_PRIORITY;
	#endif
	#ifdef IRQ_520_PRIORITY      /* ISR 520 */
	   INTC_0.PSR[520].B.PRIN=IRQ_520_PRIORITY;
	#endif
	#ifdef IRQ_521_PRIORITY      /* ISR 521 */
	   INTC_0.PSR[521].B.PRIN=IRQ_521_PRIORITY;
	#endif
	#ifdef IRQ_523_PRIORITY      /* ISR 523 */
	   INTC_0.PSR[523].B.PRIN=IRQ_523_PRIORITY;
	#endif
	#ifdef IRQ_524_PRIORITY      /* ISR 524 */
	   INTC_0.PSR[524].B.PRIN=IRQ_524_PRIORITY;
	#endif
	#ifdef IRQ_525_PRIORITY      /* ISR 525 */
	   INTC_0.PSR[525].B.PRIN=IRQ_525_PRIORITY;
	#endif
	#ifdef IRQ_526_PRIORITY      /* ISR 526 */
	   INTC_0.PSR[526].B.PRIN=IRQ_526_PRIORITY;
	#endif
	#ifdef IRQ_527_PRIORITY      /* ISR 527 */
	   INTC_0.PSR[527].B.PRIN=IRQ_527_PRIORITY;
	#endif
	#ifdef IRQ_528_PRIORITY      /* ISR 528 */
	   INTC_0.PSR[528].B.PRIN=IRQ_528_PRIORITY;
	#endif
	#ifdef IRQ_529_PRIORITY      /* ISR 529 */
	   INTC_0.PSR[529].B.PRIN=IRQ_529_PRIORITY;
	#endif
	#ifdef IRQ_530_PRIORITY      /* ISR 530 */
	   INTC_0.PSR[530].B.PRIN=IRQ_530_PRIORITY;
	#endif
	#ifdef IRQ_531_PRIORITY      /* ISR 531 */
	   INTC_0.PSR[531].B.PRIN=IRQ_531_PRIORITY;
	#endif
	#ifdef IRQ_532_PRIORITY      /* ISR 532 */
	   INTC_0.PSR[532].B.PRIN=IRQ_532_PRIORITY;
	#endif
	#ifdef IRQ_533_PRIORITY      /* ISR 533 */
	   INTC_0.PSR[533].B.PRIN=IRQ_533_PRIORITY;
	#endif
	#ifdef IRQ_534_PRIORITY      /* ISR 534 */
	   INTC_0.PSR[534].B.PRIN=IRQ_534_PRIORITY;
	#endif
	#ifdef IRQ_535_PRIORITY      /* ISR 535 */
	   INTC_0.PSR[535].B.PRIN=IRQ_535_PRIORITY;
	#endif
	#ifdef IRQ_536_PRIORITY      /* ISR 536 */
	   INTC_0.PSR[536].B.PRIN=IRQ_536_PRIORITY;
	#endif
	#ifdef IRQ_537_PRIORITY      /* ISR 537 */
	   INTC_0.PSR[537].B.PRIN=IRQ_537_PRIORITY;
	#endif
	#ifdef IRQ_538_PRIORITY      /* ISR 538 */
	   INTC_0.PSR[538].B.PRIN=IRQ_538_PRIORITY;
	#endif
	#ifdef IRQ_539_PRIORITY      /* ISR 539 */
	   INTC_0.PSR[539].B.PRIN=IRQ_539_PRIORITY;
	#endif
	#ifdef IRQ_540_PRIORITY      /* ISR 540 */
	   INTC_0.PSR[540].B.PRIN=IRQ_540_PRIORITY;
	#endif
	#ifdef IRQ_541_PRIORITY      /* ISR 541 */
	   INTC_0.PSR[541].B.PRIN=IRQ_541_PRIORITY;
	#endif
	#ifdef IRQ_542_PRIORITY      /* ISR 542 */
	   INTC_0.PSR[542].B.PRIN=IRQ_542_PRIORITY;
	#endif
	#ifdef IRQ_543_PRIORITY      /* ISR 543 */
	   INTC_0.PSR[543].B.PRIN=IRQ_543_PRIORITY;
	#endif
	#ifdef IRQ_544_PRIORITY      /* ISR 544 */
	   INTC_0.PSR[544].B.PRIN=IRQ_544_PRIORITY;
	#endif
	#ifdef IRQ_545_PRIORITY      /* ISR 545 */
	   INTC_0.PSR[545].B.PRIN=IRQ_545_PRIORITY;
	#endif
	#ifdef IRQ_546_PRIORITY      /* ISR 546 */
	   INTC_0.PSR[546].B.PRIN=IRQ_546_PRIORITY;
	#endif
	#ifdef IRQ_547_PRIORITY      /* ISR 547 */
	   INTC_0.PSR[547].B.PRIN=IRQ_547_PRIORITY;
	#endif
	#ifdef IRQ_548_PRIORITY      /* ISR 548 */
	   INTC_0.PSR[548].B.PRIN=IRQ_548_PRIORITY;
	#endif
	#ifdef IRQ_549_PRIORITY      /* ISR 549 */
	   INTC_0.PSR[549].B.PRIN=IRQ_549_PRIORITY;
	#endif
	#ifdef IRQ_550_PRIORITY      /* ISR 550 */
	   INTC_0.PSR[550].B.PRIN=IRQ_550_PRIORITY;
	#endif
	#ifdef IRQ_551_PRIORITY      /* ISR 551 */
	   INTC_0.PSR[551].B.PRIN=IRQ_551_PRIORITY;
	#endif
	#ifdef IRQ_552_PRIORITY      /* ISR 552 */
	   INTC_0.PSR[552].B.PRIN=IRQ_552_PRIORITY;
	#endif
	#ifdef IRQ_553_PRIORITY      /* ISR 553 */
	   INTC_0.PSR[553].B.PRIN=IRQ_553_PRIORITY;
	#endif
	#ifdef IRQ_554_PRIORITY      /* ISR 554 */
	   INTC_0.PSR[554].B.PRIN=IRQ_554_PRIORITY;
	#endif
	#ifdef IRQ_555_PRIORITY      /* ISR 555 */
	   INTC_0.PSR[555].B.PRIN=IRQ_555_PRIORITY;
	#endif
	#ifdef IRQ_556_PRIORITY      /* ISR 556 */
	   INTC_0.PSR[556].B.PRIN=IRQ_556_PRIORITY;
	#endif
	#ifdef IRQ_557_PRIORITY      /* ISR 557 */
	   INTC_0.PSR[557].B.PRIN=IRQ_557_PRIORITY;
	#endif
	#ifdef IRQ_558_PRIORITY      /* ISR 558 */
	   INTC_0.PSR[558].B.PRIN=IRQ_558_PRIORITY;
	#endif
	#ifdef IRQ_559_PRIORITY      /* ISR 559 */
	   INTC_0.PSR[559].B.PRIN=IRQ_559_PRIORITY;
	#endif
	#ifdef IRQ_560_PRIORITY      /* ISR 560 */
	   INTC_0.PSR[560].B.PRIN=IRQ_560_PRIORITY;
	#endif
	#ifdef IRQ_561_PRIORITY      /* ISR 561 */
	   INTC_0.PSR[561].B.PRIN=IRQ_561_PRIORITY;
	#endif
	#ifdef IRQ_562_PRIORITY      /* ISR 562 */
	   INTC_0.PSR[562].B.PRIN=IRQ_562_PRIORITY;
	#endif
	#ifdef IRQ_563_PRIORITY      /* ISR 563 */
	   INTC_0.PSR[563].B.PRIN=IRQ_563_PRIORITY;
	#endif
	#ifdef IRQ_564_PRIORITY      /* ISR 564 */
	   INTC_0.PSR[564].B.PRIN=IRQ_564_PRIORITY;
	#endif
	#ifdef IRQ_565_PRIORITY      /* ISR 565 */
	   INTC_0.PSR[565].B.PRIN=IRQ_565_PRIORITY;
	#endif
	#ifdef IRQ_566_PRIORITY      /* ISR 566 */
	   INTC_0.PSR[566].B.PRIN=IRQ_566_PRIORITY;
	#endif
	#ifdef IRQ_567_PRIORITY      /* ISR 567 */
	   INTC_0.PSR[567].B.PRIN=IRQ_567_PRIORITY;
	#endif
	#ifdef IRQ_568_PRIORITY      /* ISR 568 */
	   INTC_0.PSR[568].B.PRIN=IRQ_568_PRIORITY;
	#endif
	#ifdef IRQ_569_PRIORITY      /* ISR 569 */
	   INTC_0.PSR[569].B.PRIN=IRQ_569_PRIORITY;
	#endif
	#ifdef IRQ_570_PRIORITY      /* ISR 570 */
	   INTC_0.PSR[570].B.PRIN=IRQ_570_PRIORITY;
	#endif
	#ifdef IRQ_571_PRIORITY      /* ISR 571 */
	   INTC_0.PSR[571].B.PRIN=IRQ_571_PRIORITY;
	#endif
	#ifdef IRQ_572_PRIORITY      /* ISR 572 */
	   INTC_0.PSR[572].B.PRIN=IRQ_572_PRIORITY;
	#endif
	#ifdef IRQ_573_PRIORITY      /* ISR 573 */
	   INTC_0.PSR[573].B.PRIN=IRQ_573_PRIORITY;
	#endif
	#ifdef IRQ_574_PRIORITY      /* ISR 574 */
	   INTC_0.PSR[574].B.PRIN=IRQ_574_PRIORITY;
	#endif
	#ifdef IRQ_575_PRIORITY      /* ISR 575 */
	   INTC_0.PSR[575].B.PRIN=IRQ_575_PRIORITY;
	#endif
	#ifdef IRQ_576_PRIORITY      /* ISR 576 */
	   INTC_0.PSR[576].B.PRIN=IRQ_576_PRIORITY;
	#endif
	#ifdef IRQ_577_PRIORITY      /* ISR 577 */
	   INTC_0.PSR[577].B.PRIN=IRQ_577_PRIORITY;
	#endif
	#ifdef IRQ_578_PRIORITY      /* ISR 578 */
	   INTC_0.PSR[578].B.PRIN=IRQ_578_PRIORITY;
	#endif
	#ifdef IRQ_579_PRIORITY      /* ISR 579 */
	   INTC_0.PSR[579].B.PRIN=IRQ_579_PRIORITY;
	#endif
	#ifdef IRQ_580_PRIORITY      /* ISR 580 */
	   INTC_0.PSR[580].B.PRIN=IRQ_580_PRIORITY;
	#endif
	#ifdef IRQ_581_PRIORITY      /* ISR 581 */
	   INTC_0.PSR[581].B.PRIN=IRQ_581_PRIORITY;
	#endif
	#ifdef IRQ_582_PRIORITY      /* ISR 582 */
	   INTC_0.PSR[582].B.PRIN=IRQ_582_PRIORITY;
	#endif
	#ifdef IRQ_583_PRIORITY      /* ISR 583 */
	   INTC_0.PSR[583].B.PRIN=IRQ_583_PRIORITY;
	#endif
	#ifdef IRQ_584_PRIORITY      /* ISR 584 */
	   INTC_0.PSR[584].B.PRIN=IRQ_584_PRIORITY;
	#endif
	#ifdef IRQ_585_PRIORITY      /* ISR 585 */
	   INTC_0.PSR[585].B.PRIN=IRQ_585_PRIORITY;
	#endif
	#ifdef IRQ_586_PRIORITY      /* ISR 586 */
	   INTC_0.PSR[586].B.PRIN=IRQ_586_PRIORITY;
	#endif
	#ifdef IRQ_587_PRIORITY      /* ISR 587 */
	   INTC_0.PSR[587].B.PRIN=IRQ_587_PRIORITY;
	#endif
	#ifdef IRQ_588_PRIORITY      /* ISR 588 */
	   INTC_0.PSR[588].B.PRIN=IRQ_588_PRIORITY;
	#endif
	#ifdef IRQ_589_PRIORITY      /* ISR 589 */
	   INTC_0.PSR[589].B.PRIN=IRQ_589_PRIORITY;
	#endif
	#ifdef IRQ_590_PRIORITY      /* ISR 590 */
	   INTC_0.PSR[590].B.PRIN=IRQ_590_PRIORITY;
	#endif
	#ifdef IRQ_591_PRIORITY      /* ISR 591 */
	   INTC_0.PSR[591].B.PRIN=IRQ_591_PRIORITY;
	#endif
	#ifdef IRQ_592_PRIORITY      /* ISR 592 */
	   INTC_0.PSR[592].B.PRIN=IRQ_592_PRIORITY;
	#endif
	#ifdef IRQ_593_PRIORITY      /* ISR 593 */
	   INTC_0.PSR[593].B.PRIN=IRQ_593_PRIORITY;
	#endif
	#ifdef IRQ_594_PRIORITY      /* ISR 594 */
	   INTC_0.PSR[594].B.PRIN=IRQ_594_PRIORITY;
	#endif
	#ifdef IRQ_595_PRIORITY      /* ISR 595 */
	   INTC_0.PSR[595].B.PRIN=IRQ_595_PRIORITY;
	#endif
	#ifdef IRQ_596_PRIORITY      /* ISR 596 */
	   INTC_0.PSR[596].B.PRIN=IRQ_596_PRIORITY;
	#endif
	#ifdef IRQ_597_PRIORITY      /* ISR 597 */
	   INTC_0.PSR[597].B.PRIN=IRQ_597_PRIORITY;
	#endif
	#ifdef IRQ_598_PRIORITY      /* ISR 598 */
	   INTC_0.PSR[598].B.PRIN=IRQ_598_PRIORITY;
	#endif
	#ifdef IRQ_599_PRIORITY      /* ISR 599 */
	   INTC_0.PSR[599].B.PRIN=IRQ_599_PRIORITY;
	#endif
	#ifdef IRQ_600_PRIORITY      /* ISR 600 */
	   INTC_0.PSR[600].B.PRIN=IRQ_600_PRIORITY;
	#endif
	#ifdef IRQ_601_PRIORITY      /* ISR 601 */
	   INTC_0.PSR[601].B.PRIN=IRQ_601_PRIORITY;
	#endif
	#ifdef IRQ_602_PRIORITY      /* ISR 602 */
	   INTC_0.PSR[602].B.PRIN=IRQ_602_PRIORITY;
	#endif
	#ifdef IRQ_603_PRIORITY      /* ISR 603 */
	   INTC_0.PSR[603].B.PRIN=IRQ_603_PRIORITY;
	#endif
	#ifdef IRQ_604_PRIORITY      /* ISR 604 */
	   INTC_0.PSR[604].B.PRIN=IRQ_604_PRIORITY;
	#endif
	#ifdef IRQ_605_PRIORITY      /* ISR 605 */
	   INTC_0.PSR[605].B.PRIN=IRQ_605_PRIORITY;
	#endif
	#ifdef IRQ_606_PRIORITY      /* ISR 606 */
	   INTC_0.PSR[606].B.PRIN=IRQ_606_PRIORITY;
	#endif
	#ifdef IRQ_607_PRIORITY      /* ISR 607 */
	   INTC_0.PSR[607].B.PRIN=IRQ_607_PRIORITY;
	#endif
	#ifdef IRQ_608_PRIORITY      /* ISR 608 */
	   INTC_0.PSR[608].B.PRIN=IRQ_608_PRIORITY;
	#endif
	#ifdef IRQ_609_PRIORITY      /* ISR 609 */
	   INTC_0.PSR[609].B.PRIN=IRQ_609_PRIORITY;
	#endif
	#ifdef IRQ_610_PRIORITY      /* ISR 610 */
	   INTC_0.PSR[610].B.PRIN=IRQ_610_PRIORITY;
	#endif
	#ifdef IRQ_611_PRIORITY      /* ISR 611 */
	   INTC_0.PSR[611].B.PRIN=IRQ_611_PRIORITY;
	#endif
	#ifdef IRQ_612_PRIORITY      /* ISR 612 */
	   INTC_0.PSR[612].B.PRIN=IRQ_612_PRIORITY;
	#endif
	#ifdef IRQ_613_PRIORITY      /* ISR 613 */
	   INTC_0.PSR[613].B.PRIN=IRQ_613_PRIORITY;
	#endif
	#ifdef IRQ_614_PRIORITY      /* ISR 614 */
	   INTC_0.PSR[614].B.PRIN=IRQ_614_PRIORITY;
	#endif
	#ifdef IRQ_615_PRIORITY      /* ISR 615 */
	   INTC_0.PSR[615].B.PRIN=IRQ_615_PRIORITY;
	#endif
	#ifdef IRQ_616_PRIORITY      /* ISR 616 */
	   INTC_0.PSR[616].B.PRIN=IRQ_616_PRIORITY;
	#endif
	#ifdef IRQ_617_PRIORITY      /* ISR 617 */
	   INTC_0.PSR[617].B.PRIN=IRQ_617_PRIORITY;
	#endif
	#ifdef IRQ_618_PRIORITY      /* ISR 618 */
	   INTC_0.PSR[618].B.PRIN=IRQ_618_PRIORITY;
	#endif
	#ifdef IRQ_619_PRIORITY      /* ISR 619 */
	   INTC_0.PSR[619].B.PRIN=IRQ_619_PRIORITY;
	#endif
	#ifdef IRQ_620_PRIORITY      /* ISR 620 */
	   INTC_0.PSR[620].B.PRIN=IRQ_620_PRIORITY;
	#endif
	#ifdef IRQ_621_PRIORITY      /* ISR 621 */
	   INTC_0.PSR[621].B.PRIN=IRQ_621_PRIORITY;
	#endif
	#ifdef IRQ_623_PRIORITY      /* ISR 623 */
	   INTC_0.PSR[623].B.PRIN=IRQ_623_PRIORITY;
	#endif
	#ifdef IRQ_624_PRIORITY      /* ISR 624 */
	   INTC_0.PSR[624].B.PRIN=IRQ_624_PRIORITY;
	#endif
	#ifdef IRQ_625_PRIORITY      /* ISR 625 */
	   INTC_0.PSR[625].B.PRIN=IRQ_625_PRIORITY;
	#endif
	#ifdef IRQ_626_PRIORITY      /* ISR 626 */
	   INTC_0.PSR[626].B.PRIN=IRQ_626_PRIORITY;
	#endif
	#ifdef IRQ_627_PRIORITY      /* ISR 627 */
	   INTC_0.PSR[627].B.PRIN=IRQ_627_PRIORITY;
	#endif
	#ifdef IRQ_628_PRIORITY      /* ISR 628 */
	   INTC_0.PSR[628].B.PRIN=IRQ_628_PRIORITY;
	#endif
	#ifdef IRQ_629_PRIORITY      /* ISR 629 */
	   INTC_0.PSR[629].B.PRIN=IRQ_629_PRIORITY;
	#endif
	#ifdef IRQ_630_PRIORITY      /* ISR 630 */
	   INTC_0.PSR[630].B.PRIN=IRQ_630_PRIORITY;
	#endif
	#ifdef IRQ_631_PRIORITY      /* ISR 631 */
	   INTC_0.PSR[631].B.PRIN=IRQ_631_PRIORITY;
	#endif
	#ifdef IRQ_632_PRIORITY      /* ISR 632 */
	   INTC_0.PSR[632].B.PRIN=IRQ_632_PRIORITY;
	#endif
	#ifdef IRQ_633_PRIORITY      /* ISR 633 */
	   INTC_0.PSR[633].B.PRIN=IRQ_633_PRIORITY;
	#endif
	#ifdef IRQ_634_PRIORITY      /* ISR 634 */
	   INTC_0.PSR[634].B.PRIN=IRQ_634_PRIORITY;
	#endif
	#ifdef IRQ_635_PRIORITY      /* ISR 635 */
	   INTC_0.PSR[635].B.PRIN=IRQ_635_PRIORITY;
	#endif
	#ifdef IRQ_636_PRIORITY      /* ISR 636 */
	   INTC_0.PSR[636].B.PRIN=IRQ_636_PRIORITY;
	#endif
	#ifdef IRQ_637_PRIORITY      /* ISR 637 */
	   INTC_0.PSR[637].B.PRIN=IRQ_637_PRIORITY;
	#endif
	#ifdef IRQ_638_PRIORITY      /* ISR 638 */
	   INTC_0.PSR[638].B.PRIN=IRQ_638_PRIORITY;
	#endif
	#ifdef IRQ_639_PRIORITY      /* ISR 639 */
	   INTC_0.PSR[639].B.PRIN=IRQ_639_PRIORITY;
	#endif
	#ifdef IRQ_640_PRIORITY      /* ISR 640 */
	   INTC_0.PSR[640].B.PRIN=IRQ_640_PRIORITY;
	#endif
	#ifdef IRQ_641_PRIORITY      /* ISR 641 */
	   INTC_0.PSR[641].B.PRIN=IRQ_641_PRIORITY;
	#endif
	#ifdef IRQ_642_PRIORITY      /* ISR 642 */
	   INTC_0.PSR[642].B.PRIN=IRQ_642_PRIORITY;
	#endif
	#ifdef IRQ_643_PRIORITY      /* ISR 643 */
	   INTC_0.PSR[643].B.PRIN=IRQ_643_PRIORITY;
	#endif
	#ifdef IRQ_644_PRIORITY      /* ISR 644 */
	   INTC_0.PSR[644].B.PRIN=IRQ_644_PRIORITY;
	#endif
	#ifdef IRQ_645_PRIORITY      /* ISR 645 */
	   INTC_0.PSR[645].B.PRIN=IRQ_645_PRIORITY;
	#endif
	#ifdef IRQ_646_PRIORITY      /* ISR 646 */
	   INTC_0.PSR[646].B.PRIN=IRQ_646_PRIORITY;
	#endif
	#ifdef IRQ_647_PRIORITY      /* ISR 647 */
	   INTC_0.PSR[647].B.PRIN=IRQ_647_PRIORITY;
	#endif
	#ifdef IRQ_648_PRIORITY      /* ISR 648 */
	   INTC_0.PSR[648].B.PRIN=IRQ_648_PRIORITY;
	#endif
	#ifdef IRQ_649_PRIORITY      /* ISR 649 */
	   INTC_0.PSR[649].B.PRIN=IRQ_649_PRIORITY;
	#endif
	#ifdef IRQ_650_PRIORITY      /* ISR 650 */
	   INTC_0.PSR[650].B.PRIN=IRQ_650_PRIORITY;
	#endif
	#ifdef IRQ_651_PRIORITY      /* ISR 651 */
	   INTC_0.PSR[651].B.PRIN=IRQ_651_PRIORITY;
	#endif
	#ifdef IRQ_652_PRIORITY      /* ISR 652 */
	   INTC_0.PSR[652].B.PRIN=IRQ_652_PRIORITY;
	#endif
	#ifdef IRQ_653_PRIORITY      /* ISR 653 */
	   INTC_0.PSR[653].B.PRIN=IRQ_653_PRIORITY;
	#endif
	#ifdef IRQ_654_PRIORITY      /* ISR 654 */
	   INTC_0.PSR[654].B.PRIN=IRQ_654_PRIORITY;
	#endif
	#ifdef IRQ_655_PRIORITY      /* ISR 655 */
	   INTC_0.PSR[655].B.PRIN=IRQ_655_PRIORITY;
	#endif
	#ifdef IRQ_656_PRIORITY      /* ISR 656 */
	   INTC_0.PSR[656].B.PRIN=IRQ_656_PRIORITY;
	#endif
	#ifdef IRQ_657_PRIORITY      /* ISR 657 */
	   INTC_0.PSR[657].B.PRIN=IRQ_657_PRIORITY;
	#endif
	#ifdef IRQ_658_PRIORITY      /* ISR 658 */
	   INTC_0.PSR[658].B.PRIN=IRQ_658_PRIORITY;
	#endif
	#ifdef IRQ_659_PRIORITY      /* ISR 659 */
	   INTC_0.PSR[659].B.PRIN=IRQ_659_PRIORITY;
	#endif
	#ifdef IRQ_660_PRIORITY      /* ISR 660 */
	   INTC_0.PSR[660].B.PRIN=IRQ_660_PRIORITY;
	#endif
	#ifdef IRQ_661_PRIORITY      /* ISR 661 */
	   INTC_0.PSR[661].B.PRIN=IRQ_661_PRIORITY;
	#endif
	#ifdef IRQ_662_PRIORITY      /* ISR 662 */
	   INTC_0.PSR[662].B.PRIN=IRQ_662_PRIORITY;
	#endif
	#ifdef IRQ_663_PRIORITY      /* ISR 663 */
	   INTC_0.PSR[663].B.PRIN=IRQ_663_PRIORITY;
	#endif
	#ifdef IRQ_664_PRIORITY      /* ISR 664 */
	   INTC_0.PSR[664].B.PRIN=IRQ_664_PRIORITY;
	#endif
	#ifdef IRQ_665_PRIORITY      /* ISR 665 */
	   INTC_0.PSR[665].B.PRIN=IRQ_665_PRIORITY;
	#endif
	#ifdef IRQ_666_PRIORITY      /* ISR 666 */
	   INTC_0.PSR[666].B.PRIN=IRQ_666_PRIORITY;
	#endif
	#ifdef IRQ_667_PRIORITY      /* ISR 667 */
	   INTC_0.PSR[667].B.PRIN=IRQ_667_PRIORITY;
	#endif
	#ifdef IRQ_668_PRIORITY      /* ISR 668 */
	   INTC_0.PSR[668].B.PRIN=IRQ_668_PRIORITY;
	#endif
	#ifdef IRQ_669_PRIORITY      /* ISR 669 */
	   INTC_0.PSR[669].B.PRIN=IRQ_669_PRIORITY;
	#endif
	#ifdef IRQ_670_PRIORITY      /* ISR 670 */
	   INTC_0.PSR[670].B.PRIN=IRQ_670_PRIORITY;
	#endif
	#ifdef IRQ_671_PRIORITY      /* ISR 671 */
	   INTC_0.PSR[671].B.PRIN=IRQ_671_PRIORITY;
	#endif
	#ifdef IRQ_672_PRIORITY      /* ISR 672 */
	   INTC_0.PSR[672].B.PRIN=IRQ_672_PRIORITY;
	#endif
	#ifdef IRQ_673_PRIORITY      /* ISR 673 */
	   INTC_0.PSR[673].B.PRIN=IRQ_673_PRIORITY;
	#endif
	#ifdef IRQ_674_PRIORITY      /* ISR 674 */
	   INTC_0.PSR[674].B.PRIN=IRQ_674_PRIORITY;
	#endif
	#ifdef IRQ_675_PRIORITY      /* ISR 675 */
	   INTC_0.PSR[675].B.PRIN=IRQ_675_PRIORITY;
	#endif
	#ifdef IRQ_676_PRIORITY      /* ISR 676 */
	   INTC_0.PSR[676].B.PRIN=IRQ_676_PRIORITY;
	#endif
	#ifdef IRQ_677_PRIORITY      /* ISR 677 */
	   INTC_0.PSR[677].B.PRIN=IRQ_677_PRIORITY;
	#endif
	#ifdef IRQ_678_PRIORITY      /* ISR 678 */
	   INTC_0.PSR[678].B.PRIN=IRQ_678_PRIORITY;
	#endif
	#ifdef IRQ_679_PRIORITY      /* ISR 679 */
	   INTC_0.PSR[679].B.PRIN=IRQ_679_PRIORITY;
	#endif
	#ifdef IRQ_680_PRIORITY      /* ISR 680 */
	   INTC_0.PSR[680].B.PRIN=IRQ_680_PRIORITY;
	#endif
	#ifdef IRQ_681_PRIORITY      /* ISR 681 */
	   INTC_0.PSR[681].B.PRIN=IRQ_681_PRIORITY;
	#endif
	#ifdef IRQ_682_PRIORITY      /* ISR 682 */
	   INTC_0.PSR[682].B.PRIN=IRQ_682_PRIORITY;
	#endif
	#ifdef IRQ_683_PRIORITY      /* ISR 683 */
	   INTC_0.PSR[683].B.PRIN=IRQ_683_PRIORITY;
	#endif
	#ifdef IRQ_684_PRIORITY      /* ISR 684 */
	   INTC_0.PSR[684].B.PRIN=IRQ_684_PRIORITY;
	#endif
	#ifdef IRQ_685_PRIORITY      /* ISR 685 */
	   INTC_0.PSR[685].B.PRIN=IRQ_685_PRIORITY;
	#endif
	#ifdef IRQ_686_PRIORITY      /* ISR 686 */
	   INTC_0.PSR[686].B.PRIN=IRQ_686_PRIORITY;
	#endif
	#ifdef IRQ_687_PRIORITY      /* ISR 687 */
	   INTC_0.PSR[687].B.PRIN=IRQ_687_PRIORITY;
	#endif
	#ifdef IRQ_688_PRIORITY      /* ISR 688 */
	   INTC_0.PSR[688].B.PRIN=IRQ_688_PRIORITY;
	#endif
	#ifdef IRQ_689_PRIORITY      /* ISR 689 */
	   INTC_0.PSR[689].B.PRIN=IRQ_689_PRIORITY;
	#endif
	#ifdef IRQ_690_PRIORITY      /* ISR 690 */
	   INTC_0.PSR[690].B.PRIN=IRQ_690_PRIORITY;
	#endif
	#ifdef IRQ_691_PRIORITY      /* ISR 691 */
	   INTC_0.PSR[691].B.PRIN=IRQ_691_PRIORITY;
	#endif
	#ifdef IRQ_692_PRIORITY      /* ISR 692 */
	   INTC_0.PSR[692].B.PRIN=IRQ_692_PRIORITY;
	#endif
	#ifdef IRQ_693_PRIORITY      /* ISR 693 */
	   INTC_0.PSR[693].B.PRIN=IRQ_693_PRIORITY;
	#endif
	#ifdef IRQ_694_PRIORITY      /* ISR 694 */
	   INTC_0.PSR[694].B.PRIN=IRQ_694_PRIORITY;
	#endif
	#ifdef IRQ_695_PRIORITY      /* ISR 695 */
	   INTC_0.PSR[695].B.PRIN=IRQ_695_PRIORITY;
	#endif
	#ifdef IRQ_696_PRIORITY      /* ISR 696 */
	   INTC_0.PSR[696].B.PRIN=IRQ_696_PRIORITY;
	#endif
	#ifdef IRQ_697_PRIORITY      /* ISR 697 */
	   INTC_0.PSR[697].B.PRIN=IRQ_697_PRIORITY;
	#endif
	#ifdef IRQ_698_PRIORITY      /* ISR 698 */
	   INTC_0.PSR[698].B.PRIN=IRQ_698_PRIORITY;
	#endif
	#ifdef IRQ_699_PRIORITY      /* ISR 699 */
	   INTC_0.PSR[699].B.PRIN=IRQ_699_PRIORITY;
	#endif
	#ifdef IRQ_700_PRIORITY      /* ISR 700 */
	   INTC_0.PSR[700].B.PRIN=IRQ_700_PRIORITY;
	#endif
	#ifdef IRQ_701_PRIORITY      /* ISR 701 */
	   INTC_0.PSR[701].B.PRIN=IRQ_701_PRIORITY;
	#endif
	#ifdef IRQ_702_PRIORITY      /* ISR 702 */
	   INTC_0.PSR[702].B.PRIN=IRQ_702_PRIORITY;
	#endif
	#ifdef IRQ_703_PRIORITY      /* ISR 703 */
	   INTC_0.PSR[703].B.PRIN=IRQ_703_PRIORITY;
	#endif
	#ifdef IRQ_704_PRIORITY      /* ISR 704 */
	   INTC_0.PSR[704].B.PRIN=IRQ_704_PRIORITY;
	#endif
	#ifdef IRQ_705_PRIORITY      /* ISR 705 */
	   INTC_0.PSR[705].B.PRIN=IRQ_705_PRIORITY;
	#endif
	#ifdef IRQ_706_PRIORITY      /* ISR 706 */
	   INTC_0.PSR[706].B.PRIN=IRQ_706_PRIORITY;
	#endif
	#ifdef IRQ_707_PRIORITY      /* ISR 707 */
	   INTC_0.PSR[707].B.PRIN=IRQ_707_PRIORITY;
	#endif
	#ifdef IRQ_708_PRIORITY      /* ISR 708 */
	   INTC_0.PSR[708].B.PRIN=IRQ_708_PRIORITY;
	#endif
	#ifdef IRQ_709_PRIORITY      /* ISR 709 */
	   INTC_0.PSR[709].B.PRIN=IRQ_709_PRIORITY;
	#endif
	#ifdef IRQ_710_PRIORITY      /* ISR 710 */
	   INTC_0.PSR[710].B.PRIN=IRQ_710_PRIORITY;
	#endif
	#ifdef IRQ_711_PRIORITY      /* ISR 711 */
	   INTC_0.PSR[711].B.PRIN=IRQ_711_PRIORITY;
	#endif
	#ifdef IRQ_712_PRIORITY      /* ISR 712 */
	   INTC_0.PSR[712].B.PRIN=IRQ_712_PRIORITY;
	#endif
	#ifdef IRQ_713_PRIORITY      /* ISR 713 */
	   INTC_0.PSR[713].B.PRIN=IRQ_713_PRIORITY;
	#endif
	#ifdef IRQ_714_PRIORITY      /* ISR 714 */
	   INTC_0.PSR[714].B.PRIN=IRQ_714_PRIORITY;
	#endif
	#ifdef IRQ_715_PRIORITY      /* ISR 715 */
	   INTC_0.PSR[715].B.PRIN=IRQ_715_PRIORITY;
	#endif
	#ifdef IRQ_716_PRIORITY      /* ISR 716 */
	   INTC_0.PSR[716].B.PRIN=IRQ_716_PRIORITY;
	#endif
	#ifdef IRQ_717_PRIORITY      /* ISR 717 */
	   INTC_0.PSR[717].B.PRIN=IRQ_717_PRIORITY;
	#endif
	#ifdef IRQ_718_PRIORITY      /* ISR 718 */
	   INTC_0.PSR[718].B.PRIN=IRQ_718_PRIORITY;
	#endif
	#ifdef IRQ_719_PRIORITY      /* ISR 719 */
	   INTC_0.PSR[719].B.PRIN=IRQ_719_PRIORITY;
	#endif
	#ifdef IRQ_720_PRIORITY      /* ISR 720 */
	   INTC_0.PSR[720].B.PRIN=IRQ_720_PRIORITY;
	#endif
	#ifdef IRQ_721_PRIORITY      /* ISR 721 */
	   INTC_0.PSR[721].B.PRIN=IRQ_721_PRIORITY;
	#endif
	#ifdef IRQ_723_PRIORITY      /* ISR 723 */
	   INTC_0.PSR[723].B.PRIN=IRQ_723_PRIORITY;
	#endif
	#ifdef IRQ_724_PRIORITY      /* ISR 724 */
	   INTC_0.PSR[724].B.PRIN=IRQ_724_PRIORITY;
	#endif
	#ifdef IRQ_725_PRIORITY      /* ISR 725 */
	   INTC_0.PSR[725].B.PRIN=IRQ_725_PRIORITY;
	#endif
	#ifdef IRQ_726_PRIORITY      /* ISR 726 */
	   INTC_0.PSR[726].B.PRIN=IRQ_726_PRIORITY;
	#endif
	#ifdef IRQ_727_PRIORITY      /* ISR 727 */
	   INTC_0.PSR[727].B.PRIN=IRQ_727_PRIORITY;
	#endif
	#ifdef IRQ_728_PRIORITY      /* ISR 728 */
	   INTC_0.PSR[728].B.PRIN=IRQ_728_PRIORITY;
	#endif
	#ifdef IRQ_729_PRIORITY      /* ISR 729 */
	   INTC_0.PSR[729].B.PRIN=IRQ_729_PRIORITY;
	#endif
	#ifdef IRQ_730_PRIORITY      /* ISR 730 */
	   INTC_0.PSR[730].B.PRIN=IRQ_730_PRIORITY;
	#endif
	#ifdef IRQ_731_PRIORITY      /* ISR 731 */
	   INTC_0.PSR[731].B.PRIN=IRQ_731_PRIORITY;
	#endif
	#ifdef IRQ_732_PRIORITY      /* ISR 732 */
	   INTC_0.PSR[732].B.PRIN=IRQ_732_PRIORITY;
	#endif
	#ifdef IRQ_733_PRIORITY      /* ISR 733 */
	   INTC_0.PSR[733].B.PRIN=IRQ_733_PRIORITY;
	#endif
	#ifdef IRQ_734_PRIORITY      /* ISR 734 */
	   INTC_0.PSR[734].B.PRIN=IRQ_734_PRIORITY;
	#endif
	#ifdef IRQ_735_PRIORITY      /* ISR 735 */
	   INTC_0.PSR[735].B.PRIN=IRQ_735_PRIORITY;
	#endif
	#ifdef IRQ_736_PRIORITY      /* ISR 736 */
	   INTC_0.PSR[736].B.PRIN=IRQ_736_PRIORITY;
	#endif
	#ifdef IRQ_737_PRIORITY      /* ISR 737 */
	   INTC_0.PSR[737].B.PRIN=IRQ_737_PRIORITY;
	#endif
	#ifdef IRQ_738_PRIORITY      /* ISR 738 */
	   INTC_0.PSR[738].B.PRIN=IRQ_738_PRIORITY;
	#endif
	#ifdef IRQ_739_PRIORITY      /* ISR 739 */
	   INTC_0.PSR[739].B.PRIN=IRQ_739_PRIORITY;
	#endif
	#ifdef IRQ_740_PRIORITY      /* ISR 740 */
	   INTC_0.PSR[740].B.PRIN=IRQ_740_PRIORITY;
	#endif
	#ifdef IRQ_741_PRIORITY      /* ISR 741 */
	   INTC_0.PSR[741].B.PRIN=IRQ_741_PRIORITY;
	#endif
	#ifdef IRQ_742_PRIORITY      /* ISR 742 */
	   INTC_0.PSR[742].B.PRIN=IRQ_742_PRIORITY;
	#endif
	#ifdef IRQ_743_PRIORITY      /* ISR 743 */
	   INTC_0.PSR[743].B.PRIN=IRQ_743_PRIORITY;
	#endif
	#ifdef IRQ_744_PRIORITY      /* ISR 744 */
	   INTC_0.PSR[744].B.PRIN=IRQ_744_PRIORITY;
	#endif
	#ifdef IRQ_745_PRIORITY      /* ISR 745 */
	   INTC_0.PSR[745].B.PRIN=IRQ_745_PRIORITY;
	#endif
	#ifdef IRQ_746_PRIORITY      /* ISR 746 */
	   INTC_0.PSR[746].B.PRIN=IRQ_746_PRIORITY;
	#endif
	#ifdef IRQ_747_PRIORITY      /* ISR 747 */
	   INTC_0.PSR[747].B.PRIN=IRQ_747_PRIORITY;
	#endif
	#ifdef IRQ_748_PRIORITY      /* ISR 748 */
	   INTC_0.PSR[748].B.PRIN=IRQ_748_PRIORITY;
	#endif
	#ifdef IRQ_749_PRIORITY      /* ISR 749 */
	   INTC_0.PSR[749].B.PRIN=IRQ_749_PRIORITY;
	#endif
	#ifdef IRQ_750_PRIORITY      /* ISR 750 */
	   INTC_0.PSR[750].B.PRIN=IRQ_750_PRIORITY;
	#endif
	#ifdef IRQ_751_PRIORITY      /* ISR 751 */
	   INTC_0.PSR[751].B.PRIN=IRQ_751_PRIORITY;
	#endif
	#ifdef IRQ_752_PRIORITY      /* ISR 752 */
	   INTC_0.PSR[752].B.PRIN=IRQ_752_PRIORITY;
	#endif
	#ifdef IRQ_753_PRIORITY      /* ISR 753 */
	   INTC_0.PSR[753].B.PRIN=IRQ_753_PRIORITY;
	#endif
	#ifdef IRQ_754_PRIORITY      /* ISR 754 */
	   INTC_0.PSR[754].B.PRIN=IRQ_754_PRIORITY;
	#endif
	#ifdef IRQ_755_PRIORITY      /* ISR 755 */
	   INTC_0.PSR[755].B.PRIN=IRQ_755_PRIORITY;
	#endif
	#ifdef IRQ_756_PRIORITY      /* ISR 756 */
	   INTC_0.PSR[756].B.PRIN=IRQ_756_PRIORITY;
	#endif
	#ifdef IRQ_757_PRIORITY      /* ISR 757 */
	   INTC_0.PSR[757].B.PRIN=IRQ_757_PRIORITY;
	#endif
	#ifdef IRQ_758_PRIORITY      /* ISR 758 */
	   INTC_0.PSR[758].B.PRIN=IRQ_758_PRIORITY;
	#endif
	#ifdef IRQ_759_PRIORITY      /* ISR 759 */
	   INTC_0.PSR[759].B.PRIN=IRQ_759_PRIORITY;
	#endif
	#ifdef IRQ_760_PRIORITY      /* ISR 760 */
	   INTC_0.PSR[760].B.PRIN=IRQ_760_PRIORITY;
	#endif
	#ifdef IRQ_761_PRIORITY      /* ISR 761 */
	   INTC_0.PSR[761].B.PRIN=IRQ_761_PRIORITY;
	#endif
	#ifdef IRQ_762_PRIORITY      /* ISR 762 */
	   INTC_0.PSR[762].B.PRIN=IRQ_762_PRIORITY;
	#endif
	#ifdef IRQ_763_PRIORITY      /* ISR 763 */
	   INTC_0.PSR[763].B.PRIN=IRQ_763_PRIORITY;
	#endif
	#ifdef IRQ_764_PRIORITY      /* ISR 764 */
	   INTC_0.PSR[764].B.PRIN=IRQ_764_PRIORITY;
	#endif
	#ifdef IRQ_765_PRIORITY      /* ISR 765 */
	   INTC_0.PSR[765].B.PRIN=IRQ_765_PRIORITY;
	#endif
	#ifdef IRQ_766_PRIORITY      /* ISR 766 */
	   INTC_0.PSR[766].B.PRIN=IRQ_766_PRIORITY;
	#endif
	#ifdef IRQ_767_PRIORITY      /* ISR 767 */
	   INTC_0.PSR[767].B.PRIN=IRQ_767_PRIORITY;
	#endif
	#ifdef IRQ_768_PRIORITY      /* ISR 768 */
	   INTC_0.PSR[768].B.PRIN=IRQ_768_PRIORITY;
	#endif
	#ifdef IRQ_769_PRIORITY      /* ISR 769 */
	   INTC_0.PSR[769].B.PRIN=IRQ_769_PRIORITY;
	#endif
	#ifdef IRQ_770_PRIORITY      /* ISR 770 */
	   INTC_0.PSR[770].B.PRIN=IRQ_770_PRIORITY;
	#endif
	#ifdef IRQ_771_PRIORITY      /* ISR 771 */
	   INTC_0.PSR[771].B.PRIN=IRQ_771_PRIORITY;
	#endif
	#ifdef IRQ_772_PRIORITY      /* ISR 772 */
	   INTC_0.PSR[772].B.PRIN=IRQ_772_PRIORITY;
	#endif
	#ifdef IRQ_773_PRIORITY      /* ISR 773 */
	   INTC_0.PSR[773].B.PRIN=IRQ_773_PRIORITY;
	#endif
	#ifdef IRQ_774_PRIORITY      /* ISR 774 */
	   INTC_0.PSR[774].B.PRIN=IRQ_774_PRIORITY;
	#endif
	#ifdef IRQ_775_PRIORITY      /* ISR 775 */
	   INTC_0.PSR[775].B.PRIN=IRQ_775_PRIORITY;
	#endif
	#ifdef IRQ_776_PRIORITY      /* ISR 776 */
	   INTC_0.PSR[776].B.PRIN=IRQ_776_PRIORITY;
	#endif
	#ifdef IRQ_777_PRIORITY      /* ISR 777 */
	   INTC_0.PSR[777].B.PRIN=IRQ_777_PRIORITY;
	#endif
	#ifdef IRQ_778_PRIORITY      /* ISR 778 */
	   INTC_0.PSR[778].B.PRIN=IRQ_778_PRIORITY;
	#endif
	#ifdef IRQ_779_PRIORITY      /* ISR 779 */
	   INTC_0.PSR[779].B.PRIN=IRQ_779_PRIORITY;
	#endif
	#ifdef IRQ_780_PRIORITY      /* ISR 780 */
	   INTC_0.PSR[780].B.PRIN=IRQ_780_PRIORITY;
	#endif
	#ifdef IRQ_781_PRIORITY      /* ISR 781 */
	   INTC_0.PSR[781].B.PRIN=IRQ_781_PRIORITY;
	#endif
	#ifdef IRQ_782_PRIORITY      /* ISR 782 */
	   INTC_0.PSR[782].B.PRIN=IRQ_782_PRIORITY;
	#endif
	#ifdef IRQ_783_PRIORITY      /* ISR 783 */
	   INTC_0.PSR[783].B.PRIN=IRQ_783_PRIORITY;
	#endif
	#ifdef IRQ_784_PRIORITY      /* ISR 784 */
	   INTC_0.PSR[784].B.PRIN=IRQ_784_PRIORITY;
	#endif
	#ifdef IRQ_785_PRIORITY      /* ISR 785 */
	   INTC_0.PSR[785].B.PRIN=IRQ_785_PRIORITY;
	#endif
	#ifdef IRQ_786_PRIORITY      /* ISR 786 */
	   INTC_0.PSR[786].B.PRIN=IRQ_786_PRIORITY;
	#endif
	#ifdef IRQ_787_PRIORITY      /* ISR 787 */
	   INTC_0.PSR[787].B.PRIN=IRQ_787_PRIORITY;
	#endif
	#ifdef IRQ_788_PRIORITY      /* ISR 788 */
	   INTC_0.PSR[788].B.PRIN=IRQ_788_PRIORITY;
	#endif
	#ifdef IRQ_789_PRIORITY      /* ISR 789 */
	   INTC_0.PSR[789].B.PRIN=IRQ_789_PRIORITY;
	#endif
	#ifdef IRQ_790_PRIORITY      /* ISR 790 */
	   INTC_0.PSR[790].B.PRIN=IRQ_790_PRIORITY;
	#endif
	#ifdef IRQ_791_PRIORITY      /* ISR 791 */
	   INTC_0.PSR[791].B.PRIN=IRQ_791_PRIORITY;
	#endif
	#ifdef IRQ_792_PRIORITY      /* ISR 792 */
	   INTC_0.PSR[792].B.PRIN=IRQ_792_PRIORITY;
	#endif
	#ifdef IRQ_793_PRIORITY      /* ISR 793 */
	   INTC_0.PSR[793].B.PRIN=IRQ_793_PRIORITY;
	#endif
	#ifdef IRQ_794_PRIORITY      /* ISR 794 */
	   INTC_0.PSR[794].B.PRIN=IRQ_794_PRIORITY;
	#endif
	#ifdef IRQ_795_PRIORITY      /* ISR 795 */
	   INTC_0.PSR[795].B.PRIN=IRQ_795_PRIORITY;
	#endif
	#ifdef IRQ_796_PRIORITY      /* ISR 796 */
	   INTC_0.PSR[796].B.PRIN=IRQ_796_PRIORITY;
	#endif
	#ifdef IRQ_797_PRIORITY      /* ISR 797 */
	   INTC_0.PSR[797].B.PRIN=IRQ_797_PRIORITY;
	#endif
	#ifdef IRQ_798_PRIORITY      /* ISR 798 */
	   INTC_0.PSR[798].B.PRIN=IRQ_798_PRIORITY;
	#endif
	#ifdef IRQ_799_PRIORITY      /* ISR 799 */
	   INTC_0.PSR[799].B.PRIN=IRQ_799_PRIORITY;
	#endif
	#ifdef IRQ_800_PRIORITY      /* ISR 800 */
	   INTC_0.PSR[800].B.PRIN=IRQ_800_PRIORITY;
	#endif
	#ifdef IRQ_801_PRIORITY      /* ISR 801 */
	   INTC_0.PSR[801].B.PRIN=IRQ_801_PRIORITY;
	#endif
	#ifdef IRQ_802_PRIORITY      /* ISR 802 */
	   INTC_0.PSR[802].B.PRIN=IRQ_802_PRIORITY;
	#endif
	#ifdef IRQ_803_PRIORITY      /* ISR 803 */
	   INTC_0.PSR[803].B.PRIN=IRQ_803_PRIORITY;
	#endif
	#ifdef IRQ_804_PRIORITY      /* ISR 804 */
	   INTC_0.PSR[804].B.PRIN=IRQ_804_PRIORITY;
	#endif
	#ifdef IRQ_805_PRIORITY      /* ISR 805 */
	   INTC_0.PSR[805].B.PRIN=IRQ_805_PRIORITY;
	#endif
	#ifdef IRQ_806_PRIORITY      /* ISR 806 */
	   INTC_0.PSR[806].B.PRIN=IRQ_806_PRIORITY;
	#endif
	#ifdef IRQ_807_PRIORITY      /* ISR 807 */
	   INTC_0.PSR[807].B.PRIN=IRQ_807_PRIORITY;
	#endif
	#ifdef IRQ_808_PRIORITY      /* ISR 808 */
	   INTC_0.PSR[808].B.PRIN=IRQ_808_PRIORITY;
	#endif
	#ifdef IRQ_809_PRIORITY      /* ISR 809 */
	   INTC_0.PSR[809].B.PRIN=IRQ_809_PRIORITY;
	#endif
	#ifdef IRQ_810_PRIORITY      /* ISR 810 */
	   INTC_0.PSR[810].B.PRIN=IRQ_810_PRIORITY;
	#endif
	#ifdef IRQ_811_PRIORITY      /* ISR 811 */
	   INTC_0.PSR[811].B.PRIN=IRQ_811_PRIORITY;
	#endif
	#ifdef IRQ_812_PRIORITY      /* ISR 812 */
	   INTC_0.PSR[812].B.PRIN=IRQ_812_PRIORITY;
	#endif
	#ifdef IRQ_813_PRIORITY      /* ISR 813 */
	   INTC_0.PSR[813].B.PRIN=IRQ_813_PRIORITY;
	#endif
	#ifdef IRQ_814_PRIORITY      /* ISR 814 */
	   INTC_0.PSR[814].B.PRIN=IRQ_814_PRIORITY;
	#endif
	#ifdef IRQ_815_PRIORITY      /* ISR 815 */
	   INTC_0.PSR[815].B.PRIN=IRQ_815_PRIORITY;
	#endif
	#ifdef IRQ_816_PRIORITY      /* ISR 816 */
	   INTC_0.PSR[816].B.PRIN=IRQ_816_PRIORITY;
	#endif
	#ifdef IRQ_817_PRIORITY      /* ISR 817 */
	   INTC_0.PSR[817].B.PRIN=IRQ_817_PRIORITY;
	#endif
	#ifdef IRQ_818_PRIORITY      /* ISR 818 */
	   INTC_0.PSR[818].B.PRIN=IRQ_818_PRIORITY;
	#endif
	#ifdef IRQ_819_PRIORITY      /* ISR 819 */
	   INTC_0.PSR[819].B.PRIN=IRQ_819_PRIORITY;
	#endif
	#ifdef IRQ_820_PRIORITY      /* ISR 820 */
	   INTC_0.PSR[820].B.PRIN=IRQ_820_PRIORITY;
	#endif
	#ifdef IRQ_821_PRIORITY      /* ISR 821 */
	   INTC_0.PSR[821].B.PRIN=IRQ_821_PRIORITY;
	#endif
	#ifdef IRQ_823_PRIORITY      /* ISR 823 */
	   INTC_0.PSR[823].B.PRIN=IRQ_823_PRIORITY;
	#endif
	#ifdef IRQ_824_PRIORITY      /* ISR 824 */
	   INTC_0.PSR[824].B.PRIN=IRQ_824_PRIORITY;
	#endif
	#ifdef IRQ_825_PRIORITY      /* ISR 825 */
	   INTC_0.PSR[825].B.PRIN=IRQ_825_PRIORITY;
	#endif
	#ifdef IRQ_826_PRIORITY      /* ISR 826 */
	   INTC_0.PSR[826].B.PRIN=IRQ_826_PRIORITY;
	#endif
	#ifdef IRQ_827_PRIORITY      /* ISR 827 */
	   INTC_0.PSR[827].B.PRIN=IRQ_827_PRIORITY;
	#endif
	#ifdef IRQ_828_PRIORITY      /* ISR 828 */
	   INTC_0.PSR[828].B.PRIN=IRQ_828_PRIORITY;
	#endif
	#ifdef IRQ_829_PRIORITY      /* ISR 829 */
	   INTC_0.PSR[829].B.PRIN=IRQ_829_PRIORITY;
	#endif
	#ifdef IRQ_830_PRIORITY      /* ISR 830 */
	   INTC_0.PSR[830].B.PRIN=IRQ_830_PRIORITY;
	#endif
	#ifdef IRQ_831_PRIORITY      /* ISR 831 */
	   INTC_0.PSR[831].B.PRIN=IRQ_831_PRIORITY;
	#endif
	#ifdef IRQ_832_PRIORITY      /* ISR 832 */
	   INTC_0.PSR[832].B.PRIN=IRQ_832_PRIORITY;
	#endif
	#ifdef IRQ_833_PRIORITY      /* ISR 833 */
	   INTC_0.PSR[833].B.PRIN=IRQ_833_PRIORITY;
	#endif
	#ifdef IRQ_834_PRIORITY      /* ISR 834 */
	   INTC_0.PSR[834].B.PRIN=IRQ_834_PRIORITY;
	#endif
	#ifdef IRQ_835_PRIORITY      /* ISR 835 */
	   INTC_0.PSR[835].B.PRIN=IRQ_835_PRIORITY;
	#endif
	#ifdef IRQ_836_PRIORITY      /* ISR 836 */
	   INTC_0.PSR[836].B.PRIN=IRQ_836_PRIORITY;
	#endif
	#ifdef IRQ_837_PRIORITY      /* ISR 837 */
	   INTC_0.PSR[837].B.PRIN=IRQ_837_PRIORITY;
	#endif
	#ifdef IRQ_838_PRIORITY      /* ISR 838 */
	   INTC_0.PSR[838].B.PRIN=IRQ_838_PRIORITY;
	#endif
	#ifdef IRQ_839_PRIORITY      /* ISR 839 */
	   INTC_0.PSR[839].B.PRIN=IRQ_839_PRIORITY;
	#endif
	#ifdef IRQ_840_PRIORITY      /* ISR 840 */
	   INTC_0.PSR[840].B.PRIN=IRQ_840_PRIORITY;
	#endif
	#ifdef IRQ_841_PRIORITY      /* ISR 841 */
	   INTC_0.PSR[841].B.PRIN=IRQ_841_PRIORITY;
	#endif
	#ifdef IRQ_842_PRIORITY      /* ISR 842 */
	   INTC_0.PSR[842].B.PRIN=IRQ_842_PRIORITY;
	#endif
	#ifdef IRQ_843_PRIORITY      /* ISR 843 */
	   INTC_0.PSR[843].B.PRIN=IRQ_843_PRIORITY;
	#endif
	#ifdef IRQ_844_PRIORITY      /* ISR 844 */
	   INTC_0.PSR[844].B.PRIN=IRQ_844_PRIORITY;
	#endif
	#ifdef IRQ_845_PRIORITY      /* ISR 845 */
	   INTC_0.PSR[845].B.PRIN=IRQ_845_PRIORITY;
	#endif
	#ifdef IRQ_846_PRIORITY      /* ISR 846 */
	   INTC_0.PSR[846].B.PRIN=IRQ_846_PRIORITY;
	#endif
	#ifdef IRQ_847_PRIORITY      /* ISR 847 */
	   INTC_0.PSR[847].B.PRIN=IRQ_847_PRIORITY;
	#endif
	#ifdef IRQ_848_PRIORITY      /* ISR 848 */
	   INTC_0.PSR[848].B.PRIN=IRQ_848_PRIORITY;
	#endif
	#ifdef IRQ_849_PRIORITY      /* ISR 849 */
	   INTC_0.PSR[849].B.PRIN=IRQ_849_PRIORITY;
	#endif
	#ifdef IRQ_850_PRIORITY      /* ISR 850 */
	   INTC_0.PSR[850].B.PRIN=IRQ_850_PRIORITY;
	#endif
	#ifdef IRQ_851_PRIORITY      /* ISR 851 */
	   INTC_0.PSR[851].B.PRIN=IRQ_851_PRIORITY;
	#endif
	#ifdef IRQ_852_PRIORITY      /* ISR 852 */
	   INTC_0.PSR[852].B.PRIN=IRQ_852_PRIORITY;
	#endif
	#ifdef IRQ_853_PRIORITY      /* ISR 853 */
	   INTC_0.PSR[853].B.PRIN=IRQ_853_PRIORITY;
	#endif
	#ifdef IRQ_854_PRIORITY      /* ISR 854 */
	   INTC_0.PSR[854].B.PRIN=IRQ_854_PRIORITY;
	#endif
	#ifdef IRQ_855_PRIORITY      /* ISR 855 */
	   INTC_0.PSR[855].B.PRIN=IRQ_855_PRIORITY;
	#endif
	#ifdef IRQ_856_PRIORITY      /* ISR 856 */
	   INTC_0.PSR[856].B.PRIN=IRQ_856_PRIORITY;
	#endif
	#ifdef IRQ_857_PRIORITY      /* ISR 857 */
	   INTC_0.PSR[857].B.PRIN=IRQ_857_PRIORITY;
	#endif
	#ifdef IRQ_858_PRIORITY      /* ISR 858 */
	   INTC_0.PSR[858].B.PRIN=IRQ_858_PRIORITY;
	#endif
	#ifdef IRQ_859_PRIORITY      /* ISR 859 */
	   INTC_0.PSR[859].B.PRIN=IRQ_859_PRIORITY;
	#endif
	#ifdef IRQ_860_PRIORITY      /* ISR 860 */
	   INTC_0.PSR[860].B.PRIN=IRQ_860_PRIORITY;
	#endif
	#ifdef IRQ_861_PRIORITY      /* ISR 861 */
	   INTC_0.PSR[861].B.PRIN=IRQ_861_PRIORITY;
	#endif
	#ifdef IRQ_862_PRIORITY      /* ISR 862 */
	   INTC_0.PSR[862].B.PRIN=IRQ_862_PRIORITY;
	#endif
	#ifdef IRQ_863_PRIORITY      /* ISR 863 */
	   INTC_0.PSR[863].B.PRIN=IRQ_863_PRIORITY;
	#endif
	#ifdef IRQ_864_PRIORITY      /* ISR 864 */
	   INTC_0.PSR[864].B.PRIN=IRQ_864_PRIORITY;
	#endif
	#ifdef IRQ_865_PRIORITY      /* ISR 865 */
	   INTC_0.PSR[865].B.PRIN=IRQ_865_PRIORITY;
	#endif
	#ifdef IRQ_866_PRIORITY      /* ISR 866 */
	   INTC_0.PSR[866].B.PRIN=IRQ_866_PRIORITY;
	#endif
	#ifdef IRQ_867_PRIORITY      /* ISR 867 */
	   INTC_0.PSR[867].B.PRIN=IRQ_867_PRIORITY;
	#endif
	#ifdef IRQ_868_PRIORITY      /* ISR 868 */
	   INTC_0.PSR[868].B.PRIN=IRQ_868_PRIORITY;
	#endif
	#ifdef IRQ_869_PRIORITY      /* ISR 869 */
	   INTC_0.PSR[869].B.PRIN=IRQ_869_PRIORITY;
	#endif
	#ifdef IRQ_870_PRIORITY      /* ISR 870 */
	   INTC_0.PSR[870].B.PRIN=IRQ_870_PRIORITY;
	#endif
	#ifdef IRQ_871_PRIORITY      /* ISR 871 */
	   INTC_0.PSR[871].B.PRIN=IRQ_871_PRIORITY;
	#endif
	#ifdef IRQ_872_PRIORITY      /* ISR 872 */
	   INTC_0.PSR[872].B.PRIN=IRQ_872_PRIORITY;
	#endif
	#ifdef IRQ_873_PRIORITY      /* ISR 873 */
	   INTC_0.PSR[873].B.PRIN=IRQ_873_PRIORITY;
	#endif
	#ifdef IRQ_874_PRIORITY      /* ISR 874 */
	   INTC_0.PSR[874].B.PRIN=IRQ_874_PRIORITY;
	#endif
	#ifdef IRQ_875_PRIORITY      /* ISR 875 */
	   INTC_0.PSR[875].B.PRIN=IRQ_875_PRIORITY;
	#endif
	#ifdef IRQ_876_PRIORITY      /* ISR 876 */
	   INTC_0.PSR[876].B.PRIN=IRQ_876_PRIORITY;
	#endif
	#ifdef IRQ_877_PRIORITY      /* ISR 877 */
	   INTC_0.PSR[877].B.PRIN=IRQ_877_PRIORITY;
	#endif
	#ifdef IRQ_878_PRIORITY      /* ISR 878 */
	   INTC_0.PSR[878].B.PRIN=IRQ_878_PRIORITY;
	#endif
	#ifdef IRQ_879_PRIORITY      /* ISR 879 */
	   INTC_0.PSR[879].B.PRIN=IRQ_879_PRIORITY;
	#endif
	#ifdef IRQ_880_PRIORITY      /* ISR 880 */
	   INTC_0.PSR[880].B.PRIN=IRQ_880_PRIORITY;
	#endif
	#ifdef IRQ_881_PRIORITY      /* ISR 881 */
	   INTC_0.PSR[881].B.PRIN=IRQ_881_PRIORITY;
	#endif
	#ifdef IRQ_882_PRIORITY      /* ISR 882 */
	   INTC_0.PSR[882].B.PRIN=IRQ_882_PRIORITY;
	#endif
	#ifdef IRQ_883_PRIORITY      /* ISR 883 */
	   INTC_0.PSR[883].B.PRIN=IRQ_883_PRIORITY;
	#endif
	#ifdef IRQ_884_PRIORITY      /* ISR 884 */
	   INTC_0.PSR[884].B.PRIN=IRQ_884_PRIORITY;
	#endif
	#ifdef IRQ_885_PRIORITY      /* ISR 885 */
	   INTC_0.PSR[885].B.PRIN=IRQ_885_PRIORITY;
	#endif
	#ifdef IRQ_886_PRIORITY      /* ISR 886 */
	   INTC_0.PSR[886].B.PRIN=IRQ_886_PRIORITY;
	#endif
	#ifdef IRQ_887_PRIORITY      /* ISR 887 */
	   INTC_0.PSR[887].B.PRIN=IRQ_887_PRIORITY;
	#endif
	#ifdef IRQ_888_PRIORITY      /* ISR 888 */
	   INTC_0.PSR[888].B.PRIN=IRQ_888_PRIORITY;
	#endif
	#ifdef IRQ_889_PRIORITY      /* ISR 889 */
	   INTC_0.PSR[889].B.PRIN=IRQ_889_PRIORITY;
	#endif
	#ifdef IRQ_890_PRIORITY      /* ISR 890 */
	   INTC_0.PSR[890].B.PRIN=IRQ_890_PRIORITY;
	#endif
	#ifdef IRQ_891_PRIORITY      /* ISR 891 */
	   INTC_0.PSR[891].B.PRIN=IRQ_891_PRIORITY;
	#endif
	#ifdef IRQ_892_PRIORITY      /* ISR 892 */
	   INTC_0.PSR[892].B.PRIN=IRQ_892_PRIORITY;
	#endif
	#ifdef IRQ_893_PRIORITY      /* ISR 893 */
	   INTC_0.PSR[893].B.PRIN=IRQ_893_PRIORITY;
	#endif
	#ifdef IRQ_894_PRIORITY      /* ISR 894 */
	   INTC_0.PSR[894].B.PRIN=IRQ_894_PRIORITY;
	#endif
	#ifdef IRQ_895_PRIORITY      /* ISR 895 */
	   INTC_0.PSR[895].B.PRIN=IRQ_895_PRIORITY;
	#endif
	#ifdef IRQ_896_PRIORITY      /* ISR 896 */
	   INTC_0.PSR[896].B.PRIN=IRQ_896_PRIORITY;
	#endif
	#ifdef IRQ_897_PRIORITY      /* ISR 897 */
	   INTC_0.PSR[897].B.PRIN=IRQ_897_PRIORITY;
	#endif
	#ifdef IRQ_898_PRIORITY      /* ISR 898 */
	   INTC_0.PSR[898].B.PRIN=IRQ_898_PRIORITY;
	#endif
	#ifdef IRQ_899_PRIORITY      /* ISR 899 */
	   INTC_0.PSR[899].B.PRIN=IRQ_899_PRIORITY;
	#endif
	#ifdef IRQ_900_PRIORITY      /* ISR 900 */
	   INTC_0.PSR[900].B.PRIN=IRQ_900_PRIORITY;
	#endif
	#ifdef IRQ_901_PRIORITY      /* ISR 901 */
	   INTC_0.PSR[901].B.PRIN=IRQ_901_PRIORITY;
	#endif
	#ifdef IRQ_902_PRIORITY      /* ISR 902 */
	   INTC_0.PSR[902].B.PRIN=IRQ_902_PRIORITY;
	#endif
	#ifdef IRQ_903_PRIORITY      /* ISR 903 */
	   INTC_0.PSR[903].B.PRIN=IRQ_903_PRIORITY;
	#endif
	#ifdef IRQ_904_PRIORITY      /* ISR 904 */
	   INTC_0.PSR[904].B.PRIN=IRQ_904_PRIORITY;
	#endif
	#ifdef IRQ_905_PRIORITY      /* ISR 905 */
	   INTC_0.PSR[905].B.PRIN=IRQ_905_PRIORITY;
	#endif
	#ifdef IRQ_906_PRIORITY      /* ISR 906 */
	   INTC_0.PSR[906].B.PRIN=IRQ_906_PRIORITY;
	#endif
	#ifdef IRQ_907_PRIORITY      /* ISR 907 */
	   INTC_0.PSR[907].B.PRIN=IRQ_907_PRIORITY;
	#endif
	#ifdef IRQ_908_PRIORITY      /* ISR 908 */
	   INTC_0.PSR[908].B.PRIN=IRQ_908_PRIORITY;
	#endif
	#ifdef IRQ_909_PRIORITY      /* ISR 909 */
	   INTC_0.PSR[909].B.PRIN=IRQ_909_PRIORITY;
	#endif
	#ifdef IRQ_910_PRIORITY      /* ISR 910 */
	   INTC_0.PSR[910].B.PRIN=IRQ_910_PRIORITY;
	#endif
	#ifdef IRQ_911_PRIORITY      /* ISR 911 */
	   INTC_0.PSR[911].B.PRIN=IRQ_911_PRIORITY;
	#endif
	#ifdef IRQ_912_PRIORITY      /* ISR 912 */
	   INTC_0.PSR[912].B.PRIN=IRQ_912_PRIORITY;
	#endif
	#ifdef IRQ_913_PRIORITY      /* ISR 913 */
	   INTC_0.PSR[913].B.PRIN=IRQ_913_PRIORITY;
	#endif
	#ifdef IRQ_914_PRIORITY      /* ISR 914 */
	   INTC_0.PSR[914].B.PRIN=IRQ_914_PRIORITY;
	#endif
	#ifdef IRQ_915_PRIORITY      /* ISR 915 */
	   INTC_0.PSR[915].B.PRIN=IRQ_915_PRIORITY;
	#endif
	#ifdef IRQ_916_PRIORITY      /* ISR 916 */
	   INTC_0.PSR[916].B.PRIN=IRQ_916_PRIORITY;
	#endif
	#ifdef IRQ_917_PRIORITY      /* ISR 917 */
	   INTC_0.PSR[917].B.PRIN=IRQ_917_PRIORITY;
	#endif
	#ifdef IRQ_918_PRIORITY      /* ISR 918 */
	   INTC_0.PSR[918].B.PRIN=IRQ_918_PRIORITY;
	#endif
	#ifdef IRQ_919_PRIORITY      /* ISR 919 */
	   INTC_0.PSR[919].B.PRIN=IRQ_919_PRIORITY;
	#endif
	#ifdef IRQ_920_PRIORITY      /* ISR 920 */
	   INTC_0.PSR[920].B.PRIN=IRQ_920_PRIORITY;
	#endif
	#ifdef IRQ_921_PRIORITY      /* ISR 921 */
	   INTC_0.PSR[921].B.PRIN=IRQ_921_PRIORITY;
	#endif
	#ifdef IRQ_923_PRIORITY      /* ISR 923 */
	   INTC_0.PSR[923].B.PRIN=IRQ_923_PRIORITY;
	#endif
	#ifdef IRQ_924_PRIORITY      /* ISR 924 */
	   INTC_0.PSR[924].B.PRIN=IRQ_924_PRIORITY;
	#endif
	#ifdef IRQ_925_PRIORITY      /* ISR 925 */
	   INTC_0.PSR[925].B.PRIN=IRQ_925_PRIORITY;
	#endif
	#ifdef IRQ_926_PRIORITY      /* ISR 926 */
	   INTC_0.PSR[926].B.PRIN=IRQ_926_PRIORITY;
	#endif
	#ifdef IRQ_927_PRIORITY      /* ISR 927 */
	   INTC_0.PSR[927].B.PRIN=IRQ_927_PRIORITY;
	#endif
	#ifdef IRQ_928_PRIORITY      /* ISR 928 */
	   INTC_0.PSR[928].B.PRIN=IRQ_928_PRIORITY;
	#endif
	#ifdef IRQ_929_PRIORITY      /* ISR 929 */
	   INTC_0.PSR[929].B.PRIN=IRQ_929_PRIORITY;
	#endif
	#ifdef IRQ_930_PRIORITY      /* ISR 930 */
	   INTC_0.PSR[930].B.PRIN=IRQ_930_PRIORITY;
	#endif
	#ifdef IRQ_931_PRIORITY      /* ISR 931 */
	   INTC_0.PSR[931].B.PRIN=IRQ_931_PRIORITY;
	#endif
	#ifdef IRQ_932_PRIORITY      /* ISR 932 */
	   INTC_0.PSR[932].B.PRIN=IRQ_932_PRIORITY;
	#endif
	#ifdef IRQ_933_PRIORITY      /* ISR 933 */
	   INTC_0.PSR[933].B.PRIN=IRQ_933_PRIORITY;
	#endif
	#ifdef IRQ_934_PRIORITY      /* ISR 934 */
	   INTC_0.PSR[934].B.PRIN=IRQ_934_PRIORITY;
	#endif
	#ifdef IRQ_935_PRIORITY      /* ISR 935 */
	   INTC_0.PSR[935].B.PRIN=IRQ_935_PRIORITY;
	#endif
	#ifdef IRQ_936_PRIORITY      /* ISR 936 */
	   INTC_0.PSR[936].B.PRIN=IRQ_936_PRIORITY;
	#endif
	#ifdef IRQ_937_PRIORITY      /* ISR 937 */
	   INTC_0.PSR[937].B.PRIN=IRQ_937_PRIORITY;
	#endif
	#ifdef IRQ_938_PRIORITY      /* ISR 938 */
	   INTC_0.PSR[938].B.PRIN=IRQ_938_PRIORITY;
	#endif
	#ifdef IRQ_939_PRIORITY      /* ISR 939 */
	   INTC_0.PSR[939].B.PRIN=IRQ_939_PRIORITY;
	#endif
	#ifdef IRQ_940_PRIORITY      /* ISR 940 */
	   INTC_0.PSR[940].B.PRIN=IRQ_940_PRIORITY;
	#endif
	#ifdef IRQ_941_PRIORITY      /* ISR 941 */
	   INTC_0.PSR[941].B.PRIN=IRQ_941_PRIORITY;
	#endif
	#ifdef IRQ_942_PRIORITY      /* ISR 942 */
	   INTC_0.PSR[942].B.PRIN=IRQ_942_PRIORITY;
	#endif
	#ifdef IRQ_943_PRIORITY      /* ISR 943 */
	   INTC_0.PSR[943].B.PRIN=IRQ_943_PRIORITY;
	#endif
	#ifdef IRQ_944_PRIORITY      /* ISR 944 */
	   INTC_0.PSR[944].B.PRIN=IRQ_944_PRIORITY;
	#endif
	#ifdef IRQ_945_PRIORITY      /* ISR 945 */
	   INTC_0.PSR[945].B.PRIN=IRQ_945_PRIORITY;
	#endif
	#ifdef IRQ_946_PRIORITY      /* ISR 946 */
	   INTC_0.PSR[946].B.PRIN=IRQ_946_PRIORITY;
	#endif
	#ifdef IRQ_947_PRIORITY      /* ISR 947 */
	   INTC_0.PSR[947].B.PRIN=IRQ_947_PRIORITY;
	#endif
	#ifdef IRQ_948_PRIORITY      /* ISR 948 */
	   INTC_0.PSR[948].B.PRIN=IRQ_948_PRIORITY;
	#endif
	#ifdef IRQ_949_PRIORITY      /* ISR 949 */
	   INTC_0.PSR[949].B.PRIN=IRQ_949_PRIORITY;
	#endif
	#ifdef IRQ_950_PRIORITY      /* ISR 950 */
	   INTC_0.PSR[950].B.PRIN=IRQ_950_PRIORITY;
	#endif
	#ifdef IRQ_951_PRIORITY      /* ISR 951 */
	   INTC_0.PSR[951].B.PRIN=IRQ_951_PRIORITY;
	#endif
	#ifdef IRQ_952_PRIORITY      /* ISR 952 */
	   INTC_0.PSR[952].B.PRIN=IRQ_952_PRIORITY;
	#endif
	#ifdef IRQ_953_PRIORITY      /* ISR 953 */
	   INTC_0.PSR[953].B.PRIN=IRQ_953_PRIORITY;
	#endif
	#ifdef IRQ_954_PRIORITY      /* ISR 954 */
	   INTC_0.PSR[954].B.PRIN=IRQ_954_PRIORITY;
	#endif
	#ifdef IRQ_955_PRIORITY      /* ISR 955 */
	   INTC_0.PSR[955].B.PRIN=IRQ_955_PRIORITY;
	#endif
	#ifdef IRQ_956_PRIORITY      /* ISR 956 */
	   INTC_0.PSR[956].B.PRIN=IRQ_956_PRIORITY;
	#endif
	#ifdef IRQ_957_PRIORITY      /* ISR 957 */
	   INTC_0.PSR[957].B.PRIN=IRQ_957_PRIORITY;
	#endif
	#ifdef IRQ_958_PRIORITY      /* ISR 958 */
	   INTC_0.PSR[958].B.PRIN=IRQ_958_PRIORITY;
	#endif
	#ifdef IRQ_959_PRIORITY      /* ISR 959 */
	   INTC_0.PSR[959].B.PRIN=IRQ_959_PRIORITY;
	#endif
	#ifdef IRQ_960_PRIORITY      /* ISR 960 */
	   INTC_0.PSR[960].B.PRIN=IRQ_960_PRIORITY;
	#endif
	#ifdef IRQ_961_PRIORITY      /* ISR 961 */
	   INTC_0.PSR[961].B.PRIN=IRQ_961_PRIORITY;
	#endif
	#ifdef IRQ_962_PRIORITY      /* ISR 962 */
	   INTC_0.PSR[962].B.PRIN=IRQ_962_PRIORITY;
	#endif
	#ifdef IRQ_963_PRIORITY      /* ISR 963 */
	   INTC_0.PSR[963].B.PRIN=IRQ_963_PRIORITY;
	#endif
	#ifdef IRQ_964_PRIORITY      /* ISR 964 */
	   INTC_0.PSR[964].B.PRIN=IRQ_964_PRIORITY;
	#endif
	#ifdef IRQ_965_PRIORITY      /* ISR 965 */
	   INTC_0.PSR[965].B.PRIN=IRQ_965_PRIORITY;
	#endif
	#ifdef IRQ_966_PRIORITY      /* ISR 966 */
	   INTC_0.PSR[966].B.PRIN=IRQ_966_PRIORITY;
	#endif
	#ifdef IRQ_967_PRIORITY      /* ISR 967 */
	   INTC_0.PSR[967].B.PRIN=IRQ_967_PRIORITY;
	#endif
	#ifdef IRQ_968_PRIORITY      /* ISR 968 */
	   INTC_0.PSR[968].B.PRIN=IRQ_968_PRIORITY;
	#endif
	#ifdef IRQ_969_PRIORITY      /* ISR 969 */
	   INTC_0.PSR[969].B.PRIN=IRQ_969_PRIORITY;
	#endif
	#ifdef IRQ_970_PRIORITY      /* ISR 970 */
	   INTC_0.PSR[970].B.PRIN=IRQ_970_PRIORITY;
	#endif
	#ifdef IRQ_971_PRIORITY      /* ISR 971 */
	   INTC_0.PSR[971].B.PRIN=IRQ_971_PRIORITY;
	#endif
	#ifdef IRQ_972_PRIORITY      /* ISR 972 */
	   INTC_0.PSR[972].B.PRIN=IRQ_972_PRIORITY;
	#endif
	#ifdef IRQ_973_PRIORITY      /* ISR 973 */
	   INTC_0.PSR[973].B.PRIN=IRQ_973_PRIORITY;
	#endif
	#ifdef IRQ_974_PRIORITY      /* ISR 974 */
	   INTC_0.PSR[974].B.PRIN=IRQ_974_PRIORITY;
	#endif
	#ifdef IRQ_975_PRIORITY      /* ISR 975 */
	   INTC_0.PSR[975].B.PRIN=IRQ_975_PRIORITY;
	#endif
	#ifdef IRQ_976_PRIORITY      /* ISR 976 */
	   INTC_0.PSR[976].B.PRIN=IRQ_976_PRIORITY;
	#endif
	#ifdef IRQ_977_PRIORITY      /* ISR 977 */
	   INTC_0.PSR[977].B.PRIN=IRQ_977_PRIORITY;
	#endif
	#ifdef IRQ_978_PRIORITY      /* ISR 978 */
	   INTC_0.PSR[978].B.PRIN=IRQ_978_PRIORITY;
	#endif
	#ifdef IRQ_979_PRIORITY      /* ISR 979 */
	   INTC_0.PSR[979].B.PRIN=IRQ_979_PRIORITY;
	#endif
	#ifdef IRQ_980_PRIORITY      /* ISR 980 */
	   INTC_0.PSR[980].B.PRIN=IRQ_980_PRIORITY;
	#endif
	#ifdef IRQ_981_PRIORITY      /* ISR 981 */
	   INTC_0.PSR[981].B.PRIN=IRQ_981_PRIORITY;
	#endif
	#ifdef IRQ_982_PRIORITY      /* ISR 982 */
	   INTC_0.PSR[982].B.PRIN=IRQ_982_PRIORITY;
	#endif
	#ifdef IRQ_983_PRIORITY      /* ISR 983 */
	   INTC_0.PSR[983].B.PRIN=IRQ_983_PRIORITY;
	#endif
	#ifdef IRQ_984_PRIORITY      /* ISR 984 */
	   INTC_0.PSR[984].B.PRIN=IRQ_984_PRIORITY;
	#endif
	#ifdef IRQ_985_PRIORITY      /* ISR 985 */
	   INTC_0.PSR[985].B.PRIN=IRQ_985_PRIORITY;
	#endif
	#ifdef IRQ_986_PRIORITY      /* ISR 986 */
	   INTC_0.PSR[986].B.PRIN=IRQ_986_PRIORITY;
	#endif
	#ifdef IRQ_987_PRIORITY      /* ISR 987 */
	   INTC_0.PSR[987].B.PRIN=IRQ_987_PRIORITY;
	#endif
	#ifdef IRQ_988_PRIORITY      /* ISR 988 */
	   INTC_0.PSR[988].B.PRIN=IRQ_988_PRIORITY;
	#endif
	#ifdef IRQ_989_PRIORITY      /* ISR 989 */
	   INTC_0.PSR[989].B.PRIN=IRQ_989_PRIORITY;
	#endif
	#ifdef IRQ_990_PRIORITY      /* ISR 990 */
	   INTC_0.PSR[990].B.PRIN=IRQ_990_PRIORITY;
	#endif
	#ifdef IRQ_991_PRIORITY      /* ISR 991 */
	   INTC_0.PSR[991].B.PRIN=IRQ_991_PRIORITY;
	#endif
	#ifdef IRQ_992_PRIORITY      /* ISR 992 */
	   INTC_0.PSR[992].B.PRIN=IRQ_992_PRIORITY;
	#endif
	#ifdef IRQ_993_PRIORITY      /* ISR 993 */
	   INTC_0.PSR[993].B.PRIN=IRQ_993_PRIORITY;
	#endif
	#ifdef IRQ_994_PRIORITY      /* ISR 994 */
	   INTC_0.PSR[994].B.PRIN=IRQ_994_PRIORITY;
	#endif
	#ifdef IRQ_995_PRIORITY      /* ISR 995 */
	   INTC_0.PSR[995].B.PRIN=IRQ_995_PRIORITY;
	#endif
	#ifdef IRQ_996_PRIORITY      /* ISR 996 */
	   INTC_0.PSR[996].B.PRIN=IRQ_996_PRIORITY;
	#endif
	#ifdef IRQ_997_PRIORITY      /* ISR 997 */
	   INTC_0.PSR[997].B.PRIN=IRQ_997_PRIORITY;
	#endif
	#ifdef IRQ_998_PRIORITY      /* ISR 998 */
	   INTC_0.PSR[998].B.PRIN=IRQ_998_PRIORITY;
	#endif
	#ifdef IRQ_999_PRIORITY      /* ISR 999 */
	   INTC_0.PSR[999].B.PRIN=IRQ_999_PRIORITY;
	#endif
	#ifdef IRQ_1000_PRIORITY      /* ISR 1000 */
	   INTC_0.PSR[1000].B.PRIN=IRQ_1000_PRIORITY;
	#endif
	#ifdef IRQ_1001_PRIORITY      /* ISR 1001 */
	   INTC_0.PSR[1001].B.PRIN=IRQ_1001_PRIORITY;
	#endif
	#ifdef IRQ_1002_PRIORITY      /* ISR 1002 */
	   INTC_0.PSR[1002].B.PRIN=IRQ_1002_PRIORITY;
	#endif
	#ifdef IRQ_1003_PRIORITY      /* ISR 1003 */
	   INTC_0.PSR[1003].B.PRIN=IRQ_1003_PRIORITY;
	#endif
	#ifdef IRQ_1004_PRIORITY      /* ISR 1004 */
	   INTC_0.PSR[1004].B.PRIN=IRQ_1004_PRIORITY;
	#endif
	#ifdef IRQ_1005_PRIORITY      /* ISR 1005 */
	   INTC_0.PSR[1005].B.PRIN=IRQ_1005_PRIORITY;
	#endif
	#ifdef IRQ_1006_PRIORITY      /* ISR 1006 */
	   INTC_0.PSR[1006].B.PRIN=IRQ_1006_PRIORITY;
	#endif
	#ifdef IRQ_1007_PRIORITY      /* ISR 1007 */
	   INTC_0.PSR[1007].B.PRIN=IRQ_1007_PRIORITY;
	#endif
	#ifdef IRQ_1008_PRIORITY      /* ISR 1008 */
	   INTC_0.PSR[1008].B.PRIN=IRQ_1008_PRIORITY;
	#endif
	#ifdef IRQ_1009_PRIORITY      /* ISR 1009 */
	   INTC_0.PSR[1009].B.PRIN=IRQ_1009_PRIORITY;
	#endif
	#ifdef IRQ_1010_PRIORITY      /* ISR 1010 */
	   INTC_0.PSR[1010].B.PRIN=IRQ_1010_PRIORITY;
	#endif
	#ifdef IRQ_1011_PRIORITY      /* ISR 1011 */
	   INTC_0.PSR[1011].B.PRIN=IRQ_1011_PRIORITY;
	#endif
	#ifdef IRQ_1012_PRIORITY      /* ISR 1012 */
	   INTC_0.PSR[1012].B.PRIN=IRQ_1012_PRIORITY;
	#endif
	#ifdef IRQ_1013_PRIORITY      /* ISR 1013 */
	   INTC_0.PSR[1013].B.PRIN=IRQ_1013_PRIORITY;
	#endif
	#ifdef IRQ_1014_PRIORITY      /* ISR 1014 */
	   INTC_0.PSR[1014].B.PRIN=IRQ_1014_PRIORITY;
	#endif
	#ifdef IRQ_1015_PRIORITY      /* ISR 1015 */
	   INTC_0.PSR[1015].B.PRIN=IRQ_1015_PRIORITY;
	#endif
	#ifdef IRQ_1016_PRIORITY      /* ISR 1016 */
	   INTC_0.PSR[1016].B.PRIN=IRQ_1016_PRIORITY;
	#endif
	#ifdef IRQ_1017_PRIORITY      /* ISR 1017 */
	   INTC_0.PSR[1017].B.PRIN=IRQ_1017_PRIORITY;
	#endif
	#ifdef IRQ_1018_PRIORITY      /* ISR 1018 */
	   INTC_0.PSR[1018].B.PRIN=IRQ_1018_PRIORITY;
	#endif
	#ifdef IRQ_1019_PRIORITY      /* ISR 1019 */
	   INTC_0.PSR[1019].B.PRIN=IRQ_1019_PRIORITY;
	#endif
	#ifdef IRQ_1020_PRIORITY      /* ISR 1020 */
	   INTC_0.PSR[1020].B.PRIN=IRQ_1020_PRIORITY;
	#endif
	#ifdef IRQ_1021_PRIORITY      /* ISR 1021 */
	   INTC_0.PSR[1021].B.PRIN=IRQ_1021_PRIORITY;
	#endif
	#ifdef IRQ_1023_PRIORITY      /* ISR 1023 */
	   INTC_0.PSR[1023].B.PRIN=IRQ_1023_PRIORITY;
	#endif

}

/**************************************************************************//*!
@brief          This function enables the interrupt

@param[in,out]  void
@param[in]      void

@return         void

@details        This function sets INTC's current priority to 0.
                External interrupts to the core are enabled.

@note

@warning
******************************************************************************/
/*void enableIrq(void)
{
    // Ensure INTC's current priority is 0
    //INTC_0.CPR[0].B.PRI = 0;
	INTC_0.CPR0.B.PRI = 0;

    // Enable external interrupts
    asm("wrteei 1");

    //enable external interrupt and exceptions
//    asm ("e_lis r3,  0x0002");
//    asm ("e_or2i r3, 0x9000");
//    asm ("mfmsr r4");
//    asm ("se_or r4, r3");
//    asm ("mtmsr r4");
}

/**************************************************************************//*!
@brief  		System init function define Pictus system

@param[in,out]  void
@param[in]      void

@return         void
		
@details        The function setup Pictus system behavior

				\image latex foc.pdf "My application" width=10cm

@note			This function initializes the system and interrupt table 
				and enable interrupt
				
@warning		
******************************************************************************/
void sys_init(void)
{
    // Initialize Magic Carpet: clock, CPU mode, reset states etc.
    mc_mode_init();

    // Initialize INTC for software vector mode
    //intc_init();
}
/*****************************************************************************/
