/******************************************************************************
*
* Copyright 2006-2015 Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
***************************************************************************//*

* Project:	Dual PMSM Field Oriented Control
* MCU:		MPC5744P
* Sensor:	Resolver sensor
* Control:	Speed FOC control
*
* Last Updated By: Daniel Diamont (nxf46153)
* Date of Last Update: 8/15/2018
*
***************************************************************************/

#include "derivative.h" 			/* include peripheral declarations */
#include "MPC5744P_qs.h"
#include "MPC5744P_appconfig.h"
#include "MC33937_routines.h"
#include "MPC5744P.h"
#include "MPC5744P_CB.h"

#include "freemaster.h"
#include "freemaster_MPC57xx.h"

#include "mlib.h"
#include "gflib.h"
#include "gmclib.h"
#include "gdflib.h"

#include "uart_init.h"
#include "PMSM_struct.h"
#include "state_machine.h"

#include "MPC5744P_BSP_Faults.h"
#include "PMSM_appconfig.h"			/* generated by MCAT*/

/******************************************************************************
* Defines and macros
******************************************************************************/
#define DISABLE_IRQ()                   __asm__("wrteei 0")
#define ENABLE_IRQ()                    __asm__("wrteei 1")

/******************************************************************************
* Local function prototypes  MLIB_Abs
******************************************************************************/
void main(void);

/* Interrupt functions */
void FOC_one_Fast_ISR(void);			// 10kHz interrupt.
void FOC_two_Fast_ISR(void);			// 10kHz interrupt.

void MC33937_interrupt_ISR(void);	// ISR executed by MC33937 interrupt out signal from either power stage board.

/* Motor control functions*/
static tBool faultDetect(pmsmFOC_t *ptr);
static tBool focFastLoop(pmsmFOC_t *ptr);
static tBool focSlowLoop(pmsmFOC_t *ptr);
static tBool getMotorControlVariables(pmsmFOC_t *ptr);

extern void xcptn_xmpl(void);

/******************************************************************************
* Local variables
******************************************************************************/

/*------------------------------------
 * Application defined modules
 * ----------------------------------*/
static volatile tU8	appRunState_one, appRunState_one_last;
static volatile tU8	appSwStateCnt_one, appRunState_one_Lo, appRunState_one_Hi;

static volatile tU8	appRunState_two, appRunState_two_last;
static volatile tU8	appSwStateCnt_two, appRunState_two_Lo, appRunState_two_Hi;

/*------------------------------------
 * Currents
 * ----------------------------------*/

/*------------------------------------
 * Voltages
 * ----------------------------------*/

/*------------------------------------
 * Speed/Position
 * ----------------------------------*/

/*------------------------------------
 * SVC-PWM variables
 * ----------------------------------*/

/*------------------------------------
 * FOC variables
 * ----------------------------------*/
pmsmFOC_t 			FOC_one;
pmsmFOC_t			FOC_two;

/*------------------------------------
 * FREEMASTER SCALES
 * ----------------------------------*/

static volatile tFrac32		fm_speed_n_m;
static volatile tFrac32		fm_position_deg;

/*------------------------------------
 * General use variables
 * ----------------------------------*/

void main(void)
{
	// Disable interrupts
	DISABLE_IRQ();

	//assign motor ID for internal application use
	FOC_one.motorID = 0;
	FOC_two.motorID = 1;

	// Initializing of clock, reset, core mode etc.
	sys_init();

	// Configure and Enable Interrupts
	xcptn_xmpl ();

	// Siul2 initialize
	siul_init();

	// Initialize DSPI module
	SPI_init();

	// Initialize PIT timer module
	pit_init();

//	 Initialize SBC power source
//	MC33907_Init(&sbcDrv);

	// Initialize UART
	uart_init();

	// FreeMASTER initialization
	FMSTR_Init();

	// Initialize FlexPWM modules
	flexpwm1_init();
	flexpwm0_init();

	// Initialize eTimer modules
	etimer0_init();
	etimer1_init();

	// Initialize ADC modules with HW offset cancellation
	adc0_init();
	adc1_init();
	adc2_init();
	adc3_init();

	// Initialize CTU modules including the CTU interrupt priority
	INTC_0.PSR[713].B.PRIN = 10;	// Set CTU0-ADC Interrupt Priority
	INTC_0.PSR[729].B.PRIN = 10;	// Set CTU1-ADC Interrupt Priority
	ctu0_init();
	ctu1_init();

	// Initialize SIUL2 module - external priority [Pad GPIO13 and Pad GPIO14 -- these are mux'ed together]
	//external interrupt for Motor 1 / Power Stage Board 1
	SIUL2.IREER0.B.IREE12 	= 1;	// Rising-edge Event Enable
	SIUL2.IFER0.B.IFE12 	= 1;	// Interrupt Filter Enable
	SIUL2.DISR0.B.EIF12		= 1;	// Clear Ext. Interrupt Status Flag
	SIUL2.DIRER0.B.EIRE12	= 1;	// Interrupt Request Enable

	//external interrupt for Motor 2 / Power Stage Board 2
	SIUL2.IREER0.B.IREE13 	= 1;	// Rising-edge Event Enable
	SIUL2.IFER0.B.IFE13 	= 1;	// Interrupt Filter Enable
	SIUL2.DISR0.B.EIF13		= 1;	// Clear Ext. Interrupt Status Flag
	SIUL2.DIRER0.B.EIRE13	= 1;	// Interrupt Request Enable

	INTC_0.PSR[244].B.PRIN	= 5;	// Set Interrupt Priority for external interrupts

	//set initial state for the state machines
	FOC_one.cntrState.state   = reset;
	FOC_one.cntrState.event   = e_reset;
	FOC_two.cntrState.state   = reset;
	FOC_two.cntrState.event   = e_reset;
   // FOC.Resolver.calib.calibCntr = RES_CALIB_CNTR;

	//first call to state machines
	state_table[FOC_one.cntrState.event][FOC_one.cntrState.state](&FOC_one);
	state_table[FOC_two.cntrState.event][FOC_two.cntrState.state](&FOC_two);

	// Enable interrupts
	INTC_0.CPR0.R = 0U;
	ENABLE_IRQ();

	/* Loop forever */
	for(;;) {

		FMSTR_Poll();
	}

}

/**************************************************************************//*!
@brief          CTU0-ADC command interrupt service routine

@param[in,out]  void
@param[in]      void

@return         void

@details        The ISR runs the same frequency as defined in CTU (10KHz).
				ISR periodically performs: all ADC quantities measurement,
				state machine calling, user button/switch and LEDs control,
				FreeMASTER recorder

@note           none

@warning		none
****************************************************************** ************/
void FOC_one_Fast_ISR(void)
{
    // Read input of user buttons/switch states
	appRunState_one		= GPDI_read(FOC_one.MPC5744P_HW.buttons.flipFlop);
	FOC_one.btSpeedUp 		= GPDI_read(FOC_one.MPC5744P_HW.buttons.upButton);
	FOC_one.btSpeedDown		= GPDI_read(FOC_one.MPC5744P_HW.buttons.downButton);

	// Routine for reading the application On/Off state, resistant for overshoot of the HW button
	if (appRunState_one ^ appRunState_one_last){
		appSwStateCnt_one++;
			if (appRunState_one)	appRunState_one_Hi = appRunState_one_Hi + 1;
			else    		    appRunState_one_Lo = appRunState_one_Lo + 1;
	}

	// If 7 Hi/Lo states is red out of 10 reading, perform the switch On/Off event
	if (appSwStateCnt_one >= 10){
		// SW_APP_RUN_L is switched off - logic 1
		FOC_one.cntrState.switchAppOnOff  = (appRunState_one_Hi > 6) ? FALSE: FOC_one.cntrState.switchAppOnOff;
		// SW_APP_RUN_L is switched on - logic 0
		FOC_one.cntrState.switchAppOnOff  = (appRunState_one_Lo > 6) ? TRUE : FOC_one.cntrState.switchAppOnOff;
		// re-init the flip/flop s/w count states
		appRunState_one_last    = appRunState_one;
		appSwStateCnt_one       = 0;
		appRunState_one_Hi      = 0;
		appRunState_one_Lo      = 0;
	}

	// User accessible Flip/Flop switch for starting/stopping the application.
	if (FOC_one.cntrState.switchAppOnOff ^ FOC_one.cntrState.switchAppOnOffState) {
		FOC_one.cntrState.switchAppOnOffState	= FOC_one.cntrState.switchAppOnOff;

		if(FOC_one.cntrState.switchAppOnOff){
				GPDO_clear(FOC_one.MPC5744P_HW.ledIndic.Led_D18);
				FOC_one.cntrState.event = e_app_on;
		}
		else	FOC_one.cntrState.event	= e_app_off;
	}

	// Get the motor control quantities for FOC
	getMotorControlVariables(&FOC_one);

	// if DC-bus voltage is higher than TRIP value, open DC-bus brake transistor
    if (FOC_one.uDcbFbck.raw > U_DCB_TRIP)	GPDO_set(BRAKE_GATE_J1);
    else								GPDO_clear(BRAKE_GATE_J1);

	// Fault Detection routine
	if(faultDetect(&FOC_one)) 	FOC_one.cntrState.event 			= e_fault;

	// Application state machine calling including LED signalization
	state_table[FOC_one.cntrState.event][FOC_one.cntrState.state](&FOC_one);
	state_LED[FOC_one.cntrState.state](&FOC_one.MPC5744P_HW.ledIndic, FOC_one.motorID);

	// DC-bus voltage ripple elimination and Space Vector Modulation
    FOC_one.elimDcbRip.fltArgDcBusMsr  = FOC_one.uDcbFbck.raw;
	GMCLIB_ElimDcBusRip(&FOC_one.uAlBeReqDCB,&FOC_one.uAlBeReq,&FOC_one.elimDcbRip);
	FOC_one.svmSector = GMCLIB_SvmStd(&FOC_one.pwm.PhABC,&FOC_one.uAlBeReqDCB);

	// Recalculate FOC voltages to DutyCycle values and apply to FlexPWM
	apply_voltages(&FOC_one.MPC5744P_HW.flexPWM, &FOC_one.pwm);
	FLEXPWM_set_LDOK(&FOC_one.MPC5744P_HW.flexPWM);

	// FreeMASTER recorder
	FMSTR_Recorder();

	// Interrupt Flag Register - clear ADC command interrupt flag
	CTU_0.IFR.B.ADC_I = 0x1;

}

/**************************************************************************//*!
@brief          CTU1-ADC command interrupt service routine

@param[in,out]  void
@param[in]      void

@return         void

@details        The ISR runs the same frequency as defined in CTU (10KHz).
				ISR periodically performs: all ADC quantities measurement,
				state machine calling, user button/switch and LEDs control,
				FreeMASTER recorder

@note           none

@warning		none
****************************************************************** ************/
void FOC_two_Fast_ISR(void)
{
    // Read input of user buttons/switch states
	appRunState_two		= GPDI_read(FOC_two.MPC5744P_HW.buttons.flipFlop);
	FOC_two.btSpeedUp 		= GPDI_read(FOC_two.MPC5744P_HW.buttons.upButton);
	FOC_two.btSpeedDown		= GPDI_read(FOC_two.MPC5744P_HW.buttons.downButton);

	// Routine for reading the application On/Off state, resistant for overshoot of the HW button
	if (appRunState_two ^ appRunState_two_last){
		appSwStateCnt_two++;
			if (appRunState_two)	appRunState_two_Hi = appRunState_two_Hi + 1;
			else    		    appRunState_two_Lo = appRunState_two_Lo + 1;
	}

	// If 7 Hi/Lo states is red out of 10 reading, perform the switch On/Off event
	if (appSwStateCnt_two >= 10){
		// SW_APP_RUN_L is switched off - logic 1
		FOC_two.cntrState.switchAppOnOff  = (appRunState_two_Hi > 6) ? FALSE: FOC_two.cntrState.switchAppOnOff;
		// SW_APP_RUN_L is switched on - logic 0
		FOC_two.cntrState.switchAppOnOff  = (appRunState_two_Lo > 6) ? TRUE : FOC_two.cntrState.switchAppOnOff;
		// re-init the flip/flop s/w count states
		appRunState_two_last    = appRunState_two;
		appSwStateCnt_two       = 0;
		appRunState_two_Hi      = 0;
		appRunState_two_Lo      = 0;
	}

	// User accessible Flip/Flop switch for starting/stopping the application.
	if (FOC_two.cntrState.switchAppOnOff ^ FOC_two.cntrState.switchAppOnOffState) {
		FOC_two.cntrState.switchAppOnOffState	= FOC_two.cntrState.switchAppOnOff;

		if(FOC_two.cntrState.switchAppOnOff){
				GPDO_clear(FOC_two.MPC5744P_HW.ledIndic.Led_D18);
				FOC_two.cntrState.event = e_app_on;
		}
		else	FOC_two.cntrState.event	= e_app_off;
	}

	// Get the motor control quantities for FOC
	getMotorControlVariables(&FOC_two);

	// if DC-bus voltage is higher than TRIP value, open DC-bus brake transistor
    if (FOC_two.uDcbFbck.raw > U_DCB_TRIP)	GPDO_set(BRAKE_GATE_J200);
    else								GPDO_clear(BRAKE_GATE_J200);

	// Fault Detection routine
	if(faultDetect(&FOC_two)) 	FOC_two.cntrState.event 			= e_fault;

	// Application state machine calling including LED signalization
	state_table[FOC_two.cntrState.event][FOC_two.cntrState.state](&FOC_two);
	state_LED[FOC_two.cntrState.state](&FOC_two.MPC5744P_HW.ledIndic, FOC_two.motorID);

	// DC-bus voltage ripple elimination and Space Vector Modulation
    FOC_two.elimDcbRip.fltArgDcBusMsr  = FOC_two.uDcbFbck.raw;
	GMCLIB_ElimDcBusRip(&FOC_two.uAlBeReqDCB,&FOC_two.uAlBeReq,&FOC_two.elimDcbRip);
	FOC_two.svmSector = GMCLIB_SvmStd(&FOC_two.pwm.PhABC,&FOC_two.uAlBeReqDCB);

	// Recalculate FOC voltages to DutyCycle values and apply to FlexPWM
	apply_voltages(&FOC_two.MPC5744P_HW.flexPWM, &FOC_two.pwm);
	FLEXPWM_set_LDOK(&FOC_two.MPC5744P_HW.flexPWM);

	// FreeMASTER recorder
	FMSTR_Recorder();

	// Interrupt Flag Register - clear ADC command interrupt flag
	CTU_1.IFR.B.ADC_I = 0x1;

}

/**************************************************************************//*!
@brief          SIUL2 external interrupt service routine

@param[in,out]  void
@param[in]      void

@return         void

@details        The ISR is called by an interrupt pin from MC33937.

@note           none

@warning		none
****************************************************************** ************/
void MC33937_interrupt_ISR(void){

	if(SIUL2.DISR0.B.EIF12 == 1){
		if(faultDetect(&FOC_one)) 	FOC_one.cntrState.event 			= e_fault;

		// Read Fault from MC33937 pre-driver - performed in main() - while(forever)
		MC33937_get_status(&FOC_one.MPC5744P_HW.MC33937, &FOC_one.MC33937);

		if(FOC_one.MC33937.status.sr0.R){
			FOC_one.faultID.B.WrongHardware		= 1;
			FOC_one.faultIDp.B.WrongHardware	= 1;
		}

		// Clear interrupt flag
		SIUL2.DISR0.B.EIF12 = 1;
	}
	if(SIUL2.DISR0.B.EIF13 == 1){
		if(faultDetect(&FOC_two)) 	FOC_two.cntrState.event 			= e_fault;

		// Read Fault from MC33937 pre-driver - performed in main() - while(forever)
		MC33937_get_status(&FOC_two.MPC5744P_HW.MC33937, &FOC_two.MC33937);

		if(FOC_two.MC33937.status.sr0.R){

			FOC_two.faultID.B.WrongHardware		= 1;
			FOC_two.faultIDp.B.WrongHardware	= 1;
		}

		// Clear interrupt flag
		SIUL2.DISR0.B.EIF13 = 1;
	}
}

/***************************************************************************//*!
*
* @brief   FAULT state
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
void stateFault(pmsmFOC_t *ptr)
{
	// Application State Machine - state identification
    ptr->cntrState.state	= fault;

    // Disable the output of MC33937 pre-diver and flexPWM
    MC33937_disable_output(&ptr->MPC5744P_HW.MC33937,&ptr->MC33937);
    FLEXPWM_disable_output(&ptr->MPC5744P_HW.flexPWM);

    // Disable user application switch
    ptr->cntrState.switchAppOnOff      = FALSE;
    ptr->cntrState.switchAppOnOffState = FALSE;

    // Clear manually Fault state by pushing both user buttons on power stage board
    if (!(ptr->btSpeedUp) && !(ptr->btSpeedDown))
    	ptr->cntrState.switchFaultClear = 1;

    // does not allow to reset application before solving and clearing the FAULT state
    ptr->cntrState.switchAppReset 		= 0;

    if (ptr->cntrState.switchFaultClear)
    {
    	DISABLE_IRQ();

        // Clear FlexPWM Fault PIN flags manually
    	clear_flexpwm_fault_flag_manual(0xF, ptr->motorID);

    	// Clear Fault/Permanent Fault registers
        ptr->faultID.R          = 0x0;      // Clear Fault register
        ptr->faultIDp.R         = 0x0;      // Clear Permanent Fault register

    	MC33937_clear_faults(&ptr->MPC5744P_HW.MC33937,&ptr->MC33937);
    	MC33937_get_status(&ptr->MPC5744P_HW.MC33937,&ptr->MC33937);

    	// check faults in general status register of MC33937
	    if(ptr->MC33937.status.sr0.R){
				ptr->faultID.B.WrongHardware	= 1;
				ptr->faultIDp.B.WrongHardware	= 1;
	    }

        // Reset fault clearing switch when all sources of faults were removed
        ptr->cntrState.switchFaultClear = 0;

        // set new application event
        ptr->cntrState.event	= e_fault_clear;

        ENABLE_IRQ();
    }

}

/***************************************************************************//*!
*
* @brief   RESET state
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
* @details	clear all application state variables and load default configuration
* 			of all parameters of the control algorithm (PI controllers, etc)
******************************************************************************/
void stateReset(pmsmFOC_t *ptr)
{
	tBool statusResetPass = TRUE;

	// Application State Machine - state identification
    ptr->cntrState.state   								= reset;
    ptr->cntrState.loadDefSetting						= 0;

    // Controller board LED indicators initialization
    Led_indicator_init(&ptr->MPC5744P_HW.ledIndic);
    // Controller board user buttons initialization
    if(ptr->motorID == 0)
    	User_button_init(&ptr->MPC5744P_HW.buttons, buttons_GPDI_HW_J1);
    if(ptr->motorID == 1)
    	User_button_init(&ptr->MPC5744P_HW.buttons, buttons_GPDI_HW_J200);

    // Resolver HW part initialization
    if(ptr->motorID == 0)
    	Resolver_hw_init(&ptr->MPC5744P_HW.Resolver, RES_HW_PCIe_J1);
    if(ptr->motorID == 1)
    	Resolver_hw_init(&ptr->MPC5744P_HW.Resolver, RES_HW_PCIe_J200);

    // MA filter - smoothing factor set for DC offset calibration
    ptr->Resolver.MAF_Sin.fltLambda 					= 0.5F;
    ptr->Resolver.MAF_Cos.fltLambda 					= 0.5F;

    // Angle Tracking Observer initialization for resolver
    ATO_clear_module(&ptr->Resolver_SW);
    // Set motor pole-pairs number
    ptr->Resolver_SW.fltmotorPP							= MOTOR_PP;
    ptr->Resolver_SW.s32motorPPscale					= MOTOR_PP_GAIN;
    ptr->Resolver_SW.s16motorPPscaleShift				= MOTOR_PP_SHIFT;

    // Set integral constant of ATO
	ptr->Resolver_SW.ATO.pParamInteg.fltC1 				= POSPE_RES_TO_INTEG_GAIN;

	// Set PI controller of ATO
	ptr->Resolver_SW.ATO.pParamPI.fltCC1sc				= POSPE_RES_TO_CC1;
	ptr->Resolver_SW.ATO.pParamPI.fltCC2sc				= POSPE_RES_TO_CC2;
	ptr->Resolver_SW.ATO.pParamPI.fltLowerLimit			= -1000.F;
	ptr->Resolver_SW.ATO.pParamPI.fltUpperLimit			= 1000.F;

    // Set MA filter smoothing factor for speed filtering in ATO module
    ptr->Resolver_SW.filterMA.fltLambda 				= POSPE_SPEED_FILTER_MA_LAMBDA;

    // Phase current HW module initialization
    if(ptr->motorID == 0)
		PhCurrent_meas_hw_init(&ptr->MPC5744P_HW.phCurrents, CURR_HW_PCIe_J1);
    if(ptr->motorID == 1)
		PhCurrent_meas_hw_init(&ptr->MPC5744P_HW.phCurrents, CURR_HW_PCIe_J200);

    // MA filter - smoothing factor set for DC offset current ADC calibration
    ptr->iAbcFbck.MAF_PhA.fltLambda 					= 0.5F;
    ptr->iAbcFbck.MAF_PhB.fltLambda 					= 0.5F;
    ptr->iAbcFbck.MAF_PhC.fltLambda 					= 0.5F;

    // DC-bus voltage measurement
    if(ptr->motorID == 0)
    	DcbVoltage_meas_hw_init(&ptr->MPC5744P_HW.uDcb, VOLT_HW_PCIe_J1);
    if(ptr->motorID == 1)
    	DcbVoltage_meas_hw_init(&ptr->MPC5744P_HW.uDcb, VOLT_HW_PCIe_J200);

    // IIR1 filter setting, Fcut = 100Hz @ 100e-6 sampling
    ptr->uDcbFbck.IIR1_UdcbFilt.trFiltCoeff.fltA1 		= UDCB_IIR_A1;
    ptr->uDcbFbck.IIR1_UdcbFilt.trFiltCoeff.fltB0 		= UDCB_IIR_B0;
    ptr->uDcbFbck.IIR1_UdcbFilt.trFiltCoeff.fltB1 		= UDCB_IIR_B1;

    // DC-bus voltage ripple elimination parameters
    ptr->elimDcbRip.fltModIndex         	 			= 0.866025403784439F;
    ptr->elimDcbRip.fltArgDcBusMsr      	 			= 0.0F;

    // MC33937 HW initialization
    if(ptr->motorID == 0)
    	MC33937_hw_init(&ptr->MPC5744P_HW.MC33937, MC33937_HW_PCIe_J1);
    if(ptr->motorID == 1)
    	MC33937_hw_init(&ptr->MPC5744P_HW.MC33937, MC33937_HW_PCIe_J200);

    // MC33937 SW initialization
    /* MC33937 driver SW module constructor, linking also system timer/counter */
    ptr->MC33937.configData.deadtime                   	= 750;
    ptr->MC33937.configData.mode.B.enableFullOn        	= 0;
    ptr->MC33937.configData.mode.B.lock                	= 1;
    ptr->MC33937.configData.mode.B.disableDesat        	= 1;

    ptr->MC33937.configData.interruptEnable.R          	= 0;    /* Clear MASK0/1 registers - no interrupts enabled */
    ptr->MC33937.configData.interruptEnable.B.overCurrent= 1;    /* enable interrupt by over-current */
    ptr->MC33937.configData.interruptEnable.B.lowVls		= 1;    /* enable interrupt by low-VLS */
    ptr->MC33937.configData.interruptEnable.B.desaturation= 1;   /* enable interrupt by Desaturation Detection */

    MC33937_sw_init(&ptr->MPC5744P_HW.MC33937, &ptr->MC33937);
    MC33937_get_status(&ptr->MPC5744P_HW.MC33937, &ptr->MC33937);

    // FlexPWM initialization
    if(ptr->motorID == 0)
    	FLEXPWM_hw_init(&ptr->MPC5744P_HW.flexPWM, FLEXPWM_HW_PCIe_J1);
    if(ptr->motorID == 1)
    	FLEXPWM_hw_init(&ptr->MPC5744P_HW.flexPWM, FLEXPWM_HW_PCIe_J200);

    // Scalar control variables
	ptr->scalarControl.wRotReqRamp.fltRampUp 			= SCALAR_RAMP_UP;
	ptr->scalarControl.wRotReqRamp.fltRampDown 			= SCALAR_RAMP_DOWN;
	ptr->scalarControl.VHzRatioReq						= SCALAR_VHZ_FACTOR_GAIN;

	ptr->scalarControl.wRotElReq						= 0.0;
	ptr->scalarControl.UmReq							= 0.0;

	ptr->scalarControl.wElMax							= WEL_MAX;
	ptr->scalarControl.integ.f32C1						= SCALAR_INTEG_GAIN;
	ptr->scalarControl.integ.u16NShift 					= SCALAR_INTEG_SHIFT;

	/*------------------------------------
	 * FOC variables
	 * ----------------------------------*/
	ptr->align.duration									= ALIGN_DURATION;
	ptr->align.counter 									= ptr->align.duration;
	ptr->align.voltage									= ALIGN_VOLTAGE;

	// D-axis PI parallel current controller
	ptr->dAxisPIp.fltLowerLimit							= MLIB_Mul(-U_DCB_MAX, FLOAT_DIVBY_SQRT3); //Udc_max/sqrt3
	ptr->dAxisPIp.fltUpperLimit         				= MLIB_Mul(U_DCB_MAX, FLOAT_DIVBY_SQRT3);  //Udc_max/sqrt3
	ptr->dAxisPIp.fltPropGain							= D_KP_GAIN;
	ptr->dAxisPIp.fltIntegGain							= D_KI_GAIN;
	ptr->dAxisPIp.fltInK_1								= 0.0;
	// Q-axis PI parallel current controller
	ptr->qAxisPIp.fltLowerLimit         				= MLIB_Mul(-U_DCB_MAX, FLOAT_DIVBY_SQRT3); //Udc_max/sqrt3;
	ptr->qAxisPIp.fltUpperLimit         				= MLIB_Mul(U_DCB_MAX, FLOAT_DIVBY_SQRT3);  //Udc_max/sqrt3;
	ptr->qAxisPIp.fltPropGain							= Q_KP_GAIN;
	ptr->qAxisPIp.fltIntegGain							= Q_KI_GAIN;
	ptr->qAxisPIp.fltInK_1								= 0.0;

	// D-axis current loop - zero cancellation by 1st order filer
	ptr->dAxisZC.trFiltCoeff.fltB0						= D_B0;
	ptr->dAxisZC.trFiltCoeff.fltB1      				= D_B1;
	ptr->dAxisZC.trFiltCoeff.fltA1      				= D_A1;
	GDFLIB_FilterIIR1Init(&(ptr->dAxisZC));

	// Q-axis current loop - zero cancellation by 1st order filer
	ptr->qAxisZC.trFiltCoeff.fltB0      				= Q_B0;
	ptr->qAxisZC.trFiltCoeff.fltB1      				= Q_B1;
	ptr->qAxisZC.trFiltCoeff.fltA1      				= Q_A1;
	GDFLIB_FilterIIR1Init(&(ptr->qAxisZC));

    /*------------------------------------
     * Speed/Position
     * ----------------------------------*/
    ptr->speedPIp.fltPropGain			    			= SPEED_PI_PROP_GAIN;
    ptr->speedPIp.fltIntegGain			    			= SPEED_PI_INTEG_GAIN;

    ptr->speedPIp.fltUpperLimit							= I_MAX;		/* Limit of the current */
    ptr->speedPIp.fltLowerLimit							= - I_MAX;
    ptr->speedPIp.fltInK_1								= 0.0F;
    ptr->speedPIp.fltIntegPartK_1						= 0.0F;

    ptr->controlLoop.speedRamp.fltState					= 0.0F;
    ptr->controlLoop.speedRamp.fltRampUp				= SPEED_RAMP_UP;
    ptr->controlLoop.speedRamp.fltRampDown				= SPEED_RAMP_DOWN;

    ptr->cntrState.switchAppReset						= 0;

    // default setting of a control mode
    ptr->cntrState.controlMode							= speedCtrl;

    // FreeMASTER scales
    fm_speed_n_m    									= FM_SPEED_RPM_MEC_SCALE;
    fm_position_deg 									= FM_POSITION_DEG_SCALE;

    // Resolve the RESET state sequence
    if(statusResetPass)	    ptr->cntrState.event   		= e_reset_done;
    else					ptr->cntrState.event   		= e_reset;
}

/***************************************************************************//*!
*
* @brief   INIT state - clear state variables
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
* @details	clear all application state variables, while algorithm parameters
* 			remain unchanged. It means if user changed the parameters during last
* 			run state, those parameters will not be set to default.
******************************************************************************/
void stateInit(pmsmFOC_t *ptr)
{
	// Application State Machine - state identification
    ptr->cntrState.state   = init;

    // Disable the output of MC33937 pre-diver and flexPWM
    MC33937_disable_output(&ptr->MPC5744P_HW.MC33937,&ptr->MC33937);
    FLEXPWM_disable_output(&ptr->MPC5744P_HW.flexPWM);

    /*------------------------------------
     * Voltages
     * ----------------------------------*/
    ptr->uDQReq.fltArg1					= 0.0F;
    ptr->uDQReq.fltArg2					= 0.0F;
    ptr->uAlBeReq.fltArg1				= 0.0F;
    ptr->uAlBeReq.fltArg2				= 0.0F;

    ptr->scalarControl.integ.f32State	= 0.0F;

    /*------------------------------------
     * Currents
     * ----------------------------------*/
    ptr->iDQReq.fltArg1					= 0.0F;
    ptr->iDQReq.fltArg2					= 0.0F;
    ptr->iDQReqZC.fltArg1				= 0.0F;
    ptr->iDQReqZC.fltArg2				= 0.0F;
    ptr->iDQFbck.fltArg1				= 0.0F;
    ptr->iDQFbck.fltArg2				= 0.0F;

	ptr->dAxisPIp.fltInK_1				= 0.0F;
	ptr->dAxisPIp.fltIntegPartK_1		= 0.0F;
	ptr->qAxisPIp.fltInK_1				= 0.0F;
	ptr->qAxisPIp.fltIntegPartK_1		= 0.0F;

    PhCurrent_module_clear(&ptr->iAbcFbck);

    /*------------------------------------
     * Speed/ position
     * ----------------------------------*/
    ptr->Resolver.MAF_Cos.fltAcc		= 0.0F;
    ptr->Resolver.MAF_Sin.fltAcc		= 0.0F;
    ptr->Resolver_SW.filterMA.fltAcc	= 0.0F;
    ptr->Resolver.calib.calibDone		= 0;


    ptr->speedPIp.fltInK_1				= 0.0F;
    ptr->speedPIp.fltIntegPartK_1		= 0.0F;

    ptr->controlLoop.wRotElErr			= 0.0F;
    ptr->controlLoop.wRotElReq			= 0.0F;
    ptr->controlLoop.wRotElReqRamp		= 0.0F;
    ptr->controlLoop.speedRamp.fltState	= 0.0F;

    // Angle Tracking Observer module clear for resolver
    ATO_clear_module(&ptr->Resolver_SW);

    // Set speed loop counter to its threshold to call the speed loop in first step
    ptr->cntrState.speedLoopCntr		= SPEED_LOOP_CNTR;

    /*------------------------------------
     * FOC general
     ------------------------------------*/
    ptr->align.counter 					= ptr->align.duration;

    // MC33937 - read general status and internal IC settings
    MC33937_get_status(&ptr->MPC5744P_HW.MC33937, &ptr->MC33937);

    ptr->cntrState.event   				= e_init_done;
}

/***************************************************************************//*!
*
* @brief   READY state
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
void stateReady(pmsmFOC_t *ptr)
{
	// Application State Machine - state identification
	ptr->cntrState.state   = ready;
	ptr->cntrState.event   = e_ready;

    // Waiting for user ON/OFF-command or LOAD

	if(ptr->cntrState.loadDefSetting) ptr->cntrState.event   = e_reset;
}

/***************************************************************************//*!
*
* @brief   CALIBRATION state - ADC calibration state
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
void stateCalib(pmsmFOC_t *ptr)
{
	static tBool statusCalibStatus = FALSE;

	// Application State Machine - state identification
    ptr->cntrState.state    = calib;
    ptr->cntrState.event    = e_calib;

    statusCalibStatus = Resolver_disable_excitation(&ptr->MPC5744P_HW.Resolver);

    //Resolver excitation is enabled as soon as calibration is done
    statusCalibStatus |=    Resolver_inputs_calibration(&ptr->MPC5744P_HW.Resolver, &ptr->Resolver);

    //Phase currents DC offset calibration
    statusCalibStatus |=    PhCurrent_calibration(&ptr->MPC5744P_HW.phCurrents, &ptr->iAbcFbck);

    // Calibration sequence has successfully finished
    if(ptr->Resolver.calib.calibDone && ptr->iAbcFbck.calib.calibDone)
    			ptr->cntrState.event   = e_calib_done;
    else		ptr->cntrState.event   = e_calib;

    if (!statusCalibStatus)	ptr->faultID.B.CalibError = 1;
}

/***************************************************************************//*!
*
* @brief   ALIGNMENT state - d-axes rotor alignment, position offset calibration
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
void stateAlign(pmsmFOC_t *ptr)
{
	static tBool stateAlignStatus = FALSE;

	// Application State Machine - state identification
    ptr->cntrState.state	= align;
    ptr->cntrState.event	= e_align;

    // Enable output of FlexPWM and preDriver MC33937
    stateAlignStatus =	MC33937_enable_output(&ptr->MPC5744P_HW.MC33937,&ptr->MC33937);
    stateAlignStatus &= FLEXPWM_enable_output(&ptr->MPC5744P_HW.flexPWM);

    // Function performs Sin(angle) and Cos (angle), angle = 0;
    GFLIB_SinCos (0.0F, &ptr->thTransform);

    // Q-Axis voltage is kept at zero
	ptr->uDQReq.fltArg1			= ptr->align.voltage;
	ptr->uDQReq.fltArg2      	= 0.0F;

    GMCLIB_ParkInv(&ptr->uAlBeReq,&ptr->thTransform,&ptr->uDQReq);

    if (--ptr->align.counter <= 0)
    {
        ptr->uDQReq.fltArg1 	 = (0.0F);
        ptr->uDQReq.fltArg2 	 = (0.0F);

        // Store the non-zero electrical position as a position offset
        ptr->Resolver_SW.fltThRotElOffset	= ptr->Resolver_SW.fltThRotEl;

        ptr->cntrState.event	= e_align_done;
    }

    if (!stateAlignStatus)	ptr->faultID.B.AlignError = 1;
}

/***************************************************************************//*!
*
* @brief   RUN state
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
void stateRun(pmsmFOC_t *ptr)
{
	static tBool stateRunStatus = FALSE;

	// Application State Machine - state identification
    ptr->cntrState.state	= run;
    ptr->cntrState.event	= e_run;

    // Arbitration of executing the slow speed loop
    if (++ptr->cntrState.speedLoopCntr >= SPEED_LOOP_CNTR){
		// reset speed loop counter
		ptr->cntrState.speedLoopCntr	= 0;

		if(ptr->cntrState.controlMode==scalarCtrl){
			if (!(ptr->btSpeedUp))
				ptr->scalarControl.wRotElReq = (ptr->scalarControl.wRotElReq + 0.5F);

			if (!(ptr->btSpeedDown))
				ptr->scalarControl.wRotElReq = (ptr->scalarControl.wRotElReq - 0.5F);
		}
		else{
		// Manual speed control using Up/Down switch buttons SW2/SW3
		if (!(ptr->btSpeedUp))
			ptr->controlLoop.wRotElReq = (ptr->controlLoop.wRotElReq + 0.5F);

		if (!(ptr->btSpeedDown))
			ptr->controlLoop.wRotElReq = (ptr->controlLoop.wRotElReq - 0.5F);
		}

		// calculate FOC fast loop - speed loop
		stateRunStatus  = focSlowLoop(ptr);
    }

    // calculate FOC fast loop - current loop
    stateRunStatus  &= focFastLoop(ptr);

    if (!stateRunStatus)	ptr->faultID.B.FOCError = 1;

    // Endless loop; Waiting for user OFF-command -> cntrState.event = e_app_off
}

/***************************************************************************//*!
*
* @brief   Fault Detect function
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
static tBool faultDetect(pmsmFOC_t *ptr)
{
	static tBool faultDetectiontEvent;

	    faultDetectiontEvent = FALSE;

	    //-----------------------------
	    // Actual Faults
	    //-----------------------------
	    // Fault:  ABC - phases over-current detected on FAULT PIN FLEXPWM_0 FAULT_0
	    if(get_flexpwm_fault(((FLEXPWM_FAULT_0) || (ptr->iAbcFbck.filt.fltArg1 >= I_PH_OVER)), ptr->motorID))
	    	ptr->faultID.B.OverPhaseACurrent = 1;

	    if(get_flexpwm_fault(((FLEXPWM_FAULT_0) || (ptr->iAbcFbck.filt.fltArg2 >= I_PH_OVER)), ptr->motorID))
	    	ptr->faultID.B.OverPhaseBCurrent = 1;

	    if(get_flexpwm_fault(((FLEXPWM_FAULT_0) || (ptr->iAbcFbck.filt.fltArg3 >= I_PH_OVER)), ptr->motorID))
	    	ptr->faultID.B.OverPhaseCCurrent = 1;

	    // Fault:   DC-bus over-voltage detect on FAULT PIN FLEXPWM_0 FAULT 1
	    ptr->faultID.B.OverDCBusVoltage 	= get_flexpwm_fault(FLEXPWM_FAULT_1, ptr->motorID);

	    // Fault:   DC-bus under-voltage - s/w comparator
	    ptr->faultID.B.UnderDCBusVoltage 	= (ptr->uDcbFbck.raw < U_DCB_UNDER) ? TRUE : FALSE;

	    //-----------------------------
	    // Pending Faults - flags
	    //-----------------------------
	    // Fault flag:   Phase A,B,C over-current detect on FAULT PIN FLEXPWM_0 FAULT 0
	    if(get_flexpwm_fault_flag(((FLEXPWM_FAULT_0) || (ptr->iAbcFbck.filt.fltArg1 >= I_PH_OVER)), ptr->motorID))
	    	ptr->faultIDp.B.OverPhaseACurrent = 1;

	    if(get_flexpwm_fault_flag(((FLEXPWM_FAULT_0) || (ptr->iAbcFbck.filt.fltArg2 >= I_PH_OVER)), ptr->motorID))
	    	ptr->faultIDp.B.OverPhaseBCurrent = 1;

	    if(get_flexpwm_fault_flag(((FLEXPWM_FAULT_0) || (ptr->iAbcFbck.filt.fltArg3 >= I_PH_OVER)), ptr->motorID))
	    	ptr->faultIDp.B.OverPhaseCCurrent = 1;

	    // Fault flag:   DC-bus over-voltage detect on FAULT PIN FLEXPWM0/1 FAULT 1
	    if(!ptr->faultIDp.B.OverDCBusVoltage)
	    	ptr->faultIDp.B.OverDCBusVoltage 	 = get_flexpwm_fault_flag(FLEXPWM_FAULT_1, ptr->motorID);

	    // Fault flag:   DC-bus under-voltage - s/w comparator
	    if(!ptr->faultIDp.B.UnderDCBusVoltage)
	    	ptr->faultIDp.B.UnderDCBusVoltage 	= (ptr->uDcbFbck.raw < U_DCB_UNDER) ? TRUE : FALSE;

	    // Read Fault from MC33937 pre-driver - performed in MC33937_interrupt_ISR()

	    faultDetectiontEvent = (ptr->faultIDp.R ) ? TRUE : FALSE;
	    faultDetectiontEvent |= (ptr->faultID.R ) ? TRUE : FALSE;

	return (faultDetectiontEvent);
}

/***************************************************************************//*!
*
* @brief   Motor control variables measurement
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
static tBool getMotorControlVariables(pmsmFOC_t *ptr)
{
	// Read DC-bus voltage (with scale U_DCB_MAX)
	DcbVoltage_get_data(&ptr->MPC5744P_HW.uDcb ,&ptr->uDcbFbck, U_DCB_MAX);

	// Read 3-phase phase currents (with scale I_MAX)
	PhCurrent3Ph_get_data(&ptr->MPC5744P_HW.phCurrents ,&ptr->iAbcFbck, I_MAX);

	// Read resolver data - sin/cos inputs
	Resolver_get_data(&ptr->MPC5744P_HW.Resolver, &ptr->Resolver);

	// Calculate ATO for resolver
	ATO_calculation(&ptr->Resolver_SW);
	// Get the resolver rotor position and speed from ATO
	ATO_get_position_el(&ptr->Resolver_SW);
	ATO_get_w_speed_el(&ptr->Resolver_SW);
	// Calculate the input angle error for ATO
	ATO_theta_err_calc_resolver(&ptr->Resolver_SW, &ptr->Resolver);

	// Close the control loop with the filtered speed from resolevr_ATO
	ptr->controlLoop.wRotEl		= ptr->Resolver_SW.wRotEl.filt;
	// Control loop position for Park transforms from Resolver_SW
	ptr->controlLoop.thRotEl	= ptr->Resolver_SW.fltThRotEl;

	return (TRUE);
}

/***************************************************************************//*!
*
* @brief   Field Oriented Control - slow loop calculations
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
static tBool focSlowLoop(pmsmFOC_t *ptr)
{
    ptr->controlLoop.wRotElReqRamp    	= GFLIB_Ramp(ptr->controlLoop.wRotElReq,&(ptr->controlLoop.speedRamp));

    // Speed FO control mode from MCAT
    if(ptr->cntrState.controlMode==speedCtrl)
    {
		ptr->controlLoop.wRotElErr      = MLIB_Sub(ptr->controlLoop.wRotElReqRamp, ptr->controlLoop.wRotEl);
		ptr->iDQReq.fltArg2             = GFLIB_ControllerPIpAW(ptr->controlLoop.wRotElErr, &(ptr->speedPIp));
    }

	return (TRUE);
}

/***************************************************************************//*!
*
* @brief   Field Oriented Control - fast (current) loop calculations
*
* @param   pointer to structure of pmsmFOC_t type (defined in PMSM_struct.h)
*
* @return  none
*
******************************************************************************/
static tBool focFastLoop(pmsmFOC_t *ptr)
{
	// Transform the abc current into AlphaBeta frame
    GMCLIB_Clark(&ptr->iAlBeFbck,&ptr->iAbcFbck.filt);

    // Application Cascade Control Structure - controlled by MCAT
	switch(ptr->cntrState.controlMode){

		// Scalar (V/f) control mode - open loop control
		case scalarCtrl:
			// ramp in reference speed feed-forward path
			ptr->scalarControl.wRotElReqRamp = GFLIB_Ramp(ptr->scalarControl.wRotElReq,&(ptr->scalarControl.wRotReqRamp));

			// generated electrical position in Frac32 format
			ptr->scalarControl.thScalarEl  = GFLIB_IntegratorTR_F32(MLIB_ConvertPU_F32FLT(MLIB_Div(ptr->scalarControl.wRotElReqRamp, ptr->scalarControl.wElMax)), &(ptr->scalarControl.integ));

			// Required voltage = VHzRatio * Required Frequency (wRotEl required)
			ptr->scalarControl.UmReq	   = MLIB_Mul(ptr->scalarControl.VHzRatioReq,ptr->scalarControl.wRotElReqRamp);

			// transformation angle
		    // Function performs Sin(angle) and Cos (angle), angle = ptr->scalarControl.thScalarEl;
		    GFLIB_SinCos (MLIB_Mul(MLIB_ConvertPU_FLTF32(ptr->scalarControl.thScalarEl),FLOAT_PI), &ptr->thTransform);

			ptr->uDQReq.fltArg1          	= 0;
			ptr->uDQReq.fltArg2          	= ptr->scalarControl.UmReq;

			break;

		// Voltage FO control mode
		case voltageCtrl:
			if(ptr->uDQReq.fltArg1)	{
			    // Function performs Sin(angle) and Cos (angle), angle = 0;
			    GFLIB_SinCos (0.0F, &ptr->thTransform);
			}
			else{
			    // Function performs Sin(angle) and Cos (angle), angle = ptr->controlLoop.thRotEl;
			    GFLIB_SinCos (ptr->controlLoop.thRotEl, &ptr->thTransform);
			}

			GMCLIB_Park(&ptr->iDQFbck, &ptr->thTransform, &ptr->iAlBeFbck);
			break;

		// Current FO control mode
		case currentCtrl:

		// Speed FO control mode
		case speedCtrl:
			// Function performs Sin(angle) and Cos (angle), angle = ptr->controlLoop.thRotEl;
			GFLIB_SinCos (ptr->controlLoop.thRotEl, &ptr->thTransform);

			GMCLIB_Park(&ptr->iDQFbck, &ptr->thTransform, &ptr->iAlBeFbck);

			ptr->iDQReqZC.fltArg1        	= GDFLIB_FilterIIR1(ptr->iDQReq.fltArg1,&(ptr->dAxisZC));
			ptr->iDQReqZC.fltArg2        	= GDFLIB_FilterIIR1(ptr->iDQReq.fltArg2,&(ptr->qAxisZC));

			ptr->iDQErr.fltArg1          	= MLIB_Sub(ptr->iDQReqZC.fltArg1,ptr->iDQFbck.fltArg1);
			ptr->iDQErr.fltArg2          	= MLIB_Sub(ptr->iDQReqZC.fltArg2,ptr->iDQFbck.fltArg2);

			// 95% of available DCbus recalculated to phase voltage = 0.95*uDCB/sqrt(3)
			ptr->dAxisPIp.fltUpperLimit   	= MLIB_Mul(0.54848275573F, ptr->uDcbFbck.filt);
			ptr->dAxisPIp.fltLowerLimit   	= MLIB_Neg(ptr->dAxisPIp.fltUpperLimit);
			ptr->uDQReq.fltArg1          	= GFLIB_ControllerPIpAW(ptr->iDQErr.fltArg1,&ptr->dAxisPIp);

			ptr->qAxisPIp.fltUpperLimit		= MLIB_Mul(0.54848275573F, ptr->uDcbFbck.filt);
			ptr->qAxisPIp.fltLowerLimit		= MLIB_Neg(ptr->qAxisPIp.fltUpperLimit);
			ptr->uDQReq.fltArg2				= GFLIB_ControllerPIpAW(ptr->iDQErr.fltArg2, &ptr->qAxisPIp);
			break;
	}

	// DQ required voltages transformed to AlphaBeta frame
    GMCLIB_ParkInv(&ptr->uAlBeReq,&ptr->thTransform,&ptr->uDQReq);

	return (TRUE);
}

/*
 *######################################################################
 *                           End of File
 *######################################################################
*/

