/**
  ******************************************************************************
  * @file    flash.c
  * @author  YANDLD
  * @version V3.0.0
  * @date    2016.6.6
  * @brief   www.beyondcore.net   http://upcmcu.taobao.com 
  ******************************************************************************
  */
#include <string.h>

#include "flash.h"
#include "common.h"

unsigned long CCLK = 12000;            // CCLK in kHz

struct sIAP
{                  // IAP Structure
    unsigned long cmd;           // Command
    unsigned long par[4];        // Parameters
    unsigned long stat;          // Status
    unsigned long res[2];        // Result
} IAP;

/* IAP Call */
typedef void (*IAP_Entry) (unsigned long *cmd, unsigned long *stat);
#define IAP_Call ((IAP_Entry) 0x0F001FF1)


#define PHY_PAGE_SIZE       (64)       /* actual flash page size */
#define PAGE_COUNT      (LOG_PAGE_SIZE/PHY_PAGE_SIZE)
#define SECTOR_SIZE         (1*1024)


unsigned long GetSecNum (unsigned long adr)
{
    unsigned long n;

    n = adr / SECTOR_SIZE;
    return (n);
}

 /**
 * @brief  С
 * @note   None
 * @param  None
 * @retval Flashߴ
 */
uint32_t FLASH_GetSectorSize(void)
{
    return SECTOR_SIZE;
}

uint32_t FLASH_GetPageSize(void)
{
    return PHY_PAGE_SIZE;
}

 /**
 * @brief  ʼFlash
 * @note   None
 * @param  None
 * @retval Flashߴ
 */
void FLASH_Init(void)
{
    CCLK = GetClock(kCoreClock)/1000;
}

 /**
 * @brief  Flash
 * @note   ùܽɾһFlash
 * @param  addr: ʼַ
 * @retval ز
 */
uint8_t FLASH_EraseSector(uint32_t addr)
{
    unsigned long n;

    n = GetSecNum(addr);                          // Get Sector Number

    IAP.cmd    = 50;                             // Prepare Sector for Erase
    IAP.par[0] = n;                              // Start Sector
    IAP.par[1] = n;                              // End Sector
    __disable_irq();
    IAP_Call (&IAP.cmd, &IAP.stat);              // Call IAP Command
    __enable_irq();
    if (IAP.stat) return (1);                    // Command Failed

    IAP.cmd    = 52;                             // Erase Sector
    IAP.par[0] = n;                              // Start Sector
    IAP.par[1] = n;                              // End Sector
    IAP.par[2] = CCLK;                           // CCLK in kHz
    __disable_irq();
    IAP_Call (&IAP.cmd, &IAP.stat);              // Call IAP Command
    __enable_irq();
    if (IAP.stat) return (1);                    // Command Failed

    return (0);                                  // Finished without Errors
}

uint32_t FLASH_GetUID(void)
{
#if defined(LPC54114)
    uint32_t uid0, uid1, uid2;
    /* from 0x0100_0100 to 0x0100_010C */
    uid0 = *(uint32_t *)(0x01000100);
    uid1 = *(uint32_t *)(0x01000104);
    uid2 = *(uint32_t *)(0x01000108);
    return uid0 ^ uid1 ^ uid2;
#endif
    return 0;
}

uint8_t FLASH_ErasePage(uint32_t addr)
{
    unsigned long n;
    unsigned long page;
    
    n = GetSecNum(addr);                          // Get Sector Number

    IAP.cmd    = 50;                             // Prepare Sector for Erase
    IAP.par[0] = n;                              // Start Sector
    IAP.par[1] = n;                              // End Sector
    __disable_irq();
    IAP_Call (&IAP.cmd, &IAP.stat);              // Call IAP Command
    __enable_irq();
    if (IAP.stat) return (1);                    // Command Failed
    
    page = addr/PHY_PAGE_SIZE;
    
    IAP.cmd    = 59;
    IAP.par[0] = page;
    IAP.par[1] = page;
    IAP.par[2] = CCLK;
    __disable_irq();
    IAP_Call (&IAP.cmd, &IAP.stat);
    __enable_irq();
    if (IAP.stat) return (1);

    return (0);
}


 /**
 * @brief  write a flash page
 * @note   
 * @param  addr: start address, must be align with 256 bytes
 * @param  buf : buf pointer
 * @param  len : len of buffer
 * @retval CH_OK or CH_ERR
 */
uint8_t FLASH_WritePage(uint32_t addr, const uint8_t *buf)
{
    unsigned long n;

#if SET_VALID_CODE != 0                        // Set valid User Code Signature
  if (adr == 0) {                              // Check for Vector Table
    n = *((unsigned long *)(buf + 0x00)) +
        *((unsigned long *)(buf + 0x04)) +
        *((unsigned long *)(buf + 0x08)) +
        *((unsigned long *)(buf + 0x0C)) +
        *((unsigned long *)(buf + 0x10)) +
        *((unsigned long *)(buf + 0x14)) +
        *((unsigned long *)(buf + 0x18));
    *((unsigned long *)(buf + 0x1C)) = 0 - n;  // Signature at Reserved Vector
  }
#endif

    n = GetSecNum(addr);                          // Get Sector Number
  
    IAP.cmd    = 50;                             // Prepare Sector for Write
    IAP.par[0] = n;                              // Start Sector
    IAP.par[1] = n;                              // End Sector
    __disable_irq();
    IAP_Call (&IAP.cmd, &IAP.stat);              // Call IAP Command
    __enable_irq();
    if (IAP.stat) return (1);                    // Command Failed

    IAP.cmd    = 51;                             // Copy RAM to Flash
    IAP.par[0] = addr;                            // Destination Flash Address
    IAP.par[1] = (unsigned long)buf;             // Source RAM Address
    IAP.par[2] = PHY_PAGE_SIZE;                 // Fixed Page Size
    IAP.par[3] = CCLK;                           // CCLK in kHz
    __disable_irq();
    IAP_Call (&IAP.cmd, &IAP.stat);              // Call IAP Command
    __enable_irq();
    if (IAP.stat) return (1);                    // Command Failed

    return CH_OK;
}

static uint32_t FLASH_PageTest(uint32_t addr)
{
    int i;
    uint8_t *p;
    ALIGN(8) uint8_t buf[PHY_PAGE_SIZE];
    

    FLASH_ErasePage(addr);
    memset(buf, 0, sizeof(buf));
    
    FLASH_WritePage(addr, buf);
    
    for(i=0; i<sizeof(buf); i++)
    {
        buf[i] = i % 0xFF;
    }
    
    uint32_t ret;
    ret = FLASH_ErasePage(addr);
    if(ret)
    {
        printf("FLASH_ErasePage failed\r\n");
    }
    
    ret = FLASH_WritePage(addr, buf);
    if(ret)
    {
        printf("FLASH_WritePage failed\r\n");
    }
    
    p = (uint8_t*)(addr);
    for(i=0; i<sizeof(buf); i++)
    {
        if(p[i] != (i%0xFF))
        {
            ret++;
            printf("ERR:[%d]:0x%02X ", i, *p); 
            return CH_ERR;
        }
    }
    printf("page test: 0x%X ok\r\n", addr);
    return CH_OK;
}

uint32_t FLASH_Test(uint32_t addr, uint32_t len)
{
    int  ret;
    FLASH_Init();
    uint32_t start = addr;
    
    while(addr <= start + len)
    {
        ret = FLASH_PageTest(addr);
        if(ret)
        {
            return ret;
        }
        addr += FLASH_GetPageSize();
    }
    return ret;
}


#define TICK_BEGIN  SysTick->LOAD = 0xFFFFFF;SysTick->VAL = 0; time = SysTick->VAL;
#define TICK_END    time = time - SysTick->VAL;

void app_flash_test(void)
{
    static uint8_t buf[64];
    FLASH_Init();
    FLASH_Test(4*1024, 12*1024);

    LIB_TRACE("flash performance test\r\n");
    
    uint32_t time;
    
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk | SysTick_CTRL_CLKSOURCE_Msk;



    TICK_BEGIN;
    FLASH_EraseSector(4*1024);
    TICK_END;
    
    printf("erase sector time:%d us\r\n", (time)/(15));
    
    
    TICK_BEGIN;
    FLASH_ErasePage(4*1024);
    TICK_END;
    
    printf("erase page time:%d us\r\n", (time)/(15));
    
    
    TICK_BEGIN;
    FLASH_WritePage(4*1024, buf);
    TICK_END;
    
    printf("write page time:%d us\r\n", (time)/(15));
    
}
