/*
 * The Clear BSD License
 * Copyright (c) 2018, NXP Semiconductors, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer below) provided
 *  that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * NO EXPRESS OR IMPLIED LICENSES TO ANY PARTY'S PATENT RIGHTS ARE GRANTED BY THIS LICENSE.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_twiddlefactor.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief MATLAB script for calculating twiddle factor table for DCT is below,this gives phasor for FFTs twiddle at end
 * of DCT,
 * note that y is raw, scaled y is for fixed point multiplication.
 *
 * N=32;
 * n=0;
 * while(n<N)
 *    y(n+1)= exp(-1*i*n*(pi/(2*N)));
 *
 *    if (n==0)
 *       y(n+1) = y(n+1) * sqrt(1/N);
 *    else
 *       y(n+1) = y(n+1) * sqrt(2/N);
 *    end
 *
 *    scaledy(n+1) = y(n+1) * 1024*1024*16;
 *    n=n+1;
 * end
 *
 * Use real(int32(scaledy)) to get co-efficients for real side and imag(int32(scaledy)) to get co-efficients for
 * imaginary side
 * or simply int32(scaledy) to get list of co-efficients for both.
 */

int32_t dct16_twiddle[32] = {4194304, 0,        5903079, -581403,  5817667, -1157206, 5676227, -1721865,
                             5480122, -2269941, 5231241, -2796157, 4931980, -3295444, 4585221, -3762994,
                             4194304, -4194304, 3762994, -4585221, 3295444, -4931980, 2796157, -5231241,
                             2269941, -5480122, 1721865, -5676227, 1157206, -5817667, 581403,  -5903079};

int32_t dct32_twiddle[64] = {
    2965821,  0,        4189252,  -205805,  4174107,  -411114,  4148907,  -615432,  4113712,  -818268,  4068606,
    -1019133, 4013699,  -1217542, 3949122,  -1413018, 3875032,  -1605091, 3791606,  -1793296, 3699046,  -1977181,
    3597575,  -2156303, 3487436,  -2330230, 3368897,  -2498544, 3242241,  -2660838, 3107774,  -2816722, 2965821,
    -2965821, 2816722,  -3107774, 2660838,  -3242241, 2498544,  -3368897, 2330230,  -3487436, 2156303,  -3597575,
    1977181,  -3699046, 1793296,  -3791606, 1605091,  -3875032, 1413018,  -3949122, 1217542,  -4013699, 1019133,
    -4068606, 818268,   -4113712, 615432,   -4148907, 411114,   -4174107, 205805,   -4189252};

int32_t dct64_twiddle[128] = {
    2097152, 0,        2964928, -72785,   2962248, -145526,  2957785, -218179,  2951540, -290701,  2943516, -363048,
    2933720, -435176,  2922157, -507042,  2908833, -578603,  2893758, -649815,  2876939, -720636,  2858387, -791022,
    2838114, -860932,  2816130, -930324,  2792451, -999155,  2767089, -1067384, 2740061, -1134970, 2711382, -1201873,
    2681070, -1268052, 2649143, -1333467, 2615620, -1398078, 2580522, -1461848, 2543869, -1524737, 2505684, -1586707,
    2465990, -1647722, 2424810, -1707744, 2382170, -1766737, 2338094, -1824667, 2292610, -1881497, 2245746, -1937194,
    2197528, -1991724, 2147987, -2045054, 2097152, -2097152, 2045054, -2147987, 1991724, -2197528, 1937194, -2245746,
    1881497, -2292610, 1824667, -2338094, 1766737, -2382170, 1707744, -2424810, 1647722, -2465990, 1586707, -2505684,
    1524737, -2543869, 1461848, -2580522, 1398078, -2615620, 1333467, -2649143, 1268052, -2681070, 1201873, -2711382,
    1134970, -2740061, 1067384, -2767089, 999155,  -2792451, 930324,  -2816130, 860932,  -2838114, 791022,  -2858387,
    720636,  -2876939, 649815,  -2893758, 578603,  -2908833, 507042,  -2922157, 435176,  -2933720, 363048,  -2943516,
    290701,  -2951540, 218179,  -2957785, 145526,  -2962248, 72785,   -2964928};

int32_t dct128_twiddle[256] = {
    1482910, 0,        2096994, -25735,   2096520, -51467,   2095731, -77190,   2094626, -102902,  2093205, -128599,
    2091470, -154276,  2089419, -179930,  2087054, -205557,  2084374, -231153,  2081380, -256714,  2078073, -282236,
    2074453, -307716,  2070521, -333150,  2066277, -358533,  2061722, -383862,  2056856, -409134,  2051680, -434344,
    2046196, -459489,  2040403, -484564,  2034303, -509566,  2027897, -534492,  2021185, -559337,  2014169, -584098,
    2006849, -608771,  1999228, -633352,  1991305, -657838,  1983082, -682225,  1974561, -706509,  1965742, -730687,
    1956628, -754755,  1947218, -778709,  1937516, -802545,  1927521, -826261,  1917237, -849853,  1906663, -873316,
    1895803, -896648,  1884657, -919845,  1873227, -942903,  1861515, -965820,  1849523, -988591,  1837252, -1011213,
    1824705, -1033682, 1811882, -1055997, 1798787, -1078152, 1785421, -1100144, 1771786, -1121971, 1757885, -1143629,
    1743718, -1165115, 1729289, -1186426, 1714600, -1207557, 1699652, -1228507, 1684448, -1249272, 1668991, -1269849,
    1653282, -1290234, 1637325, -1310425, 1621120, -1330419, 1604672, -1350213, 1587982, -1369803, 1571053, -1389187,
    1553887, -1408361, 1536487, -1427324, 1518856, -1446071, 1500996, -1464601, 1482910, -1482910, 1464601, -1500996,
    1446071, -1518856, 1427324, -1536487, 1408361, -1553887, 1389187, -1571053, 1369803, -1587982, 1350213, -1604672,
    1330419, -1621120, 1310425, -1637325, 1290234, -1653282, 1269849, -1668991, 1249272, -1684448, 1228507, -1699652,
    1207557, -1714600, 1186426, -1729289, 1165115, -1743718, 1143629, -1757885, 1121971, -1771786, 1100144, -1785421,
    1078152, -1798787, 1055997, -1811882, 1033682, -1824705, 1011213, -1837252, 988591,  -1849523, 965820,  -1861515,
    942903,  -1873227, 919845,  -1884657, 896648,  -1895803, 873316,  -1906663, 849853,  -1917237, 826261,  -1927521,
    802545,  -1937516, 778709,  -1947218, 754755,  -1956628, 730687,  -1965742, 706509,  -1974561, 682225,  -1983082,
    657838,  -1991305, 633352,  -1999228, 608771,  -2006849, 584098,  -2014169, 559337,  -2021185, 534492,  -2027897,
    509566,  -2034303, 484564,  -2040403, 459489,  -2046196, 434344,  -2051680, 409134,  -2056856, 383862,  -2061722,
    358533,  -2066277, 333150,  -2070521, 307716,  -2074453, 282236,  -2078073, 256714,  -2081380, 231153,  -2084374,
    205557,  -2087054, 179930,  -2089419, 154276,  -2091470, 128599,  -2093205, 102902,  -2094626, 77190,   -2095731,
    51467,   -2096520, 25735,   -2096994};

int32_t dct256_twiddle[512] = {
    1048576, 0,        1482882, -9099,    1482799, -18198,   1482659, -27296,   1482464, -36392,   1482213, -45488,
    1481906, -54582,   1481543, -63674,   1481124, -72763,   1480650, -81850,   1480120, -90933,   1479534, -100013,
    1478892, -109090,  1478195, -118162,  1477442, -127230,  1476634, -136293,  1475770, -145351,  1474850, -154403,
    1473875, -163450,  1472844, -172490,  1471758, -181524,  1470617, -190551,  1469420, -199571,  1468168, -208584,
    1466860, -217588,  1465497, -226585,  1464080, -235572,  1462607, -244551,  1461078, -253521,  1459495, -262481,
    1457857, -271432,  1456164, -280372,  1454417, -289301,  1452614, -298220,  1450757, -307128,  1448845, -316024,
    1446879, -324908,  1444858, -333779,  1442783, -342638,  1440653, -351485,  1438469, -360318,  1436231, -369137,
    1433939, -377943,  1431593, -386734,  1429194, -395511,  1426740, -404273,  1424232, -413020,  1421671, -421751,
    1419057, -430466,  1416389, -439165,  1413667, -447848,  1410893, -456513,  1408065, -465162,  1405185, -473793,
    1402251, -482406,  1399265, -491001,  1396225, -499577,  1393134, -508135,  1389990, -516674,  1386793, -525193,
    1383545, -533692,  1380244, -542171,  1376891, -550630,  1373487, -559068,  1370031, -567485,  1366523, -575881,
    1362963, -584255,  1359353, -592607,  1355691, -600937,  1351978, -609244,  1348215, -617528,  1344400, -625789,
    1340535, -634026,  1336620, -642239,  1332654, -650429,  1328638, -658593,  1324572, -666733,  1320456, -674848,
    1316290, -682938,  1312075, -691001,  1307810, -699039,  1303496, -707051,  1299133, -715035,  1294722, -722993,
    1290261, -730924,  1285752, -738827,  1281194, -746702,  1276589, -754550,  1271935, -762368,  1267233, -770158,
    1262483, -777919,  1257686, -785651,  1252842, -793354,  1247951, -801026,  1243012, -808668,  1238027, -816280,
    1232995, -823861,  1227917, -831411,  1222792, -838930,  1217621, -846417,  1212405, -853872,  1207143, -861295,
    1201835, -868686,  1196483, -876044,  1191085, -883369,  1185642, -890660,  1180155, -897919,  1174623, -905143,
    1169047, -912333,  1163427, -919489,  1157763, -926611,  1152056, -933697,  1146305, -940748,  1140511, -947764,
    1134674, -954744,  1128795, -961689,  1122873, -968597,  1116909, -975468,  1110902, -982303,  1104854, -989101,
    1098764, -995862,  1092633, -1002585, 1086461, -1009270, 1080247, -1015918, 1073994, -1022527, 1067699, -1029097,
    1061365, -1035629, 1054990, -1042122, 1048576, -1048576, 1042122, -1054990, 1035629, -1061365, 1029097, -1067699,
    1022527, -1073994, 1015918, -1080247, 1009270, -1086461, 1002585, -1092633, 995862,  -1098764, 989101,  -1104854,
    982303,  -1110902, 975468,  -1116909, 968597,  -1122873, 961689,  -1128795, 954744,  -1134674, 947764,  -1140511,
    940748,  -1146305, 933697,  -1152056, 926611,  -1157763, 919489,  -1163427, 912333,  -1169047, 905143,  -1174623,
    897919,  -1180155, 890660,  -1185642, 883369,  -1191085, 876044,  -1196483, 868686,  -1201835, 861295,  -1207143,
    853872,  -1212405, 846417,  -1217621, 838930,  -1222792, 831411,  -1227917, 823861,  -1232995, 816280,  -1238027,
    808668,  -1243012, 801026,  -1247951, 793354,  -1252842, 785651,  -1257686, 777919,  -1262483, 770158,  -1267233,
    762368,  -1271935, 754550,  -1276589, 746702,  -1281194, 738827,  -1285752, 730924,  -1290261, 722993,  -1294722,
    715035,  -1299133, 707051,  -1303496, 699039,  -1307810, 691001,  -1312075, 682938,  -1316290, 674848,  -1320456,
    666733,  -1324572, 658593,  -1328638, 650429,  -1332654, 642239,  -1336620, 634026,  -1340535, 625789,  -1344400,
    617528,  -1348215, 609244,  -1351978, 600937,  -1355691, 592607,  -1359353, 584255,  -1362963, 575881,  -1366523,
    567485,  -1370031, 559068,  -1373487, 550630,  -1376891, 542171,  -1380244, 533692,  -1383545, 525193,  -1386793,
    516674,  -1389990, 508135,  -1393134, 499577,  -1396225, 491001,  -1399265, 482406,  -1402251, 473793,  -1405185,
    465162,  -1408065, 456513,  -1410893, 447848,  -1413667, 439165,  -1416389, 430466,  -1419057, 421751,  -1421671,
    413020,  -1424232, 404273,  -1426740, 395511,  -1429194, 386734,  -1431593, 377943,  -1433939, 369137,  -1436231,
    360318,  -1438469, 351485,  -1440653, 342638,  -1442783, 333779,  -1444858, 324908,  -1446879, 316024,  -1448845,
    307128,  -1450757, 298220,  -1452614, 289301,  -1454417, 280372,  -1456164, 271432,  -1457857, 262481,  -1459495,
    253521,  -1461078, 244551,  -1462607, 235572,  -1464080, 226585,  -1465497, 217588,  -1466860, 208584,  -1468168,
    199571,  -1469420, 190551,  -1470617, 181524,  -1471758, 172490,  -1472844, 163450,  -1473875, 154403,  -1474850,
    145351,  -1475770, 136293,  -1476634, 127230,  -1477442, 118162,  -1478195, 109090,  -1478892, 100013,  -1479534,
    90933,   -1480120, 81850,   -1480650, 72763,   -1481124, 63674,   -1481543, 54582,   -1481906, 45488,   -1482213,
    36392,   -1482464, 27296,   -1482659, 18198,   -1482799, 9099,    -1482882};

int32_t dct512_twiddle[1024] = {
    741455,  0,        1048571, -3217,    1048556, -6434,    1048532, -9651,    1048497, -12868,   1048453, -16084,
    1048398, -19301,   1048334, -22517,   1048260, -25733,   1048176, -28949,   1048083, -32165,   1047979, -35380,
    1047865, -38595,   1047742, -41810,   1047609, -45024,   1047466, -48238,   1047313, -51451,   1047150, -54664,
    1046978, -57876,   1046795, -61088,   1046603, -64299,   1046401, -67510,   1046188, -70720,   1045967, -73929,
    1045735, -77138,   1045493, -80346,   1045242, -83553,   1044981, -86759,   1044709, -89965,   1044429, -93170,
    1044138, -96374,   1043837, -99576,   1043527, -102778,  1043207, -105979,  1042877, -109179,  1042537, -112378,
    1042187, -115576,  1041828, -118773,  1041458, -121969,  1041079, -125163,  1040690, -128357,  1040292, -131549,
    1039883, -134740,  1039465, -137930,  1039037, -141118,  1038599, -144305,  1038151, -147491,  1037694, -150675,
    1037227, -153858,  1036750, -157039,  1036263, -160219,  1035767, -163398,  1035261, -166575,  1034745, -169750,
    1034219, -172924,  1033684, -176096,  1033138, -179267,  1032584, -182435,  1032019, -185602,  1031445, -188768,
    1030861, -191931,  1030267, -195093,  1029664, -198253,  1029051, -201411,  1028428, -204567,  1027795, -207721,
    1027153, -210873,  1026502, -214024,  1025840, -217172,  1025169, -220318,  1024488, -223462,  1023798, -226604,
    1023098, -229744,  1022388, -232882,  1021669, -236018,  1020940, -239151,  1020201, -242282,  1019453, -245411,
    1018696, -248537,  1017928, -251661,  1017151, -254783,  1016365, -257903,  1015569, -261020,  1014763, -264134,
    1013948, -267246,  1013124, -270356,  1012289, -273462,  1011446, -276567,  1010592, -279669,  1009730, -282768,
    1008857, -285864,  1007976, -288958,  1007084, -292049,  1006184, -295137,  1005273, -298223,  1004354, -301306,
    1003425, -304386,  1002486, -307463,  1001538, -310537,  1000581, -313608,  999614,  -316676,  998638,  -319741,
    997652,  -322804,  996657,  -325863,  995652,  -328919,  994639,  -331972,  993616,  -335022,  992583,  -338069,
    991541,  -341113,  990490,  -344153,  989429,  -347190,  988360,  -350224,  987281,  -353255,  986192,  -356282,
    985094,  -359306,  983987,  -362326,  982871,  -365343,  981746,  -368357,  980611,  -371367,  979467,  -374374,
    978314,  -377377,  977151,  -380377,  975980,  -383373,  974799,  -386365,  973609,  -389354,  972410,  -392339,
    971202,  -395321,  969984,  -398299,  968758,  -401273,  967522,  -404243,  966278,  -407209,  965024,  -410172,
    963761,  -413131,  962489,  -416085,  961208,  -419036,  959918,  -421983,  958618,  -424926,  957310,  -427865,
    955993,  -430800,  954667,  -433731,  953332,  -436658,  951988,  -439581,  950634,  -442499,  949272,  -445414,
    947901,  -448324,  946522,  -451230,  945133,  -454132,  943735,  -457029,  942328,  -459922,  940913,  -462811,
    939489,  -465696,  938056,  -468576,  936614,  -471452,  935163,  -474323,  933703,  -477190,  932235,  -480052,
    930758,  -482910,  929272,  -485763,  927777,  -488612,  926274,  -491456,  924761,  -494295,  923241,  -497130,
    921711,  -499960,  920173,  -502786,  918626,  -505606,  917071,  -508422,  915506,  -511233,  913934,  -514040,
    912352,  -516841,  910762,  -519638,  909164,  -522430,  907557,  -525216,  905941,  -527998,  904317,  -530775,
    902684,  -533547,  901043,  -536314,  899394,  -539076,  897736,  -541833,  896069,  -544584,  894394,  -547331,
    892711,  -550072,  891019,  -552808,  889319,  -555539,  887610,  -558265,  885893,  -560986,  884168,  -563701,
    882434,  -566411,  880692,  -569115,  878942,  -571815,  877184,  -574509,  875417,  -577197,  873642,  -579880,
    871859,  -582558,  870068,  -585230,  868268,  -587896,  866460,  -590557,  864645,  -593213,  862821,  -595863,
    860988,  -598507,  859148,  -601146,  857300,  -603779,  855443,  -606406,  853579,  -609028,  851706,  -611643,
    849826,  -614254,  847937,  -616858,  846041,  -619456,  844137,  -622049,  842224,  -624636,  840304,  -627217,
    838376,  -629792,  836439,  -632361,  834495,  -634924,  832544,  -637482,  830584,  -640033,  828616,  -642578,
    826641,  -645117,  824658,  -647650,  822667,  -650177,  820669,  -652698,  818662,  -655213,  816648,  -657721,
    814627,  -660224,  812597,  -662720,  810560,  -665210,  808516,  -667693,  806463,  -670171,  804403,  -672642,
    802336,  -675106,  800261,  -677565,  798179,  -680017,  796089,  -682462,  793991,  -684901,  791886,  -687334,
    789774,  -689760,  787654,  -692180,  785526,  -694593,  783392,  -697000,  781250,  -699400,  779100,  -701794,
    776944,  -704181,  774780,  -706561,  772608,  -708935,  770430,  -711302,  768244,  -713662,  766051,  -716015,
    763850,  -718362,  761643,  -720702,  759428,  -723036,  757206,  -725362,  754977,  -727682,  752741,  -729995,
    750498,  -732301,  748248,  -734600,  745991,  -736892,  743726,  -739177,  741455,  -741455,  739177,  -743726,
    736892,  -745991,  734600,  -748248,  732301,  -750498,  729995,  -752741,  727682,  -754977,  725362,  -757206,
    723036,  -759428,  720702,  -761643,  718362,  -763850,  716015,  -766051,  713662,  -768244,  711302,  -770430,
    708935,  -772608,  706561,  -774780,  704181,  -776944,  701794,  -779100,  699400,  -781250,  697000,  -783392,
    694593,  -785526,  692180,  -787654,  689760,  -789774,  687334,  -791886,  684901,  -793991,  682462,  -796089,
    680017,  -798179,  677565,  -800261,  675106,  -802336,  672642,  -804403,  670171,  -806463,  667693,  -808516,
    665210,  -810560,  662720,  -812597,  660224,  -814627,  657721,  -816648,  655213,  -818662,  652698,  -820669,
    650177,  -822667,  647650,  -824658,  645117,  -826641,  642578,  -828616,  640033,  -830584,  637482,  -832544,
    634924,  -834495,  632361,  -836439,  629792,  -838376,  627217,  -840304,  624636,  -842224,  622049,  -844137,
    619456,  -846041,  616858,  -847937,  614254,  -849826,  611643,  -851706,  609028,  -853579,  606406,  -855443,
    603779,  -857300,  601146,  -859148,  598507,  -860988,  595863,  -862821,  593213,  -864645,  590557,  -866460,
    587896,  -868268,  585230,  -870068,  582558,  -871859,  579880,  -873642,  577197,  -875417,  574509,  -877184,
    571815,  -878942,  569115,  -880692,  566411,  -882434,  563701,  -884168,  560986,  -885893,  558265,  -887610,
    555539,  -889319,  552808,  -891019,  550072,  -892711,  547331,  -894394,  544584,  -896069,  541833,  -897736,
    539076,  -899394,  536314,  -901043,  533547,  -902684,  530775,  -904317,  527998,  -905941,  525216,  -907557,
    522430,  -909164,  519638,  -910762,  516841,  -912352,  514040,  -913934,  511233,  -915506,  508422,  -917071,
    505606,  -918626,  502786,  -920173,  499960,  -921711,  497130,  -923241,  494295,  -924761,  491456,  -926274,
    488612,  -927777,  485763,  -929272,  482910,  -930758,  480052,  -932235,  477190,  -933703,  474323,  -935163,
    471452,  -936614,  468576,  -938056,  465696,  -939489,  462811,  -940913,  459922,  -942328,  457029,  -943735,
    454132,  -945133,  451230,  -946522,  448324,  -947901,  445414,  -949272,  442499,  -950634,  439581,  -951988,
    436658,  -953332,  433731,  -954667,  430800,  -955993,  427865,  -957310,  424926,  -958618,  421983,  -959918,
    419036,  -961208,  416085,  -962489,  413131,  -963761,  410172,  -965024,  407209,  -966278,  404243,  -967522,
    401273,  -968758,  398299,  -969984,  395321,  -971202,  392339,  -972410,  389354,  -973609,  386365,  -974799,
    383373,  -975980,  380377,  -977151,  377377,  -978314,  374374,  -979467,  371367,  -980611,  368357,  -981746,
    365343,  -982871,  362326,  -983987,  359306,  -985094,  356282,  -986192,  353255,  -987281,  350224,  -988360,
    347190,  -989429,  344153,  -990490,  341113,  -991541,  338069,  -992583,  335022,  -993616,  331972,  -994639,
    328919,  -995652,  325863,  -996657,  322804,  -997652,  319741,  -998638,  316676,  -999614,  313608,  -1000581,
    310537,  -1001538, 307463,  -1002486, 304386,  -1003425, 301306,  -1004354, 298223,  -1005273, 295137,  -1006184,
    292049,  -1007084, 288958,  -1007976, 285864,  -1008857, 282768,  -1009730, 279669,  -1010592, 276567,  -1011446,
    273462,  -1012289, 270356,  -1013124, 267246,  -1013948, 264134,  -1014763, 261020,  -1015569, 257903,  -1016365,
    254783,  -1017151, 251661,  -1017928, 248537,  -1018696, 245411,  -1019453, 242282,  -1020201, 239151,  -1020940,
    236018,  -1021669, 232882,  -1022388, 229744,  -1023098, 226604,  -1023798, 223462,  -1024488, 220318,  -1025169,
    217172,  -1025840, 214024,  -1026502, 210873,  -1027153, 207721,  -1027795, 204567,  -1028428, 201411,  -1029051,
    198253,  -1029664, 195093,  -1030267, 191931,  -1030861, 188768,  -1031445, 185602,  -1032019, 182435,  -1032584,
    179267,  -1033138, 176096,  -1033684, 172924,  -1034219, 169750,  -1034745, 166575,  -1035261, 163398,  -1035767,
    160219,  -1036263, 157039,  -1036750, 153858,  -1037227, 150675,  -1037694, 147491,  -1038151, 144305,  -1038599,
    141118,  -1039037, 137930,  -1039465, 134740,  -1039883, 131549,  -1040292, 128357,  -1040690, 125163,  -1041079,
    121969,  -1041458, 118773,  -1041828, 115576,  -1042187, 112378,  -1042537, 109179,  -1042877, 105979,  -1043207,
    102778,  -1043527, 99576,   -1043837, 96374,   -1044138, 93170,   -1044429, 89965,   -1044709, 86759,   -1044981,
    83553,   -1045242, 80346,   -1045493, 77138,   -1045735, 73929,   -1045967, 70720,   -1046188, 67510,   -1046401,
    64299,   -1046603, 61088,   -1046795, 57876,   -1046978, 54664,   -1047150, 51451,   -1047313, 48238,   -1047466,
    45024,   -1047609, 41810,   -1047742, 38595,   -1047865, 35380,   -1047979, 32165,   -1048083, 28949,   -1048176,
    25733,   -1048260, 22517,   -1048334, 19301,   -1048398, 16084,   -1048453, 12868,   -1048497, 9651,    -1048532,
    6434,    -1048556, 3217,    -1048571};

/*!
 * @brief MATLAB script for calculating twiddle factor table for IDCT is below.
 *
 * N = 32;
 * Q=exp(pi*1i*((0:(N-1))')/(2*N)); % Generate e^(jnPi/2N) table
 * Q(2:N)=Q(2:N)/sqrt(2);           % scale twiddle by dividing by sqrt(2)
 * int32(Q*1024*1024*16*1/sqrt(1/N))
 *
 * Note that for twiddle table, we can already scale by 1/sqrt(1/N) by running following command
 */

int32_t idct16_twiddle[32] = {67108864, 0,        47224633, 4651220,  46541334, 9257647,  45409817, 13774917,
                              43840978, 18159528, 41849927, 22369252, 39455838, 26363548, 36681768, 30103949,
                              33554432, 33554432, 30103949, 36681768, 26363548, 39455838, 22369252, 41849927,
                              18159528, 43840978, 13774917, 45409817, 9257647,  46541334, 4651220,  47224633};

int32_t idct32_twiddle[64] = {
    94906266, 0,        67028028, 3292876,  66785716, 6577819,  66382512, 9846915,  65819386, 13092290, 65097695,
    16306124, 64219179, 19480675, 63185953, 22608295, 62000506, 25681450, 60665695, 28692737, 59184734, 31634900,
    57561193, 34500851, 55798981, 37283687, 53902345, 39976704, 51875853, 42573413, 49724388, 45067559, 47453133,
    47453133, 45067559, 49724388, 42573413, 51875853, 39976704, 53902345, 37283687, 55798981, 34500851, 57561193,
    31634900, 59184734, 28692737, 60665695, 25681450, 62000506, 22608295, 63185953, 19480675, 64219179, 16306124,
    65097695, 13092290, 65819386, 9846915,  66382512, 6577819,  66785716, 3292876,  67028028};

int32_t idct64_twiddle[128] = {
    134217728, 0,        94877682, 2329116,  94791947, 4656830,  94649113, 6981738,  94449266, 9302441,  94192526,
    11617540,  93879049, 13925641, 93509022, 16225354, 93082668, 18515294, 92600246, 20794080, 92062044, 23060341,
    91468387,  25312712, 90819634, 27549835, 90116174, 29770363, 89358431, 31972958, 88546862, 34156294, 87681956,
    36319055,  86764234, 38459940, 85794248, 40577657, 84772583, 42670932, 83699854, 44738504, 82576707, 46779127,
    81403819,  48791572, 80181897, 50774626, 78911676, 52727096, 77593921, 54647805, 76229427, 56535596, 74819015,
    58389333,  73363535, 60207898, 71863864, 61990195, 70320904, 63735153, 68735586, 65441718, 67108864, 67108864,
    65441718,  68735586, 63735153, 70320904, 61990195, 71863864, 60207898, 73363535, 58389333, 74819015, 56535596,
    76229427,  54647805, 77593921, 52727096, 78911676, 50774626, 80181897, 48791572, 81403819, 46779127, 82576707,
    44738504,  83699854, 42670932, 84772583, 40577657, 85794248, 38459940, 86764234, 36319055, 87681956, 34156294,
    88546862,  31972958, 89358431, 29770363, 90116174, 27549835, 90819634, 25312712, 91468387, 23060341, 92062044,
    20794080,  92600246, 18515294, 93082668, 16225354, 93509022, 13925641, 93879049, 11617540, 94192526, 9302441,
    94449266,  6981738,  94649113, 4656830,  94791947, 2329116,  94877682};

int32_t idct128_twiddle[256] = {
    189812531, 0,         134207622, 1647058,   134177304, 3293868,   134126780, 4940182,   134056057, 6585752,
    133965145, 8230330,   133854059, 9873669,   133722815, 11515520,  133571433, 13155638,  133399935, 14793774,
    133208348, 16429683,  132996700, 18063117,  132765024, 19693831,  132513353, 21321579,  132241727, 22946116,
    131950185, 24567198,  131638772, 26184580,  131307535, 27798018,  130956523, 29407271,  130585790, 31012094,
    130195391, 32612248,  129785385, 34207490,  129355834, 35797580,  128906802, 37382280,  128438358, 38961350,
    127950571, 40534552,  127443515, 42101651,  126917267, 43662408,  126371905, 45216591,  125807513, 46763964,
    125224174, 48304294,  124621977, 49837350,  124001012, 51362901,  123361373, 52880716,  122703156, 54390568,
    122026461, 55892229,  121331389, 57385473,  120618045, 58870075,  119886536, 60345811,  119136973, 61812459,
    118369468, 63269799,  117584138, 64717610,  116781099, 66155675,  115960474, 67583778,  115122385, 69001702,
    114266960, 70409235,  113394326, 71806165,  112504616, 73192281,  111597962, 74567374,  110674503, 75931238,
    109734376, 77283667,  108777724, 78624457,  107804690, 79953407,  106815421, 81270316,  105810066, 82574986,
    104788777, 83867221,  103751707, 85146825,  102699012, 86413607,  101630851, 87667375,  100547385, 88907941,
    99448777,  90135117,  98335192,  91348720,  97206798,  92548565,  96063765,  93734473,  94906266,  94906266,
    93734473,  96063765,  92548565,  97206798,  91348720,  98335192,  90135117,  99448777,  88907941,  100547385,
    87667375,  101630851, 86413607,  102699012, 85146825,  103751707, 83867221,  104788777, 82574986,  105810066,
    81270316,  106815421, 79953407,  107804690, 78624457,  108777724, 77283667,  109734376, 75931238,  110674503,
    74567374,  111597962, 73192281,  112504616, 71806165,  113394326, 70409235,  114266960, 69001702,  115122385,
    67583778,  115960474, 66155675,  116781099, 64717610,  117584138, 63269799,  118369468, 61812459,  119136973,
    60345811,  119886536, 58870075,  120618045, 57385473,  121331389, 55892229,  122026461, 54390568,  122703156,
    52880716,  123361373, 51362901,  124001012, 49837350,  124621977, 48304294,  125224174, 46763964,  125807513,
    45216591,  126371905, 43662408,  126917267, 42101651,  127443515, 40534552,  127950571, 38961350,  128438358,
    37382280,  128906802, 35797580,  129355834, 34207490,  129785385, 32612248,  130195391, 31012094,  130585790,
    29407271,  130956523, 27798018,  131307535, 26184580,  131638772, 24567198,  131950185, 22946116,  132241727,
    21321579,  132513353, 19693831,  132765024, 18063117,  132996700, 16429683,  133208348, 14793774,  133399935,
    13155638,  133571433, 11515520,  133722815, 9873669,   133854059, 8230330,   133965145, 6585752,   134056057,
    4940182,   134126780, 3293868,   134177304, 1647058,   134207622};

int32_t idct256_twiddle[512] = {
    268435456, 0,         189808958, 1164668,   189798239, 2329292,   189780374, 3493828,   189755363, 4658233,
    189723209, 5822462,   189683911, 6986472,   189637472, 8150219,   189583894, 9313659,   189523177, 10476749,
    189455325, 11639444,  189380341, 12801701,  189298226, 13963476,  189208984, 15124725,  189112619, 16285405,
    189009133, 17445472,  188898532, 18604882,  188780819, 19763591,  188655998, 20921556,  188524074, 22078734,
    188385053, 23235080,  188238939, 24390552,  188085737, 25545105,  187925455, 26698696,  187758097, 27851283,
    187583670, 29002821,  187402181, 30153266,  187213637, 31302577,  187018043, 32450709,  186815409, 33597619,
    186605741, 34743264,  186389048, 35887602,  186165337, 37030588,  185934617, 38172180,  185696897, 39312335,
    185452185, 40451009,  185200491, 41588161,  184941825, 42723747,  184676195, 43857724,  184403613, 44990051,
    184124088, 46120683,  183837630, 47249579,  183544252, 48376696,  183243963, 49501992,  182936775, 50625424,
    182622699, 51746950,  182301748, 52866527,  181973933, 53984115,  181639267, 55099669,  181297763, 56213150,
    180949433, 57324514,  180594290, 58433719,  180232347, 59540725,  179863620, 60645489,  179488120, 61747970,
    179105863, 62848126,  178716862, 63945916,  178321133, 65041298,  177918690, 66134232,  177509549, 67224675,
    177093725, 68312588,  176671233, 69397929,  176242089, 70480656,  175806311, 71560731,  175363913, 72638111,
    174914912, 73712756,  174459327, 74784626,  173997173, 75853681,  173528468, 76919880,  173053230, 77983182,
    172571476, 79043549,  172083225, 80100940,  171588496, 81155315,  171087306, 82206634,  170579675, 83254858,
    170065622, 84299948,  169545166, 85341865,  169018326, 86380568,  168485123, 87416018,  167945577, 88448178,
    167399708, 89477008,  166847536, 90502469,  166289082, 91524522,  165724368, 92543130,  165153414, 93558254,
    164576243, 94569855,  163992875, 95577895,  163403333, 96582337,  162807639, 97583143,  162205815, 98580275,
    161597884, 99573696,  160983869, 100563367, 160363794, 101549252, 159737680, 102531315, 159105553, 103509516,
    158467436, 104483821, 157823352, 105454192, 157173326, 106420593, 156517383, 107382987, 155855547, 108341338,
    155187843, 109295610, 154514296, 110245768, 153834932, 111191774, 153149776, 112133594, 152458855, 113071193,
    151762193, 114004534, 151059817, 114933583, 150351754, 115858305, 149638031, 116778666, 148918674, 117694629,
    148193710, 118606161, 147463166, 119513228, 146727071, 120415795, 145985451, 121313829, 145238336, 122207295,
    144485752, 123096160, 143727728, 123980391, 142964293, 124859954, 142195475, 125734815, 141421304, 126604944,
    140641809, 127470305, 139857018, 128330867, 139066962, 129186598, 138271670, 130037465, 137471172, 130883436,
    136665499, 131724480, 135854680, 132560564, 135038746, 133391657, 134217728, 134217728, 133391657, 135038746,
    132560564, 135854680, 131724480, 136665499, 130883436, 137471172, 130037465, 138271670, 129186598, 139066962,
    128330867, 139857018, 127470305, 140641809, 126604944, 141421304, 125734815, 142195475, 124859954, 142964293,
    123980391, 143727728, 123096160, 144485752, 122207295, 145238336, 121313829, 145985451, 120415795, 146727071,
    119513228, 147463166, 118606161, 148193710, 117694629, 148918674, 116778666, 149638031, 115858305, 150351754,
    114933583, 151059817, 114004534, 151762193, 113071193, 152458855, 112133594, 153149776, 111191774, 153834932,
    110245768, 154514296, 109295610, 155187843, 108341338, 155855547, 107382987, 156517383, 106420593, 157173326,
    105454192, 157823352, 104483821, 158467436, 103509516, 159105553, 102531315, 159737680, 101549252, 160363794,
    100563367, 160983869, 99573696,  161597884, 98580275,  162205815, 97583143,  162807639, 96582337,  163403333,
    95577895,  163992875, 94569855,  164576243, 93558254,  165153414, 92543130,  165724368, 91524522,  166289082,
    90502469,  166847536, 89477008,  167399708, 88448178,  167945577, 87416018,  168485123, 86380568,  169018326,
    85341865,  169545166, 84299948,  170065622, 83254858,  170579675, 82206634,  171087306, 81155315,  171588496,
    80100940,  172083225, 79043549,  172571476, 77983182,  173053230, 76919880,  173528468, 75853681,  173997173,
    74784626,  174459327, 73712756,  174914912, 72638111,  175363913, 71560731,  175806311, 70480656,  176242089,
    69397929,  176671233, 68312588,  177093725, 67224675,  177509549, 66134232,  177918690, 65041298,  178321133,
    63945916,  178716862, 62848126,  179105863, 61747970,  179488120, 60645489,  179863620, 59540725,  180232347,
    58433719,  180594290, 57324514,  180949433, 56213150,  181297763, 55099669,  181639267, 53984115,  181973933,
    52866527,  182301748, 51746950,  182622699, 50625424,  182936775, 49501992,  183243963, 48376696,  183544252,
    47249579,  183837630, 46120683,  184124088, 44990051,  184403613, 43857724,  184676195, 42723747,  184941825,
    41588161,  185200491, 40451009,  185452185, 39312335,  185696897, 38172180,  185934617, 37030588,  186165337,
    35887602,  186389048, 34743264,  186605741, 33597619,  186815409, 32450709,  187018043, 31302577,  187213637,
    30153266,  187402181, 29002821,  187583670, 27851283,  187758097, 26698696,  187925455, 25545105,  188085737,
    24390552,  188238939, 23235080,  188385053, 22078734,  188524074, 20921556,  188655998, 19763591,  188780819,
    18604882,  188898532, 17445472,  189009133, 16285405,  189112619, 15124725,  189208984, 13963476,  189298226,
    12801701,  189380341, 11639444,  189455325, 10476749,  189523177, 9313659,   189583894, 8150219,   189637472,
    6986472,   189683911, 5822462,   189723209, 4658233,   189755363, 3493828,   189780374, 2329292,   189798239,
    1164668,   189808958};

int32_t idct512_twiddle[1024] = {
    379625062, 0,         268434193, 823548,    268430403, 1647089,   268424086, 2470614,   268415243, 3294116,
    268403874, 4117587,   268389978, 4941019,   268373556, 5764405,   268354608, 6587736,   268333134, 7411005,
    268309135, 8234205,   268282610, 9057327,   268253560, 9880364,   268221985, 10703307,  268187885, 11526151,
    268151262, 12348885,  268112114, 13171504,  268070442, 13993998,  268026248, 14816361,  267979530, 15638584,
    267930291, 16460660,  267878529, 17282581,  267824246, 18104339,  267767443, 18925927,  267708119, 19747337,
    267646275, 20568561,  267581912, 21389592,  267515030, 22210421,  267445630, 23031041,  267373713, 23851444,
    267299280, 24671623,  267222331, 25491569,  267142866, 26311276,  267060887, 27130735,  266976394, 27949938,
    266889388, 28768879,  266799871, 29587548,  266707842, 30405940,  266613302, 31224045,  266516254, 32041856,
    266416696, 32859365,  266314632, 33676566,  266210060, 34493449,  266102983, 35310008,  265993401, 36126234,
    265881315, 36942120,  265766727, 37757659,  265649638, 38572842,  265530048, 39387662,  265407958, 40202111,
    265283371, 41016182,  265156286, 41829867,  265026706, 42643158,  264894632, 43456048,  264760064, 44268529,
    264623004, 45080593,  264483453, 45892233,  264341413, 46703440,  264196885, 47514209,  264049870, 48324530,
    263900370, 49134396,  263748386, 49943799,  263593919, 50752733,  263436971, 51561189,  263277544, 52369160,
    263115639, 53176637,  262951257, 53983614,  262784400, 54790083,  262615070, 55596037,  262443267, 56401467,
    262268995, 57206366,  262092254, 58010727,  261913046, 58814541,  261731373, 59617802,  261547237, 60420502,
    261360638, 61222634,  261171580, 62024189,  260980063, 62825160,  260786090, 63625540,  260589662, 64425321,
    260390782, 65224495,  260189450, 66023056,  259985670, 66820995,  259779443, 67618306,  259570770, 68414980,
    259359654, 69211009,  259146097, 70006388,  258930101, 70801108,  258711668, 71595161,  258490799, 72388540,
    258267498, 73181238,  258041766, 73973247,  257813604, 74764560,  257583017, 75555169,  257350004, 76345067,
    257114570, 77134246,  256876715, 77922700,  256636443, 78710420,  256393755, 79497399,  256148654, 80283630,
    255901142, 81069105,  255651221, 81853817,  255398894, 82637759,  255144163, 83420922,  254887030, 84203301,
    254627499, 84984887,  254365570, 85765673,  254101248, 86545652,  253834534, 87324817,  253565430, 88103159,
    253293940, 88880672,  253020066, 89657349,  252743810, 90433181,  252465176, 91208163,  252184165, 91982286,
    251900781, 92755543,  251615025, 93527927,  251326901, 94299431,  251036412, 95070047,  250743560, 95839769,
    250448347, 96608588,  250150778, 97376498,  249850854, 98143492,  249548578, 98909562,  249243953, 99674700,
    248936982, 100438901, 248627669, 101202156, 248316015, 101964459, 248002024, 102725802, 247685698, 103486178,
    247367041, 104245580, 247046056, 105004000, 246722746, 105761433, 246397113, 106517870, 246069161, 107273304,
    245738894, 108027729, 245406313, 108781137, 245071422, 109533521, 244734224, 110284874, 244394723, 111035189,
    244052922, 111784459, 243708824, 112532677, 243362431, 113279835, 243013748, 114025927, 242662778, 114770946,
    242309524, 115514885, 241953989, 116257737, 241596176, 116999494, 241236090, 117740150, 240873733, 118479698,
    240509109, 119218130, 240142221, 119955441, 239773073, 120691622, 239401668, 121426668, 239028009, 122160570,
    238652101, 122893323, 238273946, 123624919, 237893549, 124355351, 237510913, 125084613, 237126041, 125812698,
    236738937, 126539598, 236349605, 127265307, 235958048, 127989819, 235564270, 128713125, 235168275, 129435221,
    234770067, 130156098, 234369649, 130875750, 233967025, 131594170, 233562198, 132311351, 233155174, 133027287,
    232745954, 133741971, 232334544, 134455396, 231920948, 135167556, 231505168, 135878443, 231087209, 136588051,
    230667075, 137296374, 230244771, 138003405, 229820298, 138709136, 229393663, 139413562, 228964869, 140116676,
    228533919, 140818471, 228100819, 141518940, 227665572, 142218078, 227228181, 142915876, 226788652, 143612330,
    226346988, 144307432, 225903194, 145001176, 225457274, 145693554, 225009231, 146384562, 224559071, 147074192,
    224106797, 147762437, 223652413, 148449292, 223195925, 149134749, 222737335, 149818802, 222276649, 150501446,
    221813871, 151182673, 221349006, 151862477, 220882056, 152540851, 220413028, 153217790, 219941925, 153893286,
    219468752, 154567334, 218993513, 155239928, 218516213, 155911060, 218036857, 156580724, 217555448, 157248915,
    217071991, 157915626, 216586491, 158580850, 216098952, 159244582, 215609380, 159906814, 215117778, 160567542,
    214624151, 161226759, 214128505, 161884457, 213630842, 162540633, 213131169, 163195278, 212629490, 163848387,
    212125810, 164499954, 211620133, 165149973, 211112464, 165798437, 210602808, 166445340, 210091170, 167090677,
    209577554, 167734442, 209061966, 168376627, 208544410, 169017228, 208024891, 169656238, 207503414, 170293651,
    206979983, 170929461, 206454605, 171563662, 205927284, 172196248, 205398024, 172827214, 204866831, 173456553,
    204333709, 174084259, 203798665, 174710327, 203261702, 175334750, 202722826, 175957523, 202182042, 176578640,
    201639355, 177198094, 201094770, 177815881, 200548292, 178431995, 199999926, 179046428, 199449678, 179659177,
    198897553, 180270234, 198343556, 180879595, 197787692, 181487253, 197229966, 182093203, 196670384, 182697439,
    196108950, 183299956, 195545671, 183900747, 194980551, 184499807, 194413596, 185097131, 193844811, 185692712,
    193274202, 186286546, 192701773, 186878626, 192127530, 187468947, 191551480, 188057504, 190973626, 188644290,
    190393975, 189229301, 189812531, 189812531, 189229301, 190393975, 188644290, 190973626, 188057504, 191551480,
    187468947, 192127530, 186878626, 192701773, 186286546, 193274202, 185692712, 193844811, 185097131, 194413596,
    184499807, 194980551, 183900747, 195545671, 183299956, 196108950, 182697439, 196670384, 182093203, 197229966,
    181487253, 197787692, 180879595, 198343556, 180270234, 198897553, 179659177, 199449678, 179046428, 199999926,
    178431995, 200548292, 177815881, 201094770, 177198094, 201639355, 176578640, 202182042, 175957523, 202722826,
    175334750, 203261702, 174710327, 203798665, 174084259, 204333709, 173456553, 204866831, 172827214, 205398024,
    172196248, 205927284, 171563662, 206454605, 170929461, 206979983, 170293651, 207503414, 169656238, 208024891,
    169017228, 208544410, 168376627, 209061966, 167734442, 209577554, 167090677, 210091170, 166445340, 210602808,
    165798437, 211112464, 165149973, 211620133, 164499954, 212125810, 163848387, 212629490, 163195278, 213131169,
    162540633, 213630842, 161884457, 214128505, 161226759, 214624151, 160567542, 215117778, 159906814, 215609380,
    159244582, 216098952, 158580850, 216586491, 157915626, 217071991, 157248915, 217555448, 156580724, 218036857,
    155911060, 218516213, 155239928, 218993513, 154567334, 219468752, 153893286, 219941925, 153217790, 220413028,
    152540851, 220882056, 151862477, 221349006, 151182673, 221813871, 150501446, 222276649, 149818802, 222737335,
    149134749, 223195925, 148449292, 223652413, 147762437, 224106797, 147074192, 224559071, 146384562, 225009231,
    145693554, 225457274, 145001176, 225903194, 144307432, 226346988, 143612330, 226788652, 142915876, 227228181,
    142218078, 227665572, 141518940, 228100819, 140818471, 228533919, 140116676, 228964869, 139413562, 229393663,
    138709136, 229820298, 138003405, 230244771, 137296374, 230667075, 136588051, 231087209, 135878443, 231505168,
    135167556, 231920948, 134455396, 232334544, 133741971, 232745954, 133027287, 233155174, 132311351, 233562198,
    131594170, 233967025, 130875750, 234369649, 130156098, 234770067, 129435221, 235168275, 128713125, 235564270,
    127989819, 235958048, 127265307, 236349605, 126539598, 236738937, 125812698, 237126041, 125084613, 237510913,
    124355351, 237893549, 123624919, 238273946, 122893323, 238652101, 122160570, 239028009, 121426668, 239401668,
    120691622, 239773073, 119955441, 240142221, 119218130, 240509109, 118479698, 240873733, 117740150, 241236090,
    116999494, 241596176, 116257737, 241953989, 115514885, 242309524, 114770946, 242662778, 114025927, 243013748,
    113279835, 243362431, 112532677, 243708824, 111784459, 244052922, 111035189, 244394723, 110284874, 244734224,
    109533521, 245071422, 108781137, 245406313, 108027729, 245738894, 107273304, 246069161, 106517870, 246397113,
    105761433, 246722746, 105004000, 247046056, 104245580, 247367041, 103486178, 247685698, 102725802, 248002024,
    101964459, 248316015, 101202156, 248627669, 100438901, 248936982, 99674700,  249243953, 98909562,  249548578,
    98143492,  249850854, 97376498,  250150778, 96608588,  250448347, 95839769,  250743560, 95070047,  251036412,
    94299431,  251326901, 93527927,  251615025, 92755543,  251900781, 91982286,  252184165, 91208163,  252465176,
    90433181,  252743810, 89657349,  253020066, 88880672,  253293940, 88103159,  253565430, 87324817,  253834534,
    86545652,  254101248, 85765673,  254365570, 84984887,  254627499, 84203301,  254887030, 83420922,  255144163,
    82637759,  255398894, 81853817,  255651221, 81069105,  255901142, 80283630,  256148654, 79497399,  256393755,
    78710420,  256636443, 77922700,  256876715, 77134246,  257114570, 76345067,  257350004, 75555169,  257583017,
    74764560,  257813604, 73973247,  258041766, 73181238,  258267498, 72388540,  258490799, 71595161,  258711668,
    70801108,  258930101, 70006388,  259146097, 69211009,  259359654, 68414980,  259570770, 67618306,  259779443,
    66820995,  259985670, 66023056,  260189450, 65224495,  260390782, 64425321,  260589662, 63625540,  260786090,
    62825160,  260980063, 62024189,  261171580, 61222634,  261360638, 60420502,  261547237, 59617802,  261731373,
    58814541,  261913046, 58010727,  262092254, 57206366,  262268995, 56401467,  262443267, 55596037,  262615070,
    54790083,  262784400, 53983614,  262951257, 53176637,  263115639, 52369160,  263277544, 51561189,  263436971,
    50752733,  263593919, 49943799,  263748386, 49134396,  263900370, 48324530,  264049870, 47514209,  264196885,
    46703440,  264341413, 45892233,  264483453, 45080593,  264623004, 44268529,  264760064, 43456048,  264894632,
    42643158,  265026706, 41829867,  265156286, 41016182,  265283371, 40202111,  265407958, 39387662,  265530048,
    38572842,  265649638, 37757659,  265766727, 36942120,  265881315, 36126234,  265993401, 35310008,  266102983,
    34493449,  266210060, 33676566,  266314632, 32859365,  266416696, 32041856,  266516254, 31224045,  266613302,
    30405940,  266707842, 29587548,  266799871, 28768879,  266889388, 27949938,  266976394, 27130735,  267060887,
    26311276,  267142866, 25491569,  267222331, 24671623,  267299280, 23851444,  267373713, 23031041,  267445630,
    22210421,  267515030, 21389592,  267581912, 20568561,  267646275, 19747337,  267708119, 18925927,  267767443,
    18104339,  267824246, 17282581,  267878529, 16460660,  267930291, 15638584,  267979530, 14816361,  268026248,
    13993998,  268070442, 13171504,  268112114, 12348885,  268151262, 11526151,  268187885, 10703307,  268221985,
    9880364,   268253560, 9057327,   268282610, 8234205,   268309135, 7411005,   268333134, 6587736,   268354608,
    5764405,   268373556, 4941019,   268389978, 4117587,   268403874, 3294116,   268415243, 2470614,   268424086,
    1647089,   268430403, 823548,    268434193};
