#ifndef _NTAG_I2C_H_
#define _NTAG_I2C_H_


#include <stdint.h>
#include "fsl_common.h"


/*! @brief QN9080SIP NTAG I2C driver version 0.0.1. */
#define FSL_I2C_DRIVER_VERSION (MAKE_VERSION(0, 0, 1))
/*@}*/


/*! @brief QN9080M NTAG I2C reg structure definition. */
typedef struct
{
  __IO uint32_t CR;
  __I  uint32_t SR;
  __O  uint32_t TXD;
  __I  uint32_t RXD;
  __IO uint32_t INT;
} NTAG_I2C_TypeDef;

#define NTAG_I2C   ((NTAG_I2C_TypeDef *)0x4000A000UL)


/*! @brief Error codes for the NTAG driver. */
typedef enum _NTAG_Status
{
    gNtagSuccess_c 						= 0U,  /*!< Execution successful. */
    gNtagInvalidParameters_c 	= 1U,  /*!< Invalid parameters. */
    gNtagNACKReceive_c 				= 2U,  /*!< Execution failure. */
		gNtagConfigurationLocked_c= 3U,  /*!< The NTAG configuration bytes can not be changed via I2C. */
		gNtagI2CProtected_c				= 4U,	 /*!< The NTAG module I2C access was protected. */
		gNtagRFLocked_c					  = 5U,	 /*!< The NTAG module RF locked. */
	  gNtagEEPROMWriteError_c		= 6U,  /*!< HV voltage error during EEPROM write or erase cycle. */
	  gNtagEEPROMWriteBusy_c		= 7U,  /*!< EEPROM write cycle in progress. */
		gNtagFieldDected_c				= 8U	 /*!< NFC field is detected. */
} NTAG_Status_t;


/*! *********************************************************************************
* \brief  Initialize the NTAG I2C as master role.
*
* \param[in\out] - none
*
* \return - none
********************************************************************************** */
void NTAG_I2cInitDevice(void);


/*! *********************************************************************************
* \brief  Enumerate to find a valid NTAG's I2C slave(NTAG) device 
* 
*					Any read to the NTAG's I2C address 
*			 		will only return 0x04 which is manufacturer ID for NXP Semiconductors.
*			 		For the master who want to access the NTAG device enumeration is needed.
*
* \param[in] none
*
* \return - NTAG's I2C address 
********************************************************************************** */
uint8_t NTAG_Enumeration(void);


/*! *********************************************************************************
* \brief send n byte to NTAG via I2C
*
*			 	 For NTAG write operation it should always be in the unit of 16 byte(one block).  			 	
*
* \param[in] address 	- I2C address 
*
* \param[in] *p_data 	- Point to address of the data to be writen to NTAG
*
* \param[in] size 		- The length of data 
*
* \return NTAG_Status_t - gNtagSuccess_c or error code
********************************************************************************** */
NTAG_Status_t NTAG_MemoryWrite(uint8_t address, uint8_t *p_data, uint8_t size);


/*! *********************************************************************************
* \brief  read n byte from NTAG I2C
*
*			 For NTAG read operation it shoud always be in the unit of 16 byte(one block).
*			 Less than 16 bytes read will cause I2C busy until power cycle the system.
*
* \param[in] address 	- I2C block address 
*
* \param[out] *p_data - Point to address of the data to be stored in host.
*
* \param[in] size 		- The length of data to be read
*
* \return NTAG_Status_t - gNtagSuccess_c or error code
********************************************************************************** */
NTAG_Status_t NTAG_MemoryRead(uint8_t address, uint8_t *p_data, uint8_t size);



#endif

