/*! *********************************************************************************
* \addtogroup FSCI_BLE
* @{
********************************************************************************** */
/*!
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
* \file
*
* This is a source file for FSCI BLE management.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/

#if gFsciIncluded_c

#include "fsci_ble.h"

#if gFsciBleHciLayerEnabled_d
    #include "fsci_ble_hci.h"
#endif 

#if gFsciBleL2capLayerEnabled_d
    #include "fsci_ble_l2cap.h"
#endif 

#if gFsciBleL2capCbLayerEnabled_d
    #include "fsci_ble_l2cap_cb.h"
#endif 

#if gFsciBleSmLayerEnabled_d
    #include "fsci_ble_sm.h"
#endif 

#if gFsciBleAttLayerEnabled_d
    #include "fsci_ble_att.h"
#endif 

#if gFsciBleGattLayerEnabled_d
    #include "fsci_ble_gatt.h"
#endif 

#if gFsciBleGattDbAppLayerEnabled_d
    #include "fsci_ble_gatt_db_app.h"
#endif 

#if gFsciBleGattDbAttLayerEnabled_d
    #include "fsci_ble_gatt_db_att.h"
#endif 

#if gFsciBleGapLayerEnabled_d
    #include "fsci_ble_gap.h"
#endif 

/************************************************************************************
*************************************************************************************
* Private constants & macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/

uint32_t fsciBleInterfaceId = 0xFF;             /* Indicates the FSCI interface that 
                                                   will be used for monitoring */

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

void fsciBleRegister(uint32_t fsciInterfaceId)
{
#if gFsciBleHciLayerEnabled_d
    /* Register HCI command handler */
    if(fsciBleRegisterOpGroup(gFsciBleHciOpcodeGroup_c, 
                              fsciBleHciHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleHciLayerEnabled_d */    
 
#if gFsciBleL2capLayerEnabled_d
    /* Register L2CAP command handler */
    if(fsciBleRegisterOpGroup(gFsciBleL2capOpcodeGroup_c, 
                              fsciBleL2capHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleL2capLayerEnabled_d */    

#if gFsciBleL2capCbLayerEnabled_d
    /* Register L2CAP CB command handler */
    if(fsciBleRegisterOpGroup(gFsciBleL2capCbOpcodeGroup_c, 
                              fsciBleL2capCbHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleL2capCbLayerEnabled_d */    

#if gFsciBleSmLayerEnabled_d
    /* Register SM command handler */
    if(fsciBleRegisterOpGroup(gFsciBleSmOpcodeGroup_c, 
                              fsciBleSmHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleSmLayerEnabled_d */    

#if gFsciBleAttLayerEnabled_d
    /* Register ATT command handler */
    if(fsciBleRegisterOpGroup(gFsciBleAttOpcodeGroup_c, 
                              fsciBleAttHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleAttLayerEnabled_d */    

#if gFsciBleGattLayerEnabled_d
    /* Register GATT command handler */
    if(fsciBleRegisterOpGroup(gFsciBleGattOpcodeGroup_c, 
                              fsciBleGattHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleGattLayerEnabled_d */    

#if gFsciBleGattDbAppLayerEnabled_d
    /* Register GATT Database (application) command handler */
    if(fsciBleRegisterOpGroup(gFsciBleGattDbAppOpcodeGroup_c, 
                              fsciBleGattDbAppHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleGattDbAppLayerEnabled_d */    

#if gFsciBleGattDbAttLayerEnabled_d
    /* Register GATT Database (ATT) command handler */
    if(fsciBleRegisterOpGroup(gFsciBleGattDbAttOpcodeGroup_c, 
                              fsciBleGattDbAttHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleGattDbAttLayerEnabled_d */    

#if gFsciBleGapLayerEnabled_d
    /* Register GAP command handler */
    if(fsciBleRegisterOpGroup(gFsciBleGapOpcodeGroup_c, 
                              fsciBleGapHandler, 
                              fsciInterfaceId) != gFsciSuccess_c)
    {
        panic(0, (uint32_t)fsciBleRegister, 0, 0);
    }
#endif /* gFsciBleGapLayerEnabled_d */  
   
    /* Save FSCI interface to be used for monitoring */
    fsciBleInterfaceId = fsciInterfaceId;
}


#if gFsciBleBBox_d || gFsciBleTest_d
void fsciBleStatusMonitor(opGroup_t opCodeGroup, uint8_t opCode, bleResult_t result)
{
    clientPacketStructured_t*   pClientPacket;
    uint8_t*                    pBuffer;

    
    /* Allocate the packet to be sent over UART */
    pClientPacket = fsciBleAllocFsciPacket(opCodeGroup, 
                                           opCode,
                                           sizeof(bleResult_t));
    
    if(NULL == pClientPacket)
    {
        return;
    }
    
    pBuffer = &pClientPacket->payload[0];
    
    /* Set status in the buffer */
    fsciBleGetBufferFromEnumValue(result, pBuffer, bleResult_t);
    
    /* Transmit the packet over UART */
    fsciBleTransmitFormatedPacket(pClientPacket, fsciBleInterfaceId);
}
#endif /* gFsciBleBBox_d || gFsciBleTest_d */


clientPacketStructured_t* fsciBleAllocFsciPacket(opGroup_t opCodeGroup, uint8_t opCode, uint16_t dataSize)
{
    /* Allocate buffer for the FSCI packet (header, data, and CRC) */
    clientPacketStructured_t* pClientPacket = (clientPacketStructured_t*)MEM_BufferAlloc(sizeof(clientPacketHdr_t) + 
                                                                                         dataSize + 
                                                                                         2 * sizeof(uint8_t));
    
    if(NULL == pClientPacket)
    {
        /* Buffer can not be allocated */
        fsciBleError(gFsciOutOfMessages_c, fsciBleInterfaceId);
        return NULL;
    }
    
    /* Create FSCI packet header */
    pClientPacket->header.opGroup   = opCodeGroup;
    pClientPacket->header.opCode    = opCode;
    pClientPacket->header.len       = dataSize;
    
    /* Return the allocated FSCI packet */
    return pClientPacket;
}


void fsciBleNoParamCmdOrEvtMonitor(opGroup_t opCodeGroup, uint8_t opCode)
{
    clientPacketStructured_t* pClientPacket;
    
    /* Allocate the FSCI packet to be transmitted over UART (with FSCI header added) */
    pClientPacket = fsciBleAllocFsciPacket(opCodeGroup, opCode, 0);
    
    if(NULL == pClientPacket)
    {
        /* FSCI packet can not be allocated */
        return;
    }
    
    /* Transmit FSCI packet over UART */
    fsciBleTransmitFormatedPacket(pClientPacket, fsciBleInterfaceId);
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

#endif /* gFsciIncluded_c */

/*! *********************************************************************************
* @}
********************************************************************************** */
