/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * \file
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*****************************************************************************
 *                               INCLUDED HEADERS                            *
 *---------------------------------------------------------------------------*
 * Add to this section all the headers that this module needs to include.    *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
#include "EmbeddedTypes.h"
#include "PWR_Configuration.h"
#include "PWRLib.h"
#include "PWR_Interface.h"
#include "TimersManager.h"
#include "fsl_device_registers.h"
#include "fsl_os_abstraction.h"
#include "board.h"
#include "clock_config.h"
#include "fsl_power.h"
#include "LED.h"
#include "controller_interface.h"

/*****************************************************************************
 *                             PRIVATE MACROS                                *
 *---------------------------------------------------------------------------*
 * Add to this section all the access macros, registers mappings, bit access *
 * macros, masks, flags etc ...                                              *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
#define ICSR (*((volatile uint32_t *)0xE000ED04))

/*****************************************************************************
 *                               PUBLIC VARIABLES                            *
 *---------------------------------------------------------------------------*
 * Add to this section all the variables and constants that have global      *
 * (project) scope.                                                          *
 * These variables / constants can be accessed outside this module.          *
 * These variables / constants shall be preceded by the 'extern' keyword in  *
 * the interface header.                                                     *
 *---------------------------------------------------------------------------*
 *****************************************************************************/

/*****************************************************************************
 *                           PRIVATE FUNCTIONS PROTOTYPES                    *
 *---------------------------------------------------------------------------*
 * Add to this section all the functions prototypes that have local (file)   *
 * scope.                                                                    *
 * These functions cannot be accessed outside this module.                   *
 * These declarations shall be preceded by the 'static' keyword.             *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
#if (cPWR_UsePowerDownMode)
#if (cPWR_EnableDeepSleepMode_1 || cPWR_EnableDeepSleepMode_2 || cPWR_EnableDeepSleepMode_3)
   static void PWR_HandleDeepSleepMode_1_2_3_4(void);
   void PWR_WakeupTimerHandler(void* param);
#endif
#endif /* (cPWR_UsePowerDownMode) */
/*****************************************************************************
 *                        PUBLIC TYPE DEFINITIONS                            *
 *---------------------------------------------------------------------------*
 * Add to this section all the data types definitions: stuctures, unions,    *
 * enumerations, typedefs ...                                                *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
typedef enum 
{
    PWR_Run = 77,
    PWR_Sleep,
    PWR_DeepSleep,
    PWR_Reset,
    PWR_OFF
} PWR_CheckForAndEnterNewPowerState_t;

typedef  void (*pfHandleDeepSleepFunc_t)(void);

/*****************************************************************************
 *                               PRIVATE VARIABLES                           *
 *---------------------------------------------------------------------------*
 * Add to this section all the variables and constants that have local       *
 * (file) scope.                                                             *
 * Each of this declarations shall be preceded by the 'static' keyword.      *
 * These variables / constants cannot be accessed outside this module.       *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
uint8_t mLPMFlag = gAllowDeviceToSleep_c;
uint8_t mLpmXcvrDisallowCnt = 0;

#if (cPWR_UsePowerDownMode)
static uint32_t mPWR_DeepSleepTimeMs = cPWR_DeepSleepDurationMs;
static bool_t mPreventEnterLowPower = FALSE;

#if (cPWR_GENFSK_LL_Enable)
static uint32_t mPWR_AbsoluteWakeupTimeInDsmTicks = mPWR_DSM_InvalidTime_c;
#endif

const pfHandleDeepSleepFunc_t maHandleDeepSleep[] =
{
#if cPWR_EnableDeepSleepMode_1  
    PWR_HandleDeepSleepMode_1_2_3_4,
#else
    (pfHandleDeepSleepFunc_t)0,
#endif  
#if cPWR_EnableDeepSleepMode_2    
    PWR_HandleDeepSleepMode_1_2_3_4,
#else
    (pfHandleDeepSleepFunc_t)0,
#endif  
#if cPWR_EnableDeepSleepMode_3                                                    
    PWR_HandleDeepSleepMode_1_2_3_4,
#else
    (pfHandleDeepSleepFunc_t)0,
#endif  
#if cPWR_EnableDeepSleepMode_3                                                    
    PWR_HandleDeepSleepMode_1_2_3_4,
#else
    (pfHandleDeepSleepFunc_t)0,
#endif 
    (pfHandleDeepSleepFunc_t)0,
    (pfHandleDeepSleepFunc_t)0,
    (pfHandleDeepSleepFunc_t)0,
};

static pfPWRCallBack_t gpfPWR_LowPowerEnterCb;
static pfPWRCallBack_t gpfPWR_LowPowerExitCb;

static uint32_t sysTickCtrl = 0;
#endif /* (cPWR_UsePowerDownMode) */

/*****************************************************************************
 *                             PRIVATE FUNCTIONS                             *
 *---------------------------------------------------------------------------*
 * Add to this section all the functions that have local (file) scope.       *
 * These functions cannot be accessed outside this module.                   *
 * These definitions shall be preceded by the 'static' keyword.              *
 *---------------------------------------------------------------------------*
 *****************************************************************************/

/*---------------------------------------------------------------------------
 * Name: PWR_HandleGenfskDsmEnter
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if ((cPWR_UsePowerDownMode) && (cPWR_GENFSK_LL_Enable))
static bool_t PWR_HandleGenfskDsmEnter(void)
{
    uint32_t currentDsmTime;
    uint32_t dsmOscOffset;
    uint32_t durationInDsmTicks = 0;
    bool_t retVal = FALSE;
    
    currentDsmTime = RSIM->DSM_TIMER;  /* Get the current DSM time. */
    
    /* Compute the remaining sleep period. */
    if (mPWR_AbsoluteWakeupTimeInDsmTicks != mPWR_DSM_InvalidTime_c)
    {
        if (currentDsmTime < mPWR_AbsoluteWakeupTimeInDsmTicks)
        {        
            durationInDsmTicks = mPWR_AbsoluteWakeupTimeInDsmTicks - currentDsmTime;
        }
        else
        {
            durationInDsmTicks = mPWR_DSM_MaxTimeCount_c - currentDsmTime + mPWR_AbsoluteWakeupTimeInDsmTicks;
        }
    }
    
    dsmOscOffset = 1024 << READ_REGISTER_FIELD(RSIM->RF_OSC_CTRL,RSIM_RF_OSC_CTRL_BB_XTAL_READY_COUNT_SEL);
    dsmOscOffset = (dsmOscOffset << 15)/BOARD_BOOTCLOCKRUN_CORE_CLOCK;
    
    if(durationInDsmTicks > dsmOscOffset)
    {
        uint32_t dsmSleep;
        
        RSIM->DSM_OSC_OFFSET = dsmOscOffset;
        dsmSleep = (RSIM->DSM_TIMER + 5) & mPWR_DSM_TimerMask_c;
        RSIM->GEN_SLEEP = dsmSleep;
        RSIM->GEN_WAKE = mPWR_AbsoluteWakeupTimeInDsmTicks;
        GENFSK->DSM_CTRL = GENFSK_DSM_CTRL_GENERIC_FSK_SLEEP_EN_MASK;
        retVal = TRUE;
    }
    
    return retVal;
}
#endif /* ((cPWR_UsePowerDownMode) && (cPWR_GENFSK_LL_Enable)) */


/*---------------------------------------------------------------------------
 * Name: PWR_HandleDeepSleepMode_1
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if (cPWR_UsePowerDownMode && (cPWR_EnableDeepSleepMode_1 || cPWR_EnableDeepSleepMode_2 || cPWR_EnableDeepSleepMode_3 || cPWR_EnableDeepSleepMode_4))
static void PWR_HandleDeepSleepMode_1_2_3_4(void)
{
#if cPWR_BLE_LL_Enable
    uint8_t   power_down_mode = 0xff;
    bool_t    enterLowPower = TRUE;
    
    (void)mPWR_DeepSleepTimeMs; //remove compiler warnings
    
    __disable_irq();

    PWRLib_ClearWakeupReason();
    
    //Try to put BLE in deep sleep mode
    power_down_mode = BLE_sleep();
    if (power_down_mode < kPmPowerDown0)
    {
        enterLowPower = false; // BLE doesn't allow deep sleep
    }
    //no else - enterLowPower is already true
    
    if(enterLowPower)
    {
        NVIC_ClearPendingIRQ(OSC_INT_LOW_IRQn);
        NVIC_EnableIRQ(OSC_INT_LOW_IRQn);
        while (SYSCON_SYS_STAT_OSC_EN_MASK & SYSCON->SYS_STAT) //wait for BLE to enter sleep
        {
            POWER_EnterSleep();
        }
        NVIC_DisableIRQ(OSC_INT_LOW_IRQn);
            
        if(gpfPWR_LowPowerEnterCb != NULL)
        {
            gpfPWR_LowPowerEnterCb();
        }

        /* Disable SysTick counter and interrupt */
        sysTickCtrl = SysTick->CTRL & (SysTick_CTRL_ENABLE_Msk | SysTick_CTRL_TICKINT_Msk);
        SysTick->CTRL &= ~(SysTick_CTRL_ENABLE_Msk | SysTick_CTRL_TICKINT_Msk);
        ICSR |= (1 << 25); // clear PendSysTick bit in ICSR, if set
        
        POWER_EnterPowerDown(0); //Nighty night!
        
        if(gpfPWR_LowPowerExitCb != NULL)
        {
            gpfPWR_LowPowerExitCb();
        }
        
        /* Restore the state of SysTick */
        SysTick->CTRL |= sysTickCtrl;
        
        PWRLib_UpdateWakeupReason();
    }
    
    __enable_irq();
    
#else
    PWRLib_ClearWakeupReason();
#endif /* cPWR_BLE_LL_Enable */
}
#endif /* (cPWR_UsePowerDownMode && cPWR_EnableDeepSleepMode_1) */

/*---------------------------------------------------------------------------
 * Name: PWR_HandleDeepSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if (cPWR_UsePowerDownMode)
static PWRLib_WakeupReason_t PWR_HandleDeepSleep(void)
{
    uint8_t lpMode;
    PWRLib_ClearWakeupReason();
        
    lpMode = PWRLib_GetDeepSleepMode();
    
    if(lpMode)
    {
        if(maHandleDeepSleep[lpMode - 1])
        {
            maHandleDeepSleep[lpMode - 1]();
        }
    }
    
    return PWRLib_MCU_WakeupReason;
}
#endif

/*---------------------------------------------------------------------------
 * Name: PWR_HandleSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if (cPWR_UsePowerDownMode)
static PWRLib_WakeupReason_t PWR_HandleSleep(void)
{
    PWRLib_WakeupReason_t  Res;
    
    Res.AllBits = 0;
    
/*---------------------------------------------------------------------------*/
#if (cPWR_SleepMode == 0)
    return Res;
    
/*---------------------------------------------------------------------------*/
#elif (cPWR_SleepMode == 1)
    /* Radio in autodoze mode by default. MCU in wait mode */
    PWRLib_ClearWakeupReason();
    PWRLib_MCU_Enter_Sleep();
    
    return Res;
/*---------------------------------------------------------------------------*/
#else
#error "*** ERROR: Not a valid cPWR_SleepMode chosen"
#endif
}
#endif /* (cPWR_UsePowerDownMode) */

/*---------------------------------------------------------------------------
 * Name: PWR_SleepAllowed
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if (cPWR_UsePowerDownMode)
static bool_t PWR_SleepAllowed(void)
{
    bool_t state = TRUE;
    
#if cPWR_Zigbee_Enable
    if((PWRLib_GetCurrentZigbeeStackPowerState != StackPS_Sleep) &&  \
        (PWRLib_GetCurrentZigbeeStackPowerState != StackPS_DeepSleep))
    {
        state = FALSE;
    }
#endif
    return state;
}
#endif /* (cPWR_UsePowerDownMode) */

/*---------------------------------------------------------------------------
 * Name: PWR_DeepSleepAllowed
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if (cPWR_UsePowerDownMode)
static bool_t PWR_DeepSleepAllowed(void)
{
    bool_t state = TRUE;
    
    if((PWRLib_GetDeepSleepMode() != 6) && mLpmXcvrDisallowCnt)
    {
        state = FALSE;
    }
    
#if cPWR_Zigbee_Enable  
    if(state)
    {
        if (PWRLib_GetCurrentZigbeeStackPowerState != StackPS_DeepSleep)
        {
            state = FALSE;
        }
    }
    
#endif      
    return state;
}
#endif /* #if (cPWR_UsePowerDownMode)*/

/*---------------------------------------------------------------------------
 * Name: PWR_CheckForAndEnterNewPowerState
 * Description: - 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
#if (cPWR_UsePowerDownMode)
static PWRLib_WakeupReason_t PWR_CheckForAndEnterNewPowerState(PWR_CheckForAndEnterNewPowerState_t NewPowerState)
{
    PWRLib_WakeupReason_t ReturnValue;
    
    ReturnValue.AllBits = 0;
    
    if (NewPowerState == PWR_Run)
    {
        /* ReturnValue = 0; */
    }
    else if(NewPowerState == PWR_OFF)
    {   
        /* Never returns */
        for(;;){}        
    }
    else if(NewPowerState == PWR_Reset)
    {
        /* Never returns */
        PWR_SystemReset();
    }
    
    else if(NewPowerState == PWR_DeepSleep)
    {
        if(PWR_CheckIfDeviceCanGoToSleep() && PWR_DeepSleepAllowed())
        {
            ReturnValue = PWR_HandleDeepSleep();
        }
    } 
    else if(NewPowerState == PWR_Sleep)
    {
        if(PWR_CheckIfDeviceCanGoToSleep() && PWR_SleepAllowed())
        {
            ReturnValue = PWR_HandleSleep();
        }
    }
    else
    {
        /* ReturnValue = FALSE; */
    }
    /* Clear wakeup reason */    
    
    return ReturnValue;
}
#endif /* (cPWR_UsePowerDownMode) */

/*****************************************************************************
 *                             PUBLIC FUNCTIONS                              *
 *---------------------------------------------------------------------------*
 * Add to this section all the functions that have global (project) scope.   *
 * These functions can be accessed outside this module.                      *
 * These functions shall have their declarations (prototypes) within the     *
 * interface header file and shall be preceded by the 'extern' keyword.      *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
 
/*---------------------------------------------------------------------------
 * Name: PWR_Init
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_Init(void)
{
#if (cPWR_UsePowerDownMode)
    PWRLib_Init();
#endif /* (cPWR_UsePowerDownMode) */
}


/*---------------------------------------------------------------------------
 * Name: PWR_SystemReset
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_SystemReset(void)
{
    SCB->AIRCR = ((uint32_t)0x5FA << SCB_AIRCR_VECTKEY_Pos) | SCB_AIRCR_SYSRESETREQ_Msk;
    while(1);
}

/*---------------------------------------------------------------------------
 * Name: PWR_SetDeepSleepTimeInMs
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_SetDeepSleepTimeInMs(uint32_t deepSleepTimeMs)
{
#if (cPWR_UsePowerDownMode)
    if(deepSleepTimeMs == 0) 
    {
        return;
    }
    
    mPWR_DeepSleepTimeMs = deepSleepTimeMs; 
#else
    (void) deepSleepTimeMs;
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_SetDeepSleepTimeInSymbols
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_SetDeepSleepTimeInSymbols(uint32_t deepSleepTimeSymbols)
{
#if (cPWR_UsePowerDownMode)
    if(deepSleepTimeSymbols == 0) 
    {
        return;
    }
    mPWR_DeepSleepTimeMs = (deepSleepTimeSymbols * 16)/1000; 
#else
    (void) deepSleepTimeSymbols;
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_AllowDeviceToSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_AllowDeviceToSleep(void)
{
#if (cPWR_UsePowerDownMode)
    OSA_InterruptDisable();
    
    if(mLPMFlag != 0)
    {
        mLPMFlag--;
    }
    
    OSA_InterruptEnable();
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_DisallowDeviceToSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_DisallowDeviceToSleep(void)
{
#if (cPWR_UsePowerDownMode)
    uint8_t prot;
    
    OSA_InterruptDisable();
    
    prot = mLPMFlag + 1;
    
    if(prot != 0)
    {
        mLPMFlag++;
    }
    
    OSA_InterruptEnable();
#endif /* (cPWR_UsePowerDownMode) */
}
  
/*---------------------------------------------------------------------------
 * Name: PWR_AllowXcvrToSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_AllowXcvrToSleep(void)
{
#if (cPWR_UsePowerDownMode)
    OSA_InterruptDisable();
    
    if(mLpmXcvrDisallowCnt != 0)
    {
        mLpmXcvrDisallowCnt--;
    }
    
    OSA_InterruptEnable();
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_DisallowXcvrToSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_DisallowXcvrToSleep(void)
{
#if (cPWR_UsePowerDownMode)
    uint8_t prot;
    
    OSA_InterruptDisable();
    
    prot = mLpmXcvrDisallowCnt + 1;
    
    if(prot != 0)
    {
        mLpmXcvrDisallowCnt++;
    }
    
    OSA_InterruptEnable();
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_CheckIfDeviceCanGoToSleep
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
bool_t PWR_CheckIfDeviceCanGoToSleep(void)
{
#if cPWR_UsePowerDownMode  
    bool_t returnValue;
    
    OSA_InterruptDisable();
    returnValue = mLPMFlag == 0 ? TRUE : FALSE;
    OSA_InterruptEnable();
    
    if ( FALSE == TMR_AreAllTimersOff() ||
         BLE_get_sleep_mode() < kPmPowerDown0 )
    {
        returnValue = FALSE;
    }
    
    return returnValue;
#else
    return FALSE;
#endif
}

/*---------------------------------------------------------------------------
 * Name: PWR_ChangeDeepSleepMode
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
bool_t PWR_ChangeDeepSleepMode(uint8_t dsMode)
{
    uint8_t result = TRUE;
    
#if (cPWR_UsePowerDownMode)
    if((dsMode > 7) ||
       (maHandleDeepSleep[dsMode - 1] == (pfHandleDeepSleepFunc_t)0)) /* keep this condition last */
    {
        result = FALSE;
    }
    else
    {
        PWRLib_SetDeepSleepMode(dsMode);
    }
#endif /* (cPWR_UsePowerDownMode) */
    
    return result;
} 

/*---------------------------------------------------------------------------
 * Name: PWR_GetDeepSleepMode
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
uint8_t PWR_GetDeepSleepMode(void)
{
#if (cPWR_UsePowerDownMode)
    return PWRLib_GetDeepSleepMode();
#else
    return 0;
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_EnterPowerOff
 * Description: - Radio on Reset, MCU on VLLS1
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_EnterPowerOff(void)
{
#if cPWR_UsePowerDownMode
    OSA_InterruptDisable();
    (void)PWR_CheckForAndEnterNewPowerState(PWR_OFF);
    OSA_InterruptEnable();
#endif    
}

/*---------------------------------------------------------------------------
 * Name: PWR_EnterLowPower
 * Description: - 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
PWRLib_WakeupReason_t PWR_EnterLowPower(void)
{ 
    PWRLib_WakeupReason_t ReturnValue;
    
    ReturnValue.AllBits = 0;
#if cPWR_UsePowerDownMode
    if(mPreventEnterLowPower)
      return ReturnValue;
    
    OSA_InterruptDisable();
        
    ReturnValue = PWR_CheckForAndEnterNewPowerState(PWR_DeepSleep);
    
    OSA_InterruptEnable();
#endif
    return ReturnValue;
}

/*---------------------------------------------------------------------------
 * Name: PWR_RegisterLowPowerEnterCallback
 * Description: - 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_RegisterLowPowerEnterCallback(pfPWRCallBack_t lowPowerEnterCallback)
{
#if (cPWR_UsePowerDownMode)
    gpfPWR_LowPowerEnterCb = lowPowerEnterCallback;
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_RegisterLowPowerExitCallback
 * Description: - 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_RegisterLowPowerExitCallback(pfPWRCallBack_t lowPowerExitCallback)
{
#if (cPWR_UsePowerDownMode)
    gpfPWR_LowPowerExitCb = lowPowerExitCallback;
#endif /* (cPWR_UsePowerDownMode) */
}

/*---------------------------------------------------------------------------
 * Name: PWR_EnterSleep
 * Description: - 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_EnterSleep(void)
{
#if cPWR_UsePowerDownMode
  if(mPreventEnterLowPower)
      return;  
    PWRLib_MCU_Enter_Sleep();
#endif /* cPWR_UsePowerDownMode */
}

void PWR_WakeupTimerHandler(void* param)
{
   PWRLib_MCU_WakeupReason.Bits.FromTMR = 1;
}

/*---------------------------------------------------------------------------
 * Name: PWR_PreventEnterLowPower
 * Description: Forced prevention of entering low-power
 * Parameters: prevent - if TRUE prevents the entering to low-power
 * Return: -
 *---------------------------------------------------------------------------*/
void PWR_PreventEnterLowPower(bool_t prevent)
{
#if (cPWR_UsePowerDownMode)
  mPreventEnterLowPower = prevent;
#endif
}
