/*!
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * \file
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*****************************************************************************
 *                               INCLUDED HEADERS                            *
 *---------------------------------------------------------------------------*
 * Add to this section all the headers that this module needs to include.    *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
#include "EmbeddedTypes.h"
#include "fsl_device_registers.h"
#include "board.h"
#include "PWRLib.h"
#include "PWR_Configuration.h"
#include "TimersManager.h"
#include "fsl_os_abstraction.h"
#include "Keyboard.h"   
#include "GPIO_Adapter.h"
#include "gpio_pins.h"
#include "fsl_power.h"
#include "clock_config.h"
#include "controller_interface.h"

/*****************************************************************************
 *                             PRIVATE MACROS                                *
 *---------------------------------------------------------------------------*
 * Add to this section all the access macros, registers mappings, bit access *
 * macros, masks, flags etc ...                                              *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
#define mLptmrTimoutMaxMs_c    (65535000)

#if (cPWR_BLE_LL_Enable)   
/* LLS3 + BLE DSM wakeup on the first next instant by BLE timer */
#define BLE_LL_CLOCK_CONFIG_LP_MOD_1  (BLE_LL_CLOCK_CONFIG_DSM_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_SYS_CLK_GATE_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_LPO_FREQ_SEL_MASK)

/* LLS3 + BLE DSM wakeup on deepSleepTimeInMs by BLE timer*/
#define BLE_LL_CLOCK_CONFIG_LP_MOD_2  (BLE_LL_CLOCK_CONFIG_DSM_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_DSM_INT_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_SYS_CLK_GATE_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_LPO_FREQ_SEL_MASK)

/* LLS3 + BLE idle wakeup on deepSleepTimeInMs by LPTMR */
#define BLE_LL_CLOCK_CONFIG_LP_MOD_3  (BLE_LL_CLOCK_CONFIG_DSM_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_DSM_INT_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_SYS_CLK_GATE_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_LPO_FREQ_SEL_MASK)

/* VLLS0 + BLE idle wakeup by keyboard */
#define BLE_LL_CLOCK_CONFIG_LP_MOD_4  (BLE_LL_CLOCK_CONFIG_DSM_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_DSM_INT_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_SYS_CLK_GATE_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_LPO_FREQ_SEL_MASK)

/* VLLS2 + BLE idle wakeup by keyboard */
#define BLE_LL_CLOCK_CONFIG_LP_MOD_5  (BLE_LL_CLOCK_CONFIG_DSM_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_DSM_INT_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_SYS_CLK_GATE_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_LPO_FREQ_SEL_MASK)

/* STOP wakeup on deepSleepTimeInMs by LPTMR */
#define BLE_LL_CLOCK_CONFIG_LP_MOD_6  (BLE_LL_CLOCK_CONFIG_DSM_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_SYS_CLK_GATE_EN_MASK | \
                                       BLE_LL_CLOCK_CONFIG_LPO_FREQ_SEL_MASK)
#endif /* (cPWR_BLE_LL_Enable) */

/*****************************************************************************
 *                               PRIVATE VARIABLES                           *
 *---------------------------------------------------------------------------*
 * Add to this section all the variables and constants that have local       *
 * (file) scope.                                                             *
 * Each of this declarations shall be preceded by the 'static' keyword.      *
 * These variables / constants cannot be accessed outside this module.       *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
#if (cPWR_UsePowerDownMode == 1)
static uint8_t mPWRLib_DeepSleepMode = cPWR_DeepSleepMode;
/* LPTMR variables */
const uint8_t maLPModeUseLPTMR[] = {0, 1, 1, 1, 1, 0, 1, 1};
#endif /* (cPWR_UsePowerDownMode == 1) */

/*****************************************************************************
 *                               PUBLIC VARIABLES                            *
 *---------------------------------------------------------------------------*
 * Add to this section all the variables and constants that have global      *
 * (project) scope.                                                          *
 * These variables / constants can be accessed outside this module.          *
 * These variables / constants shall be preceded by the 'extern' keyword in  *
 * the interface header.                                                     *
 *---------------------------------------------------------------------------*
 *****************************************************************************/
/* Zigbee STACK status */ 
volatile PWRLib_WakeupReason_t PWRLib_MCU_WakeupReason;

tmrTimerID_t s_xtal32KTimerID           = gTmrInvalidTimerID_c;
tmrTimerID_t s_SendKeyPressPD1TimerID   = gTmrInvalidTimerID_c;

extern void BleApp_HandleKeys(key_event_t events);

#if (cPWR_UsePowerDownMode == 1)
/*****************************************************************************
 *                           PRIVATE FUNCTIONS PROTOTYPES                    *
 *---------------------------------------------------------------------------*
 * Add to this section all the functions prototypes that have local (file)   *
 * scope.                                                                    *
 * These functions cannot be accessed outside this module.                   *
 * These declarations shall be preceded by the 'static' keyword.             *
 *---------------------------------------------------------------------------*
 *****************************************************************************/

/*****************************************************************************
 *                                PRIVATE FUNCTIONS                          *
 *---------------------------------------------------------------------------*
 * Add to this section all the functions that have local (file) scope.       *
 * These functions cannot be accessed outside this module.                   *
 * These definitions shall be preceded by the 'static' keyword.              *
 *---------------------------------------------------------------------------*
*****************************************************************************/

/*****************************************************************************
 *                             PUBLIC FUNCTIONS                              *
 *---------------------------------------------------------------------------*
 * Add to this section all the functions that have global (project) scope.   *
 * These functions can be accessed outside this module.                      *
 * These functions shall have their declarations (prototypes) within the     *
 * interface header file and shall be preceded by the 'extern' keyword.      *
 *---------------------------------------------------------------------------*
 *****************************************************************************/

#if (cPWR_BLE_LL_Enable)

/*---------------------------------------------------------------------------
 * Name: PWRLib_BLL_ClearInterrupts
 * Description: 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWRLib_BLL_ClearInterrupts(uint16_t bmInterrupts)
{
    BLE_LL_REG(EVENT_CLEAR) |= bmInterrupts;
}
#endif /* (cPWR_BLE_LL_Enable) */

/*---------------------------------------------------------------------------
 * Name: PWRLib_ClearRSIMDsmInt
 * Description: 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/  
bool_t PWRLib_ClearRSIMDsmInt(void)
{
#if cPWR_GENFSK_LL_Enable
    if (RSIM->DSM_CONTROL & RSIM_DSM_CONTROL_GEN_SYSCLK_REQ_INT_MASK)
    {
        RSIM->DSM_CONTROL = RSIM->DSM_CONTROL;
        
        return TRUE;
    }
#endif /* cPWR_GENFSK_LL_Enable */
    
    return FALSE;
}

/*---------------------------------------------------------------------------
 * Name: PWRLib_SetDeepSleepMode
 * Description: 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWRLib_SetDeepSleepMode(uint8_t lpMode)
{ 
    mPWRLib_DeepSleepMode = lpMode;  
}

/*---------------------------------------------------------------------------
 * Name: PWRLib_GetDeepSleepMode
 * Description: 
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
uint8_t PWRLib_GetDeepSleepMode(void)
{
    return mPWRLib_DeepSleepMode;
}

/*---------------------------------------------------------------------------
 * Name: PWRLib_MCUEnter_Sleep
 * Description: Puts the processor into Sleep .

                Mode of operation details:
                 - ARM core enters Sleep Mode
                 - ARM core is clock gated (HCLK = OFF)
                 - peripherals are functional
                
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWRLib_MCU_Enter_Sleep(void)
{
    POWER_EnterSleep();
}
#endif /* #if (cPWR_UsePowerDownMode==1) */

/*---------------------------------------------------------------------------
 * Name: PWRLib_Init
 * Description: -
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWRLib_Init(void)
{
#if (cPWR_UsePowerDownMode == 1)
uint32_t msk, val;

    POWER_Init();

    POWER_EnablePD(kPDRUNCFG_PD_FIR);
    POWER_EnablePD(kPDRUNCFG_PD_FSP);
    POWER_EnablePD(kPDRUNCFG_PD_OSC32M); 

#if defined(CFG_BLE_PRJ)
    #define SYSCON_PMU_CTRL0_OSC_INT_EN_MASK (0x4000000U)
    /* Enable OSC_EN as interrupt and wakeup source */
    SYSCON->PMU_CTRL0 = SYSCON->PMU_CTRL0 | SYSCON_PMU_CTRL0_OSC_INT_EN_MASK;
#endif

#if (defined(BOARD_XTAL1_CLK_HZ) && (BOARD_XTAL1_CLK_HZ == CLK_XTAL_32KHZ))
    s_xtal32KTimerID = TMR_AllocateTimer(); //used for PD1 only
    
    val = SYSCON_PMU_CTRL1_XTAL32K_PDM_DIS(0U)  /* switch on XTAL32K during power down */
          | SYSCON_PMU_CTRL1_RCO32K_PDM_DIS(1U) /* switch off RCO32K during power down */
          | SYSCON_PMU_CTRL1_XTAL32K_DIS(0U)    /* switch on XTAL32K */
          | SYSCON_PMU_CTRL1_RCO32K_DIS(1U);    /* switch off RCO32K at all time */
#else
    val = SYSCON_PMU_CTRL1_XTAL32K_PDM_DIS(1U)  /* switch off XTAL32K during power down */
          | SYSCON_PMU_CTRL1_RCO32K_PDM_DIS(0U) /* switch on RCO32K during power down */
          | SYSCON_PMU_CTRL1_XTAL32K_DIS(1U)    /* switch off XTAL32K at all time */
          | SYSCON_PMU_CTRL1_RCO32K_DIS(0U);    /* switch on RCO32K */
#endif

    msk = SYSCON_PMU_CTRL1_XTAL32K_PDM_DIS_MASK | SYSCON_PMU_CTRL1_RCO32K_PDM_DIS_MASK |
          SYSCON_PMU_CTRL1_XTAL32K_DIS_MASK | SYSCON_PMU_CTRL1_RCO32K_DIS_MASK;

    /* The default setting of capacitive sensor, DAC, ADC and USB PLL's power are disabled.
       User should power on these peripherals when using them. */
    POWER_WritePmuCtrl1(SYSCON, msk, val);
    
#if (USE_RTOS)
    //enable CPU registers retention in power down
    SYSCON->PMU_CTRL0 |= (SYSCON_PMU_CTRL0_RETENTION_EN_MASK);
#endif
    s_SendKeyPressPD1TimerID = TMR_AllocateTimer(); //used for PD1 only
#endif /* (cPWR_UsePowerDownMode == 1) */
}

/*---------------------------------------------------------------------------
 * Name: PWRLib_UpdateWakeupReason
 * Description: Updates wakeup reason when exiting deep sleep
 * Parameters: -
 * Return: -
 *---------------------------------------------------------------------------*/
void PWRLib_UpdateWakeupReason(void)
{
    /* GPIO */
    if (POWER_GpioActiveRequest() == TRUE)
    {
        PWRLib_MCU_WakeupReason.Bits.FromKeyBoard = 1; 
    }
    
    /* BLE */
    if (NVIC_GetPendingIRQ(OSC_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromBLE_LLTimer = 1;
    }
    
    /* Timer - RTC Free Running*/
    if (NVIC_GetPendingIRQ(RTC_FR_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromTMR = 1;
    }
    
    /* RTC Second Interrupt */
    if (NVIC_GetPendingIRQ(RTC_SEC_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromRTC_Sec = 1;
    }
    
    /* Analog comparator 0 */
    if (NVIC_GetPendingIRQ(ACMP0_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromACmp0 = 1;
    }
    
    /* Analog comparator 1 */
    if (NVIC_GetPendingIRQ(ACMP1_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromACmp1 = 1;
    }
    
    /* USART 0 */
    if (NVIC_GetPendingIRQ(FLEXCOMM0_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromUSART0 = 1;
    }
    
    /* USART 1 */
    if (NVIC_GetPendingIRQ(FLEXCOMM1_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromUSART1 = 1;
    }
    
    /* BOD */
    if (NVIC_GetPendingIRQ(BOD_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromBOD = 1;
    }
    
    /* Cap Sense */
    if (NVIC_GetPendingIRQ(CS_WAKEUP_IRQn))
    {
        PWRLib_MCU_WakeupReason.Bits.FromCap_Sense = 1;
    }
}

void PWRLib_ClearWakeupReason(void)
{
    PWRLib_MCU_WakeupReason.Bits.FromKeyBoard = 0; 
    PWRLib_MCU_WakeupReason.Bits.FromBLE_LLTimer = 0;
    PWRLib_MCU_WakeupReason.Bits.FromTMR = 0;
    PWRLib_MCU_WakeupReason.Bits.FromRTC_Sec = 0;
    PWRLib_MCU_WakeupReason.Bits.FromACmp0 = 0;
    PWRLib_MCU_WakeupReason.Bits.FromACmp1 = 0;
    PWRLib_MCU_WakeupReason.Bits.FromUSART0 = 0;
    PWRLib_MCU_WakeupReason.Bits.FromUSART1 = 0;
    PWRLib_MCU_WakeupReason.Bits.FromBOD = 0;
    PWRLib_MCU_WakeupReason.Bits.FromCap_Sense = 0;
    PWRLib_MCU_WakeupReason.Bits.Unused = 0;
}

void PWRLib_xtal32KTimerID_handler(void* param)
{
    CLOCK_AttachClk(kXTAL32K_to_32K_CLK);
    POWER_WritePmuCtrl1(SYSCON, SYSCON_PMU_CTRL1_RCO32K_DIS_MASK, SYSCON_PMU_CTRL1_RCO32K_DIS(1U));
    SYSCON->ANA_CTRL1 = SYSCON->ANA_CTRL1 & ~SYSCON_ANA_CTRL1_X32_SMT_EN_MASK;

    /* Allow BLE sleep */
    BLE_enable_sleep();
}

void PWRLib_Start_32kXTAL_ready_timer(void)
{
    TMR_StartLowPowerTimer(s_xtal32KTimerID, gTmrSingleShotTimer_c, 300, PWRLib_xtal32KTimerID_handler, NULL);
}
