/*!
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* Copyright 2016-2017 NXP
*
* \file
*
* This is the header file for the AES Manager interface.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#ifndef __AES_MANAGER_H__
#define __AES_MANAGER_H__

/*! *********************************************************************************
*************************************************************************************
* Include
*************************************************************************************
********************************************************************************** */
#include "EmbeddedTypes.h"


/*! *********************************************************************************
*************************************************************************************
* Public macros
*************************************************************************************
********************************************************************************** */
#define USE_TASK_FOR_HW_AES		(0)

/* The number of AES interfaces to be used.
0 - means that AES Manager is disabled */
#ifndef gAESManagerMaxNumbers_c
#define gAESManagerMaxNumbers_c       (6) 
#endif

/* AES Manager Task specification */
#ifndef gAESTaskStackSize_c
#define gAESTaskStackSize_c              (1024) /* bytes */
#endif
#ifndef gAESTaskPriority_c
#define gAESTaskPriority_c               (3)
#endif

/*! *********************************************************************************
*************************************************************************************
* Public macros
*************************************************************************************
********************************************************************************** */
#define gAESMgrInvalidIdx_c   (0xFF)

#define ENC_NONE				0
#define ENC_PENDING     1
#define ENC_COMPLETE		2

/*! *********************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
********************************************************************************** */
/* Define the types of AES type */
typedef enum{
	  gAESMGR_None		  					= 0,
	  gAESMGR_ECB_Enc_c						= 1,
	  gAESMGR_ECB_Dec_c						= 2,
	  gAESMGR_ECB_Block_Enc_c			= 3,
		gAESMGR_ECB_Block_Dec_c			= 4,
	  gAESMGR_CTR_Enc_c						= 5,
		gAESMGR_CTR_Dec_c						= 6,
  	gAESMGR_CMAC_Enc_c					= 7,
}AESType_t;


/* AES Manager status codes */
typedef enum{
   gAES_Success_c              = 0,
   gAES_InvalidParameter_c     = 1,
	 gAES_Non_excution           = 2,
	 gAES_MaxNumbersReached_c		 = 3,
   gAES_OsError_c              = 4,
	 gAES_TypeInUse_c						 = 5
}AESStatus_t;


/* Defines the param of encryption  */
typedef struct AES_param
{
		uint8_t								 *pPlain;
    uint8_t								 *pCipher;			
    uint8_t 							 *Key;			    /*16 byte for HW AES128 */
    uint8_t                *CTR_counter;	/*16 byte for HW AES128 */
		uint8_t 							 *pInitVector;  /*16 byte for HW AES128 */
		uint32_t 							 Len;
	  uint32_t 							 Blocks;
}AES_param_t;


/* AES Manager callback type */
typedef void (*pAESCallBack_t)(AES_param_t param);

/* User callback type for AES complete   */
typedef void (*pAESCompleteCallBack_t)( uint8_t *input, uint32_t len );

/*! *********************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
********************************************************************************** */

/*! *********************************************************************************
* \brief   Creates the AESManager's task and initializes internal data structures
*
********************************************************************************** */
void AESManager_Init( void );

/*! *********************************************************************************
* \brief   Initialize a AES type   
*
* \param[in] pAESId    pointer to a location where the encryption Id will be stored
* \param[in] type      the type of the AES: 
*																					gAESMGR_ECB_Enc_c						
*																					gAESMGR_ECB_Dec_c						
*																					gAESMGR_ECB_Block_Enc_c		
*																					gAESMGR_ECB_Block_Dec_c			
*																					gAESMGR_CTR_Enc_c						
*																					gAESMGR_CTR_Dec_c						
*																					gAESMGR_CMAC_Enc_c					
* \return The AES number if success or gAES_MaxNumbersReached_c when reach max numbers.
*
********************************************************************************** */
AESStatus_t AES_Init ( uint8_t *pAESId, AESType_t type );

/*! *********************************************************************************
* \brief   Set a AES encryption/decryption parameter
*
* \param[in] pAESId    AES encryption/decryption type Id 
* \param[in] AES_p     the parameter of the AES HW 
* \return    gAES_InvalidParameter_c if an error occured.
*
********************************************************************************** */
AESStatus_t Set_AES_Parameter( uint8_t pAESId, 
																			AES_param_t AES_p, 
																			pAESCallBack_t AEScallback);

/*! *********************************************************************************
* \brief     start encryption/decryption
*
* \param[in] pAESId   AES encryption/decryption type Id  
*
* \return 	 gAES_InvalidParameter_c and gAES_Non_excution if an error occured.
*
********************************************************************************** */
AESStatus_t AES_Start( uint8_t pAESId );

/*! *********************************************************************************
* \brief     set complete event when finish encryption/decryption
*
* \param[in] pAESId   AES encryption/decryption type Id  
*
* \return    gAES_InvalidParameter_c and gAES_Non_excution if an error occured.
*
********************************************************************************** */
AESStatus_t AES_Complete( uint8_t pAESId );

/*! *********************************************************************************
* \brief     get state of encryption/decryption
*
* \param[in] pAESId   AES encryption/decryption type Id  
*
* \return    encryption/decryption state
*
********************************************************************************** */
uint8_t	Get_AES_State( uint8_t pAESId );

/*! *********************************************************************************
* \brief     set callback function ,when encryption/decryption complete it will be execute
*
* \param[in] pAESId     AES encryption/decryption type Id   
*  
********************************************************************************** */
void	AES_Set_Completion_Callback( uint8_t pAESId, 
																 pAESCompleteCallBack_t cb );



#endif