/*!
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "genfsk_interface_qn.h"
#include "fsl_prop.h"
#include "fsl_dma.h"
#include "fsl_rtc.h"
#include "genfsk_ll_qn.h"
#include "genfsk_utils_qn.h"

#include "fsl_os_abstraction.h"


#include "MemManager.h"
#include "TimersManager.h"
#include "FunctionLib.h"
#include "Flash_Adapter.h"

#include "connectivity_test.h"
#include "fsl_rf.h"

#if gMWS_Enabled_d
#include "MWS.h"
#endif

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*! @brief GENFSK Maximum packet length. */
#define gGENFSK_MaxPacketLength_c     (2047)

/*! @brief GENFSK Maximum RAW packet length. */
#define gGENFSK_MaxRawPacketLength_c     (35)

/*! @brief GENFSK Maximum sync address size. */
#define gGENFSK_MaxSyncAddressSize_c     (4)

/*! @brief GENFSK PB_PARTITION settings for maximum TX packet length. */
#define gGENFSK_PbPartitionMaxTx_c    (1088)

/*! @brief GENFSK PB_PARTITION settings for maximum RX packet length. */
#define gGENFSK_PbPartitionMaxRx_c    (0)

/*! @brief GENFSK default enabled events. */
#define mDefaultEnabledEvents (gGenfskTxEvent | gGenfskRxEvent)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
extern void GENFSK_SwitchToInstance(uint8_t instanceId);

/*! @brief GENFSK RX timeout callback function. */
static void GENFSK_RxTimeoutCallback(void *param);

/*! @brief GENFSK TX delay callback function. */
static void GENFSK_TxTransferCallback(void *param);

/*! @brief GENFSK RX delay callback function. */
static void GENFSK_RxTransferCallback(void *param);

/*! @brief GENFSK RX DMA callback function. */
static void GENFSK_RxDmaCallback(dma_handle_t *handle, void *param, bool transferDone, uint32_t tcds);

/*! @brief GENFSK TX DMA callback function. */
static void GENFSK_TxDmaCallback(dma_handle_t *handle, void *param, bool transferDone, uint32_t tcds);

/*! @brief GENFSK LL Task. */
static void GENFSK_Task(osaTaskParam_t argument);

#if gMWS_Enabled_d
static uint32_t MWS_GENFSK_Callback(mwsEvents_t event);
#endif

/*******************************************************************************
 * Variables
 ******************************************************************************/

OSA_TASK_DEFINE(GENFSK_Task, gGENFSK_TaskPriority_c, 1, gGENFSK_TaskStackSize_c, FALSE);

static uint8_t genfskAlreadyInit = FALSE;

/*! @brief GENFSK default radio configuration. */
static const GENFSK_radio_config_t mDefaultRadioConfig =
{
    .radioMode = gGenfskGfskBt0p5h0p5,
    .dataRate = gGenfskDR1Mbps
};

/*! @brief GENFSK default packet configuration. */
static const GENFSK_packet_config_t mDefaultPacketConfig =
{
    .preambleSizeBytes = 0,
    .lengthBitOrder = gGenfskLengthBitLsbFirst,
    .syncAddrSizeBytes = 2,
};

/*! @brief GENFSK default CRC configuration. */
static const GENFSK_crc_config_t mDefaultCrcConfig =
{
    .crcEnable = gGenfskCrcEnable,
    .crcRecvInvalid = gGenfskCrcRecvInvalid,
    .crcSize = 2,
    .crcStartByte = 0,
    .crcSeed = 0xFFFFFF,
};

static const GENFSK_nwk_addr_match_t mDefaultNwkAddrConfig =
{
    .nwkAddrSizeBytes = 3,
    .nwkAddr = {0xD6, 0xBE, 0x89, 0x89, 0x00},
};


/*! @brief GENFSK local structure initial configuration. */
GENFSK_LocalStruct_t genfskLocal[gGENFSK_InstancesCnt_c];
/*! @brief GENFSK active instance. */
uint8_t mGenfskActiveInstance = gGENFSK_InvalidIdx_c;
/*! @breif GENFSK total number of allocated instances. */
uint8_t mNumberOfAllocatedInstances = 0;
/*! @brief GENFSK RX timeout timer ID. */
genfskTimerId_t rxTimeoutTimer = gGENFSK_InvalidTimerId_c;

osaTaskId_t gGenfskTaskId = 0;
osaEventId_t mGenfskTaskEvent;
tmrTimerID_t gGenfskTimeoutTimerId = gTmrInvalidTimerID_c;
tmrTimerID_t gGenfskTransferTimerId = gTmrInvalidTimerID_c;
tmrTimerID_t gGenfskRxTimerId = gTmrInvalidTimerID_c;

extern const uint8_t gUseRtos_c;

dma_handle_t g_propTxDMAHandle;
dma_handle_t g_propRxDMAHandle;


/*******************************************************************************
 * Code
 ******************************************************************************/

static void GENFSK_RxTimeoutCallback(void *param)
{
    GENFSK_AbortAll();

    if ((genfskLocal[mGenfskActiveInstance].eventNotifyCallback != NULL) && (genfskLocal[mGenfskActiveInstance].enabledEvents & gGenfskRxEvent))
    {
        genfskLocal[mGenfskActiveInstance].eventNotifyCallback(gGenfskRxEvent, gGenfskTimeout);
    }
}
static void GENFSK_TxTransferCallback(void *param)
{
     DMA_StartTransfer(&g_propTxDMAHandle);
}
static void GENFSK_RxTransferCallback(void *param)
{
     DMA_StartTransfer(&g_propRxDMAHandle);
}
static void GENFSK_RxDmaCallback(dma_handle_t *handle, void *param, bool transferDone, uint32_t tcds)
{
    uint32_t eventFlags = 0;

    if(TMR_IsTimerActive(gGenfskTimeoutTimerId))
    {
        TMR_StopTimer(gGenfskTimeoutTimerId);
    }
    if (genfskLocal[mGenfskActiveInstance].enabledEvents & gGenfskRxEvent)
    {
        eventFlags |= gGenfskRxDmaEventFlag_c;
    }
    else
    {
        /* No notification enabled */
        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;
    }
    OSA_EventSet(mGenfskTaskEvent, eventFlags);

}
static void GENFSK_TxDmaCallback(dma_handle_t *handle, void *param, bool transferDone, uint32_t tcds)
{
    uint32_t eventFlags = 0;
    if(TMR_IsTimerActive(gGenfskTimeoutTimerId))
    {
        TMR_StopTimer(gGenfskTimeoutTimerId);
    }
    if (genfskLocal[mGenfskActiveInstance].enabledEvents & gGenfskTxEvent)
    {
        eventFlags |= gGenfskTxDmaEventFlag_c;
    }
    else
    {
        /* No notification enabled */
        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;
    }
    OSA_EventSet(mGenfskTaskEvent, eventFlags);
}

genfskStatus_t GENFSK_Init(void)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (TRUE == genfskAlreadyInit)
    {
        status = gGenfskAlreadyInit_c;
    }
    else
    {
        FLib_MemSet(genfskLocal, 0, sizeof(genfskLocal));

        mNumberOfAllocatedInstances = 0;
        rxTimeoutTimer = gGENFSK_InvalidTimerId_c;

        for (uint8_t i = 0; i < gGENFSK_InstancesCnt_c; i++)
        {
            genfskLocal[i].enabledEvents = mDefaultEnabledEvents;
        }

        mGenfskTaskEvent = OSA_EventCreate(TRUE);

        if (NULL == mGenfskTaskEvent)
        {
            status = gGenfskFail_c;
        }
        else
        {
            PROP_Init(PROP);
            DMA_Init(DMA0);
            DMA_EnableChannel(DMA0, 9);
            DMA_CreateHandle(&g_propRxDMAHandle, DMA0, 9);
            DMA_SetCallback(&g_propRxDMAHandle, GENFSK_RxDmaCallback, NULL);
            DMA_EnableChannel(DMA0, 10);
            DMA_CreateHandle(&g_propTxDMAHandle, DMA0, 10);
            DMA_SetCallback(&g_propTxDMAHandle, GENFSK_TxDmaCallback, NULL);

            /* set interrupt priority */
            NVIC_SetPriority(DMA0_IRQn, gGENFSK_DMA_IsrPrio_c >> (8 - __NVIC_PRIO_BITS));
            NVIC_ClearPendingIRQ(DMA0_IRQn);
            NVIC_EnableIRQ(DMA0_IRQn);

            gGenfskTimeoutTimerId = TMR_AllocateTimer();
            gGenfskTransferTimerId = TMR_AllocateTimer();
            gGenfskRxTimerId = TMR_AllocateTimer();
            if ((gTmrInvalidTimerID_c == gGenfskTimeoutTimerId)||(gTmrInvalidTimerID_c == gGenfskTransferTimerId) || (gTmrInvalidTimerID_c == gGenfskRxTimerId))
            {
                status = gGenfskFail_c;
            }
            gGenfskTaskId = OSA_TaskCreate(OSA_TASK(GENFSK_Task), NULL);

            if (NULL == gGenfskTaskId)
            {
                status = gGenfskFail_c;
            }
            else
            {
                genfskAlreadyInit = TRUE;
            }
        }
    }

    return status;
}

genfskStatus_t GENFSK_AllocInstance(uint8_t *pInstanceId, GENFSK_radio_config_t *radioConfig, GENFSK_packet_config_t *packetConfig, GENFSK_bitproc_t *bitProcConfig)
{
    genfskStatus_t status = gGenfskSuccess_c;
    uint8_t instanceId = gGENFSK_InvalidIdx_c;

    if (pInstanceId == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        for (uint8_t i = 0; i < gGENFSK_InstancesCnt_c; i++)
        {
            if (genfskLocal[i].genfskState == gGENFSK_LL_NoInit)
            {
                genfskLocal[i].genfskState = gGENFSK_LL_Idle;
                instanceId = i;
                break;
            }
        }

        if (instanceId != gGENFSK_InvalidIdx_c)
        {
            /* Enter critical section. */
            OSA_InterruptDisable();

            /* Set current istance as active if it is the first instance initiated. */
            if (mGenfskActiveInstance == gGENFSK_InvalidIdx_c)
            {
                mGenfskActiveInstance = instanceId;
            }

            if (radioConfig == NULL)
            {
                GENFSK_RadioConfig(instanceId, (GENFSK_radio_config_t *)&mDefaultRadioConfig);
            }
            else
            {
                status = GENFSK_RadioConfig(instanceId, radioConfig);
            }

            if (status == gGenfskSuccess_c)
            {
                if (packetConfig == NULL)
                {
                    GENFSK_SetPacketConfig(instanceId, (GENFSK_packet_config_t *)&mDefaultPacketConfig);
                }
                else
                {
                    status = GENFSK_SetPacketConfig(instanceId, packetConfig);
                }

                if (status == gGenfskSuccess_c)
                {
                    if (bitProcConfig == NULL)
                    {
                        GENFSK_SetCrcConfig(instanceId, (GENFSK_crc_config_t *)&mDefaultCrcConfig);
                    }
                    else
                    {
                        if (bitProcConfig->crcConfig == NULL)
                        {
                            GENFSK_SetCrcConfig(instanceId, (GENFSK_crc_config_t *)&mDefaultCrcConfig);
                        }
                        else
                        {
                            status = GENFSK_SetCrcConfig(instanceId, bitProcConfig->crcConfig);
                        }
                    }
                }
            }

            /* GENFSK instance initialization failed. */
            if (status != gGenfskSuccess_c)
            {
                if (mGenfskActiveInstance == instanceId)
                {
                    mGenfskActiveInstance = gGENFSK_InvalidIdx_c;
                }

                genfskLocal[instanceId].genfskState = gGENFSK_LL_NoInit;
                instanceId = gGENFSK_InvalidIdx_c;
            }
            else
            {
                mNumberOfAllocatedInstances++;
            }

            /* Exit critical section. */
            OSA_InterruptEnable();
        }
        else
        {
            status = gGenfskAllocInstanceFailed;
        }

        *pInstanceId = instanceId;
    }

    return status;
}

genfskStatus_t GENFSK_RadioConfig(uint8_t instanceId, GENFSK_radio_config_t *radioConfig)
{
    genfskStatus_t status = gGenfskSuccess_c;
    prop_radio_mode_t radioMode = kPROP_RadioModeBt0p5h0p5;

    if(radioConfig == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyRx)
    {
        status = gGenfskBusyRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyTx)
    {
        status = gGenfskBusyTx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingRx)
    {
        status = gGenfskBusyPendingRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingTx)
    {
        status = gGenfskBusyPendingTx_c;
    }
    else
    {
        switch (radioConfig->radioMode)
        {
        case gGenfskGfskBt0p5h0p5:
            radioMode = kPROP_RadioModeBt0p5h0p5;
            break;
        case gGenfskGfskBt0p5h0p32:
            radioMode = kPROP_RadioModeBt0p5h0p32;
            break;
        case gGenfskGfskBt0p5h0p7:
            radioMode = kPROP_RadioModeBt0p5h0p7;
            break;
        default:
            status = gGenfskInvalidParameters_c;
            break;
        }

        if(status == gGenfskSuccess_c)
        {
            FLib_MemCpy(&genfskLocal[instanceId].radioConfig, radioConfig, sizeof(genfskLocal[instanceId].radioConfig));

            if (mGenfskActiveInstance == instanceId)
            {
                PROP_RadioConfig(PROP, radioMode, (prop_data_rate_t)genfskLocal[instanceId].radioConfig.dataRate);
            }
        }
    }

    return status;
}

genfskStatus_t GENFSK_SetPacketConfig(uint8_t instanceId, GENFSK_packet_config_t *packetConfig)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyRx)
    {
        status = gGenfskBusyRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyTx)
    {
        status = gGenfskBusyTx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingRx)
    {
        status = gGenfskBusyPendingRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingTx)
    {
        status = gGenfskBusyPendingTx_c;
    }
    else if (packetConfig == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (packetConfig->preambleSizeBytes > 7)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (packetConfig->syncAddrSizeBytes > 3)
    {
        status = gGenfskInvalidParameters_c;
    }

    else
    {
        genfskLocal[instanceId].syncAddrSizeBytes = packetConfig->syncAddrSizeBytes + 2;  /* Save the sync address bytes = size + 2 */
        genfskLocal[instanceId].packetConfig.preambleSizeBytes = packetConfig->preambleSizeBytes;
        genfskLocal[instanceId].packetConfig.syncAddrSizeBytes = packetConfig->syncAddrSizeBytes;
        genfskLocal[instanceId].packetConfig.lengthBitOrder = packetConfig->lengthBitOrder;
        if (mGenfskActiveInstance == instanceId)
        {
            PROP_SetPacketConfig(PROP, packetConfig->preambleSizeBytes, packetConfig->syncAddrSizeBytes, (uint8_t)packetConfig->lengthBitOrder);
        }

    }

    return status;
}

genfskStatus_t GENFSK_GetPacketConfig(uint8_t instanceId, GENFSK_packet_config_t *packetConfig)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (packetConfig == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        if (mGenfskActiveInstance == instanceId)
        {
            PROP_GetPacketConfig(PROP, &packetConfig->preambleSizeBytes,&packetConfig->syncAddrSizeBytes,&packetConfig->lengthBitOrder);
        }
    }

    return status;
}

genfskStatus_t GENFSK_SetCrcConfig(uint8_t instanceId, GENFSK_crc_config_t *crcConfig)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyRx)
    {
        status = gGenfskBusyRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyTx)
    {
        status = gGenfskBusyTx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingRx)
    {
        status = gGenfskBusyPendingRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingTx)
    {
        status = gGenfskBusyPendingTx_c;
    }
    else if (crcConfig == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (crcConfig->crcStartByte > 1)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (crcConfig->crcSize > 4)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        genfskLocal[instanceId].crcEnable = crcConfig->crcEnable;
        genfskLocal[instanceId].crcRecvInvalid = crcConfig->crcRecvInvalid;
        if (mGenfskActiveInstance == instanceId)
        {
            if(genfskLocal[instanceId].crcEnable == gGenfskCrcDisable)
            {
                PROP_SetCrcConfig(PROP, 0,crcConfig->crcStartByte,crcConfig->crcSeed);
            }
            else
            {
                PROP_SetCrcConfig(PROP, crcConfig->crcSize,crcConfig->crcStartByte,crcConfig->crcSeed);
            }
        }
    }
    return status;
}

genfskStatus_t GENFSK_GetCrcConfig(uint8_t instanceId, GENFSK_crc_config_t *crcConfig)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (crcConfig == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        PROP_GetCrcConfig(PROP, &crcConfig->crcSize,&crcConfig->crcStartByte,&crcConfig->crcSeed);
        crcConfig->crcEnable = genfskLocal[instanceId].crcEnable;
        crcConfig->crcRecvInvalid = genfskLocal[instanceId].crcRecvInvalid;
    }

    return status;
}

genfskStatus_t GENFSK_FreeInstance(uint8_t instanceId)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        GENFSK_ResetToDefaults(instanceId);
        genfskLocal[instanceId].genfskState = gGENFSK_LL_NoInit;
        mNumberOfAllocatedInstances--;

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

void GENFSK_ResetToDefaults(uint8_t instanceId)
{
    if (instanceId < gGENFSK_InstancesCnt_c)
    {
        if (genfskLocal[instanceId].genfskState != gGENFSK_LL_NoInit)
        {
            /* Enter critical section. */
            OSA_InterruptDisable();

            if (mGenfskActiveInstance == instanceId)
            {
                GENFSK_AbortAll();
            }

            GENFSK_SetPacketConfig(instanceId, (GENFSK_packet_config_t *)&mDefaultPacketConfig);
            GENFSK_SetCrcConfig(instanceId, (GENFSK_crc_config_t *)&mDefaultCrcConfig);
            GENFSK_SetNetworkAddress(instanceId,(GENFSK_nwk_addr_match_t *)&mDefaultNwkAddrConfig);

            genfskLocal[instanceId].genfskState = gGENFSK_LL_Idle;

            /* Exit critical section. */
            OSA_InterruptEnable();
        }
    }
}

genfskStatus_t GENFSK_SetNetworkAddress(uint8_t instanceId, GENFSK_nwk_addr_match_t *nwkAddressSettings)
{
    uint8_t *pTempNwkAddress = (uint8_t *)&nwkAddressSettings->nwkAddr;

    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyRx)
    {
        status = gGenfskBusyRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyTx)
    {
        status = gGenfskBusyTx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingRx)
    {
        status = gGenfskBusyPendingRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingTx)
    {
        status = gGenfskBusyPendingTx_c;
    }
    else if (nwkAddressSettings == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if ((nwkAddressSettings->nwkAddrSizeBytes > 3 ))
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        if (mGenfskActiveInstance == instanceId)
        {
            PROP_SetNetworkAddress(PROP, (uint8_t *)pTempNwkAddress);
        }
    }

    return status;
}

genfskStatus_t GENFSK_GetNetworkAddress(uint8_t instanceId, GENFSK_nwk_addr_match_t *nwkAddressSettings)
{
    uint8_t *pTempNwkAddress = (uint8_t *)&nwkAddressSettings->nwkAddr;

    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if ((nwkAddressSettings == NULL))
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
      PROP_GetNetworkAddress(PROP, (uint8_t *)pTempNwkAddress);
    }

    return status;
}

genfskStatus_t GENFSK_SetChannelNumber(uint8_t instanceId, uint8_t channelNum, genfskRadioType_t radioType)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyRx)
    {
        status = gGenfskBusyRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyTx)
    {
        status = gGenfskBusyTx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingRx)
    {
        status = gGenfskBusyPendingRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingTx)
    {
        status = gGenfskBusyPendingTx_c;
    }
    else if (channelNum > 79)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        genfskLocal[instanceId].channelNumberConfig.channelNumber = channelNum;
        genfskLocal[instanceId].channelNumberConfig.radioType = radioType;
        if (mGenfskActiveInstance == instanceId)
        {
            PROP_SetChannelNumber(PROP, channelNum, (prop_radio_operation_t)radioType);
        }
    }

    return status;
}

uint8_t GENFSK_GetChannelNumber(uint8_t instanceId,genfskRadioType_t *radioType)
{
    uint8_t chanNum = 0xFF;

    if (instanceId < gGENFSK_InstancesCnt_c)
    {
        if (genfskLocal[instanceId].genfskState != gGENFSK_LL_NoInit)
        {
            *radioType = genfskLocal[instanceId].channelNumberConfig.radioType;
            chanNum = (uint8_t)PROP_GetChannelNumber(PROP, (prop_radio_operation_t)genfskLocal[instanceId].channelNumberConfig.radioType);
        }
    }

    return chanNum;
}
genfskStatus_t GENFSK_SetTxPowerLevel(uint8_t instanceId, uint8_t txPowerLevel)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyRx)
    {
        status = gGenfskBusyRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyTx)
    {
        status = gGenfskBusyTx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingRx)
    {
        status = gGenfskBusyPendingRx_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_BusyPendingTx)
    {
        status = gGenfskBusyPendingTx_c;
    }
    else if (txPowerLevel > kTxPowerInvalid -1)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {

        if (mGenfskActiveInstance == instanceId)
        {
            RF_SetTxPowerLevel(SYSCON, (tx_power_t)txPowerLevel);
        }
    }

    return status;
}

uint8_t GENFSK_GetTxPowerLevel(uint8_t instanceId)
{
    uint8_t tempTxPower = 0xFF;

    if (instanceId < gGENFSK_InstancesCnt_c)
    {
        if (genfskLocal[instanceId].genfskState != gGENFSK_LL_NoInit)
        {
            tempTxPower = RF_GetTxPowerLevel(SYSCON);
        }
    }

    return tempTxPower;
}

genfskStatus_t GENFSK_StartTx(uint8_t instanceId, uint8_t *pBuffer, uint16_t bufLengthBytes, GENFSK_timestamp_t txStartTime)
{
    GENFSK_timestamp_t currentTime = 0;

    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (pBuffer == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }

    else if (bufLengthBytes > gGENFSK_MaxPacketLength_c + genfskLocal[instanceId].syncAddrSizeBytes)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        /* If GENFSK LL active sequence is Idle start TX. */
        if (genfskLocal[mGenfskActiveInstance].genfskState == gGENFSK_LL_Idle)
        {
#if gMWS_Enabled_d
            if (gMWS_Success_c == MWS_Acquire(gMWS_GENFSK_c, FALSE))
            {
#endif
                /* Change the active instance */
                if (instanceId != mGenfskActiveInstance)
                {
                    GENFSK_SwitchToInstance(instanceId);
                    mGenfskActiveInstance = instanceId;
                }
                if (gGenfskRadioTypeTx != genfskLocal[instanceId].channelNumberConfig.radioType)
                {

                    GENFSK_SetChannelNumber(instanceId,genfskLocal[instanceId].channelNumberConfig.channelNumber,gGenfskRadioTypeTx);
                }
                /* Start TX now. */
                if (!txStartTime)
                {
                    genfskLocal[instanceId].genfskState = gGENFSK_LL_BusyTx;

                    PROP_SetPayloadLen(PROP,bufLengthBytes - (genfskLocal[instanceId].packetConfig.preambleSizeBytes +1)-(genfskLocal[instanceId].packetConfig.syncAddrSizeBytes +2));

                    PROP_StartTx(PROP);

                    //config DMA .....
                    /* Transmit frame by DMA */
                    dma_transfer_config_t dmaConfig;
                    DMA_PrepareTransfer(&dmaConfig, (void *) pBuffer, (void *) &PROP->TX_BUF, 1, bufLengthBytes, kDMA_MemoryToPeripheral, NULL);
                    DMA_SubmitTransfer(&g_propTxDMAHandle, &dmaConfig);
                    DMA_StartTransfer(&g_propTxDMAHandle);
                }
                else
                {

                    currentTime = GENFSK_GetTimestamp();
                    if (txStartTime <= currentTime + gGENFSK_MinSetupTime_c)
                    {
                        status = gGenfskInstantPassed_c;
                    }
                    else
                    {
                        genfskLocal[instanceId].genfskState = gGENFSK_LL_BusyPendingTx;

                        PROP_SetPayloadLen(PROP,bufLengthBytes - (genfskLocal[instanceId].packetConfig.preambleSizeBytes +1)-(genfskLocal[instanceId].packetConfig.syncAddrSizeBytes +2));

                        PROP_StartTx(PROP);

                        //config DMA .....
                        /* Transmit frame by DMA */
                        dma_transfer_config_t dmaConfig;
                        DMA_PrepareTransfer(&dmaConfig, (void *) pBuffer, (void *) &PROP->TX_BUF, 1, bufLengthBytes, kDMA_MemoryToPeripheral, NULL);
                        DMA_SubmitTransfer(&g_propTxDMAHandle, &dmaConfig);

                        TMR_StartSingleShotTimer(gGenfskTransferTimerId,((txStartTime - currentTime)/1000),(pfTmrCallBack_t)GENFSK_TxTransferCallback,NULL);
                    }


                }
#if gMWS_Enabled_d
            }
            else
            {
                status = gGenfskFail_c;
            }
#endif
        }
        else
        {
            switch(genfskLocal[mGenfskActiveInstance].genfskState)
            {
            case gGENFSK_LL_NoInit:
                status = gGenfskFail_c;
                break;
            case gGENFSK_LL_BusyRx:
                status = gGenfskBusyRx_c;
                break;
            case gGENFSK_LL_BusyTx:
                status = gGenfskBusyTx_c;
                break;
            case gGENFSK_LL_BusyPendingRx:
                status = gGenfskBusyPendingRx_c;
                break;
            case gGENFSK_LL_BusyPendingTx:
                status = gGenfskBusyPendingTx_c;
                break;
            default:
                status = gGenfskFail_c;
                break;
            }
        }

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

genfskStatus_t GENFSK_CancelPendingTx(void)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (mGenfskActiveInstance != gGENFSK_InvalidIdx_c)
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        PROP_StopTx(PROP);

        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

genfskStatus_t GENFSK_StartRx(uint8_t instanceId, uint8_t *pBuffer, uint16_t maxBufLengthBytes, GENFSK_timestamp_t rxStartTime, GENFSK_timestamp_t rxDuration)
{
    GENFSK_timestamp_t currentTime = 0;

    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (genfskLocal[instanceId].genfskState == gGENFSK_LL_NoInit)
    {
        status = gGenfskNotInitialized_c;
    }
    else if (pBuffer == NULL)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if (maxBufLengthBytes > gGENFSK_MaxPacketLength_c + genfskLocal[instanceId].syncAddrSizeBytes)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        /* If GENFSK LL is idle start RX. */
        if (genfskLocal[mGenfskActiveInstance].genfskState == gGENFSK_LL_Idle)
        {
#if gMWS_Enabled_d
            if (gMWS_Success_c == MWS_Acquire(gMWS_GENFSK_c, FALSE))
            {
#endif
                /* Change the active instance */
                if (instanceId != mGenfskActiveInstance)
                {
                   GENFSK_SwitchToInstance(instanceId);
                    mGenfskActiveInstance = instanceId;
                }
                genfskLocal[instanceId].genfskRxLocal.rxPacketBuffer = pBuffer;
                genfskLocal[instanceId].genfskRxLocal.rxMaxPacketLength = maxBufLengthBytes;

                if (gGenfskRadioTypeTx == genfskLocal[instanceId].channelNumberConfig.radioType)
                {
                    GENFSK_SetChannelNumber(instanceId,genfskLocal[instanceId].channelNumberConfig.channelNumber,gGenfskRadioTypeRxMode2);

                }
                /* Start RX now. */
                if (!rxStartTime)
                {
                    genfskLocal[instanceId].genfskState = gGENFSK_LL_BusyRx;

                    if (rxDuration)
                    {
                        TMR_StartSingleShotTimer(gGenfskTimeoutTimerId,((rxDuration+500)/1000),(pfTmrCallBack_t)GENFSK_RxTimeoutCallback,NULL);
                    }


                    /* Start RX immediately. */
                    PROP_SetPayloadLen(PROP,maxBufLengthBytes);

                    PROP_StartRx(PROP);

                   /* Enable DMA for PROP Rx */
                    dma_transfer_config_t rxdmaConfig;
                    DMA_PrepareTransfer(&rxdmaConfig, (void *)&PROP->RX_BUF, (void *)pBuffer, 1, maxBufLengthBytes, kDMA_PeripheralToMemory, NULL);
                    DMA_SubmitTransfer(&g_propRxDMAHandle, &rxdmaConfig);
                    DMA_StartTransfer(&g_propRxDMAHandle);
                }
                else
                {
                    currentTime = GENFSK_GetTimestamp();
                    if (rxStartTime <= currentTime + gGENFSK_MinSetupTime_c)
                    {
                        status = gGenfskInstantPassed_c;
                    }
                    else
                    {
                        genfskLocal[instanceId].genfskState = gGENFSK_LL_BusyPendingRx;
                        if (rxDuration)
                        {
                            TMR_StartSingleShotTimer(gGenfskTimeoutTimerId,((rxDuration+500)/1000),(pfTmrCallBack_t)GENFSK_RxTimeoutCallback,NULL);
                        }
                         /* Start RX immediately. */
                        PROP_SetPayloadLen(PROP,maxBufLengthBytes);

                        PROP_StartRx(PROP);

                        /* Enable DMA for PROP Rx */
                        dma_transfer_config_t rxdmaConfig;
                        DMA_PrepareTransfer(&rxdmaConfig, (void *)&PROP->RX_BUF, (void *)pBuffer, 1, maxBufLengthBytes, kDMA_PeripheralToMemory, NULL);
                        DMA_SubmitTransfer(&g_propRxDMAHandle, &rxdmaConfig);
                        TMR_StartSingleShotTimer(gGenfskTransferTimerId,((rxStartTime - currentTime +500)/1000),(pfTmrCallBack_t)GENFSK_RxTransferCallback,NULL);

                    }
                }
#if gMWS_Enabled_d
            }
            else
            {
                status = gGenfskFail_c;
            }
#endif
        }
        else
        {
            switch(genfskLocal[mGenfskActiveInstance].genfskState)
            {
            case gGENFSK_LL_NoInit:
                status = gGenfskFail_c;
                break;
            case gGENFSK_LL_BusyRx:
                status = gGenfskBusyRx_c;
                break;
            case gGENFSK_LL_BusyTx:
                status = gGenfskBusyTx_c;
                break;
            case gGENFSK_LL_BusyPendingRx:
                status = gGenfskBusyPendingRx_c;
                break;
            case gGENFSK_LL_BusyPendingTx:
                status = gGenfskBusyPendingTx_c;
                break;
            default:
                status = gGenfskFail_c;
                break;
            }
        }

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

genfskStatus_t GENFSK_CancelPendingRx(void)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (mGenfskActiveInstance != gGENFSK_InvalidIdx_c)
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        if(TMR_IsTimerActive(gGenfskTimeoutTimerId))
        {
            TMR_StopTimer(gGenfskTimeoutTimerId);
        }

        PROP_StopRx(PROP);

        genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxMaxPacketLength = 0;
        genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxPacketBuffer = NULL;
        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

genfskStatus_t GENFSK_AbortAll(void)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (mGenfskActiveInstance != gGENFSK_InvalidIdx_c)
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        if(TMR_IsTimerActive(gGenfskTimeoutTimerId))
        {
            TMR_StopTimer(gGenfskTimeoutTimerId);
        }
        PROP_StopTx(PROP);
        PROP_StopRx(PROP);
        DMA_AbortTransfer(&g_propTxDMAHandle);
        DMA_AbortTransfer(&g_propRxDMAHandle);
        genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxMaxPacketLength = 0;
        genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxPacketBuffer = NULL;
        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

GENFSK_timestamp_t GENFSK_GetTimestamp(void)
{
    volatile GENFSK_timestamp_t tempTime = 0;

    if (TRUE == genfskAlreadyInit)
    {
          OSA_InterruptDisable();
          tempTime = RTC_GetFreeRunningCount(RTC);
          OSA_InterruptEnable();
    }
    /*convert cycle to us   us = cycle *31.25   (RTC 32Khz)*/

    return (tempTime*31 + tempTime/4);
}

genfskStatus_t GENFSK_RegisterCallbacks(uint8_t instanceId, genfskPacketReceivedCallBack_t packetReceivedCallback, genfskEventNotifyCallBack_t eventCallback)
{
    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        /* Enter critical section. */
        OSA_InterruptDisable();

        genfskLocal[instanceId].packetReceivedcallback = packetReceivedCallback;
        genfskLocal[instanceId].eventNotifyCallback = eventCallback;

        /* Exit critical section. */
        OSA_InterruptEnable();
    }

    return status;
}

genfskStatus_t GENFSK_PacketToByteArray(uint8_t instanceId, GENFSK_packet_t *pPacket, uint8_t *pBuffer)
{
    uint8_t preamble ;
    uint8_t preamSize = genfskLocal[instanceId].packetConfig.preambleSizeBytes;  /* 0-7 ==1-8  */
    uint8_t syncAddrSize = genfskLocal[instanceId].packetConfig.syncAddrSizeBytes; /* 1,2,3 == 3, 4, 5 */
    uint8_t i;

    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if ((pPacket == NULL) || (pBuffer == NULL) || (pPacket->payload == NULL))
    {
        status = gGenfskInvalidParameters_c;
    }
    else if ((pPacket->dataLengthByte + preamSize + 1 + syncAddrSize + 2) > 2048)
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
         preamble = (pPacket->addr[syncAddrSize+1] & 0x80)?0xaa:0x55;

         for(i=0; i<(preamSize+1); i++)
             *pBuffer++ = preamble;

         for(i=0; i<syncAddrSize+2; i++)
             *pBuffer++ = pPacket->addr[syncAddrSize +1 - i];

         FLib_MemCpy(pBuffer, pPacket->payload, pPacket->dataLengthByte);
    }

    return status;
}

genfskStatus_t GENFSK_ByteArrayToPacket(uint8_t instanceId, uint8_t *pBuffer, GENFSK_packet_t *pPacket)
{
    uint8_t crcSize;

    GENFSK_crc_config_t crcConfig;

    genfskStatus_t status = gGenfskSuccess_c;

    if (instanceId >= gGENFSK_InstancesCnt_c)
    {
        status = gGenfskInvalidParameters_c;
    }
    else if ((pPacket == NULL) || (pBuffer == NULL) || (pPacket->payload == NULL))
    {
        status = gGenfskInvalidParameters_c;
    }
    else
    {
        PROP_GetCrcConfig(PROP, &crcConfig.crcSize,&crcConfig.crcStartByte,&crcConfig.crcSeed);
        crcSize = crcConfig.crcSize;

        FLib_MemCpy(pPacket->payload, pBuffer, pPacket->dataLengthByte + crcSize);
    }

    return status;
}

static void GENFSK_Task(osaTaskParam_t argument)
{
    uint64_t tempTime = 0;
    uint16_t byteCount = 0;
    uint8_t rssi = 0;
    osaEventFlags_t ev = 0;

    while(1)
    {
        (void)OSA_EventWait(mGenfskTaskEvent, osaEventFlagsAll_c, FALSE, osaWaitForever_c, &ev);

        /* RX event. */
        if (ev & gGenfskRxDmaEventFlag_c)
        {
            byteCount = PROP_GetPayloadLen(PROP);
            /*crc disable*/
            if (genfskLocal[mGenfskActiveInstance].crcEnable == gGenfskCrcDisable)
            {
                rssi = PROP_GetRssi(PROP);
                tempTime = GENFSK_GetTimestamp();
                genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;

                /* Packet received OK. */
                if (genfskLocal[mGenfskActiveInstance].packetReceivedcallback != NULL)
                {
                    genfskLocal[mGenfskActiveInstance].packetReceivedcallback(genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxPacketBuffer, byteCount, tempTime, rssi, FALSE);
                }
            }
            else/*crc enbale*/
            {
                if(PROP_GetCrcStatus(PROP)== true)/*crc valid*/
                {
                    //rssi
                    rssi = PROP_GetRssi(PROP);
                    tempTime = GENFSK_GetTimestamp();
                    genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;

                    /* Packet received OK. */
                    if (genfskLocal[mGenfskActiveInstance].packetReceivedcallback != NULL)
                    {
                        genfskLocal[mGenfskActiveInstance].packetReceivedcallback(genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxPacketBuffer, byteCount, tempTime, rssi, TRUE);
                    }
                }
                else/*crc invalid*/
                {
                    if (genfskLocal[mGenfskActiveInstance].crcRecvInvalid == gGenfskCrcRecvInvalid)
                    {
                        rssi = PROP_GetRssi(PROP);
                        tempTime = GENFSK_GetTimestamp();
                        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;
                         /* Packet received with CRC invalid. */
                        if (genfskLocal[mGenfskActiveInstance].packetReceivedcallback != NULL)
                        {
                            genfskLocal[mGenfskActiveInstance].packetReceivedcallback(genfskLocal[mGenfskActiveInstance].genfskRxLocal.rxPacketBuffer, byteCount, tempTime, rssi, FALSE);
                        }
                    }
                    else
                    {
                        genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;
                        if (genfskLocal[mGenfskActiveInstance].eventNotifyCallback != NULL)
                        {
                            genfskLocal[mGenfskActiveInstance].eventNotifyCallback(gGenfskRxEvent, gGenfskCRCInvalid);
                        }
                    }


                }
            }
        }

        /* TX event. */
        if (ev & gGenfskTxDmaEventFlag_c)
        {
            //TX ok
            while(PROP_GetTxStatus(PROP));

            genfskLocal[mGenfskActiveInstance].genfskState = gGENFSK_LL_Idle;

            if (genfskLocal[mGenfskActiveInstance].eventNotifyCallback != NULL)
            {
                genfskLocal[mGenfskActiveInstance].eventNotifyCallback(gGenfskTxEvent, gGenfskSuccess);
            }
        }

        /* For BM break the while() loop. */
        if (gUseRtos_c == 0)
        {
            break;
        }
    }
}

#if gMWS_Enabled_d
static uint32_t MWS_GENFSK_Callback(mwsEvents_t event)
{
    uint32_t status = gMWS_Success_c;

    switch(event)
    {
    case gMWS_Init_c:
        /* Do nothing */
        break;
    case gMWS_Active_c:
        /* Do nothing */
        break;
    case gMWS_Idle_c:
        /* Do nothing */
        break;
    case gMWS_Abort_c:
        /* Enter critical section */
        OSA_InterruptDisable();

        if (genfskLocal[mGenfskActiveInstance].genfskState != gGENFSK_LL_NoInit)
        {
            if (genfskLocal[mGenfskActiveInstance].genfskState != gGENFSK_LL_Idle)
            {
                GENFSK_AbortAll();
            }
        }

        /* Exit critical section */
        OSA_InterruptEnable();
        break;
    case gMWS_Release_c:
        /* Enter critical section */
        OSA_InterruptDisable();

        if (genfskLocal[mGenfskActiveInstance].genfskState != gGENFSK_LL_NoInit)
        {
            if (genfskLocal[mGenfskActiveInstance].genfskState != gGENFSK_LL_Idle)
            {
                GENFSK_AbortAll();
            }

        }

        /* Exit critical section */
        OSA_InterruptEnable();
        break;
    default:
        status = gMWS_InvalidParameter_c;
        break;
    }

    return status;
}
#endif
