/*
 * Copyright 2018 NXP
 * All rights reserved.
 *
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _FSL_CASPER_H_
#define _FSL_CASPER_H_

#include "fsl_common.h"

/*!
 * @addtogroup casper
 * @{
 */

/*! @file */

/*******************************************************************************
 * Definitions
 *******************************************************************************/

/*!
 * @addtogroup casper_driver
 * @{
 */
/*! @name Driver version */
/*@{*/
/*! @brief CASPER driver version. Version 2.0.0.
 *
 * Current version: 2.0.0
 *
 * Change log:
 * - Version 2.0.0
 *   - Initial version
 */
#define FSL_CASPER_DRIVER_VERSION (MAKE_VERSION(2, 0, 0))
/*@}*/

/*! @brief CASPER operation
 *
 */
typedef enum _casper_operation
{
    kCASPER_OpMul6464NoSum = 0x01, /*! Walking 1 or more of J loop, doing r=a*b using 64x64=128*/
    kCASPER_OpMul6464Sum =
        0x02, /*! Walking 1 or more of J loop, doing c,r=r+a*b using 64x64=128, but assume inner j loop*/
    kCASPER_OpMul6464FullSum =
        0x03, /*! Walking 1 or more of J loop, doing c,r=r+a*b using 64x64=128, but sum all of w. */
    kCASPER_OpMul6464Reduce =
        0x04,               /*! Walking 1 or more of J loop, doing c,r[-1]=r+a*b using 64x64=128, but skip 1st write*/
    kCASPER_OpAdd64 = 0x08, /*! Walking add with off_AB, and in/out off_RES doing c,r=r+a+c using 64+64=65*/
    kCASPER_OpSub64 = 0x09, /*! Walking subtract with off_AB, and in/out off_RES doing r=r-a uding 64-64=64, with last
                               borrow implicit if any*/
    kCASPER_OpDouble64 = 0x0A, /*! Walking add to self with off_RES doing c,r=r+r+c using 64+64=65*/
    kCASPER_OpXor64 = 0x0B,    /*! Walking XOR with off_AB, and in/out off_RES doing r=r^a using 64^64=64*/
    kCASPER_OpShiftLeft32 =
        0x10, /*! Walking shift left doing r1,r=(b*D)|r1, where D is 2^amt and is loaded by app (off_CD not used)*/
    kCASPER_OpShiftRight32 = 0x11, /*! Walking shift right doing r,r1=(b*D)|r1, where D is 2^(32-amt) and is loaded by
                                      app (off_CD not used) and off_RES starts at MSW*/
    kCASPER_OpCopy = 0x14,         /*! Copy from ABoff to resoff, 64b at a time*/
    kCASPER_OpRemask = 0x15,       /*! Copy and mask from ABoff to resoff, 64b at a time*/
    kCASPER_OpCompare = 0x16,      /*! Compare two arrays, running all the way to the end*/
    kCASPER_OpCompareFast = 0x17,  /*! Compare two arrays, stopping on 1st !=*/
} casper_operation_t;

#define CASPER_CP 1
#define CASPER_CP_CTRL0 (0x0 >> 2)
#define CASPER_CP_CTRL1 (0x4 >> 2)
#define CASPER_CP_LOADER (0x8 >> 2)
#define CASPER_CP_STATUS (0xC >> 2)
#define CASPER_CP_INTENSET (0x10 >> 2)
#define CASPER_CP_INTENCLR (0x14 >> 2)
#define CASPER_CP_INTSTAT (0x18 >> 2)
#define CASPER_CP_AREG (0x20 >> 2)
#define CASPER_CP_BREG (0x24 >> 2)
#define CASPER_CP_CREG (0x28 >> 2)
#define CASPER_CP_DREG (0x2C >> 2)
#define CASPER_CP_RES0 (0x30 >> 2)
#define CASPER_CP_RES1 (0x34 >> 2)
#define CASPER_CP_RES2 (0x38 >> 2)
#define CASPER_CP_RES3 (0x3C >> 2)
#define CASPER_CP_MASK (0x60 >> 2)
#define CASPER_CP_REMASK (0x64 >> 2)
#define CASPER_CP_LOCK (0x80 >> 2)
#define CASPER_CP_ID (0xFFC >> 2)
/* mcr (cp,  opc1, value, CRn, CRm, opc2) */
#define CASPER_Wr32b(value, off) __arm_mcr(CASPER_CP, 0, value, ((off >> 4)), (off), 0)
/* mcrr(coproc, opc1, value, CRm) */
#define CASPER_Wr64b(value, off) __arm_mcrr(CASPER_CP, 0, value, off)
/* mrc(coproc, opc1, CRn, CRm, opc2) */
#define CASPER_Rd32b(off) __arm_mrc(CASPER_CP, 0, ((off >> 4)), (off), 0)

/*! @} */

/*******************************************************************************
 * API
 ******************************************************************************/
#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @addtogroup casper_driver
 * @{
 */

/*!
 * @brief Enables clock and disables reset for CASPER peripheral.
 *
 * Enable clock and disable reset for CASPER.
 *
 * @param base CASPER base address
 */
void CASPER_Init(CASPER_Type *base);

/*!
 * @brief Disables clock for CASPER peripheral.
 *
 * Disable clock and enable reset.
 *
 * @param base CASPER base address
 */
void CASPER_Deinit(CASPER_Type *base);

/*!
 *@}
 */ /* end of casper_driver */

/*******************************************************************************
 * PKHA API
 ******************************************************************************/

/*!
 * @addtogroup casper_driver_pkha
 * @{
 */

/*!
 * @brief Performs modular exponentiation - (A^E) mod N.
 *
 * This function performs modular exponentiation.
 *
 * @param base CASPER base address
 * @param signature first addend (in little endian format)
 * @param pubN modulus (in little endian format)
 * @param wordLen Size of pubN in bytes
 * @param pubE exponent
 * @param[out] plaintext Output array to store result of operation (in little endian format)
 */
void CASPER_ModExp(CASPER_Type *base,
                   const uint8_t *signature,
                   const uint8_t *pubN,
                   size_t wordLen,
                   uint32_t pubE,
                   uint8_t *plaintext);

/*!
 *@}
 */ /* end of casper_driver_pkha */

#if defined(__cplusplus)
}
#endif

#endif /* _FSL_CASPER_H_ */
