/*
 * Copyright 2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_common.h"
#include "pin_mux.h"
#include "fsl_usart.h"
#include <stdio.h>

typedef struct
{
    void (*reinvoke)(void);
    void (*set_update_flag)(void);
    void (*test)(void);
}sbl_api_t;

static sbl_api_t *sbl_api = (sbl_api_t*)(0x400);

// Global counter for SysTick
volatile uint32_t systick_counter = 0;
volatile uint32_t systick_1ms_counter = 0;

// SysTick interrupt handler - triggers every 1ms
void SysTick_Handler(void)
{
    systick_1ms_counter++;

    // Print counter every 1000ms (1 second)
    if (systick_1ms_counter % 1000 == 0) {
        systick_counter++;
        printf("SysTick counter: %lu (1s intervals)\r\n", systick_counter);
    }
}

// Initialize SysTick timer
void SysTick_Init(void)
{
    uint32_t core_freq = CLOCK_GetFreq(kCLOCK_CoreSysClk);

    // Configure SysTick for 1ms interval
    // 96MHz / 1000 = 96000 ticks per 1ms
    uint32_t ticks_per_ms = core_freq / 1000;

    printf("Core frequency: %lu Hz\r\n", core_freq);
    printf("Ticks per 1ms: %lu\r\n", ticks_per_ms);

    if (ticks_per_ms > SysTick_LOAD_RELOAD_Msk) {
        printf("Error: Ticks per ms (%lu) exceeds SysTick 24-bit limit (%lu)\r\n",
               ticks_per_ms, SysTick_LOAD_RELOAD_Msk);
        return;
    }

    SysTick_Config(ticks_per_ms);
    printf("SysTick initialized for 1ms intervals\r\n");
}

int main()
{
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    /* enable clock for GPIO*/
    CLOCK_EnableClock(kCLOCK_Gpio0);
    CLOCK_EnableClock(kCLOCK_Gpio1);

    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    BOARD_InitDebugConsole();

    printf("\r\n-----------------------\r\n");
    printf("I am golden image!!!\r\n");
    printf("CoreClock:%dHz\r\n", CLOCK_GetFreq(kCLOCK_CoreSysClk));

    // Initialize SysTick timer
    SysTick_Init();

    uint32_t app_version = *(uint32_t*)(0x10000 + 0x140 + 5*sizeof(uint32_t));

    printf("hello, golden image running: my version:%d\r\n", app_version);
    printf("press R or r to re-enter bootloader\r\n");

    while (1)
    {
        char cmd = 0;

        cmd = getchar();
        if(cmd == 'R' || cmd == 'r')
        {
          printf("call bootlaoder's reinvoke\r\n");

          /* clean up */

          // Disable SysTick timer and clear pending interrupt
          SysTick->CTRL = 0;
          SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

          sbl_api->reinvoke();
        }
    }
}

void HardFault_Handler(void)
{
    printf("HardFault_Handler from app\r\n");
    while(1);
}
