/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright (c) 2016, NXP
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "fsl_common.h"
#include "fsl_power.h"

/*! @name Driver version */
/*@{*/
/*! @brief LPC546XX power contorl version 1.0.0. */
#define FSL_LPC55SXX_POWER_CONTROL_VERSION (MAKE_VERSION(1, 0, 0))
/*@}*/

/*******************************************************************************
 * Variables
 ******************************************************************************/
/*! @brief Low Power Main Structure. */
static LPC_LOWPOWER_T s_LOWPOWER;

/*******************************************************************************
 * Definitions
 ******************************************************************************/
/* Component ID definition, used by tools. */
#ifndef FSL_COMPONENT_ID
#define FSL_COMPONENT_ID "platform.drivers.lpc_niobe4_power"
#endif
/*! @brief set and clear bit MACRO's. */
#define U32_SET_BITS(P, B) ((*(uint32_t *)P) |= (B))
#define U32_CLR_BITS(P, B) ((*(uint32_t *)P) &= ~(B))

/*******************************************************************************
 * Codes
 ******************************************************************************/

void (*CHIPLOWPOWER_PowerCycleCpuAndFlash)(void) = (void (*)())0x130124f1; 
void (*CHIPLOWPOWER_SetLowPowerMode)(LPC_LOWPOWER_T *p_lowpower_cfg) = (void (*)(LPC_LOWPOWER_T *p_lowpower_cfg))0x13012533; 

/**
 * @brief    PMC Sleep function call
 * @return   nothing
 */

void POWER_EnterSleep(void)
{ 
    uint32_t pmsk;
    pmsk = __get_PRIMASK();
    __disable_irq();
    SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    __set_PRIMASK(pmsk);   
}

/**
 * @brief    PMC Deep Sleep function call
 * @return   nothing
 */

void POWER_EnterDeepSleep(uint64_t exclude_from_pd)
{
    memset(&s_LOWPOWER, 0, sizeof(LPC_LOWPOWER_T));
    
    /* set config field and go */
    U32_SET_BITS(&s_LOWPOWER.WAKEUPIOSRC, 0x1 < LOWPOWER_WAKEUPIOSRC_PIO2_INDEX);
    U32_SET_BITS(&s_LOWPOWER.PDCTRL0, exclude_from_pd);
    U32_SET_BITS(&s_LOWPOWER.CFG, LOWPOWER_CFG_LPMODE_DEEPSLEEP);
    CHIPLOWPOWER_SetLowPowerMode(&s_LOWPOWER);
}

/**
 * @brief    PMC power Down function call
 * @return   nothing
 */

void POWER_EnterPowerDown(uint64_t exclude_from_pd)
{
    memset(&s_LOWPOWER, 0, sizeof(LPC_LOWPOWER_T));
    
    /* set config field and go */
    U32_SET_BITS(&s_LOWPOWER.WAKEUPIOSRC, 0x1 < LOWPOWER_WAKEUPIOSRC_PIO2_INDEX);
    U32_SET_BITS(&s_LOWPOWER.PDCTRL0, exclude_from_pd);
    U32_SET_BITS(&s_LOWPOWER.CFG, LOWPOWER_CFG_LPMODE_POWERDOWN);
    CHIPLOWPOWER_SetLowPowerMode(&s_LOWPOWER);
}

/**
 * @brief    PMC Deep Sleep Power Down function call
 * @return   nothing
 */

void POWER_EnterDeepPowerDown(uint64_t exclude_from_pd)
{
    memset(&s_LOWPOWER, 0, sizeof(LPC_LOWPOWER_T));
    
    /* set config field and go */
    U32_SET_BITS(&s_LOWPOWER.PDCTRL0, exclude_from_pd);
    U32_SET_BITS(&s_LOWPOWER.CFG, LOWPOWER_CFG_LPMODE_DEEPPOWERDOWN);
    CHIPLOWPOWER_SetLowPowerMode(&s_LOWPOWER);
}

/* Enter Power mode */
void POWER_EnterPowerMode(power_mode_cfg_t mode, uint64_t exclude_from_pd)
{
    switch (mode)
    {   
        case kPmu_Sleep:
            POWER_EnterSleep();
            break;

        case kPmu_Deep_Sleep:
            POWER_EnterDeepSleep(exclude_from_pd);
            break;

        case kPmu_Deep_PowerDown:
            POWER_EnterDeepPowerDown(exclude_from_pd);
            break;

        default:
            break; 
    }  
}

/* Get power lib version */
uint32_t POWER_GetLibVersion(void)
{
    return FSL_LPC55SXX_POWER_CONTROL_VERSION;
}
