/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "app_inc.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define APP_PAGES_PER_PACKAGE (MODEM_PACKAGE_BYTE_COUNT/FSL_FEATURE_SYSCON_FLASH_PAGE_SIZE_BYTES) /* 1024 / 256. */
#define APP_PAGES_PER_SECTOR  (FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES/FSL_FEATURE_SYSCON_FLASH_PAGE_SIZE_BYTES) /* 32K / 256. */


/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/
modem_struct gAppModemStruct;
char      gPrintfBuf[64];


uint32_t gAppFwWriteImageLen    ;
uint32_t gAppFwWriteSectorStart ;
uint32_t gAppFwWriteSectorCount ;
uint32_t gAppFwWriteSectorCurIdx;
//uint32_t gAppFwWritePageCount   ;
//uint32_t gAppFwWritePageCurIdx  ;
uint32_t gAppFwWriteCurAddr;


/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
* @brief Main function
*/
int main(void)
{
    uint8_t ch;
    bool bNoTimeout;
    int err;
    uint32_t errIAP;
    uint8_t *pFwInfo;

    BOARD_InitBootClocks();
    SystemCoreClockUpdate();
    BOARD_InitPins();

    Terminal_Init(115200ul);
    Terminal_PutString("enter the 2nd bootloader [y/n] ...\r\n");
    
    bNoTimeout = Terminal_GetCharTimeout(&ch, 5000u);
    if ( (!bNoTimeout) || (ch != 'y')  )
    {
        Terminal_PutString("normal boot to: ");

        /* check if the firmware is available. */
        pFwInfo = (uint8_t *)(BOOT_FIRMWARE_BASE_ADDRESS+BOOT_FIRMWARE_INFO_OFFSET);
        if (*pFwInfo == 0xff)
        {
            Terminal_PutString("No available firmware.\r\n");
            FwBoot_Exit();
        }
        else /* print the file name of the firwware. */
        {
            ch = 0u; /* reuse 'ch' as 'i'. */
            while (ch < 32u)/* the max length of filename. */
            {
                if (*pFwInfo == '\0')
                {
                    break;
                }
                Terminal_PutChar(*pFwInfo);
                pFwInfo++;
                ch++;
            }
            Terminal_PutString("\r\n\r\n");
            FwBoot_BootToFwImage(BOOT_FIRMWARE_BASE_ADDRESS); /* boot to normal image. */
        }
    }

    /* enter bootloader. */
    Terminal_PutString("[2nd bootloader] Entered.\r\n");
    Terminal_PutString("[2nd bootloader] Please start the ymodem server to transfer new firmware:\r\n");

    /* start the ymodem. */
    err = ymodem_init(&gAppModemStruct);

    gAppFwWriteImageLen    = gAppModemStruct.filelen;
    gAppFwWriteSectorStart = BOOT_FIRMWARE_BASE_ADDRESS / FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES;
    gAppFwWriteSectorCount = (gAppFwWriteImageLen + FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES-1) / FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES;

    /* erase the flash to be programmed. */
    IAP_PrepareSectorForWrite(gAppFwWriteSectorStart, gAppFwWriteSectorStart+gAppFwWriteSectorCount-1);
    IAP_EraseSector(gAppFwWriteSectorStart, gAppFwWriteSectorStart+gAppFwWriteSectorCount-1, SystemCoreClock);
    errIAP = IAP_BlankCheckSector(gAppFwWriteSectorStart, gAppFwWriteSectorStart+gAppFwWriteSectorCount-1);
    if (errIAP != kStatus_IAP_Success)
    {
        modem_cancle();
        Terminal_PutString("Sector Erase failed.\r\n");
        FwBoot_Exit();
    }

    do /* recevie the firmware data and program the flash. */
    {
        err = modem_recvdata(&gAppModemStruct);
        if (err == 1)
        {
            break; /* done. */
        }

        if (gAppModemStruct.cur_num == 1u) /* the first package would include the firmware info. */
        {
            /* copy the file name of firmware into the image. */
            strcpy( (char *)(gAppModemStruct.buf+BOOT_FIRMWARE_INFO_OFFSET), (char *)gAppModemStruct.filename);
        }


        /* 这里要使用gAppModemStruct.cur_num定位烧写地址，因为要传输包可能出现错误重传的情况。 */
        /* 需要根据gAppModemStruct.cur_num直接计算如下变量：
         * - gAppFwWriteSectorCurIdx
         * - gAppFwWriteCurAddr
         */
        gAppFwWriteCurAddr      = BOOT_FIRMWARE_BASE_ADDRESS + (gAppModemStruct.cur_num-1) * MODEM_PACKAGE_BYTE_COUNT;
        gAppFwWriteSectorCurIdx = gAppFwWriteCurAddr / FSL_FEATURE_SYSCON_FLASH_SECTOR_SIZE_BYTES;
        /* program to flash. */
        IAP_PrepareSectorForWrite(gAppFwWriteSectorCurIdx, gAppFwWriteSectorCurIdx);
        IAP_CopyRamToFlash(
                gAppFwWriteCurAddr,                /* dstaddr. */
                (uint32_t *)(gAppModemStruct.buf), /* srcAddr. */
                MODEM_PACKAGE_BYTE_COUNT,          /* numOfBytes. */
                SystemCoreClock                    /* systemCoreClock. */
            );
        errIAP = IAP_Compare(
                gAppFwWriteCurAddr,                /* dstaddr. */
                (uint32_t *)(gAppModemStruct.buf), /* srcAddr. */
                MODEM_PACKAGE_BYTE_COUNT           /* numOfBytes. */
            );
        if (errIAP != kStatus_IAP_Success)
        {
            err = 10;
        }
			
    } while (err == 0u);

    Terminal_PutString("\r\n");
    if (err != 1)
    {
        Terminal_PutString("[2nd bootloader] firmware failed.\r\n");
        while (1);
    }
    else
    {
        Terminal_PutString("[2nd bootloader] firmware updated.\r\n");
    }

    Terminal_PutString("[2nd bootloader] reboot ...\r\n");
    FwBoot_Exit();

    /* the code should never run to here. */
    while (1);
}

/* EOF. */

