/*
 *  Benchmark demonstration program
 *
 *  Copyright (C) 2006-2016, ARM Limited, All Rights Reserved
 *  SPDX-License-Identifier: Apache-2.0
 *  Copyright 2017 NXP. Not a Contribution
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"); you may
 *  not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  This file is part of mbed TLS (https://tls.mbed.org)
 */

#if !defined(MBEDTLS_CONFIG_FILE)
#include "mbedtls/config.h"
#else
#include MBEDTLS_CONFIG_FILE
#endif

#if defined(MBEDTLS_PLATFORM_C)

#if defined(FREESCALE_KSDK_BM)

#include "mbedtls/version.h"
#include <stdio.h>
#include "fsl_debug_console.h"
#include "fsl_clock.h"
#include "board.h"
#include "ksdk_mbedtls.h"

#define mbedtls_printf PRINTF
#define mbedtls_snprintf snprintf
#define mbedtls_exit(x) \
    do                  \
    {                   \
    } while (1)
#define mbedtls_free free
#define fflush(x) \
    do            \
    {             \
    } while (0)

#else
#include "mbedtls/platform.h"
#endif
#else
#include <stdio.h>
#define mbedtls_exit exit
#define mbedtls_printf printf
#define mbedtls_snprintf snprintf
#define mbedtls_free free
#endif

#if !defined(MBEDTLS_TIMING_C) && !defined(FREESCALE_KSDK_BM)
int main(void)
{
    mbedtls_printf("MBEDTLS_TIMING_C not defined.\n");
    return (0);
}
#else

#include <string.h>
#include <stdlib.h>

#include "mbedtls/timing.h"

#include "mbedtls/md4.h"
#include "mbedtls/md5.h"
#include "mbedtls/ripemd160.h"
#include "mbedtls/sha1.h"
#include "mbedtls/sha256.h"
#include "mbedtls/sha512.h"

#include "mbedtls/arc4.h"
#include "mbedtls/des.h"
#include "mbedtls/aes.h"
#include "mbedtls/aria.h"
#include "mbedtls/blowfish.h"
#include "mbedtls/camellia.h"
#include "mbedtls/chacha20.h"
#include "mbedtls/gcm.h"
#include "mbedtls/ccm.h"
#include "mbedtls/chachapoly.h"
#include "mbedtls/cmac.h"
#include "mbedtls/poly1305.h"

#include "mbedtls/havege.h"
#include "mbedtls/ctr_drbg.h"
#include "mbedtls/hmac_drbg.h"

#include "mbedtls/rsa.h"
#include "mbedtls/dhm.h"
#include "mbedtls/ecdsa.h"
#include "mbedtls/ecdh.h"

#include "mbedtls/error.h"

#if defined(MBEDTLS_MEMORY_BUFFER_ALLOC_C)
#include "mbedtls/memory_buffer_alloc.h"
#endif

#include "pin_mux.h"
#include "clock_config.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define CORE_CLK_FREQ CLOCK_GetFreq(kCLOCK_CoreSysClk)

/*
 * For heap usage estimates, we need an estimate of the overhead per allocated
 * block. ptmalloc2/3 (used in gnu libc for instance) uses 2 size_t per block,
 * so use that as our baseline.
 */
#define MEM_BLOCK_OVERHEAD (2 * sizeof(size_t))

/*
 * Size to use for the alloc buffer if MEMORY_BUFFER_ALLOC_C is defined.
 */
#define HEAP_SIZE (1u << 16) // 64k

#define BUFSIZE 1024
#define HEADER_FORMAT "  %-24s :  "
#define TITLE_LEN 25

#define OPTIONS                                         \
    "md4, md5, ripemd160, sha1, sha256, sha512,\n"      \
    "arc4, des3, des, camellia, blowfish, chacha20,\n"  \
    "aes_cbc, aes_gcm, aes_ccm, aes_ctx, chachapoly,\n" \
    "aes_cmac, des3_cmac, poly1305\n"                   \
    "havege, ctr_drbg, hmac_drbg\n"                     \
    "rsa, dhm, ecdsa, ecdh.\n"

#if defined(MBEDTLS_ERROR_C)
#define PRINT_ERROR                                  \
    mbedtls_strerror(ret, (char *)tmp, sizeof(tmp)); \
    mbedtls_printf("FAILED: %s\n", tmp);
#else
#define PRINT_ERROR mbedtls_printf("FAILED: -0x%04x\n", -ret);
#endif

#define TIME_AND_TSC(TITLE, CODE)                                                                        \
    do                                                                                                   \
    {                                                                                                    \
        uint32_t ii, jj;                                                                                 \
        uint64_t tsc1, tsc2;                                                                             \
        int ret = 0;                                                                                     \
                                                                                                         \
        mbedtls_printf(HEADER_FORMAT, TITLE);                                                            \
        fflush(stdout);                                                                                  \
                                                                                                         \
        benchmark_mbedtls_set_alarm(1);                                                                  \
        tsc1 = benchmark_mbedtls_timing_hardclock();                                                     \
        for (ii = 1; ret == 0 && !benchmark_mbedtls_timing_alarmed; ii++)                                \
        {                                                                                                \
            ret = CODE;                                                                                  \
            benchmark_mbedtls_poll_alarm();                                                              \
        }                                                                                                \
                                                                                                         \
        tsc2 = benchmark_mbedtls_timing_hardclock();                                                     \
        for (jj = 0; ret == 0 && jj < 1024; jj++)                                                        \
        {                                                                                                \
            ret = CODE;                                                                                  \
        }                                                                                                \
                                                                                                         \
        if (ret != 0)                                                                                    \
        {                                                                                                \
            PRINT_ERROR;                                                                                 \
        }                                                                                                \
        else                                                                                             \
        {                                                                                                \
            mbedtls_printf("%6.2f KB/s,  %6.2f cycles/byte\r\n",                                         \
                           (ii * BUFSIZE / 1024) / (((float)(tsc2 - tsc1)) / CLOCK_GetCoreSysClkFreq()), \
                           (((float)(benchmark_mbedtls_timing_hardclock() - tsc2)) / (jj * BUFSIZE)));   \
        }                                                                                                \
    } while (0)

#if defined(MBEDTLS_ERROR_C)
#define PRINT_ERROR                                  \
    mbedtls_strerror(ret, (char *)tmp, sizeof(tmp)); \
    mbedtls_printf("FAILED: %s\n", tmp);
#else
#define PRINT_ERROR mbedtls_printf("FAILED: -0x%04x\n", -ret);
#endif

#if defined(MBEDTLS_MEMORY_BUFFER_ALLOC_C) && defined(MBEDTLS_MEMORY_DEBUG)

#define MEMORY_MEASURE_INIT                                      \
    size_t max_used, max_blocks, max_bytes;                      \
    size_t prv_used, prv_blocks;                                 \
    mbedtls_memory_buffer_alloc_cur_get(&prv_used, &prv_blocks); \
    mbedtls_memory_buffer_alloc_max_reset();

#define MEMORY_MEASURE_PRINT(title_len)                          \
    mbedtls_memory_buffer_alloc_max_get(&max_used, &max_blocks); \
    for (ii = 12 - title_len; ii != 0; ii--)                     \
        mbedtls_printf(" ");                                     \
    max_used -= prv_used;                                        \
    max_blocks -= prv_blocks;                                    \
    max_bytes = max_used + MEM_BLOCK_OVERHEAD * max_blocks;      \
    mbedtls_printf("%6u heap bytes", (unsigned)max_bytes);

#else
#define MEMORY_MEASURE_INIT
#define MEMORY_MEASURE_PRINT(title_len)
#endif

#define TIME_PUBLIC(TITLE, TYPE, CODE)                                                                                \
    do                                                                                                                \
    {                                                                                                                 \
        uint32_t ii;                                                                                                  \
        uint64_t tsc;                                                                                                 \
        int ret;                                                                                                      \
        MEMORY_MEASURE_INIT;                                                                                          \
                                                                                                                      \
        mbedtls_printf(HEADER_FORMAT, TITLE);                                                                         \
        fflush(stdout);                                                                                               \
        benchmark_mbedtls_set_alarm(3);                                                                               \
                                                                                                                      \
        ret = 0;                                                                                                      \
        tsc = benchmark_mbedtls_timing_hardclock();                                                                   \
        for (ii = 1; !benchmark_mbedtls_timing_alarmed && !ret; ii++)                                                 \
        {                                                                                                             \
            CODE;                                                                                                     \
            benchmark_mbedtls_poll_alarm();                                                                           \
        }                                                                                                             \
                                                                                                                      \
        if (ret != 0)                                                                                                 \
        {                                                                                                             \
            PRINT_ERROR;                                                                                              \
        }                                                                                                             \
        else                                                                                                          \
        {                                                                                                             \
            mbedtls_printf("%6.2f " TYPE "/s",                                                                        \
                           ((float)ii) / ((benchmark_mbedtls_timing_hardclock() - tsc) / CLOCK_GetCoreSysClkFreq())); \
            MEMORY_MEASURE_PRINT(sizeof(TYPE) + 1);                                                                   \
            mbedtls_printf("\r\n");                                                                                   \
        }                                                                                                             \
    } while (0)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/


/*******************************************************************************
 * Code
 ******************************************************************************/
static int myrand(void *rng_state, unsigned char *output, size_t len)
{
    size_t use_len;
    int rnd;

    if (rng_state != NULL)
        rng_state = NULL;

    while (len > 0)
    {
        use_len = len;
        if (use_len > sizeof(int))
            use_len = sizeof(int);

        rnd = rand();
        memcpy(output, &rnd, use_len);
        output += use_len;
        len -= use_len;
    }

    return (0);
}

/*
 * Clear some memory that was used to prepare the context
 */
#if defined(MBEDTLS_ECP_C)
void ecp_clear_precomputed(mbedtls_ecp_group *grp)
{
    if (grp->T != NULL)
    {
        size_t i;
        for (i = 0; i < grp->T_size; i++)
            mbedtls_ecp_point_free(&grp->T[i]);
        mbedtls_free(grp->T);
    }
    grp->T      = NULL;
    grp->T_size = 0;
}
#else
#define ecp_clear_precomputed(g)
#endif

unsigned char buf[BUFSIZE];

typedef struct
{
    char md4, md5, ripemd160, sha1, sha256, sha512, arc4, des3, des, aes_cbc, aes_gcm, aes_ccm, aes_xts, chachapoly,
        aes_cmac, des3_cmac, aria, camellia, blowfish, chacha20, poly1305, havege, ctr_drbg, hmac_drbg, rsa, dhm, ecdsa,
        ecdh;
} todo_list;

#if defined(FREESCALE_KSDK_BM) && !defined(MBEDTLS_TIMING_C)

static volatile uint32_t s_MsCount = 0U;
static volatile int benchmark_mbedtls_timing_alarmed;
static uint64_t s_Timeout;

/*!
 * @brief Milliseconds counter since last POR/reset.
 */
void SysTick_Handler(void)
{
    s_MsCount++;
}

static uint64_t benchmark_mbedtls_timing_hardclock(void)
{
    uint32_t currMsCount;
    uint32_t currTick;
    uint32_t loadTick;

    do
    {
        currMsCount = s_MsCount;
        currTick    = SysTick->VAL;
    } while (currMsCount != s_MsCount);

    loadTick = CLOCK_GetCoreSysClkFreq() / 1000U;
    return (((uint64_t)currMsCount) * loadTick) + loadTick - currTick;
}

static void benchmark_mbedtls_set_alarm(int seconds)
{
    benchmark_mbedtls_timing_alarmed = 0;
    s_Timeout                        = benchmark_mbedtls_timing_hardclock() + (seconds * CLOCK_GetCoreSysClkFreq());
}

static void benchmark_mbedtls_poll_alarm(void)
{
    if (benchmark_mbedtls_timing_hardclock() > s_Timeout)
    {
        benchmark_mbedtls_timing_alarmed = 1;
    }
}
#endif

static int bench_print_features(void)
{
    char *text;
    mbedtls_printf("mbedTLS version %s\r\n", MBEDTLS_VERSION_STRING);
    mbedtls_printf("fsys=%lu\r\n", ((CORE_CLK_FREQ)));
    mbedtls_printf("Using following implementations:\r\n");
#if defined(MBEDTLS_FREESCALE_LTC_SHA256)
    text = "LTC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_MMCAU_SHA256)
    text = "MMCAU HW accelerated";
#elif defined(MBEDTLS_FREESCALE_LPC_SHA256)
    text = "LPC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_CAU3_SHA256)
    text = "CAU3 HW accelerated";
#elif defined(MBEDTLS_FREESCALE_DCP_SHA256)
    text = "DCP HW accelerated";
#elif defined(MBEDTLS_FREESCALE_HASHCRYPT_SHA256)
    text = "HASHCRYPT HW accelerated";
#else
    text = "Software implementation";
#endif
    mbedtls_printf("  SHA: %s\r\n", text);
#if defined(MBEDTLS_FREESCALE_LTC_AES)
    text = "LTC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_MMCAU_AES)
    text = "MMCAU HW accelerated";
#elif defined(MBEDTLS_FREESCALE_LPC_AES)
    text = "LPC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_CAU3_AES)
    text = "CAU3 HW accelerated";
#elif defined(MBEDTLS_FREESCALE_DCP_AES)
    text = "DCP HW accelerated";
#elif defined(MBEDTLS_FREESCALE_HASHCRYPT_AES)
    text = "HASHCRYPT HW accelerated";
#else
    text = "Software implementation";
#endif
    mbedtls_printf("  AES: %s\r\n", text);
#if defined(MBEDTLS_FREESCALE_LTC_AES_GCM)
    text = "LTC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_MMCAU_AES)
    text = "MMCAU HW accelerated";
#elif defined(MBEDTLS_FREESCALE_LPC_AES_GCM)
    text = "LPC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_CAU3_AES)
    text = "CAU3 HW accelerated";
#else
    text = "Software implementation";
#endif
    mbedtls_printf("  AES GCM: %s\r\n", text);
#if defined(MBEDTLS_FREESCALE_LTC_DES)
    text = "LTC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_MMCAU_DES)
    text = "MMCAU HW accelerated";
#elif defined(MBEDTLS_FREESCALE_CAU3_DES)
    text = "CAU3 HW accelerated";
#else
    text = "Software implementation";
#endif
    mbedtls_printf("  DES: %s\r\n", text);
#if defined(MBEDTLS_FREESCALE_LTC_PKHA)
    text = "LTC HW accelerated";
#elif defined(MBEDTLS_FREESCALE_CAU3_PKHA)
    text = "CAU3 HW accelerated";
#elif defined(MBEDTLS_FREESCALE_CASPER_PKHA)
    text = "CASPER HW accelerated";
#else
    text = "Software implementation";
#endif
    mbedtls_printf("  Asymmetric encryption: %s\r\n\n", text);
    return 0;
}

int main(int argc, char *argv[])
{
    int i;
    unsigned char tmp[200];
    char title[TITLE_LEN];
    todo_list todo;
#if defined(MBEDTLS_MEMORY_BUFFER_ALLOC_C)
    unsigned char alloc_buf[HEAP_SIZE] = {0};
#endif

#if defined(FREESCALE_KSDK_BM)
    /* HW init */
    /* attach main clock divide to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    BOARD_InitPins();
    BOARD_BootClockPLL150M();
    BOARD_InitDebugConsole();
    CRYPTO_InitHardware();

    /* Init SysTick module */
    /* call CMSIS SysTick function. It enables the SysTick interrupt at low priority */
    SysTick_Config(CLOCK_GetCoreSysClkFreq() / 1000U); /* 1 ms period */
#endif
    bench_print_features();
#if 0 /* We need to run all tests*/
    if( argc <= 1 )
    {
        memset( &todo, 1, sizeof( todo ) );
    }
    else
    {
        memset( &todo, 0, sizeof( todo ) );

        for( i = 1; i < argc; i++ )
        {
            if( strcmp( argv[i], "md4" ) == 0 )
                todo.md4 = 1;
            else if( strcmp( argv[i], "md5" ) == 0 )
                todo.md5 = 1;
            else if( strcmp( argv[i], "ripemd160" ) == 0 )
                todo.ripemd160 = 1;
            else if( strcmp( argv[i], "sha1" ) == 0 )
                todo.sha1 = 1;
            else if( strcmp( argv[i], "sha256" ) == 0 )
                todo.sha256 = 1;
            else if( strcmp( argv[i], "sha512" ) == 0 )
                todo.sha512 = 1;
            else if( strcmp( argv[i], "arc4" ) == 0 )
                todo.arc4 = 1;
            else if( strcmp( argv[i], "des3" ) == 0 )
                todo.des3 = 1;
            else if( strcmp( argv[i], "des" ) == 0 )
                todo.des = 1;
            else if( strcmp( argv[i], "aes_cbc" ) == 0 )
                todo.aes_cbc = 1;
            else if( strcmp( argv[i], "aes_xts" ) == 0 )
                todo.aes_xts = 1;
            else if( strcmp( argv[i], "aes_gcm" ) == 0 )
                todo.aes_gcm = 1;
            else if( strcmp( argv[i], "aes_ccm" ) == 0 )
                todo.aes_ccm = 1;
            else if( strcmp( argv[i], "chachapoly" ) == 0 )
                todo.chachapoly = 1;
            else if( strcmp( argv[i], "aes_cmac" ) == 0 )
                todo.aes_cmac = 1;
            else if( strcmp( argv[i], "des3_cmac" ) == 0 )
                todo.des3_cmac = 1;
            else if( strcmp( argv[i], "aria" ) == 0 )
                todo.aria = 1;
            else if( strcmp( argv[i], "camellia" ) == 0 )
                todo.camellia = 1;
            else if( strcmp( argv[i], "blowfish" ) == 0 )
                todo.blowfish = 1;
            else if( strcmp( argv[i], "chacha20" ) == 0 )
                todo.chacha20 = 1;
            else if( strcmp( argv[i], "poly1305" ) == 0 )
                todo.poly1305 = 1;
            else if( strcmp( argv[i], "havege" ) == 0 )
                todo.havege = 1;
            else if( strcmp( argv[i], "ctr_drbg" ) == 0 )
                todo.ctr_drbg = 1;
            else if( strcmp( argv[i], "hmac_drbg" ) == 0 )
                todo.hmac_drbg = 1;
            else if( strcmp( argv[i], "rsa" ) == 0 )
                todo.rsa = 1;
            else if( strcmp( argv[i], "dhm" ) == 0 )
                todo.dhm = 1;
            else if( strcmp( argv[i], "ecdsa" ) == 0 )
                todo.ecdsa = 1;
            else if( strcmp( argv[i], "ecdh" ) == 0 )
                todo.ecdh = 1;
            else
            {
                mbedtls_printf( "Unrecognized option: %s\n", argv[i] );
                mbedtls_printf( "Available options: " OPTIONS );
            }
        }
    }
#else
    /* Run all tests.*/
    memset(&todo, 1, sizeof(todo));
#endif

    mbedtls_printf("\n");

#if defined(MBEDTLS_MEMORY_BUFFER_ALLOC_C)
    mbedtls_memory_buffer_alloc_init(alloc_buf, sizeof(alloc_buf));
#endif
    memset(buf, 0xAA, sizeof(buf));
    memset(tmp, 0xBB, sizeof(tmp));


#if defined(MBEDTLS_RSA_C) && defined(MBEDTLS_GENPRIME)
    if (todo.rsa)
    {
        int keysize;
        mbedtls_rsa_context rsa;

        /* Skip 2048 and 4096 bit keys, generating takes too long */
        for (keysize = 1024; keysize <= 2048; keysize *= 2)
        {
            mbedtls_snprintf(title, sizeof(title), "RSA-%d", keysize);

            mbedtls_rsa_init(&rsa, MBEDTLS_RSA_PKCS_V15, 0);
            mbedtls_rsa_gen_key(&rsa, myrand, NULL, keysize, 65537);

            TIME_PUBLIC(title, " public", buf[0] = 0; ret = mbedtls_rsa_public(&rsa, buf, buf));

            TIME_PUBLIC(title, "private", buf[0] = 0; ret = mbedtls_rsa_private(&rsa, myrand, NULL, buf, buf));

            mbedtls_rsa_free(&rsa);
        }
    }
#endif

#if defined(MBEDTLS_ECDSA_C) && defined(MBEDTLS_SHA256_C)
    if (todo.ecdsa)
    {
        mbedtls_ecdsa_context ecdsa;
        const mbedtls_ecp_curve_info *curve_info;
        size_t sig_len;

        memset(buf, 0x2A, sizeof(buf));

        for (curve_info = mbedtls_ecp_curve_list(); curve_info->grp_id != MBEDTLS_ECP_DP_NONE; curve_info++)
        {
            mbedtls_ecdsa_init(&ecdsa);

            if (mbedtls_ecdsa_genkey(&ecdsa, curve_info->grp_id, myrand, NULL) != 0)
                mbedtls_exit(1);
            ecp_clear_precomputed(&ecdsa.grp);

            mbedtls_snprintf(title, sizeof(title), "ECDSA-%s", curve_info->name);
            TIME_PUBLIC(title, "sign",
                        ret = mbedtls_ecdsa_write_signature(&ecdsa, MBEDTLS_MD_SHA256, buf, curve_info->bit_size, tmp,
                                                            &sig_len, myrand, NULL));

            mbedtls_ecdsa_free(&ecdsa);
        }

        for (curve_info = mbedtls_ecp_curve_list(); curve_info->grp_id != MBEDTLS_ECP_DP_NONE; curve_info++)
        {
            mbedtls_ecdsa_init(&ecdsa);

            if (mbedtls_ecdsa_genkey(&ecdsa, curve_info->grp_id, myrand, NULL) != 0 ||
                mbedtls_ecdsa_write_signature(&ecdsa, MBEDTLS_MD_SHA256, buf, curve_info->bit_size, tmp, &sig_len,
                                              myrand, NULL) != 0)
            {
                mbedtls_exit(1);
            }
            ecp_clear_precomputed(&ecdsa.grp);

            mbedtls_snprintf(title, sizeof(title), "ECDSA-%s", curve_info->name);
            TIME_PUBLIC(title, "verify",
                        ret = mbedtls_ecdsa_read_signature(&ecdsa, buf, curve_info->bit_size, tmp, sig_len));

            mbedtls_ecdsa_free(&ecdsa);
        }
    }
#endif

#if defined(MBEDTLS_ECDH_C)
    if (todo.ecdh)
    {
        mbedtls_ecdh_context ecdh;
        mbedtls_mpi z;
        const mbedtls_ecp_curve_info montgomery_curve_list[] = {
#if defined(MBEDTLS_ECP_DP_CURVE25519_ENABLED)
            {MBEDTLS_ECP_DP_CURVE25519, 0, 0, "Curve25519"},
#endif
#if defined(MBEDTLS_ECP_DP_CURVE448_ENABLED)
            {MBEDTLS_ECP_DP_CURVE448, 0, 0, "Curve448"},
#endif
            {MBEDTLS_ECP_DP_NONE, 0, 0, 0}
        };
        const mbedtls_ecp_curve_info *curve_info;
        size_t olen;

        for (curve_info = mbedtls_ecp_curve_list(); curve_info->grp_id != MBEDTLS_ECP_DP_NONE; curve_info++)
        {
            mbedtls_ecdh_init(&ecdh);

            if (mbedtls_ecp_group_load(&ecdh.grp, curve_info->grp_id) != 0 ||
                mbedtls_ecdh_make_public(&ecdh, &olen, buf, sizeof(buf), myrand, NULL) != 0 ||
                mbedtls_ecp_copy(&ecdh.Qp, &ecdh.Q) != 0)
            {
                mbedtls_exit(1);
            }
            ecp_clear_precomputed(&ecdh.grp);

            mbedtls_snprintf(title, sizeof(title), "ECDHE-%s", curve_info->name);
            TIME_PUBLIC(title, "handshake",
                        ret |= mbedtls_ecdh_make_public(&ecdh, &olen, buf, sizeof(buf), myrand, NULL);
                        ret |= mbedtls_ecdh_calc_secret(&ecdh, &olen, buf, sizeof(buf), myrand, NULL));
            mbedtls_ecdh_free(&ecdh);
        }

        /* Montgomery curves need to be handled separately */
        for (curve_info = montgomery_curve_list; curve_info->grp_id != MBEDTLS_ECP_DP_NONE; curve_info++)
        {
            mbedtls_ecdh_init(&ecdh);
            mbedtls_mpi_init(&z);

            if (mbedtls_ecp_group_load(&ecdh.grp, curve_info->grp_id) != 0 ||
                mbedtls_ecdh_gen_public(&ecdh.grp, &ecdh.d, &ecdh.Qp, myrand, NULL) != 0)
            {
                mbedtls_exit(1);
            }

            mbedtls_snprintf(title, sizeof(title), "ECDHE-%s", curve_info->name);
            TIME_PUBLIC(title, "handshake", ret |= mbedtls_ecdh_gen_public(&ecdh.grp, &ecdh.d, &ecdh.Q, myrand, NULL);
                        ret |= mbedtls_ecdh_compute_shared(&ecdh.grp, &z, &ecdh.Qp, &ecdh.d, myrand, NULL));

            mbedtls_ecdh_free(&ecdh);
            mbedtls_mpi_free(&z);
        }

        for (curve_info = mbedtls_ecp_curve_list(); curve_info->grp_id != MBEDTLS_ECP_DP_NONE; curve_info++)
        {
            mbedtls_ecdh_init(&ecdh);

            if (mbedtls_ecp_group_load(&ecdh.grp, curve_info->grp_id) != 0 ||
                mbedtls_ecdh_make_public(&ecdh, &olen, buf, sizeof(buf), myrand, NULL) != 0 ||
                mbedtls_ecp_copy(&ecdh.Qp, &ecdh.Q) != 0 ||
                mbedtls_ecdh_make_public(&ecdh, &olen, buf, sizeof(buf), myrand, NULL) != 0)
            {
                mbedtls_exit(1);
            }
            ecp_clear_precomputed(&ecdh.grp);

            mbedtls_snprintf(title, sizeof(title), "ECDH-%s", curve_info->name);
            TIME_PUBLIC(title, "handshake",
                        ret |= mbedtls_ecdh_calc_secret(&ecdh, &olen, buf, sizeof(buf), myrand, NULL));
            mbedtls_ecdh_free(&ecdh);
        }

        /* Montgomery curves need to be handled separately */
        for (curve_info = montgomery_curve_list; curve_info->grp_id != MBEDTLS_ECP_DP_NONE; curve_info++)
        {
            mbedtls_ecdh_init(&ecdh);
            mbedtls_mpi_init(&z);

            if (mbedtls_ecp_group_load(&ecdh.grp, curve_info->grp_id) != 0 ||
                mbedtls_ecdh_gen_public(&ecdh.grp, &ecdh.d, &ecdh.Qp, myrand, NULL) != 0 ||
                mbedtls_ecdh_gen_public(&ecdh.grp, &ecdh.d, &ecdh.Q, myrand, NULL) != 0)
            {
                mbedtls_exit(1);
            }

            mbedtls_snprintf(title, sizeof(title), "ECDH-%s", curve_info->name);
            TIME_PUBLIC(title, "handshake",
                        ret |= mbedtls_ecdh_compute_shared(&ecdh.grp, &z, &ecdh.Qp, &ecdh.d, myrand, NULL));

            mbedtls_ecdh_free(&ecdh);
            mbedtls_mpi_free(&z);
        }
    }
#endif

    mbedtls_printf("\r\n");

#if defined(MBEDTLS_MEMORY_BUFFER_ALLOC_C)
    mbedtls_memory_buffer_alloc_free();
#endif

#if defined(_WIN32)
    mbedtls_printf("  Press Enter to exit this program.\n");
    fflush(stdout);
    getchar();
#endif

    while (1)
    {
        char ch = GETCHAR();
        PUTCHAR(ch);
    }
}

#endif /* MBEDTLS_TIMING_C */
