/*
 * Copyright 2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */



#define _GNU_SOURCE
/*=========================================================================

Include Files

===========================================================================*/
#include <arpa/inet.h>
#include <assert.h>
#include <linux/if_tun.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>   
#include <unistd.h>

// netlink includes
#include <asm/types.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <net/if.h>

#include "Framer.h"
#include "FSCIFrame.h"
#include "PhysicalDevice.h"
#include "UARTConfiguration.h"
#include "Zigbee_BlackBox_HSDK.h"

/*=========================================================================

Public type definitions

===========================================================================*/

/* FSCI standard payloads */
static uint8_t set_ch_buf[]             = {0x00, 0x00, 0x00, 0x00};
static uint8_t set_device_type_buf[] 	= {ROUTER_DEVICE_TYPE};

/*! ************************************************************************
\fn uint8_t IntToHexChannel(uint8_t value)

\brief  converts an integet channel value to hexadecimal.

\param      value   integer value

\retval     uint8_t     hexadecimal
 ************************************************************************* */
static uint8_t IntToHexChannel(uint8_t value)
{
   	uint8_t hex = 0x0B;
   	hex += (value - hex);
   	return hex;
}

/*! ************************************************************************
\fn int GetBaudrate(int value)

\brief  converts baudrate value to enumeration.

\param      value   selected baudrate value by the user

\retval     int     enumaration baudrate value
 ************************************************************************* */
static int GetBaudrate(int value)
{

    if (value == 115200)
        return BR115200;

    return -1;
}

/*! ************************************************************************
\fn void ReverseByteArray(uint8_t *buf, int size)

\brief  This function reverses an array positions. 
        [0x11, 0x22, 0x33, 0x44] -> [0x44, 0x33, 0x22, 0x11]

\param      buf     pointer to the buffer that wants to be reversed.
\param      size    size of the buffer.
 ************************************************************************* */
static void ReverseByteArray(uint8_t *buf, int size)
{
    int i;
    uint8_t *temp = malloc(size);

    memcpy(temp, buf, size);

    for (i = 0; i < size; i++) {
        buf[i] = temp[size - i - 1];
    }

    free(temp);
}

/*! ************************************************************************
\fn void FrameReceived(void *callee, void *response)

\brief  Executes on every RX packet.

\param      response    data with the received frame. 
 ************************************************************************* */
static void FrameReceived(void *callee, void *response)
{
    int16_t i;
    uint8_t opCode;
    FSCIFrame *frame = (FSCIFrame *)response;               /* Cast the received frame */ 

    if (frame->opGroup != RX_OG && frame->opGroup != RX_DISC_OG && frame->opGroup != TX_OG && frame->opGroup != RX_RSP_OG) {
    	printf("Destroy frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, frame->opCode);
        DestroyFSCIFrame(frame);
        return;
    }

    opCode = frame->opCode;
    if(opCode == 0x00 && frame->opGroup == RX_RSP_OG){
    }
    else if(opCode == 0x01 && frame->opGroup == RX_RSP_OG){
    	opCode = ON_OFF_CMD_OC;
    }
    else if(opCode == 0x02 && frame->opGroup == RX_RSP_OG){
    	opCode = READ_ATT_OC;
    }
    else if(opCode == 0x00){
    	opCode = frame->data[frame->length - 1];
    }
    
	switch (opCode) {

	case FACTORY_RST_OC:
		printf("RX: FactoryReset.Status");
		if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        }
		break;

	case FACTORY_NEW_RESTART_OC:
		printf("RX: FactoryNewRestart\n");
		if (frame->data[0] == 0x00) {
            printf("\t STARTUP\n");
        }
		break;

	case NODE_CMD_ID_LIST_OC:
		printf("RX: NodeCommandIDList\n");
		break;

     case ROUTER_DISCOVERY_OC:
        break;   

	case NODE_CLUSTER_ATT_LIST_OC:
		printf("RX: NodeClusterAttributeList:\n");
#if SEE_ATTRIBUTE_LIST
		printf("\t Endpoint: %02x\n",frame->data[0]);
		printf("\t Profile ID: %02x%02x\n",frame->data[1],frame->data[2]);
		printf("\t Cluster ID: %02x%02x\n",frame->data[3],frame->data[4]);
		printf("\t Attribute List: ");
		for(i = 5; i < frame->length ; i+= 2){
			printf("%02x%02x,",frame->data[i],frame->data[i+1]);
		}
		printf("\n");
#endif
		break;

	case NODE_CLUSTER_LIST_OC:
		printf("RX: NodeClusterList\n");
		break;

	case SET_CHANNEL_MASK_OC:
		printf("RX: SetChannel.Status");
        if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        }
		break;

	case SET_DEVICE_TYPE_OC:
		printf("RX: SetDeviceType.Status");
		if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        }
		break;

	case NETWORK_SCAN_OC:
		printf("RX: NetworkScan.Status");
		if (frame->data[0] == 0x00) {
            printf(" -> Success\n");
        }
		break;

	case CREATE_NWK_OC:
		printf("NetworkJoinedFormed.Response");
		if(frame->data[0] == 0x01){
			printf("-> Joined to Existing Network\n");
			printf("\t Short Address: %02x%02x\n",frame->data[1],frame->data[2]);
			printf("\t Extended Address:");
			for(i = 0; i < 8 ; i++){
				printf("%02x", frame->data[3 + i]);
			}
			printf("\n\t Channel: %02x\n", frame->data[3 + i]);
		}
		else if(frame->data[0] == 0x00){
			printf(" -> Formed New Network\n");
		}
		
		break;

	case NODE_DESCRIPTOR_OC:
		printf("RX: Node Descriptor\n");
		break;

	default:
        printf("RX: unknown frame (OG, OC) = (%02x, %02x)\n", frame->opGroup, opCode);
        break;
	}

	DestroyFSCIFrame(frame);
}

int main(int argc, char **argv)
{
    /* Check number of arguments. */
    if (argc < 4) {
        printf("Usage UART: # %s {/dev/ttyACMx | /dev/ttymxcx} channel [baudrate bps]\n", argv[0]);
        exit(1);
    } 

    /* Begin UART configuration */
    DeviceType dev_type = UART;
    void *serial_config = defaultConfigurationData();
    int baudrate = GetBaudrate(atoi(argv[3]));

    if (baudrate == -1) {
        printf("Wrong baudrate value.\n");
        exit(1);
    } else {
        setBaudrate(serial_config, baudrate);
    }
    /* End UART configuration */

    /* Get selected channel */
    uint8_t channel = atoi(argv[2]);
    assert(channel >= 11 && channel <= 26);
	set_ch_buf[3] = IntToHexChannel(channel);


    PhysicalDevice *device = InitPhysicalDevice(dev_type, serial_config, argv[1], GLOBAL);              /* Initialize device with UART configuration and the selected port */
    Framer *framer = InitializeFramer(device, FSCI, LENGTH_FIELD_SIZE, CRC_FIELD_SIZE, _LITTLE_ENDIAN); /* Configure framer */
    OpenPhysicalDevice(device);                                                                         /* Open kinetis device */
    AttachToFramer(framer, NULL, FrameReceived);                                                        /* Set callback */

    /* Frames */
    FSCIFrame *factory_reset        = CreateFSCIFrame(framer, TX_OG, FACTORY_RST_OC,               NULL, 0, VIF);
    FSCIFrame *set_channel          = CreateFSCIFrame(framer, TX_OG, SET_CHANNEL_MASK_OC,          set_ch_buf, sizeof(set_ch_buf), VIF);
    FSCIFrame *set_device_type      = CreateFSCIFrame(framer, TX_OG, SET_DEVICE_TYPE_OC,           set_device_type_buf, sizeof(set_device_type_buf), VIF);
    FSCIFrame *network_scan         = CreateFSCIFrame(framer, TX_OG, NETWORK_SCAN_OC,              NULL, 0, VIF);

    /* Factory reset */
    printf("TX: FactoryReset\n");
    SendFrame(framer, factory_reset);
    sleep(1);

    /* Set channel */
    printf("TX: SetChannelMask\n");
    SendFrame(framer, set_channel);
    sleep(1);

    /* Set device type */
    printf("TX: SetDeviceType\n");
    SendFrame(framer, set_device_type);
    sleep(1);

    /* Network Scan */
    printf("TX: StartNetworkScan\n");
    SendFrame(framer, network_scan);
    sleep(1);


    while(TRUE)
    {

    }
}