/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "rpmsg_lite.h"
#include "rpmsg_queue.h"
#include "rpmsg_ns.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "board.h"
#include "fsl_debug_console.h"
#include "FreeRTOS.h"
#include "task.h"

#include "app_srtm.h"
#include "coremark.h"

#include "fsl_msmc.h"
#include "fsl_lpuart.h"
#include "fsl_iomuxc.h"
#include "fsl_pmc0.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define APP_DEBUG_UART_BAUDRATE       (115200U)             /* Debug console baud rate. */
#define APP_DEBUG_UART_DEFAULT_CLKSRC kCLOCK_IpSrcSircAsync /* SCG SIRC clock. */

#define APP_TASK_STACK_SIZE (256)

//#define PRINT_DEBUG

/* Enable GPIO PAD low power operation.
 * NOTE: ONLY ON EVK BOARD CAN THIS OPTION SET TO 1.
 * Rational: i.MX7ULP supports voltage detection circuitry on pads which results in additional power consumption.
 *           We can disable this feature to save power, but it may cause malfunction or even SoC pad damage.
 *           Please read "GPIO pads operating range configuration" in Reference Manual SIM module carefully
 *           before turn APP_ENABLE_GPIO_PAD_LOW_POWER to "1". If there's any change on board, the pad range setting
 *           code need to be modified accordingly.
 */
#define APP_ENABLE_GPIO_PAD_LOW_POWER (1)

extern bool BOARD_IsRunOnQSPI(void);
extern void APP_UpdateSimDgo(uint32_t gpIdx, uint32_t mask, uint32_t value);

/* Power mode definition of low power management.
 * Waken up duration Off > Dsm > Idle > Wait > Run.
 */
typedef enum _lpm_power_mode
{
    LPM_PowerModeRun = 0, /* Normal RUN mode */
    LPM_PowerModeWait,    /* WAIT mode. */
    LPM_PowerModeStop,    /* STOP mode. */
    LPM_PowerModeVlpr,    /* VLPR mode. */
    LPM_PowerModeVlpw,    /* VLPW mode. */
    LPM_PowerModeVlps,    /* VLPS mode. */
    LPM_PowerModeHsrun,   /* HighSpeed RUN mode */
    LPM_PowerModeLls,     /* LLS mode */
    LPM_PowerModeVlls,    /* VLLS mode */
} lpm_power_mode_t;

static uint32_t IOMUX_PTA_ARRAY[32];       /*Array to store the IOMUX config for PTA*/
static uint32_t IOMUX_PTB_ARRAY[20];       /*Array to store the IOMUX config for PTB*/

extern const scg_sys_clk_config_t g_sysClkConfigVlprFirc;
extern const scg_sys_clk_config_t g_sysClkConfigVlprSirc;

typedef enum 
{
    sysClkConfigVlprFirc,
    sysClkConfigVlprFircAN12573,
    sysClkConfigVlprSirc,
} VlprTargetClockConfigs_t;

/* Default VLPR frequency = 48Mhz from SIRC, default dividers config */
static uint32_t VlprTargetClockConfig = sysClkConfigVlprFirc;
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/* Set the clock configuration for HSRUN mode. */
void APP_SetClockHsrun(void);
/* Set the clock configuration for RUN mode from HSRUN mode. */
void APP_SetClockRunFromHsrun(void);
/* Set the clock configuration for RUN mode from VLPR mode. */
void APP_SetClockRunFromVlpr(void);
/* Set the clock configuration for VLPR mode. */
void APP_SetClockVlpr(uint32_t VlprTargetClockConfig);

static const pmc0_hsrun_mode_config_t s_pmc0HsrunModeConfig = {
    .coreRegulatorVoltLevel = 33U, /* 0.596 + 33 * 0.01083 = 0.95339 */
    .enableForwardBias      = 1U};

static const pmc0_run_mode_config_t s_pmc0RunModeConfig = {.coreRegulatorVoltLevel =
                                                               28U}; /* 0.596 + 28 * 0.01083 = 0.89924 */

static const pmc0_vlpr_mode_config_t s_pmc0VlprModeConfig = {
    .arrayRegulatorSelect   = 0U,
    .coreRegulatorSelect    = 0U,
    .lvdMonitorSelect       = 0U,
    .hvdMonitorSelect       = 0U,
    .enableForceHpBandgap   = 0U,
    .coreRegulatorVoltLevel = 24U, /* 0.596 + 24 * 0.01083 = 0.85592 */
    .enableReverseBackBias  = 1U};

static const pmc0_stop_mode_config_t s_pmc0StopModeConfig = {.coreRegulatorVoltLevel =
                                                                 28U}; /* 0.596 + 28 * 0.01083 = 0.89924 */

static const pmc0_vlps_mode_config_t s_pmc0VlpsModeConfig = {
    .arrayRegulatorSelect   = 0U,
    .coreRegulatorSelect    = 0U,
    .lvdMonitorSelect       = 0U,
    .hvdMonitorSelect       = 0U,
    .enableForceHpBandgap   = 0U,
    .coreRegulatorVoltLevel = 28U, /* 0.596 + 28 * 0.01083 = 0.89924 */
    .enableReverseBackBias  = 1U};

static const pmc0_lls_mode_config_t s_pmc0LlsModeConfig = {
    .arrayRegulatorSelect   = 0U,
    .coreRegulatorSelect    = 0U,
    .lvdMonitorSelect       = 0U,
    .hvdMonitorSelect       = 0U,
    .enableForceHpBandgap   = 0U,
    .coreRegulatorVoltLevel = 13U, /* 0.596 + 13 * 0.01083 = 0.73679 */
    .enableReverseBackBias  = 1U};

static const pmc0_vlls_mode_config_t s_pmc0VllsModeConfig = {
    .arrayRegulatorSelect = 2U, .lvdMonitorSelect = 0U, .hvdMonitorSelect = 0U, .enableForceHpBandgap = 0U};

/*******************************************************************************
 * Code
 ******************************************************************************/
static void APP_InitDebugConsole(void)
{
    CLOCK_SetIpSrc(BOARD_DEBUG_UART_PCC_ADDRESS, APP_DEBUG_UART_DEFAULT_CLKSRC);
    uint32_t uartClkSrcFreq = CLOCK_GetIpFreq(BOARD_DEBUG_UART_PCC_ADDRESS);
    DbgConsole_Init(BOARD_DEBUG_UART_INSTANCE, APP_DEBUG_UART_BAUDRATE, BOARD_DEBUG_UART_TYPE, uartClkSrcFreq);
}

static TaskHandle_t app_task_handle = NULL;

void APP_InitPMC0(void)
{
    pmc0_bias_config_t bcfg = {.RBBNWellVoltageLevelSelect = 5,
                               .RBBPWellVoltageLevelSelect = 5,
                               .DisableRBBPullDown         = 0,
                               .FBBNWellVoltageLevelSelect = 5,
                               .FBBPWellVoltageLevelSelect = 5};

    /* Initialize PMC0 */
    PMC0_SetBiasConfig(&bcfg);
    PMC0_ConfigureHsrunMode(&s_pmc0HsrunModeConfig);
    PMC0_ConfigureRunMode(&s_pmc0RunModeConfig);
    PMC0_ConfigureVlprMode(&s_pmc0VlprModeConfig);
    PMC0_ConfigureStopMode(&s_pmc0StopModeConfig);
    PMC0_ConfigureVlpsMode(&s_pmc0VlpsModeConfig);
    PMC0_ConfigureLlsMode(&s_pmc0LlsModeConfig);
    PMC0_ConfigureVllsMode(&s_pmc0VllsModeConfig);
}

static void BOARD_InitClock(void)
{
    CLOCK_EnableClock(kCLOCK_PctlA);
    CLOCK_EnableClock(kCLOCK_PctlB);
    CLOCK_EnableClock(kCLOCK_Rgpio2p0);

    BOARD_BootClockRUN();
    CLOCK_SetVlprModeSysClkConfig(&g_sysClkConfigVlprSirc);

    CLOCK_SetIpSrc(kCLOCK_Lpi2c3, kCLOCK_IpSrcSircAsync);
    CLOCK_SetIpSrc(kCLOCK_Lpi2c0, kCLOCK_IpSrcSystem);
}

static void BOARD_InitClockAndPins(void)
{
    BOARD_InitClock();
    BOARD_InitPins_Core1();
    APP_SRTM_I2C_ReleaseBus();
    BOARD_I2C_ConfigurePins();
}

static void APP_ShowPowerMode(smc_power_state_t powerMode)
{
    switch (powerMode)
    {
        case kSMC_PowerStateRun:
            PRINTF("    Power mode: RUN\r\n");
            break;
        case kSMC_PowerStateVlpr:
            PRINTF("    Power mode: VLPR\r\n");
            break;
        case kSMC_PowerStateHsrun:
            PRINTF("    Power mode: HSRUN\r\n");
            break;
        default:
            PRINTF("    Power mode wrong\r\n");
            break;
    }
}

void APP_PowerPreSwitchHook(smc_power_state_t originPowerState, lpm_power_mode_t targetMode)
{
    if ((LPM_PowerModeRun != targetMode) && (LPM_PowerModeHsrun != targetMode) && (LPM_PowerModeVlpr != targetMode))
    {
        /* Wait for debug console output finished. */
        while (!(kLPUART_TransmissionCompleteFlag & LPUART_GetStatusFlags((LPUART_Type *)BOARD_DEBUG_UART_BASEADDR)))
        {
        }
        DbgConsole_Deinit();
        /*
         * Set pin for current leakage.
         * Debug console RX pin: Set to pinmux to analog.
         * Debug console TX pin: Set to pinmux to analog.
         */
        IOMUXC_SetPinMux(IOMUXC_PTA19_CMP1_IN3_3V, 0);
        IOMUXC_SetPinConfig(IOMUXC_PTA19_CMP1_IN3_3V, 0);
        IOMUXC_SetPinMux(IOMUXC_PTA18_CMP1_IN1_3V, 0);
        IOMUXC_SetPinConfig(IOMUXC_PTA18_CMP1_IN1_3V, 0);
    }
}

void APP_PowerPostSwitchHook(smc_power_state_t originPowerState, lpm_power_mode_t targetMode, bool result)
{
    if ((LPM_PowerModeRun != targetMode) && (LPM_PowerModeHsrun != targetMode) && (LPM_PowerModeVlpr != targetMode))
    {
        /*
         * Debug console RX pin was set to disable for current leakage, need to re-configure pinmux.
         * Debug console TX pin was set to disable for current leakage, need to re-configure pinmux.
         */
        IOMUXC_SetPinMux(IOMUXC_PTA18_LPUART0_TX, 0U);
        IOMUXC_SetPinConfig(IOMUXC_PTA18_LPUART0_TX, IOMUXC0_SW_MUX_CTL_PAD_PE_MASK | IOMUXC0_SW_MUX_CTL_PAD_PS_MASK);
        IOMUXC_SetPinMux(IOMUXC_PTA19_LPUART0_RX, 0U);
        IOMUXC_SetPinConfig(IOMUXC_PTA19_LPUART0_RX, IOMUXC0_SW_MUX_CTL_PAD_PE_MASK | IOMUXC0_SW_MUX_CTL_PAD_PS_MASK);
        APP_InitDebugConsole();
    }
    PRINTF("== Power switch %s ==\r\n", result ? "OK" : "FAIL");
}

static status_t APP_PowerModeSwitch(smc_power_state_t curPowerState, lpm_power_mode_t targetPowerMode, uint32_t VlprTargetClockConfig)
{
    status_t status = kStatus_Success;

    switch (targetPowerMode)
    {
        case LPM_PowerModeVlpr:
            APP_SetClockVlpr(VlprTargetClockConfig);
            status = SMC_SetPowerModeVlpr(MSMC0);
            while (kSMC_PowerStateVlpr != SMC_GetPowerModeState(MSMC0))
            {
            }
            break;

        case LPM_PowerModeRun:
            /* If enter RUN from HSRUN, fisrt change clock. */
            if (kSMC_PowerStateHsrun == curPowerState)
            {
                APP_SetClockRunFromHsrun();
            }

            /* Power mode change. */
            status = SMC_SetPowerModeRun(MSMC0);
            while (kSMC_PowerStateRun != SMC_GetPowerModeState(MSMC0))
            {
            }

            /* If enter RUN from VLPR, change clock after the power mode change. */
            if (kSMC_PowerStateVlpr == curPowerState)
            {
                APP_SetClockRunFromVlpr();
            }
            break;

        case LPM_PowerModeHsrun:
            status = SMC_SetPowerModeHsrun(MSMC0);
            while (kSMC_PowerStateHsrun != SMC_GetPowerModeState(MSMC0))
            {
            }

            APP_SetClockHsrun(); /* Change clock setting after power mode change. */
            break;

        default:
            PRINTF("Wrong value\r\n");
            break;
    }

    if (status != kStatus_Success)
    {
        PRINTF("!!!! Power switch failed !!!!!\r\n");
    }

    return status;
}

void PowerOpmizeGPIOAandB(void)
{
    uint32_t pad;
   
    /* Enable RGPIO clock gate */
    CLOCK_EnableClock(kCLOCK_Rgpio2p0);
    
    /* Enable port A and B clock gates */
    CLOCK_EnableClock(kCLOCK_PctlA);
    CLOCK_EnableClock(kCLOCK_PctlB);

    /*Configure all PTAs as GPIO (IOMUX = 0x1)*/
    for(pad = kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTA0; pad <= kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTA31; pad++) 
    {
        IOMUXC0->SW_MUX_CTL_PAD[pad] = 0x100; 
    } 

    FGPIOA->PDOR = 0;     //set all GPIOA pins as 0
    FGPIOA->PDDR = 0;     //Configure GPIOA as Input

    //GPIOB controls QSPI. Should be optimized only when run from RAM
    if (!BOARD_IsRunOnQSPI())
    {
        /*Configure all PTBs as GPIO (IOMUX = 0x1)*/
        for(pad = kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB0; pad <= kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB19; pad++) 
        {
            IOMUXC0->SW_MUX_CTL_PAD[pad] = 0x100; 
        } 
  
        FGPIOB->PDOR = 0;     //set all GPIOB pins as 0
        FGPIOB->PDDR = 0;     //Configure GPIOB as Input
    }
}

void SaveContextBeforeVLLS_LLS(void)
{
    uint32_t pad, index;
 
    /*Copy IOMUX values for all PTAs*/
    for(pad = kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTA0; pad <= kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTA31; pad++) 
    {
      IOMUX_PTA_ARRAY[pad] = IOMUXC0->SW_MUX_CTL_PAD[pad];
    } 
    
    /*Copy IOMUX values for all PTBs*/
    for(pad = kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB0; pad <= kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB19; pad++) 
    {
      index = pad - kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB0;
      IOMUX_PTB_ARRAY[index] = IOMUXC0->SW_MUX_CTL_PAD[pad];
    } 
  
}

static void PowerOpmizeVLPRandRUN(void)
{
    taskENTER_CRITICAL();
    SaveContextBeforeVLLS_LLS();
    PowerOpmizeGPIOAandB();
    //restore IOMUX for UART0 as it was disabled on function PowerOpmizeGPIOAandB
    IOMUXC0->SW_MUX_CTL_PAD[18] = IOMUX_PTA_ARRAY[18];
    IOMUXC0->SW_MUX_CTL_PAD[19] = IOMUX_PTA_ARRAY[19];
    /* PCC 0 */
    CLOCK_DisableClock(kCLOCK_Dma0);
//	CLOCK_DisableClock(kCLOCK_Rgpio2p0);
    CLOCK_DisableClock(kCLOCK_Xrdc0);
    CLOCK_DisableClock(kCLOCK_Sema420);
    CLOCK_DisableClock(kCLOCK_Dmamux0);
//    CLOCK_DisableClock(kCLOCK_MuA);                 
    CLOCK_DisableClock(kCLOCK_Wdog0);
    CLOCK_DisableClock(kCLOCK_Crc0);
    CLOCK_DisableClock(kCLOCK_Ltc0);
    CLOCK_DisableClock(kCLOCK_Lpit0);
    CLOCK_DisableClock(kCLOCK_Lptmr0);
    CLOCK_DisableClock(kCLOCK_Lptmr1);
    CLOCK_DisableClock(kCLOCK_Tpm0);
    CLOCK_DisableClock(kCLOCK_Tpm1);
    CLOCK_DisableClock(kCLOCK_Flexio0);
    CLOCK_DisableClock(kCLOCK_Lpi2c0);
    CLOCK_DisableClock(kCLOCK_Lpi2c1);
    CLOCK_DisableClock(kCLOCK_Lpi2c2);             
    CLOCK_DisableClock(kCLOCK_Lpi2c3);
    CLOCK_DisableClock(kCLOCK_Sai0);
    CLOCK_DisableClock(kCLOCK_Lpspi0);
    CLOCK_DisableClock(kCLOCK_Lpspi1);
//	CLOCK_DisableClock(kCLOCK_Lpuart0);               
    CLOCK_DisableClock(kCLOCK_Lpuart1);
//	CLOCK_DisableClock(kCLOCK_PctlA);
    CLOCK_DisableClock(kCLOCK_PctlB);
    CLOCK_DisableClock(kCLOCK_Adc0);      
    CLOCK_DisableClock(kCLOCK_Cmp0);
    CLOCK_DisableClock(kCLOCK_Cmp1);
    CLOCK_DisableClock(kCLOCK_Dac0);           
    CLOCK_DisableClock(kCLOCK_Dac1);
    CLOCK_DisableClock(kCLOCK_Snvs);

    /* PCC 1 */
    CLOCK_DisableClock(kCLOCK_Tpiu);  
    if (!BOARD_IsRunOnQSPI())
    {
        CLOCK_DisableClock(kCLOCK_Qspi);
    }    
    CLOCK_DisableClock(kCLOCK_Tpm2);               
    CLOCK_DisableClock(kCLOCK_Tpm3);               
    CLOCK_DisableClock(kCLOCK_Sai1);               
    CLOCK_DisableClock(kCLOCK_Lpuart2);               
    CLOCK_DisableClock(kCLOCK_Lpuart3);               
    CLOCK_DisableClock(kCLOCK_Adc1);  
    taskEXIT_CRITICAL();  
}

#ifdef PRINT_DEBUG
static void IsPcc0Pcc1ClockEnabled(void)
{
    taskENTER_CRITICAL();
    PRINTF("IsPCC0PCC1ClockEnabled:\r\n");
    /* PCC 0 */
    PRINTF("kCLOCK_Dma0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Dma0));
    PRINTF("kCLOCK_Rgpio2p0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Rgpio2p0));
    PRINTF("kCLOCK_Xrdc0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Xrdc0));
    PRINTF("kCLOCK_Sema420=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Sema420));
    PRINTF("kCLOCK_Dmamux0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Dmamux0));
    PRINTF("kCLOCK_MuA=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_MuA));
    PRINTF("kCLOCK_Wdog0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Wdog0));
    PRINTF("kCLOCK_Crc0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Crc0));
    PRINTF("kCLOCK_Ltc0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Ltc0));
    PRINTF("kCLOCK_Lpit0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpit0));
    PRINTF("kCLOCK_Lptmr0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lptmr0));
    PRINTF("kCLOCK_Lptmr1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lptmr1));
    PRINTF("kCLOCK_Tpm0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Tpm0));
    PRINTF("kCLOCK_Tpm1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Tpm1));
    PRINTF("kCLOCK_Flexio0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Flexio0));
    PRINTF("kCLOCK_Lpi2c0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpi2c0));
    PRINTF("kCLOCK_Lpi2c1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpi2c1));
    PRINTF("kCLOCK_Lpi2c2=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpi2c2));       
    PRINTF("kCLOCK_Lpi2c3=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpi2c3));
    PRINTF("kCLOCK_Sai0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Sai0));
    PRINTF("kCLOCK_Lpspi0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpspi0));
    PRINTF("kCLOCK_Lpspi1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpspi1));
    PRINTF("kCLOCK_Lpuart0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpuart0));
    PRINTF("kCLOCK_Lpuart1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpuart1));
    PRINTF("kCLOCK_PctlA=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_PctlA));
    PRINTF("kCLOCK_PctlB=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_PctlB));
    PRINTF("kCLOCK_Adc0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Adc0));
    PRINTF("kCLOCK_Cmp0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Cmp0));
    PRINTF("kCLOCK_Cmp1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Cmp1));
    PRINTF("kCLOCK_Dac0=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Dac0));
    PRINTF("kCLOCK_Dac1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Dac1));
    PRINTF("kCLOCK_Snvs=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Snvs));

    /* PCC 1 */
    PRINTF("kCLOCK_Tpiu=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Tpiu));
    PRINTF("kCLOCK_Qspi=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Qspi));
    PRINTF("kCLOCK_Tpm2=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Tpm2));
    PRINTF("kCLOCK_Tpm3=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Tpm3));
    PRINTF("kCLOCK_Sai1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Sai1));
    PRINTF("kCLOCK_Lpuart2=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpuart2));
    PRINTF("kCLOCK_Lpuart3=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Lpuart3));
    PRINTF("kCLOCK_Adc1=0x%x\r\n", (*(volatile uint32_t *)kCLOCK_Adc1));
    taskEXIT_CRITICAL();
}
#endif

void RestoreContextAfterVLPR(void)
{
     static uint32_t pad, index;

    /* Enable RGPIO clock gate */
    CLOCK_EnableClock(kCLOCK_Rgpio2p0);

    /* Enable port A and B clock gates */
    CLOCK_EnableClock(kCLOCK_PctlA);
    CLOCK_EnableClock(kCLOCK_PctlB);

    /*Restore IOMUX values for all PTAs*/
    for(pad = kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTA0; pad <= kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTA31; pad++) 
    {
        IOMUXC0->SW_MUX_CTL_PAD[pad] = IOMUX_PTA_ARRAY[pad];
    }

    /*Restore IOMUX values for all PTBs*/
    for(pad = kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB0; pad <= kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB19; pad++) 
    {
        index = pad - kIOMUXC0_IOMUXC0_SW_MUX_CTL_PAD_PTB0;
        IOMUXC0->SW_MUX_CTL_PAD[pad] = IOMUX_PTB_ARRAY[index];
    }
}

void EnableALLPerithClockPCC0andPCC1(void)
{
    /* PCC 0 */
    CLOCK_EnableClock(kCLOCK_Dma0);
    CLOCK_EnableClock(kCLOCK_Rgpio2p0);
    CLOCK_EnableClock(kCLOCK_Xrdc0);
    CLOCK_EnableClock(kCLOCK_Sema420);
    CLOCK_EnableClock(kCLOCK_Dmamux0);
    CLOCK_EnableClock(kCLOCK_MuA);                 
    CLOCK_EnableClock(kCLOCK_Wdog0);
    CLOCK_EnableClock(kCLOCK_Crc0);
    CLOCK_EnableClock(kCLOCK_Ltc0);
    CLOCK_EnableClock(kCLOCK_Lpit0);
    CLOCK_EnableClock(kCLOCK_Lptmr0);
    CLOCK_EnableClock(kCLOCK_Lptmr1);
    CLOCK_EnableClock(kCLOCK_Tpm0);
    CLOCK_EnableClock(kCLOCK_Tpm1);
    CLOCK_EnableClock(kCLOCK_Flexio0);
    CLOCK_EnableClock(kCLOCK_Lpi2c0);
    CLOCK_EnableClock(kCLOCK_Lpi2c1);
    CLOCK_EnableClock(kCLOCK_Lpi2c2);             
    CLOCK_EnableClock(kCLOCK_Lpi2c3);
    CLOCK_EnableClock(kCLOCK_Sai0);
    CLOCK_EnableClock(kCLOCK_Lpspi0);
    CLOCK_EnableClock(kCLOCK_Lpspi1);
    CLOCK_EnableClock(kCLOCK_Lpuart0);               
    CLOCK_EnableClock(kCLOCK_Lpuart1);
    CLOCK_EnableClock(kCLOCK_PctlA);
    CLOCK_EnableClock(kCLOCK_PctlB);
    CLOCK_EnableClock(kCLOCK_Adc0);      
    CLOCK_EnableClock(kCLOCK_Cmp0);
    CLOCK_EnableClock(kCLOCK_Cmp1);
    CLOCK_EnableClock(kCLOCK_Dac0);           
    CLOCK_EnableClock(kCLOCK_Dac1);
    CLOCK_EnableClock(kCLOCK_Snvs);

    /* PCC 1 */
    CLOCK_EnableClock(kCLOCK_Tpiu);               
    CLOCK_EnableClock(kCLOCK_Qspi);   
    CLOCK_EnableClock(kCLOCK_Tpm2);               
    CLOCK_EnableClock(kCLOCK_Tpm3);               
    CLOCK_EnableClock(kCLOCK_Sai1);               
    CLOCK_EnableClock(kCLOCK_Lpuart2);               
    CLOCK_EnableClock(kCLOCK_Lpuart3);               
    CLOCK_EnableClock(kCLOCK_Adc1);    
}

static void app_task(void *param)
{
    char ch;
    status_t status = kStatus_Success;
    smc_power_state_t curPowerState;
    uint32_t freq = 0U;

    if(OCOTP_CTRL->HW_OCOTP_PDN == 1)
    {
        /* Power down OCOTP. */
        OCOTP_CTRL->HW_OCOTP_PDN = 1;                   //4.8mW improvement
    }

    for (;;)
    {
        freq = CLOCK_GetFreq(kCLOCK_CoreSysClk);
        PRINTF("\r\n####################  Coremark Test Menu ####################\n\r\n");
        PRINTF("    Build Time: %s--%s \r\n", __DATE__, __TIME__);
        PRINTF("    SysPll Clock: %dHz \r\n", CLOCK_GetFreq(kCLOCK_ScgSysPllClk));
        PRINTF("    Core Clock: %dHz \r\n", freq);
        PRINTF("    Platform Clock: %dHz \r\n", CLOCK_GetFreq(kCLOCK_PlatClk));
        PRINTF("    Bus Clock: %dHz \r\n", CLOCK_GetFreq(kCLOCK_BusClk));
        PRINTF("    Slow Clock: %dHz \r\n", CLOCK_GetFreq(kCLOCK_SlowClk));
        curPowerState = SMC_GetPowerModeState(MSMC0);
        APP_ShowPowerMode(curPowerState);
        PRINTF("\r\nSelect the desired operation \n\r\n");
        PRINTF("Press a to switch to RUN mode\r\n");
        PRINTF("Press b to switch to VLPR mode\r\n");
        PRINTF("Press c to run coremark code\r\n");
        PRINTF("Press d to set VLPR frequency to 48Mhz (FIRC) (default clock dividers config)\r\n");
        PRINTF("Press e to set VLPR frequency to 48Mhz (FIRC) (AN12573 optimized clock dividers config)\r\n");
        PRINTF("Press f to set VLPR frequency to 16Mhz (SIRC) (default clock dividers config)\r\n");
        PRINTF("\r\nWaiting for select..\r\n\r\n");

        /* Wait for user response */
        do
        {
            ch = GETCHAR();
        }
        while ((ch == '\r') || (ch == '\n'));

#ifdef PRINT_DEBUG
        PRINTF("HW_OCOTP_PDN = 0x%x\r\n", OCOTP_CTRL->HW_OCOTP_PDN);
        PRINTF("SPLLCSR = 0x%x\r\n", SCG0->SPLLCSR);            
        PRINTF("APLLCSR = 0x%x\r\n", SCG0->APLLCSR);
        PRINTF("SOSCCSR = 0x%x\r\n", SCG0->SOSCCSR);            
        PRINTF("SPLLPFD = 0x%x\r\n", SCG0->SPLLPFD);
        PRINTF("PMC0 HSRUN = 0x%x\r\n", PMC0->HSRUN);
        PRINTF("PMC0 RUN = 0x%x\r\n", PMC0->RUN);
        PRINTF("PMC0 VLPR = 0x%x\r\n", PMC0->VLPR);
        PRINTF("PMC0 STOP = 0x%x\r\n", PMC0->STOP);
        PRINTF("PMC0 VLPS = 0x%x\r\n", PMC0->VLPS);
        PRINTF("PMC0 LLS = 0x%x\r\n", PMC0->LLS);
        PRINTF("PMC0 VLLS = 0x%x\r\n", PMC0->VLLS);
#endif

        switch (ch)
        {
        case 'a':
            curPowerState = SMC_GetPowerModeState(MSMC0);
            /* If enter RUN from VLPR, change clock after the power mode change. */
            if (kSMC_PowerStateVlpr == curPowerState)
            {
                RestoreContextAfterVLPR();
                PRINTF("Not in RUN mode, transfer to RUN mode first\r\n");
                APP_PowerPreSwitchHook(curPowerState, LPM_PowerModeRun);
                status = APP_PowerModeSwitch(curPowerState, LPM_PowerModeRun, VlprTargetClockConfig);
                APP_PowerPostSwitchHook(curPowerState, LPM_PowerModeRun, status == kStatus_Success);
                if (status == kStatus_Success)
                {
                    freq = CLOCK_GetFreq(kCLOCK_CoreSysClk);
                    PRINTF("Transfered to RUN mode, Core Clock: %dHz!\r\n", freq);
                }
                else
                {
                    PRINTF("Failed to transfer to RUN mode! Pls restart APP!\r\n");
                    break;
                }
                EnableALLPerithClockPCC0andPCC1();
            }
            //SCG0->SOSCCSR &= ~SCG_SOSCCSR_SOSCEN_MASK;          //0.8mW improvement
            PowerOpmizeVLPRandRUN();
            break;
        case 'b':
            curPowerState = SMC_GetPowerModeState(MSMC0);
            if (kSMC_PowerStateVlpr != curPowerState)
            {
                PRINTF("Not in VLPR mode, transfer to VLPR mode first\r\n");
                APP_PowerPreSwitchHook(curPowerState, LPM_PowerModeVlpr);
                status = APP_PowerModeSwitch(curPowerState, LPM_PowerModeVlpr, VlprTargetClockConfig);
                APP_PowerPostSwitchHook(curPowerState, LPM_PowerModeVlpr, status == kStatus_Success);
                if (status == kStatus_Success)
                {
                    freq = CLOCK_GetFreq(kCLOCK_CoreSysClk);
                    PRINTF("Transfered to VLPR mode, Core Clock: %dHz!\r\n", freq);
                }
                else
                {
                    PRINTF("Failed to transfer to VLPR mode! Pls restart APP!\r\n");
                    break;
                }

                /* Power optmization for VLPR */
                /* Disable SPLL. */
                SCG0->SPLLCSR &= ~SCG_SPLLCSR_SPLLEN_MASK;          //no relevant power improvement
                /* Disable APLL. As APLL is not initialized in clock_config.c */
                //SCG0->APLLCSR &= ~SCG_APLLCSR_APLLEN_MASK;          //0.2mW improvement
                /* Disable SOSC. */
                SCG0->SOSCCSR &= ~SCG_SOSCCSR_SOSCEN_MASK;          //0.8mW improvement
                /* Disable SIRC */
                //SCG0->SIRCCSR &= ~SCG_SIRCCSR_SIRCEN_MASK; 
                PowerOpmizeVLPRandRUN();                            //3mW improvement
#ifdef PRINT_DEBUG
                IsPcc0Pcc1ClockEnabled();
#endif
            }
            break;
        case 'c':
            PRINTF("Running coremark code ... ...\r\n");
            coremark_main();
            break;
        case 'd':
            PRINTF("Selected VLPR frequency @48Mhz, use FIRC with default clock dividers config (1112)\r\n");
            VlprTargetClockConfig = sysClkConfigVlprFirc;
            break;
        case 'e':
            PRINTF("Selected VLPR frequency @48Mhz, use FIRC with optimized clock dividers config (1126)\r\n");
            VlprTargetClockConfig = sysClkConfigVlprFircAN12573;
            break;
        case 'f':
            PRINTF("Selected VLPR frequency @16Mhz, use SIRC with default clock dividers config (1112)\r\n");
            VlprTargetClockConfig = sysClkConfigVlprSirc;
            break;
        default:
            PRINTF("Invalid selection!\r\n");
        }
    }
}

/*!
 * @brief Main function
 */
int main(void)
{
    /* Power related. */
    SMC_SetPowerModeProtection(MSMC0, kSMC_AllowPowerModeAll);

    /* Initialize standard SDK demo application pins */
    BOARD_InitClockAndPins();
    APP_InitPMC0();
    APP_InitDebugConsole();

    BOARD_InitBootPins();
    BOARD_BootClockRUN();

    CLOCK_SetIpSrc(kCLOCK_Lpi2c3, kCLOCK_IpSrcSircAsync);
    CLOCK_SetIpSrc(kCLOCK_Lpi2c0, kCLOCK_IpSrcSystem);

    /* Use AUXPLL main clock source */
    CLOCK_SetIpSrcDiv(kCLOCK_Sai0, kCLOCK_IpSrcRtcAuxPllAsync, 0, 0);

    APP_SRTM_Init();

    #if APP_ENABLE_GPIO_PAD_LOW_POWER
    /* NOTE: Please see the definition of APP_ENABLE_GPIO_PAD_LOW_POWER before using the DGO update here. */
    /* Set PTB/PTC/PTE to low range, PTA/PTF to high range to save power. Need to align with board design. */
    APP_UpdateSimDgo(11,
                     SIM_SIM_DGO_GP11_PTA_RANGE_CTRL_MASK | SIM_SIM_DGO_GP11_PTB_RANGE_CTRL_MASK |
                         SIM_SIM_DGO_GP11_PTC_RANGE_CTRL_MASK | SIM_SIM_DGO_GP11_PTE_RANGE_CTRL_MASK |
                         SIM_SIM_DGO_GP11_PTF_RANGE_CTRL_MASK,
                     SIM_SIM_DGO_GP11_PTA_RANGE_CTRL(2) | SIM_SIM_DGO_GP11_PTB_RANGE_CTRL(1) |
                         SIM_SIM_DGO_GP11_PTC_RANGE_CTRL(1) | SIM_SIM_DGO_GP11_PTE_RANGE_CTRL(1) |
                         SIM_SIM_DGO_GP11_PTF_RANGE_CTRL(2));
    #endif

    APP_SRTM_BootCA7();

#ifdef MCMGR_USED
    /* Initialize MCMGR before calling its API */
    (void)MCMGR_Init();
#endif /* MCMGR_USED */

    if (xTaskCreate(app_task, "APP_TASK", APP_TASK_STACK_SIZE, NULL, tskIDLE_PRIORITY + 1, &app_task_handle) != pdPASS)
    {
        PRINTF("\r\nFailed to create application task\r\n");
        for (;;)
            ;
    }

    vTaskStartScheduler();

    PRINTF("Failed to start FreeRTOS on core0.\n");
    for (;;)
        ;
}
