/*
 * Copyright (c) 2019, Freescale Semiconductor, Inc.
 * Copyright 2018-2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "freemaster.h"
#include "board.h"
#include "mcdrv.h"
#include "m1_sm_demo.h"
#include "fsl_lpuart.h"
#include "fm_tsa_pmsm.h"
#include "pin_mux.h"

#include "fsl_common.h"
#include "fsl_gpio.h"

#include "GUI.h"
#include "BUTTON.h"
#include "CHECKBOX.h"
#include "SLIDER.h"
#include "DROPDOWN.h"
#include "RADIO.h"
#include "MULTIPAGE.h"
#include "emwin_support.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

#ifndef GUI_CONTROL
#define GUI_CONTROL 
#endif

#ifndef GUI_NORMAL_FONT
#define GUI_NORMAL_FONT (&GUI_Font16_ASCII)
#endif

#ifndef GUI_LARGE_FONT
#define GUI_LARGE_FONT (&GUI_Font16B_ASCII)
#endif

#ifndef GUI_SCALE_FACTOR
#define GUI_SCALE_FACTOR 1
#endif

#ifndef GUI_SCALE_FACTOR_X
#define GUI_SCALE_FACTOR_X GUI_SCALE_FACTOR
#endif

#ifndef GUI_SCALE_FACTOR_Y
#define GUI_SCALE_FACTOR_Y GUI_SCALE_FACTOR
#endif

#define GUI_SCALE(a) ((int)((a) * (GUI_SCALE_FACTOR)))
#define GUI_SCALE_X(x) ((int)((x) * (GUI_SCALE_FACTOR_X)))
#define GUI_SCALE_Y(y) ((int)((y) * (GUI_SCALE_FACTOR_Y)))
#define GUI_SCALE_COORDS(x, y) GUI_SCALE_X(x), GUI_SCALE_Y(y)
#define GUI_SCALE_RECT(x0, y0, xs, ys) GUI_SCALE_X(x0), GUI_SCALE_Y(y0), GUI_SCALE_X(xs), GUI_SCALE_Y(ys)

#define GUI_ID_DRAWAREA (GUI_ID_USER + 0)
#define GUI_ID_PAGEWIN1 (GUI_ID_USER + 1)
#define GUI_ID_PAGEWIN2 (GUI_ID_USER + 2)


/* Three instruction added after interrupt flag clearing as required */
#define M1_END_OF_ISR \
    {                 \
        __DSB(); \
        __ISB(); \
    }

/* CPU load measurement SysTick START / STOP macros */
#define SYSTICK_START_COUNT() (SysTick->VAL = SysTick->LOAD)
#define SYSTICK_STOP_COUNT(par1) uint32_t val = SysTick->VAL; uint32_t load = SysTick->LOAD; par1 = load - val
/*******************************************************************************
 * Variables
 ******************************************************************************/

/* CPU load measurement using Systick*/
uint32_t g_ui32NumberOfCycles = 0;
uint32_t g_ui32MaxNumberOfCycles = 0;

/* Demo mode enabled/disabled */
bool_t bDemoMode = FALSE;

/* Counters used for demo mode */
static uint32_t ui32SpeedStimulatorCnt = 0;

/* Counter for button pressing */
static uint32_t ui32ButtonFilter = 0;

frac16_t SpeedSet = 0;
/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/* Demo Speed Stimulator */
static void DemoSpeedStimulator(void);
static void InitUART(uint32_t);
static void InitSysTick(void);

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
* @brief   Application main function processing peripheral function calling and
*          infinite loop
*
* @param   void
*
* @return  none
*/

uint32_t LPSPI1_GetFreq(void)
{
   return CLOCK_GetFreq(kCLOCK_OscClk) / 3;
}

uint32_t LPI2C1_GetFreq(void)
{
    return CLOCK_GetFreq(kCLOCK_OscClk) / 2;
}

static DROPDOWN_Handle hDropdown0;
static RADIO_Handle hRadio0;
static CHECKBOX_Handle hCheck0;
static WM_HWIN hDrawArea;

//static 
SLIDER_Handle hSlider0;
//static 
SLIDER_Handle hSlider1;
TEXT_Handle hTEXT0;
TEXT_Handle hTEXT1;
TEXT_Handle hTEXT2;
TEXT_Handle hTEXT3;
TEXT_Handle hTEXT4;
TEXT_Handle hTEXT5;
TEXT_Handle hTEXT6;
TEXT_Handle hTEXT7;
TEXT_Handle hTEXT8;

static SPINBOX_Handle hSpinbox0;
static PROGBAR_Handle hProgbar0;
static BUTTON_Handle hButton0;
static BUTTON_Handle hButton1;
static BUTTON_Handle hButton2;
static const GUI_COLOR color_list[]      = {GUI_BLACK,   GUI_YELLOW, GUI_ORANGE, GUI_RED,
                                       GUI_MAGENTA, GUI_BLUE,   GUI_CYAN,   GUI_GREEN};
static const GUI_POINT triangle_points[] = {
    {GUI_SCALE(0), GUI_SCALE(0)}, {GUI_SCALE(-50), GUI_SCALE(100)}, {GUI_SCALE(50), GUI_SCALE(100)}};

static void cbDrawArea(WM_MESSAGE *pMsg)
{
    switch (pMsg->MsgId)
    {
        case WM_PAINT:
            GUI_SetColor(color_list[DROPDOWN_GetSel(hDropdown0)]);
            switch (RADIO_GetValue(hRadio0))
            {
                case 0:
                    if (CHECKBOX_GetState(hCheck0))
                    {
                        GUI_FillRect(GUI_SCALE_X(70) - GUI_SCALE(50), GUI_SCALE_Y(70) - GUI_SCALE(50),
                                     GUI_SCALE_X(70) + GUI_SCALE(50), GUI_SCALE_Y(70) + GUI_SCALE(50));
                    }
                    else
                    {
                        GUI_DrawRect(GUI_SCALE_X(70) - GUI_SCALE(50), GUI_SCALE_Y(70) - GUI_SCALE(50),
                                     GUI_SCALE_X(70) + GUI_SCALE(50), GUI_SCALE_Y(70) + GUI_SCALE(50));
                    }
                    break;
                case 1:
                    if (CHECKBOX_GetState(hCheck0))
                    {
                        GUI_FillPolygon(triangle_points, 3, GUI_SCALE_COORDS(70, 20));
                    }
                    else
                    {
                        GUI_DrawPolygon(triangle_points, 3, GUI_SCALE_COORDS(70, 20));
                    }
                    break;
                case 2:
                    if (CHECKBOX_GetState(hCheck0))
                    {
                        GUI_FillEllipse(GUI_SCALE_COORDS(70, 70), GUI_SCALE(50), GUI_SCALE(50));
                    }
                    else
                    {
                        GUI_DrawEllipse(GUI_SCALE_COORDS(70, 70), GUI_SCALE(50), GUI_SCALE(50));
                    }
                    break;
            }
            break;
        default:
            WM_DefaultProc(pMsg);
            break;
    }
}

static void cbPageWin1(WM_MESSAGE *pMsg)
{
    int NCode;
    int Id;

    switch (pMsg->MsgId)
    {
        case WM_NOTIFY_PARENT:
            Id    = WM_GetId(pMsg->hWinSrc);
            NCode = pMsg->Data.v;

            switch (Id)
            {
                case GUI_ID_RADIO0:
                    switch (NCode)
                    {
                        case WM_NOTIFICATION_VALUE_CHANGED:
                            WM_InvalidateWindow(hDrawArea);
                            break;
                    }
                    break;

                case GUI_ID_DROPDOWN0:
                    switch (NCode)
                    {
                        case WM_NOTIFICATION_SEL_CHANGED:
                            WM_InvalidateWindow(hDrawArea);
                            break;
                    }
                    break;

                case GUI_ID_CHECK0:
                    switch (NCode)
                    {
                        case WM_NOTIFICATION_VALUE_CHANGED:
                            WM_InvalidateWindow(hDrawArea);
                            break;
                    }
                    break;
            }
            break;

        default:
            WM_DefaultProc(pMsg);
            break;
    }
}

static void cbPageWin2(WM_MESSAGE *pMsg)
{
    int NCode;
    int Id;

    switch (pMsg->MsgId)
    {
        case WM_NOTIFY_PARENT:
            Id    = WM_GetId(pMsg->hWinSrc);
            NCode = pMsg->Data.v;

            switch (Id)
            {
                case GUI_ID_SLIDER0:
                    switch (NCode)
                    {
                        case WM_NOTIFICATION_VALUE_CHANGED:
                            PROGBAR_SetValue(hProgbar0, SLIDER_GetValue(hSlider0));
                            break;
                    }
                    break;

                case GUI_ID_SLIDER1:
                    switch (NCode)
                    {
                        case WM_NOTIFICATION_VALUE_CHANGED:
                            SPINBOX_SetValue(hSpinbox0, SLIDER_GetValue(hSlider1));
                            break;
                    }
                    break;

                case GUI_ID_SPINBOX0:
                    switch (NCode)
                    {
                        case WM_NOTIFICATION_VALUE_CHANGED:
                            SLIDER_SetValue(hSlider1, SPINBOX_GetValue(hSpinbox0));
                            break;
                    }
                    break;
            }
            break;

        default:
            WM_DefaultProc(pMsg);
            break;
    }
}

int Speed_bar = 0;
int Speed_act = 29800;
uint8_t Button_start = 0;
uint8_t Button_stop = 0;
uint8_t Button_demomode = 0;

int main(void)
{
    uint32_t ui32PrimaskReg = 0;
    int cntr;

    /* Init pins set in pin_mux file */
    BOARD_InitBootPins();
    
    /* Initialize clock configuration */
    BOARD_BootClockRUN();
#ifdef GUI_CONTROL    
    //SCB_DisableDCache();
    /* Set LPSPI_PODF. */
    CLOCK_SetDiv(kCLOCK_LpspiDiv, 2);
    /* Set Lpspi clock source. */
    CLOCK_SetMux(kCLOCK_LpspiMux, 0);
    
    /* Set LPI2C_CLK_PODF. */
    CLOCK_SetDiv(kCLOCK_Lpi2cDiv, 1);
    /* Set Lpi2c clock source. */
    CLOCK_SetMux(kCLOCK_Lpi2cMux, 1); 

    GUI_Init();
    
    CHECKBOX_SetDefaultFont(GUI_NORMAL_FONT);
    DROPDOWN_SetDefaultFont(GUI_NORMAL_FONT);
    MULTIPAGE_SetDefaultFont(GUI_NORMAL_FONT);
    RADIO_SetDefaultFont(GUI_NORMAL_FONT);

    /* Create multipage widget */
    MULTIPAGE_Handle hMultipage0;
    hMultipage0 = MULTIPAGE_CreateEx(GUI_SCALE_RECT(10, 10, 300, 220), 0, WM_CF_SHOW, 0, GUI_ID_MULTIPAGE0);

    WM_HWIN hPageWin;
    /* Create window for page 2 and add it */
    hPageWin = WINDOW_CreateEx(GUI_SCALE_RECT(0, 0, 300, 200), WM_HBKWIN, 0, 0, GUI_ID_PAGEWIN2, cbPageWin2);
    MULTIPAGE_AddPage(hMultipage0, hPageWin, "PMSM Control");

    /* Create widgets on page 2 */
    hSlider0 = SLIDER_CreateEx(GUI_SCALE_RECT(100, 60, 190, 30), hPageWin, WM_CF_SHOW, 0, GUI_ID_SLIDER0);
    SLIDER_SetWidth(hSlider0, GUI_SCALE(10));
    SLIDER_SetRange(hSlider0, -29000, 29000);
    SLIDER_SetValue(hSlider0, 0);
    
    hTEXT0 = TEXT_CreateEx(100, 90, 70, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT0, "-4000/rpm");
    hTEXT1 = TEXT_CreateEx(190, 90, 15, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT1, "0");
    hTEXT2 = TEXT_CreateEx(240, 90, 70, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT2, "4000/rpm");
    hTEXT3 = TEXT_CreateEx(10, 100, 300, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT3, "NXP RT1010 Inside");
    hTEXT4 = TEXT_CreateEx(10, 115, 300, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT4, "High performing Arm Cortex-M7");
    hTEXT5 = TEXT_CreateEx(10, 130, 300, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT5, "2517 CoreMark/1070 DMIPS @ 500 MHz");
    hTEXT6 = TEXT_CreateEx(10, 145, 300, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT6, "Up to 128KB Tightly Coupled Memory (TCM)");
    hTEXT7 = TEXT_CreateEx(100, 10, 100, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT7, "Actual Speed:");
    hTEXT8 = TEXT_CreateEx(180, 30, 35, 15, hPageWin, WM_CF_SHOW, 0, GUI_ID_TEXT8, "    0");
    
    PROGBAR_SKINFLEX_PROPS pProps = {0};
    PROGBAR_GetSkinFlexProps(&pProps, 0);
    pProps.ColorText = GUI_BLACK;
    PROGBAR_SetSkinFlexProps(&pProps, 0);
#if 0
    hButton0 = 
        BUTTON_CreateEx(GUI_SCALE_RECT(5, 25, 90, 30), hPageWin, WM_CF_SHOW, PROGBAR_CF_HORIZONTAL, GUI_ID_BUTTON0);
    BUTTON_SetFont(hButton0, GUI_LARGE_FONT);
    BUTTON_SetText(hButton0 , "Start");
#endif
    hButton1 = 
        BUTTON_CreateEx(GUI_SCALE_RECT(5, 25, 90, 30), hPageWin, WM_CF_SHOW, PROGBAR_CF_HORIZONTAL, GUI_ID_BUTTON1);
    BUTTON_SetFont(hButton1, GUI_LARGE_FONT);
    BUTTON_SetText(hButton1 , "Stop");
    
    hButton2 = 
        BUTTON_CreateEx(GUI_SCALE_RECT(5, 60, 90, 30), hPageWin, WM_CF_SHOW, PROGBAR_CF_HORIZONTAL, GUI_ID_BUTTON2);
    BUTTON_SetFont(hButton2, GUI_LARGE_FONT);
    BUTTON_SetText(hButton2 , "Demo Mode");
    
    WM_SetDesktopColor(GUI_WHITE);
    WM_Exec();
    
#endif
    
    /* Initialize peripheral motor control driver for motor M1*/
    MCDRV_Init_M1();
    
    /* Init UART for 
 communication */ 
    InitUART(BOARD_FMSTR_UART_BAUDRATE);

    /* FreeMaster init */
    FMSTR_Init();

    /* SysTick initialization for CPU load measurement */
    InitSysTick();
    
    /* Turn off application */
    M1_SetAppSwitch(0);

    /* Disable demo mode after reset */
    bDemoMode = FALSE;
    ui32SpeedStimulatorCnt = 0;

    /* Enable interrupts  */
    EnableGlobalIRQ(ui32PrimaskReg);

    /* Infinite loop */
    while (1)
    {

        /* FreeMASTER Polling function */      
        FMSTR_Poll();
#ifdef GUI_CONTROL
        /* Poll touch controller for update */
        if (BOARD_Touch_Poll())
        {
          Speed_bar = SLIDER_GetValue(hSlider0);

          Button_start = BUTTON_IsPressed(hButton0);
          Button_stop = BUTTON_IsPressed(hButton1);
          Button_demomode = BUTTON_IsPressed(hButton2);

          if (Button_demomode)
          {
            g_bM1SwitchAppOnOff = 1;
            bDemoMode = TRUE;
            SLIDER_SetValue(hSlider0, SpeedSet);
          }
          else 
          {
            M1_SetSpeed(Speed_bar);
            g_bM1SwitchAppOnOff = 1;
          }
          if(Button_stop)
          {
            g_bM1SwitchAppOnOff = 0;
          }
          Speed_act = (int)(g_sM1Drive.sSpeed.f16SpeedFilt* 0.134f);
         cntr++;
          if(cntr > 100)
          {
            TEXT_SetDec(hTEXT8, Speed_act, 5, 0, 1, 5);
            cntr = 0;
          }

#ifdef GUI_BUFFERS
            GUI_MULTIBUF_Begin();
#endif
            GUI_Exec();
#ifdef GUI_BUFFERS
            GUI_MULTIBUF_End();
#endif
        }
#endif
    }
}

/*!
*@brief      Initialization of the UART module 
*
*@param      u32BaudRate         Baud rate
*            
*@return     none
*/
void InitUART(uint32_t u32BaudRate)
{
    lpuart_config_t config;
    
    LPUART_GetDefaultConfig(&config);
    config.baudRate_Bps = BOARD_FMSTR_UART_BAUDRATE;
    config.enableTx = true;
    config.enableRx = true;

    LPUART_Init(BOARD_FMSTR_UART_PORT, &config, BOARD_UARTSrcFreq());
}


/*!
*@brief      SysTick initialization for CPU cycle measurement
*
*@param      none
*            
*@return     none
*/
void InitSysTick(void)
{
    /* Initialize SysTick core timer to run free */
    /* Set period to maximum value 2^24*/
    SysTick->LOAD = 0xFFFFFF;
    
    /*Clock source - System Clock*/
    SysTick->CTRL |= SysTick_CTRL_CLKSOURCE_Msk;
    
    /*Start Sys Timer*/
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
}


/*!
* @brief   ADC conversion complete ISR called with 100us period processes
*           - motor M1 fast application machine function
*           - demo mode if enabled
*
* @param   void
*
* @return  none
*/
void ADC1_IRQHandler(void)
{
     static int16_t ui16slow = 0;
    /* Start CPU tick number couting */
    SYSTICK_START_COUNT();    
    
    /* M1 State machine */
    SM_StateMachineFast(&g_sM1Ctrl); 
    if (ui16slow-- < 0)
        {
         /* M1 Slow StateMachine call */
          SM_StateMachineSlow(&g_sM1Ctrl); 
        /* Demo speed stimulator */
          DemoSpeedStimulator();
            ui16slow = 10;
        }
    
    /* Stop CPU tick number couting and store actual and maximum ticks */
    SYSTICK_STOP_COUNT(g_ui32NumberOfCycles);
    g_ui32MaxNumberOfCycles = g_ui32NumberOfCycles>g_ui32MaxNumberOfCycles ? g_ui32NumberOfCycles : g_ui32MaxNumberOfCycles;   
   
    /* Call FreeMASTER recorder */
    FMSTR_Recorder();
      
    /* Add empty instructions for correct interrupt flag clearing */
    M1_END_OF_ISR;
}

/*!
* @brief   TMR1 reload ISR 
*
* @param   void
*
* @return  none
*/
void TMR1_IRQHandler(void)
{   
#if 0
/*reserved for use*/
#endif
}

/*!
* @brief   SW Button interrupt handler
*
* @param   void
*
* @return  none
*/
void GPIO2_Combined_0_15_IRQHandler(void)
{
#if 0
  /*reserved for use*/
#endif
}

/*!
* @brief   DemoSpeedStimulator
*           - When demo mode is enabled it changes the required speed according
*             to predefined profile
*
* @param   void
*
* @return  none
*/
static void DemoSpeedStimulator(void)
{
    /* Increase push button pressing counter  */
    if (ui32ButtonFilter < 1000)
        ui32ButtonFilter++;
    
    if (bDemoMode)
    {
        ui32SpeedStimulatorCnt++;
        switch (ui32SpeedStimulatorCnt)
        {
                             
            case 100:
              {
                SpeedSet = FRAC16(1000.0 / M1_N_MAX);
                M1_SetSpeed(SpeedSet);
                break;
              }
            case 3000:
              {
                SpeedSet = FRAC16(2000.0 / M1_N_MAX);
                M1_SetSpeed(SpeedSet);
                break;
              }
            case 6000:
              {
                SpeedSet = FRAC16(4000.0 / M1_N_MAX);
                M1_SetSpeed(SpeedSet);
                break;
              }
            case 9000:
              {
                SpeedSet = -FRAC16(1000.0 / M1_N_MAX);
                M1_SetSpeed(SpeedSet);
                break;
              }
            case 12000:
              {
                SpeedSet = -FRAC16(2000.0 / M1_N_MAX);
                M1_SetSpeed(SpeedSet);
                break;
              }
            case 15000:
              {
                SpeedSet = -FRAC16(4000.0 / M1_N_MAX);
                M1_SetSpeed(SpeedSet);
                break;
              }
            case 18000:
              {
                ui32SpeedStimulatorCnt = 0;
                  break;
              }
            default:
                break;
        }
    }
    else
    {
        ui32SpeedStimulatorCnt = 0;
    }
}
