/*
 * Copyright 2019-2020 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/* msy: This is a modified version of the framebuffer device fsl_fbdev.c that
   is now able to work without FreeRTOS. The usage of FreeRTOS is controlled
   by the compiler flag FSL_RTOS_FREE_RTOS.
 */

#include "fsl_fbdev.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

static void FBDEV_BufferSwitchOffCallback(void *param, void *switchOffBuffer);

/*******************************************************************************
 * Variables
 ******************************************************************************/
#if defined(FSL_RTOS_FREE_RTOS)
#else
  static __IO uint32_t          FramePending = 0;
#endif

/*******************************************************************************
 * Code
 ******************************************************************************/

status_t FBDEV_Open(fbdev_t *fbdev, const dc_fb_t *dc, uint8_t layer)
{
    status_t status;

    assert(fbdev);

    (void)memset(fbdev, 0, sizeof(fbdev_t));

    fbdev->dc = dc;

    VIDEO_MEMPOOL_InitEmpty(&fbdev->fbManager);

    /* Initialize the display controller. */
    status = dc->ops->init(dc);
    if (kStatus_Success != status)
    {
        return status;
    }

    fbdev->layer = layer;

    /* Initializes the dc_fb_info_t to the display controller default setting. */
    status = dc->ops->getLayerDefaultConfig(dc, layer, &fbdev->fbInfo.bufInfo);
    if (kStatus_Success != status)
    {
        return status;
    }

#if defined(FSL_RTOS_FREE_RTOS)

    fbdev->semaFramePending = xSemaphoreCreateBinary();
    if (NULL == fbdev->semaFramePending)
    {
        return kStatus_Fail;
    }

    /* No frame pending. */
    (void)xSemaphoreGive(fbdev->semaFramePending);

#else

   FramePending = 0;

#endif

    dc->ops->setCallback(dc, layer, FBDEV_BufferSwitchOffCallback, (void *)fbdev);

    return kStatus_Success;
}

status_t FBDEV_Close(fbdev_t *fbdev)
{
    const dc_fb_t *dc = fbdev->dc;

    (void)dc->ops->deinit(dc);

#if defined(FSL_RTOS_FREE_RTOS)

    if (NULL != fbdev->semaFbManager)
    {
        vSemaphoreDelete(fbdev->semaFbManager);
        fbdev->semaFbManager = NULL;
    }

    if (NULL != fbdev->semaFramePending)
    {
        vSemaphoreDelete(fbdev->semaFramePending);
        fbdev->semaFramePending = NULL;
    }

#endif

    return kStatus_Success;
}

status_t FBDEV_Enable(fbdev_t *fbdev)
{
    status_t status = kStatus_Success;

    const dc_fb_t *dc = fbdev->dc;

    if (!fbdev->enabled)
    {
        /* Wait for frame buffer sent to display controller video memory. */
        if ((dc->ops->getProperty(dc) & (uint32_t)kDC_FB_ReserveFrameBuffer) == 0U)
        {
#if defined(FSL_RTOS_FREE_RTOS)
            if (pdTRUE != xSemaphoreTake(fbdev->semaFramePending, portMAX_DELAY))
            {
                status = kStatus_Fail;
            }
#endif
        }

        if (kStatus_Success == status)
        {
            /* No frame is pending. */
#if defined(FSL_RTOS_FREE_RTOS)
            (void)xSemaphoreGive(fbdev->semaFramePending);
#endif
            status = dc->ops->enableLayer(dc, fbdev->layer);

            if (kStatus_Success == status)
            {
                fbdev->enabled = true;
            }
        }
    }

#if defined(FSL_RTOS_FREE_RTOS)
#else
    FramePending = 0;
#endif
    return status;
}

status_t FBDEV_Disable(fbdev_t *fbdev)
{
    status_t status = kStatus_Success;

    const dc_fb_t *dc = fbdev->dc;

    if (!fbdev->enabled)
    {
#if defined(FSL_RTOS_FREE_RTOS)
        /* Wait until no frame pending. */
        if (pdTRUE != xSemaphoreTake(fbdev->semaFramePending, portMAX_DELAY))
        {
            status = kStatus_Fail;
        }
#else
#endif
        if (kStatus_Success == status)
        {
#if defined(FSL_RTOS_FREE_RTOS)
            (void)xSemaphoreGive(fbdev->semaFramePending);
#endif
            (void)dc->ops->disableLayer(dc, fbdev->layer);

            fbdev->enabled = false;
        }
    }

    return status;
}

void FBDEV_GetFrameBufferInfo(fbdev_t *fbdev, fbdev_fb_info_t *info)
{
    *info = fbdev->fbInfo;
}

status_t FBDEV_SetFrameBufferInfo(fbdev_t *fbdev, fbdev_fb_info_t *info)
{
    status_t status;
    const dc_fb_t *dc = fbdev->dc;

    /* Should only change the frame buffer setting before enabling the fbdev. */
    if (fbdev->enabled)
    {
        return kStatus_Fail;
    }

    fbdev->fbInfo = *info;

    status = dc->ops->setLayerConfig(dc, fbdev->layer, &fbdev->fbInfo.bufInfo);

    if (kStatus_Success != status)
    {
        return status;
    }

#if defined(FSL_RTOS_FREE_RTOS)
    fbdev->semaFbManager = xSemaphoreCreateCounting(info->bufferCount, 0);
    if (NULL == fbdev->semaFbManager)
    {
        return kStatus_Fail;
    }
#else
#endif
    for (uint8_t i = 0; i < info->bufferCount; i++)
    {
        /* Don't need to disable interrupt for the MEMPOOL operation, because
           the fbdev is not working, this is the only function to access MEMPOOL.
         */
        VIDEO_MEMPOOL_Put(&fbdev->fbManager, info->buffers[i]);
#if defined(FSL_RTOS_FREE_RTOS)
        (void)xSemaphoreGive(fbdev->semaFbManager);
#endif
    }

    return kStatus_Success;
}

void *FBDEV_GetFrameBuffer(fbdev_t *fbdev, uint32_t flags)
{
#if defined(FSL_RTOS_FREE_RTOS)
    TickType_t tick;
#endif
    void *fb;

#if defined(FSL_RTOS_FREE_RTOS)
    tick = ((flags & (uint32_t)kFBDEV_NoWait) != 0U) ? 0U : portMAX_DELAY;

    if (pdTRUE == xSemaphoreTake(fbdev->semaFbManager, tick))
    {
        /* Disable interrupt to protect the MEMPOOL. */
        portENTER_CRITICAL();
        fb = VIDEO_MEMPOOL_Get(&fbdev->fbManager);
        portEXIT_CRITICAL();
    }
    else
    {
        fb = NULL;
    }
#else
    while ( FramePending )
    {
    }

    fb = VIDEO_MEMPOOL_Get(&fbdev->fbManager);
#endif

    return fb;
}

status_t FBDEV_SetFrameBuffer(fbdev_t *fbdev, void *frameBuffer, uint32_t flags)
{
#if defined(FSL_RTOS_FREE_RTOS)
    TickType_t tick;
#endif
    const dc_fb_t *dc = fbdev->dc;

#if defined(FSL_RTOS_FREE_RTOS)
    tick = ((flags & (uint32_t)kFBDEV_NoWait) != 0U) ? 0U : portMAX_DELAY;

    if (pdTRUE == xSemaphoreTake(fbdev->semaFramePending, tick))
    {
        return dc->ops->setFrameBuffer(dc, fbdev->layer, frameBuffer);
    }
    else
    {
        return kStatus_Fail;
    }
#else
    if ( FramePending )
      return kStatus_Fail;
    FramePending = 1;
    return dc->ops->setFrameBuffer(dc, fbdev->layer, frameBuffer);
#endif
}

static void FBDEV_BufferSwitchOffCallback(void *param, void *switchOffBuffer)
{
    fbdev_t *fbdev              = (fbdev_t *)param;
#if defined(FSL_RTOS_FREE_RTOS)
    BaseType_t fbManagerWake    = pdFALSE;
    BaseType_t framePendingWake = pdFALSE;
#endif

    /* This function should only be called in ISR, so don't need to protect the MEMPOOL */
    VIDEO_MEMPOOL_Put(&fbdev->fbManager, switchOffBuffer);

#if defined(FSL_RTOS_FREE_RTOS)
    (void)xSemaphoreGiveFromISR(fbdev->semaFbManager, &fbManagerWake);

    (void)xSemaphoreGiveFromISR(fbdev->semaFramePending, &framePendingWake);

    if ((fbManagerWake == pdTRUE) || (framePendingWake == pdTRUE))
    {
        portYIELD_FROM_ISR(pdTRUE);
    }
#else
  FramePending = 0;
#endif
}
