/*
 * Copyright (c) 2013-2015 Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#define exit exit_default
#include <stdbool.h>
#include "utilities/fsl_assert.h"
#include "bootloader/bl_context.h"
#include "bootloader/bl_peripheral.h"
#include "bootloader/bl_shutdown_cleanup.h"
#include "bootloader_common.h"
#include "bootloader/bootloader.h"
#if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
#if !BL_DEVICE_IS_LPC_SERIES
#include "fsl_flash.h"
#else
#include "flashiap_wrapper/fsl_flashiap_wrapper.h"
#endif
#endif // #if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
//#include "smc.h"
#include "microseconds.h"
#include "utilities/fsl_rtos_abstraction.h"
#if BL_FEATURE_CRC_CHECK
#include "bootloader/bl_app_crc_check.h"
#endif
#include "memory/memory.h"

#if BL_FEATURE_RELIABLE_UPDATE
#include "bootloader/bl_reliable_update.h"
#endif

#include "fsl_flexspi.h"
#include "app.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "board.h"
#include "clock_config.h"
#include "fsl_common.h"
#include "fsl_dcp.h"
#include "fsl_bee.h"
#include "fsl_cache.h"
#include "image_info.h"
#include "enc_operations.h"
//! @addtogroup bl_core
//! @{

#define SECTOR_SIZE					0x1000
#define PAGE_SIZE					256


////////////////////////////////////////////////////////////////////////////////
// Prototypes
////////////////////////////////////////////////////////////////////////////////

#if defined(DEBUG) && !defined(DEBUG_PRINT_DISABLE)
static const char *get_peripheral_name(uint32_t peripheralTypeMask);
#endif
static bool is_application_ready_for_executing(uint32_t applicationAddress);
#if !BL_FEATURE_TIMEOUT
static void get_user_application_entry(uint32_t *appEntry, uint32_t *appStack);
static void jump_to_application(void);
#endif // !BL_FEATURE_TIMEOUT
static peripheral_descriptor_t const *get_active_peripheral(void);
void bootloader_init(void);
void bootloader_run(void);
#if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
static void bootloader_flash_init(void);
#endif // #if !BL_FEATURE_HAS_NO_INTERNAL_FLASH
#if BL_FEATURE_QSPI_MODULE
static void configure_quadspi_as_needed(void);
#endif
static status_t decryption_config(void);
AT_QUICKACCESS_SECTION_CODE(void bee_config(protect_region_block_t prdb));
int main(void);

extern AT_QUICKACCESS_SECTION_CODE(void AHBPrefetchDisable(FLEXSPI_Type *base));
extern AT_QUICKACCESS_SECTION_CODE(void AHBPrefetchEnable(FLEXSPI_Type *base));
extern AT_QUICKACCESS_SECTION_CODE(void FLEXSPI_SoftwareResetRamfunc(FLEXSPI_Type *base));
extern AT_QUICKACCESS_SECTION_CODE(void FLEXSPI_UpdateLUTRamfunc(FLEXSPI_Type *base, uint32_t index, const uint32_t *cmd, uint32_t count));

// Not used, but needed to resolve reference in startup.s.
// uint32_t g_bootloaderTree;

////////////////////////////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////////////////////////////

#if defined(DEBUG) && !defined(DEBUG_PRINT_DISABLE)
static const char *const kPeripheralNames[] = {
    "UART", // kPeripheralType_UART
    "I2C",  // kPeripheralType_I2CSlave
    "SPI",  // kPeripheralType_SPISlave
    "CAN",  // kPeripheralType_CAN
    "HID",  // kPeripheralType_USB_HID
    "CDC",  // kPeripheralType_USB_CDC
    "DFU",  // kPeripheralType_USB_DFU
    "MSD"   // kPeripheralType_USB_MSC
};
#endif // DEBUG


uint32_t g_customLUT[CUSTOM_LUT_LENGTH] = {
        /* Normal read mode -SDR */

        [4 * NOR_CMD_LUT_SEQ_IDX_READ_NORMAL] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x03, kFLEXSPI_Command_RADDR_SDR, kFLEXSPI_1PAD, 0x18),
        [4 * NOR_CMD_LUT_SEQ_IDX_READ_NORMAL + 1] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_READ_SDR, kFLEXSPI_1PAD, 0x04, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Fast read mode - SDR */
        [4 * NOR_CMD_LUT_SEQ_IDX_READ_FAST] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x0B, kFLEXSPI_Command_RADDR_SDR, kFLEXSPI_1PAD, 0x18),
        [4 * NOR_CMD_LUT_SEQ_IDX_READ_FAST + 1] = FLEXSPI_LUT_SEQ(
            kFLEXSPI_Command_DUMMY_SDR, kFLEXSPI_1PAD, 0x08, kFLEXSPI_Command_READ_SDR, kFLEXSPI_1PAD, 0x04),

        /* Fast read quad mode - SDR */
        [4 * NOR_CMD_LUT_SEQ_IDX_READ_FAST_QUAD] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0xEB, kFLEXSPI_Command_RADDR_SDR, kFLEXSPI_4PAD, 0x18),
        [4 * NOR_CMD_LUT_SEQ_IDX_READ_FAST_QUAD + 1] = FLEXSPI_LUT_SEQ(
            kFLEXSPI_Command_DUMMY_SDR, kFLEXSPI_4PAD, 0x06, kFLEXSPI_Command_READ_SDR, kFLEXSPI_4PAD, 0x04),

        /* Read extend parameters */
        [4 * NOR_CMD_LUT_SEQ_IDX_READSTATUS] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x81, kFLEXSPI_Command_READ_SDR, kFLEXSPI_1PAD, 0x04),

        /* Write Enable */
        [4 * NOR_CMD_LUT_SEQ_IDX_WRITEENABLE] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x06, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Erase Sector  */
        [4 * NOR_CMD_LUT_SEQ_IDX_ERASESECTOR] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x20, kFLEXSPI_Command_RADDR_SDR, kFLEXSPI_1PAD, 0x18),

        /* Page Program - single mode */
        [4 * NOR_CMD_LUT_SEQ_IDX_PAGEPROGRAM_SINGLE] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x02, kFLEXSPI_Command_RADDR_SDR, kFLEXSPI_1PAD, 0x18),
        [4 * NOR_CMD_LUT_SEQ_IDX_PAGEPROGRAM_SINGLE + 1] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_WRITE_SDR, kFLEXSPI_1PAD, 0x04, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Page Program - quad mode */
        [4 * NOR_CMD_LUT_SEQ_IDX_PAGEPROGRAM_QUAD] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x32, kFLEXSPI_Command_RADDR_SDR, kFLEXSPI_1PAD, 0x18),
        [4 * NOR_CMD_LUT_SEQ_IDX_PAGEPROGRAM_QUAD + 1] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_WRITE_SDR, kFLEXSPI_4PAD, 0x04, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Read ID */
        [4 * NOR_CMD_LUT_SEQ_IDX_READID] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0xAB, kFLEXSPI_Command_DUMMY_SDR, kFLEXSPI_1PAD, 0x18),
        [4 * NOR_CMD_LUT_SEQ_IDX_READID + 1] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_READ_SDR, kFLEXSPI_1PAD, 0x04, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Enable Quad mode */
        [4 * NOR_CMD_LUT_SEQ_IDX_WRITESTATUSREG] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x01, kFLEXSPI_Command_WRITE_SDR, kFLEXSPI_1PAD, 0x04),

        /* Enter QPI mode */
        [4 * NOR_CMD_LUT_SEQ_IDX_ENTERQPI] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x35, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Exit QPI mode */
        [4 * NOR_CMD_LUT_SEQ_IDX_EXITQPI] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_4PAD, 0xF5, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),

        /* Read status register */
        [4 * NOR_CMD_LUT_SEQ_IDX_READSTATUSREG] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0x05, kFLEXSPI_Command_READ_SDR, kFLEXSPI_1PAD, 0x04),

        /* Erase whole chip */
        [4 * NOR_CMD_LUT_SEQ_IDX_ERASECHIP] =
            FLEXSPI_LUT_SEQ(kFLEXSPI_Command_SDR, kFLEXSPI_1PAD, 0xC7, kFLEXSPI_Command_STOP, kFLEXSPI_1PAD, 0),
};

////////////////////////////////////////////////////////////////////////////////
// Code
////////////////////////////////////////////////////////////////////////////////

#if defined(DEBUG) && !defined(DEBUG_PRINT_DISABLE)
//! @brief Returns the name of a peripheral given its type mask.
const char *get_peripheral_name(uint32_t peripheralTypeMask)
{
    uint32_t i;
    for (i = 0; i < ARRAY_SIZE(kPeripheralNames); ++i)
    {
        if (peripheralTypeMask & (1 << i))
        {
            return kPeripheralNames[i];
        }
    }

    return "Unknown peripheral";
}
#endif // DEBUG

#if !BL_FEATURE_TIMEOUT
//! @brief Returns the user application address and stack pointer.
//!
//! For flash-resident and rom-resident target, gets the user application address
//! and stack pointer from the APP_VECTOR_TABLE.
//! Ram-resident version does not support jumping to application address.
static void get_user_application_entry(uint32_t *appEntry, uint32_t *appStack)
{
    assert(appEntry);
    assert(appStack);

    *appEntry = APP_VECTOR_TABLE[kInitialPC];
    *appStack = APP_VECTOR_TABLE[kInitialSP];
}
#endif // BL_FEATURE_TIMEOUT



#if !BL_FEATURE_TIMEOUT
//! @brief Exits bootloader and jumps to the user application.
static void jump_to_application(void)
{
#if BL_FEATURE_OTFAD_MODULE
    quadspi_cache_clear();
    oftfad_resume_as_needed();
#endif
    uint32_t applicationAddress, stackPointer;
    /*
    // if get the correct tag on parameter address, then do BEE configuration.
    // otherwise, not configure BEE.
    */
    if(decryption_config() != kStatus_Success)
    {
      // this is raw image, continue to run.
    }
        
    get_user_application_entry(&applicationAddress, &stackPointer);
    if (!is_application_ready_for_executing(applicationAddress))
    {
      return;
    }
   
    shutdown_cleanup(kShutdownType_Shutdown);
    // Create the function call to the user application.
    // Static variables are needed since changed the stack pointer out from under the compiler
    // we need to ensure the values we are using are not stored on the previous stack
    static uint32_t s_stackPointer = 0;
    s_stackPointer = stackPointer;
    static void (*farewellBootloader)(void) = 0;
    farewellBootloader = (void (*)(void))applicationAddress;

    // Set the VTOR to the application vector table address.
    SCB->VTOR = (uint32_t)BL_APP_VECTOR_TABLE_ADDRESS;

    // Set stack pointers to the application stack pointer.
    __set_MSP(s_stackPointer);
    __set_PSP(s_stackPointer);

    // Jump to the application.
    farewellBootloader();
    // Dummy fcuntion call, should never go to this fcuntion call
    shutdown_cleanup(kShutdownType_Shutdown);
}
#endif // !BL_FEATURE_TIMEOUT

//! A given jump address is considered valid if:
//! - Not 0x00000000
//! - Not 0xffffffff
//! - Not the reset handler entry point for the bootloader
//! - Is in flash or is in RAM or QuadSPI (if available)
//! @note this interface is also used by the configure_quadspi command
bool is_valid_application_location(uint32_t applicationAddress)
{
    // Verify that the jumpLocation is non zero and then either within flash or RAM, both calculations are:
    // (jumpLocation >= startAddress) && (jumpLocation < (startAddress + size))
    if ((!applicationAddress) ||              // address is not null AND
        (applicationAddress == 0xffffffff) || // address is not blank Flash (0xff) AND
        (applicationAddress == (uint32_t)&Reset_Handler)||
         (*(uint32_t *)applicationAddress == 0xffffffff))
    {
        return false;
    }

    bool isValid = true;
    //const uint32_t minThumb2InstructionSize = 2; // smallest thumb2 instruction size is 16-bit.
    // Check if the application address is in valid executable memory range
   // status_t status = find_map_entry(applicationAddress, minThumb2InstructionSize, &map);
  //  if ((status == kStatus_Success) && (map->memoryProperty & kMemoryIsExecutable))
    {
   //     isValid = true;
    }

    return isValid;
}

bool is_valid_stackpointer_location(uint32_t stackpointerAddress)
{
    bool isValid = false;

    return isValid;
}

#if !BL_FEATURE_TIMEOUT
//! @brief Jump application is considered ready for executing if the location is valid and crc check is passed
static bool is_application_ready_for_executing(uint32_t applicationAddress)
{
    bool result = is_valid_application_location(applicationAddress);
    
    

#if BL_FEATURE_OTFAD_MODULE
    if (result && is_qspi_present())
    {
        quadspi_cache_clear();
        status_t status = otfad_init_as_needed();
        if (status != kStatus_Success)
        {
            result = false;
        }
        update_qspi_otfad_init_status(status);
    }
#endif

#if BL_FEATURE_CRC_CHECK
    // Validate application crc only if its location is valid
    if (result)
    {
        result = is_application_crc_check_pass();
    }

#if BL_FEATURE_OTFAD_MODULE
    otfad_bypass_as_needed();
#endif // BL_FEATURE_OTFAD_MODULE

#endif

    return result;
}
#endif // !BL_FEATURE_TIMEOUT

enc_region_hdr_t phdr;
bool is_phdr_valid(enc_region_hdr_t * phdr,uint32_t Address)
{
  memcpy(phdr,(void *)Address,sizeof(enc_region_hdr_t));
  if(phdr->tag == HDR_TAG)
  {
    return true;
  }
  else
  {
    return false;
  }
}


AT_QUICKACCESS_SECTION_CODE(void BEE_SetConfigRamfunc(BEE_Type *base, const bee_region_config_t *config))
{
    uint32_t beeCtrlVal;
 
    /* Wait until BEE is in idle state */
    while (!(base->STATUS & kBEE_IdleFlag))
    {
    }

    /* Preserve CTRL bit values that are not set by this function */
    beeCtrlVal = base->CTRL & 0xFFFF0037;

    /* Set variable according to configuration */
    beeCtrlVal |= BEE_CTRL_AC_PROT_EN(config->accessPermission) | BEE_CTRL_LITTLE_ENDIAN(config->endianSwapEn) |
                  BEE_CTRL_SECURITY_LEVEL_R0(config->region0SecLevel) | BEE_CTRL_CTRL_AES_MODE_R0(config->region0Mode) |
                  BEE_CTRL_SECURITY_LEVEL_R1(config->region1SecLevel) | BEE_CTRL_CTRL_AES_MODE_R1(config->region1Mode);

    /* Load values to registers */
    base->CTRL = beeCtrlVal;
    base->ADDR_OFFSET0 = config->region0AddrOffset;
    base->ADDR_OFFSET1 = config->region1AddrOffset;
    base->REGION1_BOT = config->region1Bot;
    base->REGION1_TOP = config->region1Top;

}

AT_QUICKACCESS_SECTION_CODE(status_t BEE_SetRegionNonceRamfunc(BEE_Type *base, bee_region_t region, const uint32_t *nonce, size_t nonceSize))
{
    /* Nonce must be 32-bit aligned */
    if (((uintptr_t)nonce & 0x3u) || (nonceSize != 16))
    {
        return kStatus_InvalidArgument;
    }

    /* Wait until BEE is in idle state */
    while (!(base->STATUS & kBEE_IdleFlag))
    {
    }

    /* Write nonce registers, nonce is stored in little-endian format in memory */
    uint32_t *dest32;
    if (region == kBEE_Region0)
    {
        dest32 = (uint32_t *)&base->CTR_NONCE0_W0;
    }

    else if (region == kBEE_Region1)
    {
        dest32 = (uint32_t *)&base->CTR_NONCE1_W0;
    }
    else
    {
        return kStatus_InvalidArgument;
    }
    while (nonceSize >= sizeof(uint32_t))
    {
        *dest32 = *nonce;
        nonceSize -= sizeof(uint32_t);
        dest32++;
        nonce++;
    }
    return kStatus_Success;
}
void bee_config(protect_region_block_t prdb)
{
    bee_region_config_t beeConfig={0};
    
    /* Read the BEE_KEY1_SEL value */
    uint32_t bee_key_sel1 = OCOTP->CFG5 & 0x0000C000;

    /* Initializes the configure structure to zero. */
    beeConfig.region0Mode = kBEE_AesCtrMode;
    beeConfig.region1Mode = kBEE_AesCtrMode;
    beeConfig.region0AddrOffset = 0U;
    beeConfig.region1AddrOffset = 0U;
    beeConfig.region0SecLevel = kBEE_SecurityLevel3;
    beeConfig.region1SecLevel = kBEE_SecurityLevel3;
    beeConfig.region1Bot = prdb.encrypt_region.start;
    beeConfig.region1Top = prdb.encrypt_region.end;
    beeConfig.accessPermission = kBEE_AccessProtDisabled;
    beeConfig.endianSwapEn = kBEE_EndianSwapEnabled;

   /* Init BEE driver and apply the configuration */
   // BEE_Init(BEE);
    BEE->CTRL = BEE_CTRL_CTRL_SFTRST_N_MASK | BEE_CTRL_CTRL_CLK_EN_MASK;
    BEE_SetConfigRamfunc(BEE, &beeConfig);
     
    /* Configure Start address and end address of access protected region-0 */
    IOMUXC_GPR->GPR21 = prdb.encrypt_region.end;
    IOMUXC_GPR->GPR20 = prdb.encrypt_region.start;
    /* Enable BEE data decryption of memory region-1 */
    IOMUXC_GPR->GPR11 |= IOMUXC_GPR_GPR11_BEE_DE_RX_EN(2);
  
   // status = BEE_SetRegionKey(BEE, kBEE_Region1, aesKey, AES_KEY_LEN);
    BEE->CTRL |= BEE_CTRL_KEY_REGION_SEL_MASK;
    BEE->CTRL &= ~BEE_CTRL_KEY_VALID_MASK;
    BEE->CTRL |= BEE_CTRL_KEY_VALID_MASK;
    BEE_SetRegionNonceRamfunc(BEE, kBEE_Region1, (uint32_t *)prdb.encrypt_region.counter, sizeof(prdb.encrypt_region.counter));
    //enable BEE
    BEE->CTRL |= BEE_CTRL_BEE_ENABLE_MASK;
    
   // DCACHE_InvalidateByRange(prdb.encrypt_region.start,prdb.encrypt_region.end + 1 - prdb.encrypt_region.start);
    //ICACHE_InvalidateByRange(prdb.encrypt_region.start,prdb.encrypt_region.end + 1 - prdb.encrypt_region.start);
}
status_t decryption_config(void)
{
  status_t status = kStatus_Success;
  dcp_alg_ctx_t dcp_ctx;
  dcp_aes_init(&dcp_ctx);
  aes_key_sel_t key_sel;
  key_info_block_t sKIB;
  protect_region_block_t sPRDB;
  if(is_phdr_valid(&phdr,BL_PARAMETERS_ADDRESS))
  {
      // Enecrypt KIB
      key_sel.option = SW_GP2_FLAG_BE;
      dcp_aes_set_key(&dcp_ctx, key_sel, 128);
      dcp_aes_ecb_crypt(&dcp_ctx, kAesMode_Decrypt, (uint8_t *)&phdr.kib,
                        (uint8_t *)&sKIB, sizeof(key_info_block_t));
      key_sel.key = &sKIB.key[0];
      dcp_aes_set_key(&dcp_ctx, key_sel, 128);
      dcp_aes_cbc_crypt(&dcp_ctx, kAesMode_Decrypt, &sKIB.iv[0], (uint8_t *)&phdr.prdb,
                        (uint8_t *)&sPRDB, sizeof(protect_region_block_t));
      
      // configure BEE
      /* Get default configuration. */
      CLOCK_EnableClock(kCLOCK_Bee);
      bee_config(sPRDB);
  }
  else
  {
      status = kStatus_Fail;
  }
  
  return status;
}

//! @brief Determines the active peripheral.
//!
//! This function has several stages:
//! - Init enabled peripherals.
//! - Compute timeout.
//! - Wait for peripheral activity with timeout.
//! - Shutdown inactive peripherals.
//!
//! If peripheral detection times out, then this function will call jump_to_application() to
//! directly enter the user application.
//!
//! The timeout value comes from the BCA if set, or the #BL_DEFAULT_PERIPHERAL_DETECT_TIMEOUT
//! configuration macro. If the boot pin is asserted, or if there is not a valid user application
//! in flash, then the timeout is disabled and peripheral detection will continue infinitely.
static peripheral_descriptor_t const *get_active_peripheral(void)
{
    peripheral_descriptor_t const *peripheral;
    peripheral_descriptor_t const *activePeripheral = NULL;

    // Bring up all the peripherals
    for (peripheral = g_peripherals; peripheral->typeMask != 0; ++peripheral)
    {
        // Check that the peripheral is enabled in the user configuration data
        if (peripheral->typeMask)
        {
            assert(peripheral->controlInterface->init);

#if defined(DEBUG) && !defined(DEBUG_PRINT_DISABLE)
            debug_printf("Initing %s\r\n", get_peripheral_name(peripheral->typeMask));
#endif
            peripheral->controlInterface->init(peripheral, peripheral->packetInterface->byteReceivedCallback);
        }
    }

    const uint64_t ticksPerMillisecond = microseconds_convert_to_ticks(1000);
    uint64_t lastTicks = 0;    // Value of our last recorded ticks second marker
    uint64_t timeoutTicks = 0; // The number of ticks we will wait for timeout, 0 means no timeout
    // Get the user application entry point and stack pointer.
    uint32_t milliseconds;
    milliseconds = BL_DEFAULT_PERIPHERAL_DETECT_TIMEOUT;
    timeoutTicks = milliseconds * ticksPerMillisecond;

    // save how many ticks we're currently at before the detection loop starts
    lastTicks = microseconds_get_ticks();
    // decryption_config();
    // Wait for a peripheral to become active
    while (activePeripheral == NULL)
    {

        // If timeout is enabled, check to see if we've exceeded it.
        if (timeoutTicks)
        {
            // Note that we assume that the tick counter won't overflow and wrap back to 0.
            // The timeout value is only up to 65536 milliseconds, and the tick count starts
            // at zero when when inited the microseconds driver just a few moments ago.
            uint64_t currentticks = microseconds_get_ticks();
            uint64_t elapsedTicks = currentticks - lastTicks;

            // Check if the elapsed time is longer than the timeout.
            if (elapsedTicks >= timeoutTicks)
            {
                  debug_printf("ready to jump to apps code\r\n");
                   // In the case of the typical peripheral timeout, jump to the user application.
                  jump_to_application();
                  
                  // fail to run apps code, continue to check and wait the next timeout
                  lastTicks = currentticks;

            }
        }

        // Traverse through all the peripherals
        for (peripheral = g_peripherals; peripheral->typeMask != 0; ++peripheral)
        {
            // Check that the peripheral is enabled in the user configuration data
            if (peripheral->typeMask)
            {
                assert(peripheral->controlInterface->pollForActivity);

                if (peripheral->controlInterface->pollForActivity(peripheral))
                {
#if defined(DEBUG) && !defined(DEBUG_PRINT_DISABLE)
                    debug_printf("%s is active\r\n", get_peripheral_name(peripheral->typeMask));
#endif
                    activePeripheral = peripheral;
                    break;
                }
            }
        }
    }

    // Shut down all non active peripherals
    for (peripheral = g_peripherals; peripheral->typeMask != 0; ++peripheral)
    {
        // Check that the peripheral is enabled in the user configuration data
        if (peripheral->typeMask)
        {
            if (activePeripheral != peripheral)
            {
#if defined(DEBUG) && !defined(DEBUG_PRINT_DISABLE)
                debug_printf("Shutting down %s\r\n", get_peripheral_name(peripheral->typeMask));
#endif
                assert(peripheral->controlInterface->shutdown);
                peripheral->controlInterface->shutdown(peripheral);
            }
        }
    }

    return activePeripheral;
}

//! @brief Initialize the bootloader and peripherals.
//!
//! This function initializes hardware and clocks, loads user configuration data, and initialzes
//! a number of drivers. It then enters the active peripheral detection phase by calling
//! get_active_peripheral(). Once the peripheral is detected, the packet and comand interfaces
//! are initialized.
//!
//! Note that this routine may not return if peripheral detection times out and the bootloader
//! jumps directly to the user application in flash.
void bootloader_init(void)
{
    // Init the global irq lock
    lock_init();

    // Init pinmux and other hardware setup.
    CLOCK_EnableClock(kCLOCK_UsbOh3);

    // Load the user configuration data so that we can configure the clocks
    //g_bootloaderContext.propertyInterface->load_user_config();

    // Configure clocks.
    configure_clocks(kClockOption_EnterBootloader);

    // Start the lifetime counter
    microseconds_init();
    
#if BL_FEATURE_BYPASS_WATCHDOG
    bootloader_watchdog_init();
#endif // BL_FEATURE_BYPASS_WATCHDOG

#if BL_FEATURE_RELIABLE_UPDATE
    bootloader_reliable_update_as_requested(kReliableUpdateOption_Normal, 0);
#endif // BL_FEATURE_RELIABLE_UPDATE

    // Message so python instantiated debugger can tell the
    // bootloader application is running on the target.
    debug_printf("\r\n\r\nRunning bootloader...\r\n");

    // Wait for a peripheral to become active.
    g_bootloaderContext.activePeripheral = get_active_peripheral();
    assert(g_bootloaderContext.activePeripheral);

    // Validate required active peripheral interfaces.
    assert(g_bootloaderContext.activePeripheral->controlInterface);

    // Init the active peripheral.
    if (g_bootloaderContext.activePeripheral->byteInterface &&
        g_bootloaderContext.activePeripheral->byteInterface->init)
    {
        g_bootloaderContext.activePeripheral->byteInterface->init(g_bootloaderContext.activePeripheral);
    }
    if (g_bootloaderContext.activePeripheral->packetInterface &&
        g_bootloaderContext.activePeripheral->packetInterface->init)
    {
        g_bootloaderContext.activePeripheral->packetInterface->init(g_bootloaderContext.activePeripheral);
    }

    // Initialize the command processor component.
    g_bootloaderContext.commandInterface->init();
}

//! @brief Bootloader outer loop.
//!
//! Infinitely calls the command interface and active peripheral control interface pump routines.
void bootloader_run(void)
{
    const peripheral_descriptor_t *activePeripheral = g_bootloaderContext.activePeripheral;

    assert(g_bootloaderContext.commandInterface->pump);

    // Read and execute commands.
    while (1)
    {
        g_bootloaderContext.commandInterface->pump();

        // Pump the active peripheral.
        if (activePeripheral->controlInterface->pump)
        {
            activePeripheral->controlInterface->pump(activePeripheral);
        }
    }
}

//! @brief Entry point for the bootloader.

int main(void)
{
    BOARD_ConfigMPU();
    BOARD_BootClockRUN();
#ifdef DEBUG
    BOARD_InitPins();
    BOARD_InitDebugConsole();
    debug_printf("FlexSPI Program test start-mdk(boot Loader)\r\n");
#endif
    FLEXSPI_UpdateLUTRamfunc(EXAMPLE_FLEXSPI, 4, &g_customLUT[4], CUSTOM_LUT_LENGTH-4);  
    bootloader_init();
    bootloader_run();
    // Should never end up here.
    debug_printf("Warning: reached end of main()\r\n");
    return 0;
}

//! Since we never exit this gets rid of the C standard functions that cause
//! extra ROM size usage.
#undef exit
void exit(int arg)
{
}

#if defined(__CC_ARM)
#define ITM_Port8(n) (*((volatile unsigned char *)(0xE0000000 + 4 * n)))
#define ITM_Port16(n) (*((volatile unsigned short *)(0xE0000000 + 4 * n)))
#define ITM_Port32(n) (*((volatile unsigned long *)(0xE0000000 + 4 * n)))

#define DEMCR (*((volatile unsigned long *)(0xE000EDFC)))
#define TRCENA 0x01000000

struct __FILE
{
    int handle; /* Add whatever needed */
};
FILE __stdout;
FILE __stdin;

int fputc(int ch, FILE *f)
{
    if (DEMCR & TRCENA)
    {
        while (ITM_Port32(0) == 0)
            ;
        ITM_Port8(0) = ch;
    }
    return (ch);
}
#endif

//! @}

////////////////////////////////////////////////////////////////////////////////
// EOF
////////////////////////////////////////////////////////////////////////////////
