/*
 * @brief State Configurable Timer (SCT) example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2018
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */


#include <stdio.h>
#include <stdbool.h>
#include "LPC8xx.h"
#include "lpc8xx_syscon.h"
#include "lpc8xx_sct.h"
#include "lpc8xx_swm.h"
#include "lpc8xx_gpio.h"
#include "utilities.h"
#include "SCT_UART.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
#define UART_PARITY_NONE            (0)
#define UART_PARITY_ODD             (1)
#define UART_PARITY_EVEN            (2)
/**
 * UART configuration.
 * Configure UART parameters as needed (by modifying macros).
 */
#define SCT_UART_BAUD_RATE          (9600)                      ///< UART baud rate.
#define SCT_UART_DATA_BITS          (8)                         ///< UART data bits. Currently only supports 8.
#define SCT_UART_STOP_BITS          (1)                         ///< UART stop bits.
#define SCT_UART_PARITY             UART_PARITY_NONE            ///< UART parity. It is currently unavailable.
#define SCT_UART_FLOW_CONTROL                                   ///< UART flow control. It is currently unavailable.

#define SCT_UART_TX_PORT             0                          ///< The port of TX pin
#define SCT_UART_TX_PIN              0                          ///< The pin number of TX
#define SCT_UART_RX_PORT             0                          ///< The port of RX pin
#define SCT_UART_RX_PIN              4                          ///< The pin number of RX

/**
 * The brief description.
 * The detail description.
 */
typedef struct
{
    int8_t send_bit_seq;            ///< 0:start bit	1-9:data bits	(parity bit may be added later.)	<0:stop bit.
    int8_t receive_bit_seq;         ///< For receiving data.
    int8_t tx_send_status;          ///< 1: sending  0:idle.
    uint8_t send_tempbuf;           ///< Tempbuf for sending data.
    uint8_t receive_tempbuf;        ///< Tempbuf for reading data.
    int8_t rx_receive_status;       ///< 1: received  0:idle.
    int8_t tx_parity_temp;
    int8_t rx_parity_temp;
    int8_t rx_parity_res;
} _SoftSerial_t;

volatile _SoftSerial_t SoftSerial = {.send_bit_seq = 0,
                                     .tx_send_status = 0,
                                     .receive_bit_seq = 0,
                                     .rx_receive_status = 0,
#if SCT_UART_PARITY == UART_PARITY_ODD
                                     .tx_parity_temp = 1,
                                     .rx_parity_temp = 1,
#elif SCT_UART_PARITY == UART_PARITY_EVEN
                                     .tx_parity_temp = 0,
                                     .rx_parity_temp = 0,
#endif
                                     };
/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
#if SCT_UART_PARITY != UART_PARITY_NONE
#define SCT_UART_PARITY_BITS         1
#else
#define SCT_UART_PARITY_BITS         0
#endif
/**
 * @brief   Send one byte by bits.
 *          This function is private.
 * @param   Nothing.
 * @return  Nothing.
 */
static void __SCT_UART_Send_Bits(void)
{
    if (SoftSerial.tx_send_status == 0)
    {
        return;
    }

    if (SoftSerial.send_bit_seq == 0) //start bit
    {
        LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 0;
    }
    else if (SoftSerial.send_bit_seq < (SCT_UART_DATA_BITS + 1)) //data bits
    {
        if (SoftSerial.send_tempbuf & 0x01)
        {
            LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 1;
#if SCT_UART_PARITY != UART_PARITY_NONE
            SoftSerial.tx_parity_temp = SoftSerial.tx_parity_temp ^(0x01);
#endif
        }
        else
        {
            LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 0;
        }
        SoftSerial.send_tempbuf >>= 1;
    }
#if SCT_UART_PARITY != UART_PARITY_NONE
    else if (SoftSerial.send_bit_seq == (SCT_UART_DATA_BITS + 1))
    {
        if (SoftSerial.tx_parity_temp == 1)
        {
            LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 1;
        }
        else
        {
            LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 0;
        }
    }
#endif
    else if (SoftSerial.send_bit_seq == (SCT_UART_DATA_BITS + SCT_UART_PARITY_BITS + 1)) //stop bit
    {
        LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 1;
    }

    if (SoftSerial.send_bit_seq == (SCT_UART_DATA_BITS + SCT_UART_STOP_BITS + SCT_UART_PARITY_BITS + 1))
    {
        LPC_SCT->CTRL_H |= (1 << Halt_L);
        SoftSerial.tx_send_status = 0;
        SoftSerial.send_bit_seq = 0;
#if SCT_UART_PARITY == UART_PARITY_ODD
        SoftSerial.tx_parity_temp = 1;
#elif SCT_UART_PARITY == UART_PARITY_EVEN
        SoftSerial.tx_parity_temp = 0;
#endif
        return;
    }

    SoftSerial.send_bit_seq++;
}

/**
 * @brief   Receive one byte by bits.
 *          This function is private.
 * @param   Nothing.
 * @return  Nothing.
 */
static void __SCT_UART_Receive_Bits(void)
{
    if (SoftSerial.receive_bit_seq < SCT_UART_DATA_BITS) //data bits bits: 0--->(SCT_UART_DATA_BITS-1)
    {
        if (LPC_GPIO_PORT->B0[SCT_UART_RX_PIN] != 0)
        {
            SoftSerial.receive_tempbuf |= (1 << SoftSerial.receive_bit_seq);
#if SCT_UART_PARITY != UART_PARITY_NONE
            SoftSerial.rx_parity_temp = SoftSerial.rx_parity_temp ^(0x01);
#endif
        }
        else if (LPC_GPIO_PORT->B0[SCT_UART_RX_PIN] == 0)
        {
            SoftSerial.receive_tempbuf &= ~(1 << SoftSerial.receive_bit_seq);
        }
        SoftSerial.receive_bit_seq++;
    }
#if SCT_UART_PARITY != UART_PARITY_NONE
    else if (SoftSerial.receive_bit_seq == SCT_UART_DATA_BITS) //parity bits
    {
        if (LPC_GPIO_PORT->B0[SCT_UART_RX_PIN] != 0)
        {
            SoftSerial.rx_parity_res = 1;
        }
        else if (LPC_GPIO_PORT->B0[SCT_UART_RX_PIN] == 0)
        {
            SoftSerial.rx_parity_res = 0;
        }
        SoftSerial.receive_bit_seq++;
    }
#endif
    else if (SoftSerial.receive_bit_seq == SCT_UART_DATA_BITS + SCT_UART_PARITY_BITS)
    {
        LPC_SCT->CTRL_L |= (1 << Halt_L);
        LPC_SCT->STATE_L = 0;
        LPC_SCT->CTRL_L &= ~(1 << Halt_L);

#if SCT_UART_PARITY != UART_PARITY_NONE
        if (SoftSerial.rx_parity_temp == SoftSerial.rx_parity_res)
        {
            SCT_UART_Receive_Byte_CallBack(SoftSerial.receive_tempbuf);
        }
#else
        SCT_UART_Receive_Byte_CallBack(SoftSerial.receive_tempbuf);
#endif
        SoftSerial.receive_tempbuf = 0;
        SoftSerial.receive_bit_seq = 0;
        SoftSerial.rx_receive_status = 1;
#if SCT_UART_PARITY == UART_PARITY_ODD
        SoftSerial.rx_parity_temp = 1;
#elif SCT_UART_PARITY == UART_PARITY_EVEN
        SoftSerial.rx_parity_temp = 0;
#endif
    }
}

/**
 * @brief   Handle interrupt from State Configurable Timer
 * @return  Nothing
 */
void SCT_IRQHandler(void)
{
    if (LPC_SCT->EVFLAG & (1 << event0)) // Event 0 ?
    {
        LPC_SCT->EVFLAG = (1 << event0);
        __SCT_UART_Send_Bits();
    }
    if (LPC_SCT->EVFLAG & (1 << event3)) // Event 3 ?
    {
        LPC_SCT->EVFLAG = (1 << event3);
        __SCT_UART_Receive_Bits();
    }
}

/**
 * @brief   Initialize IO that are used for UART.
 * @param   Nothing.
 * @return  Nothing.
 */
static void __SCT_Uart_IO_Init(void)
{
    GPIOInit();
    LPC_GPIO_PORT->DIRSET0 = 1UL << SCT_UART_TX_PIN;
    LPC_GPIO_PORT->B0[SCT_UART_TX_PIN] = 1;
}

/**
 * @brief   Initialize SCT.
 * @param   Nothing.
 * @return  Nothing.
 */
static void __SCT_Uart_SCT_Init(void)
{
    /* Enable clocks to relevant peripherals. */
    LPC_SYSCON->SYSAHBCLKCTRL |= (SWM | SCT);
    /* Configure the SWM (see utilities_lib and lpc8xx_swm.h) */
    /* SCT input 0 on RX, for input transition events (default internal pull-up remains on) */
    ConfigSWM(SCT_PIN0, SCT_UART_RX_PIN);
    /* Write 0x0 to SCT0_INMUX[0] to select SCT_PIN0 function (which was connected to RX in the switch matrix) as SCT input SCT_IN0 */
    LPC_INPUTMUX->SCT0_INMUX0 = 0;

    /* Configure the SCT ... */
    /* Give the module a reset */
    LPC_SYSCON->PRESETCTRL &= (SCT0_RST_N);
    LPC_SYSCON->PRESETCTRL |= ~(SCT0_RST_N);

    /* Get the System Clock frequency for the BRG calculation */
    SystemCoreClockUpdate();

    /* Configure the SCT counter as two 16 bit counters using the bus clock. Others use the reset value */
    LPC_SCT->CONFIG |= (0 << UNIFY) |
                       (Bus_clock << CLKMODE);

    /* Set the match count for match register 0 */
    LPC_SCT->MATCH[0].H = (uint16_t)(SystemCoreClock / SCT_UART_BAUD_RATE);

    /* Set the match reload value for match reload register 0*/
    LPC_SCT->MATCHREL[0].H = (uint16_t)(SystemCoreClock / SCT_UART_BAUD_RATE);

    /* Event 0 only happens on a match condition and choose the MATCH[0] High part*/
    LPC_SCT->EVENT[0].CTRL = (1 << HEVENT) |           // HEVENT[4] = Selects the H state and the H match register selected by MATCHSEL.
                             (Match_Only << COMBMODE); // COMBMODE[13:12] = uses match condition only

    /* Event 0 happens in state 0 and state 1 */
    LPC_SCT->EVENT[0].STATE = (uint32_t)((1 << state0) | (1 << state1));

    /* Event 0 is used as the H counter limit */
    LPC_SCT->LIMIT_H = (uint16_t)(1 << event0);

    LPC_SCT->EVENT[1].CTRL = (0 << HEVENT) |               // HEVENT[4]       = Selects the L state and the L match register selected by MATCHSEL.
                             (0 << IOSEL) |                // IOSEL   [9:6]   = CTIN_0
                             (Fall << IOCOND) |            // IOCOND  [11:10] = falling edge
                             (IO_Only << COMBMODE) |       // COMBMODE[13:12] = uses IO condition only
                             (1 << STATELD) |              // STATELD [14]    = STATEV is loaded into state
                             (1 << STATEV);                // STATEV[15]      = new state is 1

    /* Event 1 only happens in state 0 */
    LPC_SCT->EVENT[1].STATE = (uint32_t)(1 << state0);

    /* Set the match count for match register 0 */
    LPC_SCT->MATCH[0].L = (uint16_t)(SystemCoreClock / SCT_UART_BAUD_RATE / 8);

    /* Set the match reload value for match reload register 0*/
    LPC_SCT->MATCHREL[0].L = (uint16_t)(SystemCoreClock / SCT_UART_BAUD_RATE / 8);

    LPC_SCT->EVENT[2].CTRL = (0 << MATCHSEL) |
                             (0 << HEVENT) |               // HEVENT[4]       = Selects the L state and the L match register selected by MATCHSEL.
                             (Match_Only << COMBMODE) |    // COMBMODE[13:12] = uses match condition only
                             (1 << STATELD) |              // STATELD [14]    = STATEV is loaded into state
                             (2 << STATEV);                // STATEV[15]      = new state is 2

    /* Event 2 only happens in state 1 */
    LPC_SCT->EVENT[2].STATE = (uint32_t)(1 << state1);

    /* Set the match count for match register 0 */
    LPC_SCT->MATCH[1].L = (uint16_t)(SystemCoreClock / SCT_UART_BAUD_RATE);

    /* Set the match reload value for match reload register 0*/
    LPC_SCT->MATCHREL[1].L = (uint16_t)(SystemCoreClock / SCT_UART_BAUD_RATE);

    LPC_SCT->EVENT[3].CTRL = (1 << MATCHSEL) |
                             (0 << HEVENT) |               // HEVENT[4]       = Selects the L state and the L match register selected by MATCHSEL.
                             (Match_Only << COMBMODE) |    // COMBMODE[13:12] = uses match condition only
                             (0 << STATELD) |              // STATELD [14]    = STATEV is loaded into state!!!!!!!!!!!!!!!!!!!!!!!!
                             (0 << STATEV);                // STATEV[15]      = new state is 1!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    /* Event 3 only happens in state 2 */
    LPC_SCT->EVENT[3].STATE = (uint32_t)(1 << 2);

    /* Event 1 2 3 is used as the L counter limit */
    LPC_SCT->LIMIT_L = (uint16_t)((1 << event1) | (1 << event2) | (1 << event3));

    /* Enable Event flag interrupts for Events 0 and 3 */
    LPC_SCT->EVEN = (1 << event0) | (1 << event3);

    /* Enable the interrupt for the SCT */
    NVIC_EnableIRQ(SCT_IRQn);

    /* Start the SCT low 16 bit counter by clearing Halt_L in the SCT control register */
    LPC_SCT->CTRL_L &= ~(1 << Halt_L);
    /* No need to start the SCT high 16 bit counter, 
       because the high 16 bit counter is only started when there is data to send, 
       and halts after the data transmission is complete.   
    */
    //LPC_SCT->CTRL_H &= ~(1 << Halt_L);
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/**
 * @brief   Initialize SCT UART.
 * @param   Nothing.
 * @return  Nothing.
 */
void SCT_UART_Init(void)
{
    __SCT_Uart_IO_Init();
    __SCT_Uart_SCT_Init();
}

/**
 * @brief   Send one byte of data through SCT UART.
 * @param   Nothing.
 * @return  Nothing.
 */
void SCT_UART_Send_Byte(uint8_t BytetoSend)
{
    while (SoftSerial.tx_send_status != 0)
    {
        ;
    }

    __disable_irq();
    SoftSerial.send_tempbuf = BytetoSend;
    SoftSerial.tx_send_status = 1;
    LPC_SCT->CTRL_H &= ~(1 << Halt_L);
    __enable_irq();

    return;
}

/**
 * @brief   This function will call back when SCT UART receive one byte data.
 *          Please implement it according to your needs.
 *          Because it will run in the ISR, make sure it run as fast as possible and without blocking.
 *          This function has its name emitted as a weak symbol instead of a global name. This is primarily for the naming of library routines that can be overridden by user code.
 * @param   ReceiveByte     One byte of data that has been received.
 *                          When this function is called, The received one byte data will be passed as actual parameter to this function.
 * @return  Nothing.
 */
void __attribute__((weak)) SCT_UART_Receive_Byte_CallBack(uint8_t ReceiveByte)
{
}
