/* Copyright 2017 The TensorFlow Authors. All Rights Reserved.
   Copyright 2019 NXP. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#include "converted_model.h"
#include <vibration_data.h>
#include "board.h"
#include "pin_mux.h"
#include "clock_config.h"
#include "timer.h"

#include <iostream>
#include <string>
#include <vector>

#include "tensorflow/contrib/lite/kernels/register.h"
#include "tensorflow/contrib/lite/model.h"
#include "tensorflow/contrib/lite/optional_debug_tools.h"
#include "tensorflow/contrib/lite/string_util.h"
#include "tensorflow/contrib/lite/op_resolver.h"

#define LOG(x) std::cout

/*! *******************************************************************************
* \brief        This function checks the smallest of 3 numbers given as input
*
* \param[in]    n1	:  This is the output of the tensorflowlite model. This gives
* 						   array of distances of each point from the 3 centroids
* 				n2	:  Size of the total output
* 				n3	:  X,Y or Z Axis
*/
int findsmallest( int n1, int n2, int n3)
{

		if(n1 <= n2 && n1 <= n3)
	    {
	        return 0;
	    }
	    if(n2 <= n1 && n2 <= n3)
	    {
	        return 1;
	    }
	    if(n3 <= n1 && n3 <= n2)
	    {
	        return 2;
	    }
	    return -1;
}
/*! *******************************************************************************
* \brief        This function takes the array of distances of each point from the 3 centroids
* 				as input and assign the cluster number for each point based on the smallest
* 				distance of each point from 3 centroids.
* 				Note: This function needs to be used
*
* \param[in]    output	:  This is the output of the tensorflowlite model. This gives
* 						   array of distances of each point from the 3 centroids
* 				size	:  Size of the total output
* 				dim		:  X,Y or Z Axis
*/
void Calculate_clusters( int* output, int size, int axis_data)
{
	int list[size];
	int clst_0=0;
	int clst_1=0;
	int clst_2=0;

	int index_clst;
	uint8_t flag=0;
	/* index of output which gives pointer to distance of points from centroid 2 */
	clst_1 = clst_0 + size;
	/* index of output which gives pointer to distance of points from centroid 3 */
	clst_2 = clst_1 + size;

	for(index_clst = 0; index_clst < size; index_clst++)
	{
		list[index_clst] = findsmallest( output[clst_0],output[clst_1],output[clst_2]);
		/* As per the developed model if any point is assigned in cluster 2 then it is considered as anomaly */
		if(list[index_clst] == 2)
		{
			flag = 1;
		}
		clst_0++;
		clst_1++;
		clst_2++;
	}
	switch (axis_data)
	{
	case X_AXIS:
		if(flag ==1)
			LOG(INFO) << "X:Anomaly\n";
		else
			LOG(INFO) << "X:Healthy\n";
		break;
	case Y_AXIS:
		if(flag ==1)
			LOG(INFO) << "Y:Anomaly\n";
		else
			LOG(INFO) << "Y:Healthy\n";
		break;
	case Z_AXIS:
		if(flag ==1)
			LOG(INFO) << "Z:Anomaly\n";
		else
			LOG(INFO) << "Z:Healthy\n";
		break;
	default:;
	}
	flag = 0;
}



void RunInference(void)
{
  std::unique_ptr<tflite::FlatBufferModel> model;
  std::unique_ptr<tflite::Interpreter> interpreter;
  tflite::StderrReporter error_reporter;

  model = tflite::FlatBufferModel::BuildFromBuffer(__converted_model_new_tflite, __converted_model_new_tflite_len);

  if (!model) {
    LOG(FATAL) << "Failed to load model\r\n";
    return;
  }
  model->error_reporter();

  tflite::ops::builtin::BuiltinOpResolver resolver;

  tflite::InterpreterBuilder(*model, resolver)(&interpreter);
  if (!interpreter) {
    LOG(FATAL) << "Failed to construct interpreter\r\n";
    return;
  }
  int input_x = interpreter->inputs()[0];
  int input_y = interpreter->inputs()[1];
  int input_z = interpreter->inputs()[2];

  const std::vector<int> inputs = interpreter->inputs();
  const std::vector<int> outputs = interpreter->outputs();
  interpreter->ResizeInputTensor(input_x, {1,200,1});
  interpreter->ResizeInputTensor(input_y, {1,200,1});
  interpreter->ResizeInputTensor(input_z, {1,200,1});


  if (interpreter->AllocateTensors() != kTfLiteOk) {
    LOG(FATAL) << "Failed to allocate tensors!\r\n";
  }

  /* Get input dimension from the input tensor metadata
     assuming one input only */
  TfLiteTensor* input_tensor_x = interpreter->tensor(input_x);

  int input_bytes_x = input_tensor_x->bytes;
  int* inputn_x = interpreter->typed_input_tensor<int>(0);
  int* inputn_y = interpreter->typed_input_tensor<int>(1);
  int* inputn_z = interpreter->typed_input_tensor<int>(2);

  for (int i = 0; i < (input_bytes_x/4); i++) {
		  inputn_x[i] = vibration_data_x[i];
		  inputn_y[i] = vibration_data_y[i];
		  inputn_z[i] = vibration_data_z[i];

   }
  for( int i=0 ; i < (input_bytes_x/4); i++){
	  interpreter->typed_output_tensor<int>(0)[i] = 0;
	  interpreter->typed_output_tensor<int>(1)[i] = 0;
	  interpreter->typed_output_tensor<int>(2)[i] = 0;
  }
  if (interpreter->Invoke() != kTfLiteOk) {
	LOG(FATAL) << "Failed to invoke tflite!\r\n";
	return;
  }
  std::vector<std::pair<float, int>> top_results;

  int output = interpreter->outputs()[0];

  TfLiteTensor* output_tensor = interpreter->tensor(output);
  TfLiteIntArray* output_dims = output_tensor->dims;

  /* Assume output dims to be something like (1, 1, ... , size) */
  auto output_size = output_dims->data[output_dims->size - 1];
  int* outputn_x;
  int* outputn_y;
  int* outputn_z;

  outputn_x = interpreter->typed_output_tensor<int>(0);
  outputn_y = interpreter->typed_output_tensor<int>(1);
  outputn_z = interpreter->typed_output_tensor<int>(2);

  Calculate_clusters(outputn_x, output_size,X_AXIS);
  Calculate_clusters(outputn_y, output_size,Y_AXIS);
  Calculate_clusters(outputn_z, output_size,Z_AXIS);
}

/*! *******************************************************************************
* \brief   		Calculates the moving average of all 3 X,Y, & Z axis
*               from the raw samples collected from accelerometer.
*               This is done for data filtering.
*
******************************************************************************** */
void Calculate_AVG(void)
{
  int mean_x = 0,mean_y =0, mean_z=0,j=0;
  for(int i=0; i < (sizeof(raw_data_x)/sizeof(raw_data_x[0])); i++)
  {
	  /* Here we are taking moving average of 5 samples so for the first 5 samples it is simple average */
	  if(i <=4)
	  {
			mean_x = mean_x + raw_data_x[i];
			mean_y = mean_y + raw_data_y[i];
			mean_z = mean_z + raw_data_y[i];
	  }

	  if(i==4)
	  {
			mean_x = mean_x/5;
			mean_y = mean_y/5;
			mean_z = mean_z/5;

			avg_x[j]=mean_x;
			avg_y[j]=mean_y;
			avg_z[j]=mean_z;

			j++;
	 }
	  /* After first 5 samples we have to take moving average of each 5 samples so it is like LIFO(last in first out) */
	 else if(i > 4)
	 {
			mean_x = mean_x - raw_data_x[(i-5)];
			mean_x = mean_x + raw_data_x[i];
			avg_x[j] = mean_x/5;
			mean_y = mean_y - raw_data_y[(i-5)];
			mean_y = mean_y + raw_data_y[i];
			avg_y[j] = mean_y/5;
			mean_z = mean_z - raw_data_z[(i-5)];
			mean_z = mean_z + raw_data_z[i];
			avg_z[j] = mean_z/5;
			j++;
	}
  }
}

/*! *******************************************************************************
* \brief   Calculates the RMS values of the filtered samples for X,Y and Z axis
******************************************************************************** */
void Calculate_RMS(void)
{
	int j = 0;
	int mean_x = 0,mean_y = 0,mean_z=0;
	int square_x = 0,square_y = 0,square_z=0;
	for (int i=0; i < (sizeof(avg_x)/sizeof(avg_x[0])); i++)
	{
		/*
		 * avg_x[] is the array of filtered x axis data
		 * avg_y[] is the array of filtered y axis data
		 * avg_z[] is the array of filtered z axis data
		 *
		 * */
		square_x = (int)(pow(avg_x[i],2) + 0.5);
		square_y = (int)(pow(avg_y[i],2) + 0.5);
		square_z = (int)(pow(avg_z[i],2) + 0.5);
		mean_x = square_x + mean_x;
		mean_y = square_y + mean_y;
		mean_z = square_z + mean_z;
		/* Here we calculate the RMS of every 10 samples */
		if(!(i%10))
		{
			mean_x = mean_x/10;
			mean_y = mean_y/10;
			mean_z = mean_z/10;
			vibration_data_x[j] = int(sqrt(mean_x));
			vibration_data_y[j] = int(sqrt(mean_y));
			vibration_data_z[j] = int(sqrt(mean_z));
			j++;
			mean_x =0;
			mean_y =0;
			mean_z =0;
		}
	}
}

int main(void)
{
  /* Init board hardware */
  BOARD_ConfigMPU();
  BOARD_InitPins();
  BOARD_BootClockRUN();
  Init_FXOS_Accelerometer();
  Init_GPT();

  std::cout << "Anomaly Detection example using a TensorFlow Lite model\r\n";
  for (;;)
  {
	  /* For sensor data collection, disable macros in timer.h */
#ifdef RUN_INFERENCE
	  read_sensor_data();
	  if(counter == no_of_samples)
	  {
		 /* Apply moving average filter to the chunk of data for further processing*/
		 Calculate_AVG();
		 /* Extract the RMS feature of the filtered data */
		 Calculate_RMS();
		 RunInference();
		 counter = 0;
	  }
#else
	  read_sensor_data();
#endif
	  std::flush(std::cout);
  }
}



