/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "fsl_sdif.h"
#include "fsl_sdif1.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Component ID definition, used by tools. */
#ifndef FSL_COMPONENT_ID
#define FSL_COMPONENT_ID "platform.drivers.sdif"
#endif

///* Typedef for interrupt handler. */
//typedef void (*sdif_isr_t)(SDIF_Type *base, sdif_handle_t *handle);

///*! @brief convert the name here, due to RM use SDIO */
//#define SDIF_DriverIRQHandler SDIO_DriverIRQHandler
///*! @brief define the controller support sd/sdio card version 2.0 */
//#define SDIF_SUPPORT_SD_VERSION (0x20)
///*! @brief define the controller support mmc card version 4.4 */
//#define SDIF_SUPPORT_MMC_VERSION (0x44)

#ifndef SDIF_TIMEOUT_VALUE
/*! @brief define the timeout counter, used to polling the start bit auto-cleared when sending clock sync command */
#define SDIF_TIMEOUT_VALUE (~0U)
#endif

//#ifndef SDIF_RESET_TIMEOUT_VALUE
///*! @brief define the reset timeout counter, two AHB clock cycle, the reset should auto-cleared. */
//#define SDIF_RESET_TIMEOUT_VALUE (100U)
//#endif

///*! @brief this value can be any value */
//#define SDIF_POLL_DEMAND_VALUE (0xFFU)
///*! @brief DMA descriptor buffer1 size */
//#define SDIF_DMA_DESCRIPTOR_BUFFER1_SIZE(x) ((x)&0x1FFFU)
///*! @brief DMA descriptor buffer2 size */
//#define SDIF_DMA_DESCRIPTOR_BUFFER2_SIZE(x) (((x)&0x1FFFU) << 13U)
///*! @brief RX water mark value */
//#define SDIF_RX_WATERMARK (15U)
///*! @brief TX water mark value */
//#define SDIF_TX_WATERMARK (16U)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*!
 * @brief Get the instance.
 *
 * @param base SDIF peripheral base address.
 * @return Instance number.
 */
static uint32_t SDIF_GetInstance(SDIF_Type *base);

/*
 * @brief config the SDIF interface before transfer between the card and host
 * @param SDIF base address
 * @param transfer config structure
 * @param enDMA DMA enable flag
 */
static status_t SDIF_TransferConfig(SDIF_Type *base, sdif_transfer_t *transfer, bool enDMA);

/*
 * @brief wait the command done function and check error status
 * @param SDIF base address
 * @param command config structure
 */
static status_t SDIF_WaitCommandDone(SDIF_Type *base, sdif_command_t *command);

/*
 * @brief transfer data in a blocking way
 * @param SDIF base address
 * @param data config structure
 * @param indicate current transfer mode:DMA or polling
 */
static status_t SDIF_TransferDataBlocking(SDIF_Type *base, sdif_data_t *data, bool isDMA);

/*
 * @brief read the command response
 * @param SDIF base address
 * @param sdif command pointer
 */
static status_t SDIF_ReadCommandResponse(SDIF_Type *base, sdif_command_t *command);

/*
 * @brief handle transfer command interrupt
 * @param SDIF base address
 * @param sdif handle
 * @param interrupt mask flags
 */
static void SDIF_TransferHandleCommand(SDIF_Type *base, sdif_handle_t *handle, uint32_t interruptFlags);

/*
 * @brief handle transfer data interrupt
 * @param SDIF base address
 * @param sdif handle
 * @param interrupt mask flags
 */
static void SDIF_TransferHandleData(SDIF_Type *base, sdif_handle_t *handle, uint32_t interruptFlags);

/*
 * @brief handle DMA transfer
 * @param SDIF base address
 * @param sdif handle
 * @param interrupt mask flag
 */
static void SDIF_TransferHandleDMA(SDIF_Type *base, sdif_handle_t *handle, uint32_t interruptFlags);

/*
 * @brief driver IRQ handler
 * @param SDIF base address
 * @param sdif handle
 */
static void SDIF_TransferHandleIRQ(SDIF_Type *base, sdif_handle_t *handle);

/*
 * @brief read data port
 * @param SDIF base address
 * @param sdif data
 * @param the number of data been transferred
 */
static uint32_t SDIF_ReadDataPort(SDIF_Type *base, sdif_data_t *data, uint32_t transferredWords);

/*
 * @brief write data port
 * @param SDIF base address
 * @param sdif data
 * @param the number of data been transferred
 */
static uint32_t SDIF_WriteDataPort(SDIF_Type *base, sdif_data_t *data, uint32_t transferredWords);

/*
 * @brief read data by blocking way
 * @param SDIF base address
 * @param sdif data
 */
static status_t SDIF_ReadDataPortBlocking(SDIF_Type *base, sdif_data_t *data);

/*
 * @brief write data by blocking way
 * @param SDIF base address
 * @param sdif data
 */
static status_t SDIF_WriteDataPortBlocking(SDIF_Type *base, sdif_data_t *data);

/*
 * @brief handle sdio interrupt
 * This function will call the SDIO interrupt callback
 * @param SDIF base address
 * @param SDIF handle
 */
static void SDIF_TransferHandleSDIOInterrupt(SDIF_Type *base, sdif_handle_t *handle);

/*
 * @brief handle card detect
 * This function will call the cardInserted callback
 * @param SDIF base addres
 * @param SDIF handle
 */
static void SDIF_TransferHandleCardDetect(SDIF_Type *base, sdif_handle_t *handle);

/*
 * @brief set command register
 * This api include polling the status of the bit START_COMMAND, if 0 used as timeout value, then this function
 * will return directly without polling the START_CMD status.
 *
 * @param base SDIF base addres
 * @param cmdIndex command index
 * @param argument command argument
 * @param timeout timeout value
 *
 * @return kStatus_Success, kStatus_SDIF_SyncCmdTimeout
 */
static status_t SDIF1_SetCommandRegister(SDIF_Type *base, uint32_t cmdIndex, uint32_t argument, uint32_t timeout);

/*******************************************************************************
 * Variables
 ******************************************************************************/


static status_t SDIF1_SetCommandRegister(SDIF_Type *base, uint32_t cmdIndex, uint32_t argument, uint32_t timeout)
{
    uint32_t syncTimeout = timeout;

    base->CMDARG = argument;
    base->CMD    = cmdIndex | SDIF_CMD_CARD_NUMBER(1) | SDIF_CMD_START_CMD_MASK; //send command to sd card 1

    while (((base->CMD & SDIF_CMD_START_CMD_MASK) == SDIF_CMD_START_CMD_MASK) && syncTimeout)
    {
        --syncTimeout;

        if (!syncTimeout)
        {
            return kStatus_SDIF_SyncCmdTimeout;
        }
    }

    return kStatus_Success;
}

/*!
 * brief SDIF send initialize 80 clocks for SD card 1 after initial
 * param base SDIF peripheral base address.
 * param timeout value
 */
bool SDIF_SendCard1Active(SDIF_Type *base, uint32_t timeout)
{
    bool enINT = false;

    /* add for conflict with interrupt mode,close the interrupt temporary */
    if ((base->CTRL & SDIF_CTRL_INT_ENABLE_MASK) == SDIF_CTRL_INT_ENABLE_MASK)
    {
        enINT = true;
        base->CTRL &= ~SDIF_CTRL_INT_ENABLE_MASK;
    }

    /* send initialization command */
    if (SDIF1_SetCommandRegister(base, SDIF_CMD_SEND_INITIALIZATION_MASK, 0U, timeout) != kStatus_Success)
    {
        return false;
    }

    /* wait command done */
    while ((SDIF_GetInterruptStatus(base) & kSDIF_CommandDone) != kSDIF_CommandDone)
    {
    }

    /* clear status */
    SDIF_ClearInterruptStatus(base, kSDIF_CommandDone);

    /* add for conflict with interrupt mode */
    if (enINT)
    {
        base->CTRL |= SDIF_CTRL_INT_ENABLE_MASK;
    }

    return true;
}

/*!
 * brief Sets the card bus clock frequency.
 *
 * param base SDIF peripheral base address.
 * param srcClock_Hz SDIF source clock frequency united in Hz.
 * param target_HZ card bus clock frequency united in Hz.
 * return The nearest frequency of busClock_Hz configured to SD bus.
 */
uint32_t SDIF_SetCard1Clock(SDIF_Type *base, uint32_t srcClock_Hz, uint32_t target_HZ)
{
    uint32_t divider = 0U, targetFreq = target_HZ;
    uint32_t syncTimeout = SDIF_TIMEOUT_VALUE;

    /* if target freq bigger than the source clk, set the target_HZ to
     src clk, this interface can run up to 52MHZ with card */
    if (srcClock_Hz < targetFreq)
    {
        targetFreq = srcClock_Hz;
    }

    /* disable the clock first,need sync to CIU*/
    SDIF_EnableCard1Clock(base, false);
    /* update the clock register and wait the pre-transfer complete */
    if (SDIF1_SetCommandRegister(base, kSDIF_CmdUpdateClockRegisterOnly | kSDIF_WaitPreTransferComplete, 0U,
                                syncTimeout) != kStatus_Success)
    {
        return 0U;
    }

    /*calculate the divider*/
    if (targetFreq != srcClock_Hz)
    {
        divider = srcClock_Hz / targetFreq;
        while (srcClock_Hz / divider > targetFreq)
        {
            divider++;
        }

        if (divider > (SDIF_CLKDIV_CLK_DIVIDER0_MASK * 2))
        {
            /* Note: if assert occur here, it means that the source clock is too big, the suggestion is reconfigure the
             * SDIF divider in SYSCON to get a properly source clock */
            assert(divider <= (SDIF_CLKDIV_CLK_DIVIDER0_MASK * 2));
            divider = (SDIF_CLKDIV_CLK_DIVIDER0_MASK * 2);
        }

        divider = (divider + 1) / 2U;
    }
    /* load the clock divider */
    base->CLKDIV = SDIF_CLKDIV_CLK_DIVIDER0(divider);
    /* update the divider to CIU */
    if (SDIF1_SetCommandRegister(base, kSDIF_CmdUpdateClockRegisterOnly | kSDIF_WaitPreTransferComplete, 0U,
                                syncTimeout) != kStatus_Success)
    {
        return 0U;
    }

    /* enable the card clock and sync to CIU */
    SDIF_EnableCard1Clock(base, true);
    if (SDIF1_SetCommandRegister(base, kSDIF_CmdUpdateClockRegisterOnly | kSDIF_WaitPreTransferComplete, 0U,
                                syncTimeout) != kStatus_Success)
    {
        return 0U;
    }

    /* config the clock delay to meet the hold time and setup time */
    SDIF_ConfigClockDelay(target_HZ, divider);

    /* return the actual card clock freq */
    return (divider != 0U) ? (srcClock_Hz / (divider * 2U)) : srcClock_Hz;
}
