/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <stdio.h>
#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_sd1.h"
#include "pin_mux.h"
#include <stdbool.h>
#include "fsl_iocon.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*! @brief Data block count accessed in card */
#define DATA_BLOCK_COUNT (5U)
/*! @brief Start data block number accessed in card */
#define DATA_BLOCK_START (2U)
/*! @brief Data buffer size. */
#define DATA_BUFFER_SIZE (FSL_SDMMC_DEFAULT_BLOCK_SIZE * DATA_BLOCK_COUNT)

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/*!
 * @brief printf the card information log.
 *
 * @param card Card descriptor.
 */
static void CardInformationLog(sd_card_t *card);
/*******************************************************************************
 * Variables
 ******************************************************************************/

/*! @brief Card descriptor. */
sd_card_t g_sd;
sd_card_t g_sd1;

/* @brief decription about the read/write buffer
 * The size of the read/write buffer should be a multiple of 512, since SDHC/SDXC card uses 512-byte fixed
 * block length and this driver example is enabled with a SDHC/SDXC card.If you are using a SDSC card, you
 * can define the block length by yourself if the card supports partial access.
 * The address of the read/write buffer should align to the specific DMA data buffer address align value if
 * DMA transfer is used, otherwise the buffer address is not important.
 * At the same time buffer address/size should be aligned to the cache line size if cache is supported.
 */
/*! @brief Data written to the card */
SDK_ALIGN(uint8_t g_dataWrite[SDK_SIZEALIGN(DATA_BUFFER_SIZE, SDMMC_DATA_BUFFER_ALIGN_CACHE)],
          MAX(SDMMC_DATA_BUFFER_ALIGN_CACHE, SDMMCHOST_DMA_BUFFER_ADDR_ALIGN));
/*! @brief Data read from the card */
SDK_ALIGN(uint8_t g_dataRead[SDK_SIZEALIGN(DATA_BUFFER_SIZE, SDMMC_DATA_BUFFER_ALIGN_CACHE)],
          MAX(SDMMC_DATA_BUFFER_ALIGN_CACHE, SDMMCHOST_DMA_BUFFER_ADDR_ALIGN));

/*! @brief SDMMC host detect card configuration */
static const sdmmchost_detect_card_t s_sdCardDetect = {
#ifndef BOARD_SD_DETECT_TYPE
    .cdType = kSDMMCHOST_DetectCardByGpioCD,
#else
    .cdType = BOARD_SD_DETECT_TYPE,
#endif
    .cdTimeOut_ms = (~0U),
	
		.userData = kSD_Card_0,
};

static const sdmmchost_detect_card_t s_sdCard1Detect = {
#ifndef BOARD_SD_DETECT_TYPE
    .cdType = kSDMMCHOST_DetectCardByGpioCD,
#else
    .cdType = BOARD_SD_DETECT_TYPE,
#endif
    .cdTimeOut_ms = (~0U),
	
		.userData = kSD_Card_1,
};

/*! @brief SDMMC card power control configuration */
#if defined DEMO_SDCARD_POWER_CTRL_FUNCTION_EXIST
static const sdmmchost_pwr_card_t s_sdCardPwrCtrl = {
    .powerOn          = BOARD_PowerOnSDCARD,
    .powerOnDelay_ms  = 500U,
//    .powerOff         = BOARD_PowerOffSDCARD,
    .powerOffDelay_ms = 0U,
};
static const sdmmchost_pwr_card_t s_sdCard1PwrCtrl = {
    .powerOn          = BOARD_PowerOnSDCARD1,
    .powerOnDelay_ms  = 500U,
//    .powerOff         = BOARD_PowerOffSDCARD,
    .powerOffDelay_ms = 0U,
};
#endif
#if defined DEMO_SDCARD_SWITCH_VOLTAGE_FUNCTION_EXIST
static const sdmmchost_card_switch_voltage_func_t s_sdCardVoltageSwitch = {
    .cardSignalLine1V8 = BOARD_USDHC_Switch_VoltageTo1V8,
    .cardSignalLine3V3 = BOARD_USDHC_Switch_VoltageTo3V3,
};
#endif
/*******************************************************************************
 * Code
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/
static status_t AccessCard(sd_card_t *card, bool isReadOnly)
{
    if (isReadOnly)
    {
        PRINTF("\r\nRead one data block......\r\n");
        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, 1U))
        {
            PRINTF("Read one data block failed.\r\n");
            return kStatus_Fail;
        }

        PRINTF("Read multiple data blocks......\r\n");
        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Read multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }
    }
    else
    {
        memset(g_dataWrite, 0x67U, sizeof(g_dataWrite));

        PRINTF("\r\nWrite/read one data block......\r\n");
        if (kStatus_Success != SD_WriteBlocks(card, g_dataWrite, DATA_BLOCK_START, 1U))
        {
            PRINTF("Write one data block failed.\r\n");
            return kStatus_Fail;
        }

        memset(g_dataRead, 0U, sizeof(g_dataRead));
        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, 1U))
        {
            PRINTF("Read one data block failed.\r\n");
            return kStatus_Fail;
        }

        PRINTF("Compare the read/write content......\r\n");
        if (memcmp(g_dataRead, g_dataWrite, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
        {
            PRINTF("The read/write content isn't consistent.\r\n");
            return kStatus_Fail;
        }
        PRINTF("The read/write content is consistent.\r\n");

        PRINTF("Write/read multiple data blocks......\r\n");
        if (kStatus_Success != SD_WriteBlocks(card, g_dataWrite, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Write multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }

        memset(g_dataRead, 0U, sizeof(g_dataRead));

        if (kStatus_Success != SD_ReadBlocks(card, g_dataRead, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Read multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }

        PRINTF("Compare the read/write content......\r\n");
        if (memcmp(g_dataRead, g_dataWrite, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
        {
            PRINTF("The read/write content isn't consistent.\r\n");
            return kStatus_Fail;
        }
        PRINTF("The read/write content is consistent.\r\n");

        PRINTF("Erase multiple data blocks......\r\n");
        if (kStatus_Success != SD_EraseBlocks(card, DATA_BLOCK_START, DATA_BLOCK_COUNT))
        {
            PRINTF("Erase multiple data blocks failed.\r\n");
            return kStatus_Fail;
        }
    }

    return kStatus_Success;
}

/*!
 * @brief Main function
 */
int main(void)
{
    sd_card_t *card = &g_sd;
		sd_card_t *card1 = &g_sd1;
    char ch         = '0';
    bool isReadOnly, isReadOnly1;
	

    CLOCK_EnableClock(kCLOCK_InputMux);
    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);

    /* attach main clock to SDIF */
    CLOCK_AttachClk(BOARD_SDIF_CLK_ATTACH);

    BOARD_InitPins();
    BOARD_BootClockPLL150M();
    BOARD_InitDebugConsole();
	
//		BOARD_SDIF_PW_GPIO_INIT();

    /* need call this function to clear the halt bit in clock divider register */
    CLOCK_SetClkDiv(kCLOCK_DivSdioClk, (uint32_t)(SystemCoreClock / FSL_FEATURE_SDIF_MAX_SOURCE_CLOCK + 1U), true);

    card->host.base           = SD_HOST_BASEADDR;
    card->host.sourceClock_Hz = SD_HOST_CLK_FREQ;
    /* card detect type */
    card->usrParam.cd = &s_sdCardDetect;
#if defined DEMO_SDCARD_POWER_CTRL_FUNCTION_EXIST
    card->usrParam.pwr = &s_sdCardPwrCtrl;
		card1->usrParam.pwr = &s_sdCard1PwrCtrl;
#endif
#if defined DEMO_SDCARD_SWITCH_VOLTAGE_FUNCTION_EXIST
    card->usrParam.cardVoltage = &s_sdCardVoltageSwitch;
#endif
		//2nd sd card info
		card1->host.base           = SD_HOST_BASEADDR;
    card1->host.sourceClock_Hz = SD_HOST_CLK_FREQ;
    /* 2nd card detect type */
    card1->usrParam.cd = &s_sdCard1Detect;
		
		PRINTF("\r\n\r\n\r\n");
		PRINTF("/*********************************************************************\r\n");
    PRINTF("           < Dual SD Cards Example (block polling mode) >\r\n");
		PRINTF("*********************************************************************/\r\n");

		/* SD interface reset */
		SD_Reset(card->host.base);
    /* SD host init function */
    if (SD_HostInit(card) != kStatus_Success)
    {
        PRINTF("\r\nSD host init fail\r\n");
        return -1;
    }
		
		if (SD1_HostInit(card1) != kStatus_Success)
    {
        PRINTF("\r\nSD 1 host init fail\r\n");
        return -1;
    }
		
		
		PRINTF("\r\nPlease insert 2 cards individually...\r\n");
		
		PRINTF("waiting card 0...");			
		/* detect card 0 until detected */
		if (SD_WaitCardDetectStatus(SD_HOST_BASEADDR, &s_sdCardDetect, true) == kStatus_Success)
		{
				PRINTF(" inserted!\r\n");

				/* power on the card 0*/
				SD_PowerOnCard0(card->host.base, card->usrParam.pwr);
			
				/* Init card 0 */
				if (SD_CardInit(card))
				{
						PRINTF("\r\nSD card 0 init failed.\r\n");
						return -1;
				}
										
				/* card information log */
				CardInformationLog(card);

				/* Check if card is readonly. */
				isReadOnly = SD_CheckReadOnly(card);
		}
		else
		{
				PRINTF(" detect fail!\r\n");
				return -1;
		}
		
		PRINTF("waiting card 1...");			
		/* detect card 1 until detected */
		if (SD_WaitCard1DetectStatus(SD_HOST_BASEADDR, &s_sdCard1Detect, true) == kStatus_Success)
		{
				PRINTF(" inserted!\r\n");
			
				/* power on the card 1*/
				SD_PowerOnCard1(card1->host.base, card1->usrParam.pwr);	

				/* Init card 1 */
				if (SD_Card1Init(card1))
				{
						PRINTF("\r\nSD card 1 init failed.\r\n");
						return -1;
				}
				/* card information log */
				CardInformationLog(card1);

				/* Check if card is readonly. */
				isReadOnly1 = SD_CheckReadOnly(card1);
		}
		else
		{
				PRINTF(" detect fail!\r\n");
				return -1;
		}
		
		
		PRINTF("\r\nIndividually Read/Write/Erase both cards continuously until error occurs......\r\n");

		while (ch != 'q')
		{
				PRINTF("\r\nCard 0 access starting firstly with block mode...\r\n");
				if (kStatus_Success != AccessCard(card, isReadOnly))
				{
						/* access card fail, due to card 0 remove. */
						if (SD_WaitCardDetectStatus(SD_HOST_BASEADDR, &s_sdCardDetect, false) == kStatus_Success)
						{
								PRINTF("\r\nCard 0 removed\r\n");
								PRINTF(
										"\r\nInput 'q' to quit read/write/erase process.\
						\r\nInput other char to wait card re-insert.\r\n");
								ch = GETCHAR();
								PUTCHAR(ch);
						}
						/* access card fail, due to transfer error */
						else
						{
								PRINTF("\r\nCard 0 transfer failed!\r\n");
						}
				}
				else
				{
						PRINTF("\r\nCard 0 access OK!\r\n");
				}
				
				PRINTF("\r\nCard 1 access starting with block mode...\r\n");
				if (kStatus_Success != AccessCard(card1, isReadOnly1))
				{
						/* access card fail, due to card 1 remove. */
						if (SD_WaitCard1DetectStatus(SD_HOST_BASEADDR, &s_sdCard1Detect, false) == kStatus_Success)
						{
								PRINTF("\r\nCard 1 removed\r\n");
								PRINTF(
										"\r\nInput 'q' to quit read/write/erase process.\
						\r\nInput other char to wait card re-insert.\r\n");
								ch = GETCHAR();
								PUTCHAR(ch);
						}
						/* access card fail, due to transfer error */
						else
						{
								PRINTF("\r\nCard 1 transfer failed!\r\n");
								ch = 'q';
						}

						break;
				}
				else
				{
						PRINTF("\r\nCard 1 access OK!\r\n");
					
						PRINTF(
								"\r\nInput 'q' to quit read/write/erase process.\
						\r\nInput other char to read/write/erase data blocks again.\r\n");
						ch = GETCHAR();
						PUTCHAR(ch);
						if (ch == 'q')
						{
								break;
						}
				}
		}
    

    PRINTF("\r\nThe example completed! Need to boot board again for the example running...\r\n");
    SD_Deinit(card);
		SD1_Deinit(card1);

    while (true)
    {
    }
}

static void CardInformationLog(sd_card_t *card)
{
		uint32_t card_no;
	
    assert(card);
	  
		card_no = (uint32_t *)(card->usrParam.cd->userData);
		
		PRINTF("\r\n===================Card %d Information Log====================\r\n", card_no);
	
		PRINTF("\r\nCard size = block count: %d * block size: %d bytes", card->blockCount, card->blockSize);
		PRINTF(" (~%dGB)\r\n", card->blockCount/1000*card->blockSize/1000000);
																									
    PRINTF("\r\nWorking condition:\r\n");
    if (card->operationVoltage == kCARD_OperationVoltage330V)
    {
        PRINTF("\r\n  Voltage : 3.3V\r\n");
    }
    else if (card->operationVoltage == kCARD_OperationVoltage180V)
    {
        PRINTF("\r\n  Voltage : 1.8V\r\n");
    }

    if (card->currentTiming == kSD_TimingSDR12DefaultMode)
    {
        if (card->operationVoltage == kCARD_OperationVoltage330V)
        {
            PRINTF("\r\n  Timing mode: Default mode\r\n");
        }
        else if (card->operationVoltage == kCARD_OperationVoltage180V)
        {
            PRINTF("\r\n  Timing mode: SDR12 mode\r\n");
        }
    }
    else if (card->currentTiming == kSD_TimingSDR25HighSpeedMode)
    {
        if (card->operationVoltage == kCARD_OperationVoltage180V)
        {
            PRINTF("\r\n  Timing mode: SDR25\r\n");
        }
        else
        {
            PRINTF("\r\n  Timing mode: High Speed\r\n");
        }
    }
    else if (card->currentTiming == kSD_TimingSDR50Mode)
    {
        PRINTF("\r\n  Timing mode: SDR50\r\n");
    }
    else if (card->currentTiming == kSD_TimingSDR104Mode)
    {
        PRINTF("\r\n  Timing mode: SDR104\r\n");
    }
    else if (card->currentTiming == kSD_TimingDDR50Mode)
    {
        PRINTF("\r\n  Timing mode: DDR50\r\n");
    }

    PRINTF("\r\n  Freq : %d HZ\r\n", card->busClock_Hz);
		
		PRINTF("\r\n==============================================================\r\n");
}
