/*
 * Copyright (c) 2015, Freescale Semiconductor, Inc.
 * Copyright 2016-2018 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */
#include "fsl_sd1.h"

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*******************************************************************************
 * Code
 ******************************************************************************/
status_t SD1_SelectBusTiming(sd_card_t *card)
{	
    assert(card);
	  
		
    status_t error = kStatus_SDMMC_SwitchBusTimingFailed;

    if (card->operationVoltage != kCARD_OperationVoltage180V)
    {
        /* Switch the card to high speed mode */
        if (card->host.capability.flags & kSDMMCHOST_SupportHighSpeed)
        {
            /* group 1, function 1 ->high speed mode*/
            error = SD_SelectFunction(card, kSD_GroupTimingMode, kSD_FunctionSDR25HighSpeed);
            /* If the result isn't "switching to high speed mode(50MHZ) successfully or card doesn't support high speed
             * mode". Return failed status. */
            if (error == kStatus_Success)
            {
                card->currentTiming = kSD_TimingSDR25HighSpeedMode;
							  card->busClock_Hz = SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
            }
            else if (error == kStatus_SDMMC_NotSupportYet)
            {
                /* if not support high speed, keep the card work at default mode */
                SDMMC_LOG("\r\nNote: High speed mode is not supported by card");
                return kStatus_Success;
            }
        }
        else
        {
            /* if not support high speed, keep the card work at default mode */
            return kStatus_Success;
        }
    }
    /* card is in UHS_I mode */
    else if ((kSDMMCHOST_SupportSDR104 != SDMMCHOST_NOT_SUPPORT) ||
             (kSDMMCHOST_SupportSDR50 != SDMMCHOST_NOT_SUPPORT) || (kSDMMCHOST_SupportDDR50 != SDMMCHOST_NOT_SUPPORT))
    {
        switch (card->currentTiming)
        {
            /* if not select timing mode, sdmmc will handle it automatically*/
            case kSD_TimingSDR12DefaultMode:
            case kSD_TimingSDR104Mode:
                error = SD_SelectFunction(card, kSD_GroupTimingMode, kSD_FunctionSDR104);
                if (error == kStatus_Success)
                {
                    card->currentTiming = kSD_TimingSDR104Mode;
										card->busClock_Hz = SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, 
																																		SDMMCHOST_SUPPORT_SDR104_FREQ);

										break;
                }
                SDMMC_LOG("\r\nNote: SDR104 mode is not supported by card");

            case kSD_TimingDDR50Mode:
                error = SD_SelectFunction(card, kSD_GroupTimingMode, kSD_FunctionDDR50);
                if (error == kStatus_Success)
                {
                    card->currentTiming = kSD_TimingDDR50Mode;
										card->busClock_Hz = SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, 
																																		SD_CLOCK_50MHZ);

										SDMMCHOST_ENABLE_DDR_MODE(card->host.base, true, 0U);
                    break;
                }
                SDMMC_LOG("\r\nNote: DDR50 mode is not supported by card");

            case kSD_TimingSDR50Mode:
                error = SD_SelectFunction(card, kSD_GroupTimingMode, kSD_FunctionSDR50);
                if (error == kStatus_Success)
                {
                    card->currentTiming = kSD_TimingSDR50Mode;
										card->busClock_Hz = 
													SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_100MHZ);

                    break;
                }
                SDMMC_LOG("\r\nNote: SDR50 mode is not supported by card");

            case kSD_TimingSDR25HighSpeedMode:
                error = SD_SelectFunction(card, kSD_GroupTimingMode, kSD_FunctionSDR25HighSpeed);
                if (error == kStatus_Success)
                {
                    card->currentTiming = kSD_TimingSDR25HighSpeedMode;
										card->busClock_Hz =
												SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);

                }
								break;

            default:
                SDMMC_LOG("\r\nWarning: unknown timing mode");
                break;
        }
    }
    else
    {
    }

    if (error == kStatus_Success)
    {
        /* SDR50 and SDR104 mode need tuning */
        if ((card->currentTiming == kSD_TimingSDR50Mode) || (card->currentTiming == kSD_TimingSDR104Mode))
        {
            /* config IO strength in IOMUX*/
            if (card->currentTiming == kSD_TimingSDR50Mode)
            {
                SDMMCHOST_CONFIG_SD_IO(CARD_BUS_FREQ_100MHZ1, CARD_BUS_STRENGTH_7);
            }
            else
            {
                SDMMCHOST_CONFIG_SD_IO(CARD_BUS_FREQ_200MHZ, CARD_BUS_STRENGTH_7);
            }
            /* execute tuning */
            if (SD_ExecuteTuning(card) != kStatus_Success)
            {
                SDMMC_LOG("\r\nError: tuning failed for mode %d", card->currentTiming);
                return kStatus_SDMMC_TuningFail;
            }
        }
        else
        {
            /* set default IO strength to 4 to cover card adapter driver strength difference */
            SDMMCHOST_CONFIG_SD_IO(CARD_BUS_FREQ_100MHZ1, CARD_BUS_STRENGTH_4);
        }
    }

    return error;
} 
/* set timing to swicth clock/speed for SD0 or SD1*/ 
status_t SD_SwitchBusTiming(sd_card_t *card)
{
		uint32_t card_no;
	
    assert(card);
	  
		card_no = *(uint32_t *)(card->usrParam.cd->userData);

    status_t error = kStatus_SDMMC_SwitchBusTimingFailed;

    if (card->operationVoltage != kCARD_OperationVoltage180V)
    {
        /* Switch the card to high speed mode */
        if (card->host.capability.flags & kSDMMCHOST_SupportHighSpeed)
        {
						if(card_no == kSD_Card_0)
						{
							SDMMCHOST_SET_CARD_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
						}
						else
						{
								SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
						}
        }
        else
        {
            /* if not support high speed, keep the card work at default mode */
            return kStatus_Success;
        }
    }
    /* card is in UHS_I mode */
    else if ((kSDMMCHOST_SupportSDR104 != SDMMCHOST_NOT_SUPPORT) ||
             (kSDMMCHOST_SupportSDR50 != SDMMCHOST_NOT_SUPPORT) || (kSDMMCHOST_SupportDDR50 != SDMMCHOST_NOT_SUPPORT))
    {
        switch (card->currentTiming)
        {
            /* if not select timing mode, sdmmc will handle it automatically*/
            case kSD_TimingSDR12DefaultMode:
            case kSD_TimingSDR104Mode:               
									if(card_no == kSD_Card_0)
									{
											SDMMCHOST_SET_CARD_CLOCK(card->host.base, card->host.sourceClock_Hz, SDMMCHOST_SUPPORT_SDR104_FREQ);
									}
									else
									{
											SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SDMMCHOST_SUPPORT_SDR104_FREQ);
									}
									break;
                SDMMC_LOG("\r\nNote: SDR104 mode is not supported by card");

            case kSD_TimingDDR50Mode:
									if(card_no == kSD_Card_0)
									{
										SDMMCHOST_SET_CARD_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
									}
									else
									{
											SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
									}
                    break;
                SDMMC_LOG("\r\nNote: DDR50 mode is not supported by card");

            case kSD_TimingSDR50Mode:
									if(card_no == kSD_Card_0)
									{
										SDMMCHOST_SET_CARD_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_100MHZ);
									}
									else
									{
											SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_100MHZ);
									}
                    break;
                SDMMC_LOG("\r\nNote: SDR50 mode is not supported by card");

            case kSD_TimingSDR25HighSpeedMode:
									if(card_no == kSD_Card_0)
									{
											SDMMCHOST_SET_CARD_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
									}
									else
									{
											SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_50MHZ);
									}
                break;

            default:
                SDMMC_LOG("\r\nWarning: unknown timing mode");
                break;
        }
    }
    else
    {
    }

    if (error == kStatus_Success)
    {
        /* SDR50 and SDR104 mode need tuning */
        if ((card->currentTiming == kSD_TimingSDR50Mode) || (card->currentTiming == kSD_TimingSDR104Mode))
        {
            /* config IO strength in IOMUX*/
            if (card->currentTiming == kSD_TimingSDR50Mode)
            {
                SDMMCHOST_CONFIG_SD_IO(CARD_BUS_FREQ_100MHZ1, CARD_BUS_STRENGTH_7);
            }
            else
            {
                SDMMCHOST_CONFIG_SD_IO(CARD_BUS_FREQ_200MHZ, CARD_BUS_STRENGTH_7);
            }
            /* execute tuning */
            if (SD_ExecuteTuning(card) != kStatus_Success)
            {
                SDMMC_LOG("\r\nError: tuning failed for mode %d", card->currentTiming);
                return kStatus_SDMMC_TuningFail;
            }
        }
        else
        {
            /* set default IO strength to 4 to cover card adapter driver strength difference */
            SDMMCHOST_CONFIG_SD_IO(CARD_BUS_FREQ_100MHZ1, CARD_BUS_STRENGTH_4);
        }
    }

    return error;
}
 
status_t SD_Card1Init(sd_card_t *card)
{
    assert(card);
    assert(card->isHostReady == true);

    /* reset variables */
    card->flags = 0U;
    /* set DATA bus width */
    SDMMCHOST_SET_CARD1_BUS_WIDTH(card->host.base, kSDMMCHOST_DATABUSWIDTH1BIT);
    /*set card freq to 400KHZ*/
    card->busClock_Hz = SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SDMMC_CLOCK_400KHZ);
	/* send card active */
    SDMMCHOST_SEND_CARD1_ACTIVE(card->host.base, 100U);
    /* Get host capability. */
    GET_SDMMCHOST_CAPABILITY(card->host.base, &(card->host.capability));

    /* probe bus voltage*/
    if (SD_ProbeBusVoltage(card) == kStatus_SDMMC_SwitchVoltageFail)
    {
        return kStatus_SDMMC_SwitchVoltageFail;
    }

    /* Initialize card if the card is SD card. */
    if (kStatus_Success != SD_AllSendCid(card))
    {
        return kStatus_SDMMC_AllSendCidFailed;
    }
    if (kStatus_Success != SD_SendRca(card))
    {
        return kStatus_SDMMC_SendRelativeAddressFailed;
    }
    if (kStatus_Success != SD_SendCsd(card))
    {
        return kStatus_SDMMC_SendCsdFailed;
    }
    if (kStatus_Success != SD_SelectCard(card, true))
    {
        return kStatus_SDMMC_SelectCardFailed;
    }

    /* Set to max frequency in non-high speed mode. */
    card->busClock_Hz = SDMMCHOST_SET_CARD1_CLOCK(card->host.base, card->host.sourceClock_Hz, SD_CLOCK_25MHZ);

    if (kStatus_Success != SD_SendScr(card))
    {
        return kStatus_SDMMC_SendScrFailed;
    }
    /* Set to 4-bit data bus mode. */
    if (((card->host.capability.flags) & kSDMMCHOST_Support4BitBusWidth) && (card->flags & kSD_Support4BitWidthFlag))
    {
        if (kStatus_Success != SD_SetDataBusWidth(card, kSD_DataBusWidth4Bit))
        {
            return kStatus_SDMMC_SetDataBusWidthFailed;
        }
        SDMMCHOST_SET_CARD1_BUS_WIDTH(card->host.base, kSDMMCHOST_DATABUSWIDTH4BIT);
    }

    /* set block size */
    if (SD_SetBlockSize(card, FSL_SDMMC_DEFAULT_BLOCK_SIZE))
    {
        return kStatus_SDMMC_SetCardBlockSizeFailed;
    }

    /* select bus timing */
    if (kStatus_Success != SD1_SelectBusTiming(card))
    {
        return kStatus_SDMMC_SwitchBusTimingFailed;
    }

    /* try to get card current status */
    SD_ReadStatus(card);

    return kStatus_Success;
}


//void SD_CardDeinit(sd_card_t *card)
//{
//    assert(card);

//    SD_SelectCard(card, false);
//}

status_t SD1_HostInit(sd_card_t *card)
{
    assert(card);

    if ((!card->isHostReady) && SDMMC1HOST_Init(&(card->host), (void *)(&(card->usrParam))) != kStatus_Success)
    {
        return kStatus_Fail;
    }

    /* set the host status flag, after the card re-plug in, don't need init host again */
    card->isHostReady = true;

    return kStatus_Success;
}

void SD1_HostDeinit(sd_card_t *card)
{
    assert(card);

    SDMMC1HOST_Deinit(&(card->host));
    /* should re-init host */
    card->isHostReady = false;
}

void SD1_HostReset(SDMMCHOST_CONFIG *host)
{
    SDMMC1HOST_Reset(host->base);
}
void SD_Reset(SDMMCHOST_TYPE *base)
{
		/* Bug Fixed:
     * when SDIF host source clock is attached to 150M or 180M clock, the DATA_BUSY bit in STATUS register
     * maybe set, it will cause SDIF CIU not work, need to perform controller/fifo/dma reset after clock attached,
     * then DATA_BUSY bit will be cleared.
     */
    SDIF_Reset(base, kSDIF_ResetAll, 100U);
}
void SD_PowerOnCard0(SDMMCHOST_TYPE *base, const sdmmchost_pwr_card_t *pwr)
{
    SDMMCHOST_PowerOnCard0(base, pwr);
}
void SD_PowerOnCard1(SDMMCHOST_TYPE *base, const sdmmchost_pwr_card_t *pwr)
{
    SDMMCHOST_PowerOnCard1(base, pwr);
}

void SD_PowerOffCard1(SDMMCHOST_TYPE *base, const sdmmchost_pwr_card_t *pwr)
{
    SDMMCHOST_PowerOffCard1(base, pwr);
}

//void SD_DisableClkCard1(SDMMCHOST_TYPE *base, bool enable)
//{
//		SDMMCHOST_ENABLE_CARD1_CLOCK(base, enable);
//}

status_t SD_WaitCard1DetectStatus(SDMMCHOST_TYPE *hostBase, const sdmmchost_detect_card_t *cd, bool waitCardStatus)
{
    return SDMMCHOST_WaitCard1DetectStatus(hostBase, cd, waitCardStatus);
}

bool SD_IsCard1Present(sd_card_t *card)
{
    return SDMMCHOST_IsCard1Present();
}

status_t SD1_Init(sd_card_t *card)
{
    assert(card);

    if (!card->isHostReady)
    {
        if (SD1_HostInit(card) != kStatus_Success)
        {
            return kStatus_SDMMC_HostNotReady;
        }
    }
    else
    {
        SD1_HostReset(&(card->host));
    }
    SD_PowerOffCard1(card->host.base, card->usrParam.pwr);

    if (SD_WaitCard1DetectStatus(card->host.base, card->usrParam.cd, true) != kStatus_Success)
    {
        return kStatus_SDMMC_CardDetectFailed;
    }
    SD_PowerOnCard1(card->host.base, card->usrParam.pwr);

    return SD_Card1Init(card);
}

void SD1_Deinit(sd_card_t *card)
{
    /* card deinitialize */
    SD_CardDeinit(card);
    /* host deinitialize */
    SD1_HostDeinit(card);
}
