/*
 * Copyright 2014-2016 Freescale Semiconductor, Inc.
 * Copyright 2016-2020 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 */

/* Compiler includes. */
#if defined(__ICCARM__)
#include <intrinsics.h>
#endif

/* Scheduler includes. */
#include "FreeRTOS.h"
#include "task.h"

#if configUSE_TICKLESS_IDLE == 2
#include "fsl_rtc.h"
#else
#include "fsl_device_registers.h"
#endif

#include "fsl_tickless_rtc.h"
#include "fsl_power.h"

extern uint32_t SystemCoreClock; /* in Kinetis SDK, this contains the system core clock speed */

/*
 * LPT timer base address and interrupt number
 */
#if configUSE_TICKLESS_IDLE == 2
extern RTC_Type *vPortGetRtcBase(void);
extern IRQn_Type vPortGetRtcIrqn(void);
#endif /* configUSE_TICKLESS_IDLE */

/*
 * The number of SysTick increments that make up one tick period.
 */
#if configUSE_TICKLESS_IDLE == 2
static uint32_t ulTimerCountsForOneTick = 0;
static uint32_t ulStoppedTimerCompensation = 0;
#endif /* configUSE_TICKLESS_IDLE */

/*
 * The maximum number of tick periods that can be suppressed is limited by the
 * 24 bit resolution of the SysTick timer.
 */
#if configUSE_TICKLESS_IDLE == 2
static uint32_t xMaximumPossibleSuppressedSysTicks = 0;
static uint32_t xMaximumPossibleSuppressedTicks = 0;
static uint32_t xExpectedIdleTimeForRTC = 0;
static uint32_t xDeepSleepCompensation = 0;
#endif /* configUSE_TICKLESS_IDLE */

#if configUSE_TICKLESS_IDLE == 2
#define APP_DEEPSLEEP_RUNCFG0 0x00000000U  /*!< Power down all unnecessary blocks during deep sleep*/
#define APP_DEEPSLEEP_RAM_APD 0x3FFFFFFFU
#define APP_DEEPSLEEP_RAM_PPD 0x3FFFFFFFU
#define APP_EXCLUDE_FROM_DEEPSLEEP                                                                          \
    (((const uint32_t[]){APP_DEEPSLEEP_RUNCFG0,                                                             \
						 (SYSCTL0_PDSLEEPCFG1_FLEXSPI_SRAM_APD_MASK | SYSCTL0_PDSLEEPCFG1_FLEXSPI_SRAM_PPD_MASK), \
                         APP_DEEPSLEEP_RAM_APD, APP_DEEPSLEEP_RAM_PPD}))

#define APP_EXCLUDE_FROM_DEEP_POWERDOWN (((const uint32_t[]){0, 0, 0, 0}))
#define APP_EXCLUDE_FROM_FULL_DEEP_POWERDOWN (((const uint32_t[]){0, 0, 0, 0}))

void vPortRtcIsr(void)
{
}

void vPortSuppressTicksAndSleep(TickType_t xExpectedIdleTime)
{
    eSleepModeStatus eSleepStatus;
    RTC_Type *pxRtcBase;
    uint32_t ulSystickLoadvalue;
    volatile bool bTicklessRTC = false;
    uint32_t ulReloadValue = 0, ulCompleteTickPeriods = 0;
    uint32_t ulRemain;
    uint16_t uRTCsec1 = 0, uRTCsubsec1 = 0;

    pxRtcBase = vPortGetRtcBase();
    if (pxRtcBase == 0)
        return;

    eSleepStatus = eTaskConfirmSleepModeStatus();
    if (eSleepStatus == eAbortSleep)
        return;

    if (xExpectedIdleTime == 0)
        return;

    /* Stop the RTC and systick momentarily.  The time the RTC and systick is stopped for
    is accounted for as best it can be, but using the tickless mode will
    inevitably result in some tiny drift of the time maintained by the
    kernel with respect to calendar time. */
    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    uRTCsec1 = RTC->COUNT;
    uRTCsubsec1 = RTC->SUBSEC;

    /* Enter a critical section but don't use the taskENTER_CRITICAL()
    method as that will mask interrupts that should exit sleep mode. */
    __asm volatile( "cpsid i" ::: "memory" );
    __asm volatile( "dsb" );
    __asm volatile( "isb" );

    /* Calculate the reload value required to wait xExpectedIdleTime
    tick periods.  -1 is used because this code will execute part way
    through one of the tick periods. */
    ulTimerCountsForOneTick = SysTick->LOAD + 1;
    ulSystickLoadvalue = SysTick->VAL;

    /* If a context switch is pending or a task is waiting for the scheduler
    to be unsuspended then abandon the low power entry. */
    if (eSleepStatus == eNoTasksWaitingTimeout)
    {
        POWER_EnterDeepPowerDown(APP_EXCLUDE_FROM_FULL_DEEP_POWERDOWN);
    }
    else
    {
        if ( xExpectedIdleTime >= xExpectedIdleTimeForRTC )
        {
            uint32_t ulRTCCompleteTickPeriods, ulRTCWakePeriods, ulTemp;
            uint16_t uRTCsubsec2, uRTCsec2, secs;

            /*
             * xMaximumPossibleSuppressedTicks is calculated in vPortTimerUpdate()
             * so that it won't pass 65.535 Secs
             */
            if (xExpectedIdleTime > xMaximumPossibleSuppressedTicks)
                xExpectedIdleTime = xMaximumPossibleSuppressedTicks;

            /*
             * deep sleep
             * up to 1ms for OSC & PLL startup time
             * ulRTCWakePeriods needs to be coverted in mS for RTC WAKE
             */
            ulRTCWakePeriods = ( ( xExpectedIdleTime * configTICK_RATE_HZ ) / RTC_WAKE_COUNT_IN_MILLISEC ) - 1UL;

            /* determines whether it is using RTC or sysTick timer */
            bTicklessRTC = true;

            /*
             * RTC & subsec must be enabled 1 sec prior to be used here,
             * subsec may take up to 1 sec before the counter started.
             */
            RTC_SetWakeupCount( RTC, ulRTCWakePeriods );
            POWER_EnterDeepSleep( APP_EXCLUDE_FROM_DEEPSLEEP );
            uRTCsubsec2 = RTC->SUBSEC;
            uRTCsec2 = RTC->COUNT;

            secs = uRTCsec2 - uRTCsec1;
            if (secs)
            {
                ulRTCCompleteTickPeriods = ( uRTCsubsec2 + (32768U - uRTCsubsec1) );
                if ( ulRTCCompleteTickPeriods > 32768U )
                    ulRTCCompleteTickPeriods -= 32768U;
                else
                    secs -= 1UL;
            }
            else
                ulRTCCompleteTickPeriods = ( uRTCsubsec2 - uRTCsubsec1 );

           /*
            * value of 1 subsec is 30.51757 uS, the closer value to
            * 1,000,000,000 nS (1 S ) is 61,035 (2 x 30,517)
            * - 32,768 * 30,517 = 999,981,056 nS
            * - 32,768 * 61,035 = 1,999,994,880 div 2 = 999,997,440 nS
            */

            /* Convert RTC count to uS */
            ulRTCCompleteTickPeriods = ( ( ulRTCCompleteTickPeriods * 61035U ) >> 1 ) / 1000UL;
            /* Convert sec to tick */
            ulCompleteTickPeriods = ( secs * 1000000UL ) / configTICK_RATE_HZ;
            /* rounding up RTC uS for tick */
            ulTemp = ulRTCCompleteTickPeriods + 999U - (ulRTCCompleteTickPeriods - 1) % 1000UL;
            /* convert RTC uS to tick and add it up */
            ulCompleteTickPeriods += ( ulTemp / configTICK_RATE_HZ );

            /* difference, ulRemain will have uS value */
            ulRemain = ( ulTemp - ulRTCCompleteTickPeriods );
            /* convert uS to sysTick value */
            ulRemain = ( ( ulRemain * ulTimerCountsForOneTick ) / configTICK_RATE_HZ );
            ulRemain += ( ( xExpectedIdleTime - ulCompleteTickPeriods ) * ulTimerCountsForOneTick);

            /* some adjust in uS if needed */
            ulRemain += xDeepSleepCompensation;

            /* remaining time that needs to be spent in systick */
            SysTick->VAL = 0;
            SysTick->LOAD = ulRemain - 1;
            SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

            __asm volatile( "dsb" ::: "memory" );
            SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
            __asm volatile( "isb" );
            __asm volatile( "wfi" );
        }
        else
        {
            if (xExpectedIdleTime > xMaximumPossibleSuppressedSysTicks)
                xExpectedIdleTime = xMaximumPossibleSuppressedSysTicks;

            ulReloadValue = ( ulTimerCountsForOneTick * ( xExpectedIdleTime - 1UL ) );
            ulReloadValue += ulSystickLoadvalue;

            if( ulReloadValue > ulStoppedTimerCompensation )
                ulReloadValue -= ulStoppedTimerCompensation;

            /* WFI only */
            SysTick->LOAD = ulReloadValue;
            SysTick->VAL = 0UL;
            SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

            ulCompleteTickPeriods = xExpectedIdleTime - 1UL;

            __asm volatile( "dsb" ::: "memory" );
            SCB->SCR &= ~SCB_SCR_SLEEPDEEP_Msk;
            __asm volatile( "isb" );
            __asm volatile( "wfi" );
        }

        __asm volatile( "cpsie i" ::: "memory" );
        __asm volatile( "dsb" );
        __asm volatile( "isb" );

        /* Disable interrupts again because the clock is about to be stopped
        and interrupts that execute while the clock is stopped will increase
        any slippage between the time maintained by the RTOS and calendar
        time. */
        __asm volatile( "cpsid i" ::: "memory" );
        __asm volatile( "dsb" );
        __asm volatile( "isb" );

        /*
         * separate RTC and SysTick interrrupt. The calculation should be based
         * on selected timer (RTC or SysTick) not both.
         */
        if( bTicklessRTC )
        {
            /* Disable the SysTick clock without reading the
            portNVIC_SYSTICK_CTRL_REG register to ensure the
            SysTick_CTRL_COUNTFLAG_Msk is not cleared if it is set.  Again,
            the time the SysTick is stopped for is accounted for as best it can
            be, but using the tickless mode will inevitably result in some tiny
            drift of the time maintained by the kernel with respect to calendar
            time*/
            SysTick->CTRL = ( SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_TICKINT_Msk );

            bTicklessRTC = false;
        }
        else
        {
            uint32_t ulCompletedSysTickDecrements;

            /* Disable the SysTick clock without reading the
            SysTick->CTRL register to ensure the
            SysTick_CTRL_COUNTFLAG_Msk is not cleared if it is set.  Again,
            the time the SysTick is stopped for is accounted for as best it can
            be, but using the tickless mode will inevitably result in some tiny
            drift of the time maintained by the kernel with respect to calendar
            time*/
            SysTick->CTRL = ( SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_TICKINT_Msk );

            /* Determine if the SysTick clock has already counted to zero and
            been set back to the current reload value (the reload back being
            correct for the entire expected idle time) or if the SysTick is yet
            to count to zero (in which case an interrupt other than the SysTick
            must have brought the system out of sleep mode). */
            if( ( SysTick->CTRL & SysTick_CTRL_COUNTFLAG_Msk ) != 0 )
            {
                uint32_t ulCalculatedLoadValue;

                /* The tick interrupt is already pending, and the SysTick count
                reloaded with ulReloadValue.  Reset the
                SysTick->LOAD with whatever remains of this tick
                period. */
                ulCalculatedLoadValue = ( ulTimerCountsForOneTick - 1UL ) -
                                ( ulReloadValue - SysTick->VAL );

                /* Don't allow a tiny value, or values that have somehow
                underflowed because the post sleep hook did something
                that took too long. */
                if( ( ulCalculatedLoadValue < ulStoppedTimerCompensation ) ||
                    ( ulCalculatedLoadValue > ulTimerCountsForOneTick ) )
                {
                    ulCalculatedLoadValue = ( ulTimerCountsForOneTick - 1UL );
                }

                SysTick->LOAD = ulCalculatedLoadValue;

                /* As the pending tick will be processed as soon as this
                function exits, the tick value maintained by the tick is stepped
                forward by one less than the time spent waiting. */
                ulCompleteTickPeriods = xExpectedIdleTime - 1UL;
            }
            else
            {
                /* Something other than the tick interrupt ended the sleep.
                Work out how long the sleep lasted rounded to complete tick
                periods (not the ulReload value which accounted for part
                ticks). */
                ulCompletedSysTickDecrements = ( xExpectedIdleTime *
                        ulTimerCountsForOneTick ) - SysTick->VAL;

                /* How many complete tick periods passed while the processor
                was waiting? */
                ulCompleteTickPeriods = (ulCompletedSysTickDecrements / ulTimerCountsForOneTick);

                /* The reload value is set to whatever fraction of a single tick
                period remains. */
                SysTick->LOAD = ( ( ulCompleteTickPeriods + 1UL ) *
                        ulTimerCountsForOneTick ) - ulCompletedSysTickDecrements;
            }
        }
    }

    /* Restart SysTick so it runs from SysTick->LOAD
    again, then set SysTick->LOAD back to its standard
    value. */
    SysTick->VAL = 0;
    vTaskStepTick( ulCompleteTickPeriods );
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
    SysTick->LOAD = ulTimerCountsForOneTick - 1UL;

    /* Exit with interrpts enabled. */
    __asm volatile( "cpsie i" ::: "memory" );
}

/* Setup the variables */
void vPortSetupTimerInterrupt(void)
{
    /* RTC wake count is in 1 mS increment, converting FreeRTOS tick to 1 mS */
    xMaximumPossibleSuppressedTicks = ( portMAX_16_BIT_NUMBER * RTC_WAKE_COUNT_IN_MILLISEC ) / configTICK_RATE_HZ;
    /* maximum systick ticks allowed */
    xMaximumPossibleSuppressedSysTicks =( portMAX_24_BIT_NUMBER / ( ( configCPU_CLOCK_HZ / configTICK_RATE_HZ ) - 1UL ) ) - 1UL;

    xExpectedIdleTimeForRTC = ( 8UL * RTC_WAKE_COUNT_IN_MILLISEC ) / configTICK_RATE_HZ;
    xDeepSleepCompensation = 0;

    NVIC_EnableIRQ(vPortGetRtcIrqn());
}
#endif /* configUSE_TICKLESS_IDLE */
